##
# Procedural model of a Colomban MC-15 Cri-Cri electrical system.
# Reference: Engine Manual Chapter 9


# Consumers:
#

##
# Initialize Properties
#

var delta_sec		=	props.globals.getNode("/sim/time/delta-sec", 1);

var electrical		=	props.globals.getNode("/systems/electrical", 1);

var serviceable		=	electrical.initNode( "serviceable", 1, "BOOL");

var amps		= 	electrical.getNode( "amps", 1);
var volts		=	electrical.initNode( "volts", 0.0, "DOUBLE");

var switches		=	props.globals.getNode("/controls/switches", 1);
var master_switch		=	switches.getNode("master",	1);	# ref. HB p. 9
var ignition_sw		=	[ switches.initNode("ignition[0]", 0, "BOOL"), switches.initNode("ignition[1]", 0, "BOOL") ];
	

var outputs		=	electrical.getNode("outputs", 1);

var ign_out		=	[ outputs.initNode("ignition[0]", 0.0, "DOUBLE"), outputs.initNode("ignition[1]", 0.0, "DOUBLE") ];

var starter_volts	=	outputs.initNode("starter", 0.0, "DOUBLE");

var breakers		=	props.globals.getNode("/controls/circuit-breakers", 1);
var cb = {
	batt	:	breakers.initNode("batt",	1,	"BOOL"),	# 5 A
	gen1	:	breakers.initNode("gen[0]",	1,	"BOOL"),	# ?? A
	gen2	:	breakers.initNode("gen[1]",	1,	"BOOL"),	# ?? A
};
	
##
# Initialize properties used to determine electrical load
#
var com_ptt	= props.globals.getNode("/instrumentation/comm[0]/ptt", 1);
var com_start	= props.globals.getNode("/instrumentation/comm[0]/start", 1);

##
# Battery model class.
#

var BatteryClass = {
	new : func( volt, amps, amp_hours, charge_percent, charge_amps, n){
		m = { 
			parents : [BatteryClass],
			ideal_volts:	volt,
			ideal_amps:	amps,
			volt_p:		electrical.initNode("battery-volts["~n~"]", 0.0, "DOUBLE"),
			amp_hours:	amp_hours,
			charge_percent:	charge_percent, 
			charge_amps:	charge_amps,
		};
		return m;
	},
	apply_load : func( load ) {
		var dt = delta_sec.getDoubleValue();
		var amphrs_used = load * dt / 3600.0;
		var percent_used = amphrs_used / me.amp_hours;
		me.charge_percent -= percent_used;
		if ( me.charge_percent < 0.0 ) {
			me.charge_percent = 0.0;
		} elsif ( me.charge_percent > 1.0 ) {
			me.charge_percent = 1.0;
		}
		var output =me.amp_hours * me.charge_percent;
		return output;
	},
	
	get_output_volts : func {
		var x = 1.0 - me.charge_percent;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		var output =me.ideal_volts * factor;
		me.volt_p.setDoubleValue( output );
		return output;
	},
	
	get_output_amps : func {
		var x = 1.0 - me.charge_percent;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		var output =me.ideal_amps * factor;
		return output;
	},
	
	reset_to_full_charge : func {
		me.charge_percent = 1.0;
	},
};


##
# Alternator model class.
#


var AlternatorClass = {
	new: func ( rpm_source, rpm_threshold, ideal_volts, ideal_amps ) {
		var obj = { 
			parents : [AlternatorClass],
			rpm_source : props.globals.getNode( rpm_source, 1),
			rpm_threshold : rpm_threshold,
			ideal_volts : ideal_volts,
			ideal_amps : ideal_amps,
			amps_p:	props.globals.initNode( "/systems/electrical/alternator-amps", 0.0, "DOUBLE"),
		};
		obj.rpm_source.setDoubleValue( 0.0 );
		return obj;
	},
	apply_load: func( amps ){
		var dt = delta_sec.getDoubleValue();
		
		me.amps_p.setDoubleValue( amps );
		
		# Computes available amps and returns remaining amps after load is applied
		# Scale alternator output for rpms < 800.  For rpms >= 800
		# give full output.  This is just a WAG, and probably not how
		# it really works but I'm keeping things "simple" to start.
		var factor = math.min( me.rpm_source.getDoubleValue() / me.rpm_threshold, 1.0 );
		
		# print( "alternator amps = ", me.ideal_amps * factor );
		var available_amps = me.ideal_amps * factor;
		return available_amps - amps;
	},
	get_output_volts: func {
		# Return output volts based on rpm
		
		# scale alternator output for rpms < 800.  For rpms >= 800
		# give full output.  This is just a WAG, and probably not how
		# it really works but I'm keeping things "simple" to start.
		var factor = math.min( me.rpm_source.getDoubleValue() / me.rpm_threshold, 1.0 );
		
		# print( "alternator volts = ", me.ideal_volts * factor );
		return me.ideal_volts * factor;
	},
	get_output_amps: func {
		# Return output amps available based on rpm.
		
		# scale alternator output for rpms < 800.  For rpms >= 800
		# give full output.  This is just a WAG, and probably not how
		# it really works but I'm keeping things "simple" to start.
		var factor = math.min( me.rpm_source.getDoubleValue() / me.rpm_threshold, 1.0 );
		
		# print( "alternator amps = ", ideal_amps * factor );
		return me.ideal_amps * factor;
	},

};

#		Battery
#
#		voltage: 12 V
#		capacity: 1.8 Ah
#		mass: 0.8 kg
var battery = BatteryClass.new( 12.0, 15, 1.8, 1.0, 7.0, 0);

#		Alternator
#
#		voltage: 	12 V
#		power:	30 W
#		amps:		power / voltage = 2.5 A
var alternator = [
	AlternatorClass.new( "/engines/engine[0]/rpm", 800.0, 12.0, 2.5 ),
	AlternatorClass.new( "/engines/engine[1]/rpm", 800.0, 12.0, 2.5 ),
];

var reset_battery_and_circuit_breakers = func {
	# Charge battery to 100 %
	battery.reset_to_full_charge();
	
	# Reset circuit breakers
	foreach( var b; keys(cb) ) {
		b.setBoolValue(1);
	}
	
	foreach( var b; soaring_bus.consumers ){
		b.cb.setBoolValue( 1 );
	}
	foreach( var b; main_bus.consumers ){
		b.cb.setBoolValue( 1 );
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_virtual_bus(dt);
	}
};

##
# Model the system of relays and connections that join the battery,
# alternator, starter, master/alt switches, external power supply.
#

# Sparkplug: NGK type B9 EV

var update_virtual_bus = func (dt) {
	var load = 0.0;
	var battery_volts = 0.0;
	var alternator_volts = [ 0.0, 0.0 ];
	if ( serviceable.getBoolValue() ) {
		battery_volts = battery.get_output_volts();
		alternator_volts = [
			alternator[0].get_output_volts(),
			alternator[1].get_output_volts(),
		];
	}
	
	# switch state
	var master = master_switch.getBoolValue();
	
	# determine power source
	var bus_volts = 0.0;
	var power_source = nil;
	if ( master and cb.batt.getBoolValue() ) {
		bus_volts = battery_volts;
		power_source = "battery";
	}
	# TODO Alternator 2 (RH Engine)
	if ( master and (alternator_volts[0] > bus_volts) and cb.gen.getBoolValue() ) {
		bus_volts = alternator_volts[0];
		power_source = "alternator0";
	}
	
	# Spark Plugs
	
	# TODO Load
	if( ignition_sw[0].getBoolValue() ){
		ign_out[0].setDoubleValue( bus_volts );
	} else {
		ign_out[0].setDoubleValue( 0.0 );
	}
	
	if( ignition_sw[1].getBoolValue() ){
		ign_out[1].setDoubleValue( bus_volts );
	} else {
		ign_out[1].setDoubleValue( 0.0 );
	}
	
	load += instrument_bus.update( bus_volts );
	
	# system loads and ammeter gauge
	var ammeter = 0.0;
	if ( bus_volts > 1.0 ) {
		# ammeter gauge
		if ( power_source == "battery" ) {
			ammeter = -load;
		} else {
			ammeter = battery.charge_amps;
		}
	}
	# print( "ammeter = ", ammeter );
	
	# charge/discharge the battery
	if ( power_source == "battery" ) {
		battery.apply_load( load, dt );
		if( load > 50 ){
			cb.batt.setBoolValue( 0 );
		}
	} elsif ( power_source == "alternator0" ) {
		battery.apply_load( -alternator[0].apply_load( load, dt ), dt );
		if( load > 50 ){
			cb.gen1.setBoolValue( 0 );
		}
	}
	
	# outputs
	amps.setDoubleValue( ammeter );
	volts.setDoubleValue( bus_volts );
}

var consumer = {
	new: func( name, volt_threshold, consumption, breaker_rating, switch ){
		var obj = {
			parents: [consumer],
			name: name,
			volt_threshold: volt_threshold or 9,	# 9 is used as the standard volt threshold for this implementation
			consumption: consumption,
			breaker_rating: breaker_rating or 999.9,
			switch: switch,
			cb: breakers.initNode( name, 1, "BOOL"),
			output: outputs.initNode( name, 0.0, "DOUBLE"),
		};
		return obj;
	},
	update: func( bv ){
		if( bv > me.volt_threshold and me.cb.getBoolValue() and ( me.switch == nil or me.switch.getBoolValue() ) ){
			me.output.setDoubleValue( bv );
			
			var load = me.consumption / bv;
			# print( me.name ~ " uses " ~ me.consumption ~ "W at "~ bv ~ "V: " ~ load ~ "A" );
			if( load > me.breaker_rating ){
				me.cb.setBoolValue( 0 );
				return 0.0;
			} else {
				return me.consumption / bv;
			}
		} else {
			me.output.setDoubleValue( 0.0 );
			return 0.0;
		}
	},
};

#Load sources:
#	com:		https://www.skyfox.com/becker-ar6201-022-vhf-am-sprechfunkgeraet-8-33.html

var instrument_bus = {
	init: func {
		instrument_bus.consumers[0].update = func( bv ){
			if( me.cb.getBoolValue() ){
				me.output.setDoubleValue( bv );
				if(com_ptt.getBoolValue() and com_start.getValue()==1){
					return 19.2 / bv;
				}else{
					return 1.02 * com_start.getDoubleValue() / bv;
				}
			} else {
				me.output.setDoubleValue( 0.0 );
				return 0.0;
			}
		};
	},
	consumers: [
		consumer.new( "comm", 9, nil, 5, nil ),
	],
	update: func( bv ) {
		
		var bus_volts = bv;
		var load = 0.0;
		
		foreach( var el; me.consumers ){
			load += el.update( bus_volts );
		}
		
		
		return load;
		
	},
};

##
# Initialize the electrical system
#
var system_updater = ElectricalSystemUpdater.new();
instrument_bus.init();

# checking if battery should be automatically recharged
if (!getprop("/systems/electrical/save-battery-charge")) {
	battery.reset_to_full_charge();
};

system_updater.enable();

print("Electrical system initialized");

