/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'dart:convert';

import 'package:clima/core/date_time.dart' as date_time_utils;
import 'package:clima/data/models/full_weather_model.dart';
import 'package:clima/domain/entities/city.dart';
import 'package:clima/domain/entities/daily_forecast.dart';
import 'package:clima/domain/entities/hourly_forecast.dart';
import 'package:clima/domain/entities/unit_system.dart';
import 'package:clima/domain/entities/weather.dart';
import 'package:clima/domain/entities/wind_direction.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:timezone/data/latest.dart';
import 'package:timezone/timezone.dart';

void main() {
  group('FullWeatherModel', () {
    const city =
        City(name: 'London', long: 23, lat: 23, id: 123, language: 'en');
    late final Location location;

    setUpAll(() {
      initializeTimeZones();
      location = getLocation('Europe/London');
    });

    test('fromJson', () {
      final json = jsonDecode(
        // Weather data fetched from Open-Meteo (https://open-meteo.com/)
        // under the CC BY 4.0. See https://open-meteo.com/en/license for
        // data sources.
        '{"latitude":51.5,"longitude":-0.120000124,"generationtime_ms":3.939986228942871,"utc_offset_seconds":3600,"timezone":"Europe/London","timezone_abbreviation":"BST","elevation":29.0,"current_weather":{"temperature":11.9,"windspeed":14.0,"winddirection":64.0,"weathercode":0,"is_day":0,"time":1685134800},"hourly_units":{"time":"unixtime","temperature_2m":"°C","relativehumidity_2m":"%","dewpoint_2m":"°C","apparent_temperature":"°C","precipitation_probability":"%","precipitation":"mm","rain":"mm","showers":"mm","snowfall":"cm","snow_depth":"m","weathercode":"wmo code","pressure_msl":"hPa","surface_pressure":"hPa","cloudcover":"%","cloudcover_low":"%","cloudcover_mid":"%","cloudcover_high":"%","visibility":"m","evapotranspiration":"mm","et0_fao_evapotranspiration":"mm","vapor_pressure_deficit":"kPa","windspeed_10m":"km/h","windspeed_80m":"km/h","windspeed_120m":"km/h","windspeed_180m":"km/h","winddirection_10m":"°","winddirection_80m":"°","winddirection_120m":"°","winddirection_180m":"°","windgusts_10m":"km/h","temperature_80m":"°C","temperature_120m":"°C","temperature_180m":"°C","soil_temperature_0cm":"°C","soil_temperature_6cm":"°C","soil_temperature_18cm":"°C","soil_temperature_54cm":"°C","soil_moisture_0_1cm":"m³/m³","soil_moisture_1_3cm":"m³/m³","soil_moisture_3_9cm":"m³/m³","soil_moisture_9_27cm":"m³/m³","soil_moisture_27_81cm":"m³/m³","uv_index":"","is_day":""},"hourly":{"time":[1685055600,1685059200,1685062800,1685066400,1685070000,1685073600,1685077200,1685080800,1685084400,1685088000,1685091600,1685095200,1685098800,1685102400,1685106000,1685109600,1685113200,1685116800,1685120400,1685124000,1685127600,1685131200,1685134800,1685138400,1685142000,1685145600,1685149200,1685152800,1685156400,1685160000,1685163600,1685167200,1685170800,1685174400,1685178000,1685181600,1685185200,1685188800,1685192400,1685196000,1685199600,1685203200,1685206800,1685210400,1685214000,1685217600,1685221200,1685224800,1685228400,1685232000,1685235600,1685239200,1685242800,1685246400,1685250000,1685253600,1685257200,1685260800,1685264400,1685268000,1685271600,1685275200,1685278800,1685282400,1685286000,1685289600,1685293200,1685296800,1685300400,1685304000,1685307600,1685311200,1685314800,1685318400,1685322000,1685325600,1685329200,1685332800,1685336400,1685340000,1685343600,1685347200,1685350800,1685354400,1685358000,1685361600,1685365200,1685368800,1685372400,1685376000,1685379600,1685383200,1685386800,1685390400,1685394000,1685397600,1685401200,1685404800,1685408400,1685412000,1685415600,1685419200,1685422800,1685426400,1685430000,1685433600,1685437200,1685440800,1685444400,1685448000,1685451600,1685455200,1685458800,1685462400,1685466000,1685469600,1685473200,1685476800,1685480400,1685484000,1685487600,1685491200,1685494800,1685498400,1685502000,1685505600,1685509200,1685512800,1685516400,1685520000,1685523600,1685527200,1685530800,1685534400,1685538000,1685541600,1685545200,1685548800,1685552400,1685556000,1685559600,1685563200,1685566800,1685570400,1685574000,1685577600,1685581200,1685584800,1685588400,1685592000,1685595600,1685599200,1685602800,1685606400,1685610000,1685613600,1685617200,1685620800,1685624400,1685628000,1685631600,1685635200,1685638800,1685642400,1685646000,1685649600,1685653200,1685656800],"temperature_2m":[11.1,10.2,9.4,8.8,8.7,8.5,8.5,9.5,10.5,11.5,13.1,14.5,15.8,17.2,18.0,18.2,18.4,18.0,17.1,16.0,14.6,13.0,11.9,11.0,10.2,9.6,9.2,8.8,8.6,8.2,8.3,9.3,11.0,12.8,14.1,15.2,16.4,17.5,18.5,19.1,19.5,19.5,19.1,17.7,16.3,14.7,13.3,12.5,11.5,10.8,10.2,9.9,9.5,9.1,9.1,10.3,12.3,14.5,16.6,18.4,19.6,20.6,21.1,21.5,21.5,21.0,20.1,18.6,15.4,13.7,12.4,11.2,10.3,9.7,9.2,9.0,8.3,7.9,7.9,8.9,10.6,12.0,13.0,13.9,14.9,15.6,16.4,17.2,17.3,16.7,16.2,15.2,14.0,12.6,11.3,10.4,9.6,9.0,8.5,8.3,8.2,8.2,8.4,9.1,10.4,12.1,13.6,14.7,15.7,16.4,16.8,17.0,16.9,16.5,15.9,15.2,14.3,13.4,12.6,12.0,11.5,11.1,10.8,10.6,10.5,10.4,10.5,11.1,12.9,15.3,17.4,19.0,20.3,21.2,21.3,21.6,21.5,21.2,20.6,19.7,18.4,17.0,15.6,14.3,13.2,12.3,11.6,11.1,10.8,10.6,10.7,11.3,12.8,14.8,16.7,18.3,19.8,21.0,21.7,22.0,22.0,21.6,20.8,19.8,18.5,17.0,15.5,14.3],"relativehumidity_2m":[80,78,82,84,86,86,84,81,73,68,58,53,52,49,39,35,40,41,48,53,55,55,62,70,79,86,90,91,92,93,91,84,76,67,62,56,56,53,46,43,42,44,44,50,53,58,67,65,68,73,75,77,83,87,88,81,67,47,35,30,30,34,38,40,40,42,46,55,66,65,62,69,76,77,81,78,78,80,78,74,65,64,57,58,51,50,47,50,45,47,47,46,49,53,58,65,73,79,83,85,86,86,85,83,78,70,65,63,63,63,62,60,60,61,63,65,66,67,69,72,76,80,83,85,87,88,89,87,82,74,65,54,43,35,34,33,33,35,39,42,44,46,49,55,63,69,74,77,79,80,79,77,72,65,59,55,53,50,46,41,40,44,51,58,64,70,74,76],"dewpoint_2m":[7.8,6.6,6.5,6.3,6.5,6.3,6.0,6.4,5.9,5.8,5.0,5.1,6.0,6.4,3.8,2.5,4.6,4.6,6.0,6.4,5.7,4.2,4.9,5.7,6.7,7.4,7.7,7.5,7.4,7.2,6.9,6.8,6.9,6.9,7.0,6.5,7.6,7.8,6.7,6.3,6.2,6.9,6.6,7.1,6.7,6.6,7.3,6.1,5.8,6.1,6.0,6.1,6.8,7.0,7.2,7.2,6.3,3.3,1.1,0.5,1.6,4.2,6.3,7.4,7.4,7.6,8.2,9.4,9.0,7.2,5.3,5.7,6.3,5.8,6.1,5.3,4.6,4.6,4.3,4.5,4.3,5.4,4.7,5.8,4.8,5.2,5.0,6.7,5.3,5.3,4.8,3.7,3.5,3.2,3.3,4.1,5.0,5.5,5.8,5.9,6.0,6.0,6.0,6.3,6.7,6.8,7.1,7.7,8.6,9.3,9.5,9.1,9.1,9.0,8.9,8.6,8.0,7.4,7.0,7.1,7.4,7.7,8.0,8.1,8.4,8.5,8.7,9.0,9.9,10.7,10.7,9.4,7.3,5.1,4.8,4.6,4.6,5.1,6.1,6.4,5.9,5.3,4.9,5.4,6.3,6.7,7.1,7.2,7.3,7.3,7.2,7.4,7.8,8.3,8.6,9.1,10.0,10.2,9.5,8.2,7.8,8.8,10.3,11.3,11.6,11.5,10.9,10.1],"apparent_temperature":[9.4,7.9,7.1,6.5,6.2,6.0,5.9,6.9,7.3,8.2,9.7,11.9,13.7,15.5,15.8,15.3,15.8,14.7,13.7,12.8,11.3,9.5,8.7,8.1,7.7,7.4,7.3,7.1,6.8,6.4,6.4,7.3,8.7,10.2,11.8,13.1,15.4,16.8,17.7,18.0,17.9,17.3,16.6,15.3,14.0,12.7,11.8,10.6,9.7,9.2,8.8,8.5,8.3,7.9,7.9,8.9,10.5,11.5,12.9,15.5,16.9,18.5,19.2,19.4,18.7,17.8,17.3,15.5,12.5,10.5,9.0,8.1,7.3,6.8,6.3,5.9,5.2,5.0,4.9,5.8,7.2,8.4,9.2,10.4,11.5,12.7,13.6,15.0,14.4,13.2,12.5,11.3,10.2,8.9,7.8,7.1,6.5,6.1,5.7,5.5,5.4,5.5,5.7,6.4,7.6,9.1,10.5,12.0,13.4,14.3,14.7,14.4,14.4,13.9,13.2,12.4,11.4,10.3,9.5,9.1,8.9,8.8,8.7,8.6,8.7,8.6,8.7,9.4,11.2,13.6,15.5,17.2,18.2,18.7,18.8,18.9,18.5,17.8,17.0,16.3,15.2,13.8,12.5,11.5,10.9,10.1,9.6,9.2,9.0,8.8,8.8,9.4,10.8,12.7,14.7,17.1,19.3,20.6,21.1,20.8,20.2,19.5,18.9,18.3,17.3,16.0,14.5,13.1],"precipitation_probability":[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,4,6,9,13,16,15,14,13,9,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,2,3,2,1,0,0,0,0,0],"precipitation":[0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00],"rain":[0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00],"showers":[0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00],"snowfall":[0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00],"snow_depth":[0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00],"weathercode":[0,0,1,2,2,2,1,2,2,2,2,1,2,1,0,0,0,0,1,1,0,0,0,0,1,2,2,2,3,3,3,2,2,2,2,2,2,1,1,0,1,1,0,0,0,0,0,0,0,0,0,0,2,2,3,1,3,3,0,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,2,1,1,1,1,2,1,2,3,3,3,2,2,2,2,1,1,3,0,0,0,1,1,1,1,1,1,2,2,2,2,2,2,3,3,3,3,3,3,2,2,2,2,2,2,3,3,3,3,3,3,2,2,2,1,1,1,0,0,0,0,0,0,0,0,0,1,1,1,2,2,2,0,0,0,0,0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,0,0,0],"pressure_msl":[1031.2,1031.6,1031.4,1031.4,1031.2,1031.1,1031.9,1031.4,1032.0,1032.1,1032.2,1031.9,1032.0,1031.2,1031.3,1031.1,1031.2,1030.5,1030.5,1030.2,1030.8,1031.2,1031.0,1031.0,1031.0,1030.5,1030.2,1029.8,1029.6,1029.3,1029.3,1029.0,1028.9,1028.5,1027.9,1027.6,1026.9,1026.4,1025.4,1024.9,1024.4,1024.2,1023.7,1023.7,1023.7,1023.8,1024.0,1024.1,1024.0,1023.8,1023.4,1023.1,1023.0,1023.1,1023.4,1023.5,1023.8,1023.8,1023.6,1023.5,1023.4,1023.2,1023.0,1022.9,1022.8,1023.0,1023.3,1023.9,1024.8,1025.7,1026.4,1026.8,1027.1,1027.2,1027.3,1027.3,1027.3,1027.4,1027.6,1027.8,1028.0,1028.2,1028.2,1028.4,1028.5,1028.5,1028.3,1028.0,1027.7,1027.7,1027.8,1028.0,1028.4,1028.9,1029.3,1029.6,1029.8,1029.9,1030.0,1030.0,1030.1,1030.3,1030.5,1030.7,1030.8,1030.8,1030.7,1030.5,1030.3,1030.0,1029.7,1029.4,1029.1,1028.8,1028.6,1028.5,1028.7,1029.0,1029.2,1029.3,1029.3,1029.3,1029.2,1029.0,1028.9,1028.9,1029.0,1029.1,1029.1,1029.0,1028.9,1028.7,1028.4,1028.2,1028.0,1027.6,1027.2,1026.8,1026.5,1026.3,1026.3,1026.5,1026.7,1026.9,1027.1,1027.3,1027.3,1027.2,1027.2,1027.2,1027.3,1027.3,1027.3,1027.2,1027.1,1026.8,1026.4,1026.0,1025.7,1025.4,1025.1,1024.8,1024.6,1024.4,1024.4,1024.5,1024.7,1024.9],"surface_pressure":[1027.6,1028.0,1027.8,1027.8,1027.6,1027.5,1028.3,1027.8,1028.4,1028.5,1028.6,1028.4,1028.5,1027.7,1027.8,1027.6,1027.7,1027.0,1027.0,1026.7,1027.3,1027.6,1027.4,1027.4,1027.4,1026.9,1026.6,1026.2,1026.0,1025.7,1025.7,1025.4,1025.3,1024.9,1024.4,1024.1,1023.4,1022.9,1021.9,1021.4,1020.9,1020.7,1020.2,1020.2,1020.2,1020.3,1020.5,1020.6,1020.4,1020.2,1019.8,1019.5,1019.4,1019.5,1019.8,1019.9,1020.3,1020.3,1020.1,1020.0,1019.9,1019.8,1019.6,1019.5,1019.4,1019.6,1019.9,1020.4,1021.3,1022.2,1022.8,1023.2,1023.5,1023.6,1023.7,1023.7,1023.7,1023.8,1024.0,1024.2,1024.4,1024.6,1024.6,1024.9,1025.0,1025.0,1024.8,1024.5,1024.2,1024.2,1024.3,1024.5,1024.9,1025.3,1025.7,1026.0,1026.2,1026.3,1026.4,1026.4,1026.5,1026.7,1026.9,1027.1,1027.2,1027.2,1027.1,1027.0,1026.8,1026.5,1026.2,1025.9,1025.6,1025.3,1025.1,1025.0,1025.2,1025.4,1025.6,1025.7,1025.7,1025.7,1025.6,1025.4,1025.3,1025.3,1025.4,1025.5,1025.5,1025.5,1025.4,1025.2,1024.9,1024.7,1024.5,1024.2,1023.8,1023.4,1023.0,1022.8,1022.8,1023.0,1023.2,1023.4,1023.6,1023.7,1023.7,1023.6,1023.6,1023.6,1023.7,1023.7,1023.7,1023.7,1023.6,1023.3,1022.9,1022.6,1022.3,1022.0,1021.7,1021.4,1021.2,1020.9,1020.9,1021.0,1021.2,1021.4],"cloudcover":[0,0,35,55,82,59,50,83,82,71,43,32,48,44,0,0,3,0,0,0,0,0,0,0,16,54,60,79,90,94,84,86,82,60,63,63,50,40,2,0,0,0,0,0,0,0,0,0,1,0,0,0,82,78,92,15,93,100,1,83,96,92,100,100,100,100,100,100,100,98,61,25,39,35,58,34,31,24,24,51,39,66,85,92,78,66,53,58,50,45,41,85,57,28,0,11,22,33,33,32,32,37,42,47,51,54,58,66,74,82,83,85,86,83,81,78,73,67,62,71,79,88,92,96,100,84,67,51,47,44,40,27,13,0,0,0,0,0,0,0,8,16,24,32,41,49,33,16,0,0,0,0,0,0,0,2,5,7,5,2,0,0,0,0,0,0,0,0],"cloudcover_low":[0,0,35,55,82,59,50,83,82,71,43,32,48,44,0,0,3,0,0,0,0,0,0,0,16,54,58,59,54,59,71,62,64,55,57,47,46,40,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,75,84,100,64,20,7,21,39,35,58,34,31,24,24,40,32,60,62,83,63,61,47,52,34,29,25,18,12,6,0,11,22,33,33,32,32,37,42,47,51,54,58,66,74,82,83,85,86,83,81,78,73,67,62,66,69,73,67,62,56,54,53,51,47,44,40,27,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,5,7,5,2,0,0,0,0,0,0,0,0],"cloudcover_mid":[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,91,0,0,84,78,75,76,78,77,57,60,5,25,34,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],"cloudcover_high":[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,44,67,80,57,57,45,11,14,26,6,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,82,78,92,15,93,100,1,83,95,88,100,100,100,100,100,100,100,95,59,0,0,0,0,0,0,0,0,17,9,28,54,51,37,12,11,11,21,19,19,74,49,25,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,36,54,69,85,100,67,34,1,1,0,0,0,0,0,0,0,0,0,0,0,8,16,24,32,41,49,33,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],"visibility":[24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24120.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00,24140.00],"evapotranspiration":[0.02,0.02,0.02,0.02,0.02,0.02,0.02,0.03,0.04,0.06,0.08,0.09,0.11,0.13,0.14,0.14,0.13,0.12,0.09,0.07,0.05,0.04,0.03,0.03,0.02,0.02,0.01,0.01,0.01,0.01,0.01,0.02,0.03,0.04,0.06,0.08,0.10,0.11,0.13,0.13,0.13,0.11,0.10,0.08,0.05,0.04,0.02,0.02,0.02,0.02,0.01,0.01,0.01,0.01,0.01,0.01,0.03,0.06,0.09,0.12,0.14,0.15,0.15,0.13,0.12,0.11,0.09,0.07,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00],"et0_fao_evapotranspiration":[0.00,0.01,0.01,0.00,0.00,0.00,0.02,0.06,0.11,0.17,0.25,0.36,0.41,0.46,0.50,0.49,0.44,0.37,0.28,0.19,0.12,0.07,0.05,0.03,0.01,0.00,0.00,0.00,0.00,0.00,0.01,0.05,0.12,0.20,0.29,0.34,0.41,0.46,0.50,0.49,0.45,0.38,0.29,0.19,0.11,0.05,0.01,0.01,0.01,0.00,0.00,0.00,0.00,0.00,0.01,0.06,0.13,0.24,0.35,0.46,0.50,0.53,0.52,0.49,0.44,0.34,0.24,0.16,0.10,0.07,0.05,0.03,0.02,0.02,0.01,0.02,0.01,0.01,0.03,0.07,0.14,0.21,0.25,0.29,0.34,0.38,0.41,0.42,0.40,0.33,0.27,0.19,0.13,0.07,0.05,0.04,0.02,0.01,0.01,0.01,0.00,0.00,0.02,0.06,0.11,0.18,0.25,0.30,0.33,0.33,0.33,0.32,0.28,0.23,0.17,0.13,0.09,0.07,0.04,0.03,0.02,0.01,0.01,0.00,0.00,0.00,0.02,0.05,0.11,0.20,0.30,0.40,0.49,0.55,0.56,0.55,0.50,0.43,0.35,0.25,0.16,0.09,0.07,0.05,0.03,0.02,0.01,0.01,0.00,0.00,0.02,0.07,0.13,0.22,0.32,0.40,0.47,0.52,0.54,0.53,0.49,0.41,0.31,0.21,0.12,0.04,0.02,0.01],"vapor_pressure_deficit":[0.26,0.28,0.21,0.18,0.16,0.16,0.18,0.23,0.34,0.44,0.63,0.78,0.86,1.00,1.26,1.36,1.27,1.22,1.01,0.86,0.75,0.68,0.53,0.39,0.26,0.17,0.12,0.10,0.09,0.08,0.10,0.19,0.32,0.49,0.61,0.76,0.82,0.94,1.15,1.26,1.32,1.27,1.24,1.01,0.87,0.71,0.51,0.51,0.43,0.35,0.31,0.28,0.20,0.15,0.14,0.24,0.47,0.88,1.23,1.49,1.60,1.60,1.56,1.54,1.54,1.44,1.27,0.96,0.59,0.55,0.55,0.41,0.30,0.28,0.22,0.25,0.24,0.21,0.23,0.30,0.45,0.51,0.65,0.67,0.83,0.88,0.99,0.98,1.09,1.01,0.97,0.93,0.82,0.68,0.56,0.44,0.32,0.24,0.19,0.16,0.15,0.15,0.17,0.20,0.28,0.42,0.54,0.62,0.66,0.69,0.73,0.77,0.77,0.73,0.67,0.60,0.56,0.51,0.45,0.39,0.33,0.26,0.22,0.19,0.17,0.15,0.14,0.17,0.27,0.45,0.69,1.01,1.36,1.64,1.67,1.73,1.72,1.63,1.48,1.33,1.19,1.05,0.90,0.74,0.56,0.44,0.35,0.30,0.27,0.26,0.27,0.31,0.41,0.59,0.78,0.95,1.09,1.24,1.40,1.56,1.59,1.45,1.21,0.97,0.77,0.58,0.46,0.39],"windspeed_10m":[7.5,9.9,9.7,9.6,11.3,10.8,11.5,12.3,15.0,15.6,15.0,15.7,16.7,16.6,16.8,17.8,16.2,16.5,16.6,15.9,15.5,14.6,14.0,12.9,11.5,10.6,8.9,7.7,7.4,7.4,7.4,8.7,10.5,12.4,12.0,12.1,12.1,12.1,12.1,11.8,11.7,11.8,11.3,11.2,9.7,8.3,6.2,6.0,5.2,4.5,3.3,2.5,2.8,3.3,3.4,4.4,5.9,9.7,13.0,13.8,14.8,15.4,16.1,17.0,17.4,17.3,16.1,20.1,17.8,16.6,15.0,14.0,14.3,12.8,13.1,13.3,11.8,10.7,11.1,12.6,14.3,17.1,17.6,17.7,17.2,17.4,17.1,17.4,17.1,16.7,16.6,16.4,15.5,14.5,13.5,13.0,12.5,12.3,12.1,11.9,11.9,11.7,11.5,12.0,13.1,14.5,15.8,16.0,15.5,15.3,15.3,15.9,15.9,16.1,16.5,16.5,16.6,16.6,16.0,14.8,12.7,11.5,10.4,10.2,9.8,9.6,9.9,10.3,11.2,12.8,14.4,16.3,18.0,19.5,19.9,19.1,18.6,18.1,17.7,16.7,15.5,13.9,12.4,11.3,9.8,9.0,8.4,7.6,7.4,7.4,7.9,8.4,9.7,10.9,12.2,12.8,13.6,14.2,14.8,15.4,15.6,15.3,14.1,13.1,11.6,9.7,8.7,8.7],"windspeed_80m":[16.0,19.1,18.9,19.1,21.8,20.5,21.5,20.9,25.3,25.3,22.3,23.3,25.0,24.1,25.0,26.2,23.8,24.6,26.0,25.2,26.7,27.1,26.6,25.1,22.1,21.2,17.9,15.5,16.5,16.8,16.1,14.0,15.6,18.6,17.0,16.6,16.5,16.1,16.1,16.2,16.1,16.2,16.3,15.8,14.8,14.0,12.3,13.1,12.6,11.1,8.7,7.7,8.7,10.4,10.4,10.9,8.4,13.5,18.9,20.5,21.5,22.3,23.3,25.5,26.6,28.0,27.5,35.6,35.5,33.4,31.7,29.4,29.5,27.0,27.1,28.6,24.9,23.8,23.8,24.9,27.7,31.8,32.3,31.3,30.3,30.4,29.8,29.9,30.1,29.6,30.5,31.3,30.7,30.1,29.0,28.3,27.3,26.3,25.8,25.9,25.6,24.7,23.5,23.0,24.1,26.0,27.5,27.5,26.5,26.1,26.4,27.1,28.1,29.5,31.3,32.6,33.5,34.4,33.9,31.4,27.0,24.0,22.1,21.2,20.5,19.8,19.6,19.7,20.6,22.4,24.8,28.2,32.2,35.1,34.6,33.5,32.5,32.4,32.7,32.4,31.1,29.3,27.7,25.6,23.6,22.1,21.5,20.9,20.2,18.6,16.5,15.8,16.6,18.1,19.4,20.3,20.9,21.9,23.2,24.4,25.3,24.9,24.1,23.1,21.9,20.8,20.1,20.5],"windspeed_120m":[18.6,21.5,21.9,23.3,25.5,24.2,24.7,23.1,27.2,26.7,23.5,24.1,25.9,24.9,26.1,27.4,24.6,25.7,27.2,26.5,28.6,30.2,30.1,28.7,25.8,25.3,22.6,20.9,21.5,21.6,21.7,20.3,16.4,19.7,17.8,17.3,16.9,16.6,16.9,16.6,16.9,17.0,16.7,16.2,15.5,14.8,14.5,14.9,15.5,14.0,11.3,10.1,11.4,13.3,13.6,14.5,12.7,14.5,19.9,21.8,22.5,23.1,24.3,26.5,27.9,29.6,29.6,38.4,38.5,36.7,35.5,33.2,33.4,31.6,31.9,33.4,29.1,31.0,31.6,27.1,30.1,33.9,34.3,33.2,32.1,32.0,31.4,31.4,31.9,31.4,32.3,33.6,33.7,33.6,33.3,32.6,31.8,31.3,32.1,33.4,33.4,31.2,27.7,25.5,25.6,27.2,28.5,28.5,27.8,27.6,27.9,28.6,29.6,31.5,33.9,35.7,37.3,37.9,38.0,35.2,31.1,27.8,26.2,25.5,24.7,23.5,22.0,21.2,21.9,23.5,26.3,30.0,34.2,37.4,36.6,35.4,34.4,34.4,34.8,34.9,34.7,34.3,33.7,33.1,32.3,31.7,31.3,30.9,30.5,29.7,28.6,27.4,24.8,21.9,19.9,20.0,21.2,22.7,24.0,25.5,26.5,26.2,25.0,24.5,24.8,25.4,25.8,26.0],"windspeed_180m":[20.9,26.6,27.8,29.3,32.8,31.2,31.6,25.0,28.5,27.9,24.3,24.9,26.7,25.8,26.6,28.2,25.5,26.3,28.1,27.2,29.9,32.1,32.9,32.0,28.8,30.4,28.9,26.2,26.0,26.4,26.9,26.3,17.2,20.4,18.5,17.7,17.3,17.1,17.4,17.1,17.0,17.1,17.2,16.6,15.8,15.5,17.3,17.7,18.9,17.3,13.9,12.6,13.7,15.7,17.0,18.4,19.1,14.8,20.4,22.3,23.3,23.9,25.1,27.3,28.7,30.6,31.1,40.0,40.3,38.3,38.3,36.6,35.9,37.2,41.2,43.1,43.5,45.7,44.9,46.0,30.9,35.8,35.6,34.5,33.4,32.9,32.4,32.2,32.9,32.4,33.1,34.9,35.5,36.2,37.7,40.5,43.6,46.2,47.0,46.5,46.0,45.7,44.9,43.4,39.0,33.6,29.5,28.0,28.1,28.4,28.9,29.6,30.7,32.8,35.2,37.5,39.1,40.5,40.8,38.3,34.5,31.9,32.3,34.4,35.8,36.4,36.5,35.5,32.6,28.8,27.1,29.8,34.5,38.5,38.0,36.7,35.7,35.6,36.1,36.5,37.4,38.8,40.0,41.6,43.6,44.6,43.9,41.8,40.0,38.9,38.3,36.1,30.7,24.4,20.4,20.0,21.7,23.5,25.0,26.5,27.5,26.8,26.0,25.6,27.6,30.4,32.6,33.2],"winddirection_10m":[55,46,39,34,37,37,32,32,48,61,69,70,65,68,67,72,73,74,72,77,78,74,64,60,58,55,50,41,39,43,43,38,49,68,69,67,67,67,67,70,72,70,73,92,92,90,83,65,65,61,49,45,50,49,58,55,52,48,48,51,51,53,51,54,52,59,56,43,48,48,48,48,45,40,37,38,31,33,36,37,43,51,55,57,57,56,55,56,55,53,51,50,48,46,44,42,39,38,37,35,35,34,32,33,37,42,45,44,42,39,39,39,39,39,36,36,39,42,46,47,47,46,44,39,36,34,33,36,42,52,58,65,70,74,73,70,68,67,66,65,62,59,54,53,54,53,50,45,43,39,39,40,42,44,47,52,58,63,67,69,71,71,71,69,68,68,66,60],"winddirection_80m":[54,49,42,36,39,38,35,36,50,62,69,70,65,68,67,72,72,72,71,76,76,73,65,61,60,58,56,48,46,48,50,42,52,68,69,66,67,66,66,69,70,69,72,90,91,90,85,69,70,65,52,37,30,34,44,46,50,48,48,51,52,52,52,54,52,58,56,44,49,49,50,49,46,43,40,41,34,35,38,39,46,52,57,57,58,56,55,56,56,53,52,52,50,47,45,43,42,41,40,39,38,37,36,35,39,44,46,45,42,39,39,40,40,39,38,37,40,45,47,48,48,47,46,42,39,37,36,38,44,53,60,67,72,75,73,71,69,67,66,64,62,58,55,55,56,56,55,54,51,49,44,41,42,44,48,53,58,63,66,68,70,70,68,67,67,67,66,62],"winddirection_120m":[54,50,44,38,42,41,37,39,51,63,69,70,65,68,67,71,72,72,70,75,75,73,65,62,62,62,61,58,55,57,58,53,52,69,69,67,66,66,66,68,70,69,71,89,91,90,83,70,72,67,53,35,28,33,40,44,47,48,48,51,51,53,52,53,52,58,56,45,49,49,50,50,46,44,41,42,36,39,41,40,47,53,57,58,57,56,56,56,56,53,52,52,49,48,45,44,44,43,42,42,41,39,38,36,39,43,46,45,42,40,39,40,40,39,37,37,40,45,48,49,49,49,47,44,42,40,38,40,46,54,61,67,72,74,73,71,69,67,66,64,61,58,56,57,59,61,61,61,60,57,54,51,49,48,48,52,58,63,66,68,69,69,67,66,64,63,62,59],"winddirection_180m":[54,54,50,45,51,50,47,43,53,63,69,70,64,67,67,71,71,71,69,75,75,72,65,63,64,68,71,69,66,66,67,65,53,69,69,66,67,65,66,68,69,68,70,88,91,90,78,73,72,67,53,37,30,32,36,42,46,47,48,51,52,53,52,54,52,58,55,46,49,50,50,51,47,46,46,49,47,49,49,51,48,54,57,58,57,56,55,56,55,53,53,52,50,47,47,48,49,50,50,50,50,50,50,49,48,48,46,44,41,39,39,40,40,39,37,37,40,45,48,49,50,50,51,53,54,54,55,56,57,58,61,67,72,74,73,70,68,67,65,64,61,59,58,61,65,68,69,70,69,68,66,63,59,53,48,51,58,63,65,68,68,68,66,64,61,59,57,56],"windgusts_10m":[18.4,19.4,20.5,20.2,20.9,23.4,23.8,25.6,32.0,34.2,34.6,35.3,38.5,37.8,37.8,39.2,38.9,36.0,36.4,36.0,33.8,33.1,30.6,29.2,27.0,23.8,22.3,18.4,15.8,15.8,15.8,19.1,22.7,27.4,27.7,28.1,28.1,28.1,28.4,28.4,27.7,27.4,27.0,25.6,24.5,21.2,16.9,12.2,12.2,10.4,8.6,6.1,4.7,6.1,6.5,9.4,14.0,22.0,28.8,31.3,33.1,34.2,35.3,37.1,38.2,37.8,37.8,42.1,43.9,42.8,39.6,36.4,34.2,34.2,31.0,32.0,31.7,27.7,26.6,30.6,35.3,41.4,43.6,43.6,43.2,43.6,43.2,43.2,43.2,43.2,41.4,41.4,39.6,37.8,36.0,34.2,32.0,30.2,29.5,29.2,28.4,28.8,29.5,29.9,32.8,36.0,38.9,38.5,38.5,38.2,38.5,39.2,39.6,40.0,40.0,40.3,39.6,39.2,38.5,35.6,32.4,29.5,28.1,26.6,25.2,25.2,25.2,25.2,28.8,32.0,35.6,40.0,43.9,48.2,48.6,47.5,46.8,45.7,44.3,43.2,39.2,35.3,31.3,29.2,26.6,24.5,22.3,19.8,17.6,18.7,20.2,21.2,24.8,28.4,32.0,33.5,34.9,36.4,37.4,38.5,39.6,38.2,36.7,35.3,31.3,27.7,23.8,23.8],"temperature_80m":[10.6,9.6,8.8,8.2,8.1,7.9,7.8,8.4,9.1,9.9,11.1,12.2,13.3,14.7,15.4,15.8,16.2,16.0,15.3,14.5,13.5,12.3,11.4,10.4,9.6,9.1,8.6,8.2,8.1,7.8,7.6,8.3,9.6,11.1,12.1,13.0,14.1,15.1,16.1,16.8,17.3,17.5,17.4,16.3,15.2,14.0,12.8,12.2,11.4,10.9,10.7,10.6,10.6,10.4,10.2,10.1,11.1,13.0,14.7,16.3,17.4,18.2,18.8,19.3,19.5,19.3,18.8,17.4,14.1,12.6,11.5,10.4,9.4,8.9,8.4,8.2,7.5,7.2,7.1,7.7,9.0,10.0,10.7,11.5,12.2,12.8,13.6,14.4,14.7,14.4,14.2,13.7,12.8,11.7,10.6,9.7,8.9,8.3,7.9,7.6,7.5,7.4,7.4,7.8,8.7,10.0,11.2,12.2,13.1,13.8,14.3,14.6,14.7,14.6,14.3,13.8,13.1,12.3,11.6,11.0,10.6,10.2,9.9,9.8,9.7,9.5,9.4,9.9,11.3,13.3,15.1,16.5,17.6,18.4,18.6,18.9,19.1,19.1,18.8,18.3,17.4,16.3,15.1,13.9,12.8,11.8,11.2,10.8,10.5,10.1,9.9,10.1,11.2,12.8,14.4,15.8,17.1,18.2,18.9,19.3,19.5,19.4,19.0,18.3,17.3,16.1,14.9,13.8],"temperature_120m":[10.2,9.2,8.3,7.9,7.6,7.5,7.3,7.9,8.6,9.3,10.5,11.6,12.7,14.1,14.8,15.2,15.6,15.4,14.7,14.0,13.0,11.9,10.9,10.0,9.2,8.7,8.4,8.2,8.1,7.8,7.7,7.9,9.1,10.5,11.6,12.5,13.5,14.5,15.5,16.2,16.8,17.0,16.9,15.8,14.7,13.5,12.4,11.8,11.2,10.9,10.7,10.7,10.8,10.8,10.6,10.6,10.6,12.4,14.2,15.7,16.8,17.6,18.2,18.7,18.9,18.7,18.3,16.9,13.4,11.9,10.8,9.7,8.8,8.2,7.8,7.6,7.0,6.8,6.6,7.0,8.2,9.2,10.0,10.7,11.4,12.0,12.8,13.6,13.9,13.6,13.4,12.9,12.0,10.9,9.9,9.0,8.3,7.7,7.4,7.2,7.1,6.9,6.8,7.1,8.0,9.3,10.5,11.5,12.3,13.0,13.5,13.9,14.0,13.9,13.6,13.1,12.4,11.6,10.9,10.3,9.9,9.5,9.3,9.2,9.1,8.9,8.8,9.2,10.6,12.5,14.3,15.7,16.8,17.6,17.8,18.1,18.3,18.3,18.0,17.5,16.7,15.6,14.5,13.5,12.4,11.6,11.2,10.9,10.7,10.2,9.8,9.8,10.7,12.2,13.6,15.0,16.3,17.4,18.1,18.6,18.8,18.6,18.2,17.5,16.6,15.6,14.5,13.3],"temperature_180m":[9.6,8.7,8.0,7.6,7.5,7.4,7.2,7.4,8.0,8.7,9.9,11.0,12.1,13.4,14.2,14.5,15.0,14.8,14.1,13.4,12.4,11.3,10.4,9.5,8.8,8.3,8.2,8.2,8.2,8.0,7.9,8.1,8.5,9.9,11.0,11.8,12.9,13.8,14.9,15.6,16.1,16.4,16.3,15.2,14.2,13.0,12.1,11.4,11.2,11.0,10.8,10.9,11.0,11.0,11.0,10.9,10.8,11.9,13.6,15.1,16.2,17.0,17.6,18.1,18.3,18.1,17.7,16.3,12.5,11.0,9.9,8.9,7.9,7.5,7.1,7.0,6.7,7.0,6.8,7.0,7.4,8.3,9.0,9.8,10.5,11.1,11.8,12.7,13.0,12.7,12.5,12.0,11.1,10.1,9.1,8.4,7.7,7.3,7.1,7.2,7.2,7.0,6.9,7.0,7.7,8.6,9.6,10.5,11.4,12.1,12.6,13.0,13.1,13.0,12.7,12.2,11.6,10.8,10.1,9.6,9.1,8.8,8.7,8.6,8.7,8.7,8.7,9.2,10.4,11.9,13.4,14.7,15.9,16.7,16.8,17.2,17.3,17.3,17.1,16.7,15.9,14.8,13.9,13.2,12.6,12.1,11.8,11.5,11.3,10.9,10.5,10.4,10.9,11.7,12.7,13.9,15.4,16.5,17.2,17.6,17.8,17.7,17.3,16.7,16.0,15.1,14.1,12.9],"soil_temperature_0cm":[10.8,10.3,9.6,9.3,9.5,9.1,9.5,11.8,13.1,15.4,18.7,20.9,22.1,23.8,24.0,23.4,23.1,21.5,19.4,17.2,14.7,12.6,11.7,11.0,10.3,10.0,9.6,9.4,9.1,8.7,9.4,11.6,14.7,17.4,19.8,21.7,23.7,25.0,25.6,25.4,24.7,23.4,21.5,18.9,16.2,14.0,12.8,11.9,11.1,10.4,9.7,9.0,8.6,8.4,9.4,12.5,15.9,18.4,21.8,24.1,25.0,26.2,26.1,25.9,24.7,22.7,20.7,18.4,15.7,13.8,12.5,11.6,10.9,10.2,9.9,9.7,8.8,8.5,9.2,11.4,14.2,16.4,17.7,18.8,20.6,21.0,22.2,22.6,22.0,20.2,18.5,16.2,14.3,12.6,11.2,10.4,9.8,9.4,9.0,8.7,8.7,9.2,10.1,11.5,13.7,16.5,18.8,20.2,21.2,21.7,21.6,21.1,20.2,18.9,17.3,15.8,14.5,13.4,12.5,12.0,11.8,11.7,11.3,10.9,10.9,11.2,11.9,13.3,16.0,19.5,22.4,24.2,25.5,26.3,26.1,25.9,25.2,23.7,21.8,19.8,17.9,15.9,14.3,13.2,12.4,11.7,11.0,10.4,10.3,10.8,11.8,13.4,16.1,19.4,22.2,24.4,26.2,27.3,27.5,27.1,26.2,24.7,22.7,20.6,18.5,16.4,14.7,13.5],"soil_temperature_6cm":[13.3,12.8,12.1,11.5,11.1,10.9,10.6,11.2,12.0,12.9,14.7,16.4,17.8,19.5,20.4,20.9,21.2,21.0,20.2,19.4,17.9,16.3,14.9,13.8,13.0,12.3,11.8,11.4,11.0,10.7,10.4,10.8,11.8,13.4,15.2,16.8,18.5,19.9,21.1,21.9,22.2,22.1,21.6,20.6,19.1,17.5,16.0,14.9,13.9,13.1,12.4,11.8,11.1,10.7,10.4,10.9,12.3,14.2,16.0,18.1,19.9,21.2,22.2,22.8,22.8,22.4,21.5,20.3,18.3,16.9,15.6,14.5,13.6,12.9,12.3,11.8,11.4,10.9,10.5,10.8,11.8,13.2,14.4,15.6,16.6,17.6,18.4,19.3,19.6,19.5,18.9,18.0,16.8,15.4,14.1,13.2,12.4,11.8,11.2,10.8,10.6,10.3,10.2,10.6,11.6,13.1,14.5,15.7,16.9,17.8,18.4,18.8,18.8,18.4,17.7,16.9,16.1,15.3,14.5,13.9,13.4,13.0,12.6,12.4,12.2,12.0,11.8,12.1,13.2,14.9,16.5,18.2,20.0,21.2,21.7,22.3,22.4,22.2,21.6,20.7,19.6,18.2,17.0,16.0,15.1,14.3,13.6,13.0,12.6,12.2,12.0,12.3,13.4,15.1,16.8,18.5,20.3,21.7,22.7,23.3,23.5,23.2,22.5,21.5,20.3,18.9,17.6,16.5],"soil_temperature_18cm":[15.9,15.6,15.3,15.0,14.7,14.4,14.2,13.9,13.7,13.6,13.7,13.8,14.0,14.4,14.7,15.1,15.5,15.9,16.2,16.4,16.5,16.4,16.3,16.1,15.9,15.6,15.3,15.0,14.7,14.4,14.1,13.9,13.7,13.6,13.7,13.9,14.1,14.5,14.9,15.3,15.7,16.1,16.5,16.7,16.9,16.9,16.8,16.6,16.4,16.2,15.9,15.6,15.2,14.9,14.6,14.3,14.1,14.0,14.1,14.3,14.6,15.0,15.4,15.9,16.3,16.7,17.0,17.2,17.2,17.2,17.0,16.8,16.6,16.3,16.0,15.7,15.3,15.0,14.7,14.4,14.2,14.1,14.0,14.1,14.3,14.5,14.7,15.0,15.3,15.5,15.8,15.9,16.0,15.9,15.8,15.6,15.4,15.1,14.9,14.6,14.3,14.0,13.7,13.5,13.4,13.3,13.4,13.5,13.7,14.0,14.3,14.6,14.9,15.1,15.3,15.4,15.4,15.4,15.3,15.2,15.1,14.9,14.8,14.6,14.4,14.2,14.1,13.9,13.9,13.9,14.0,14.3,14.7,15.0,15.2,15.6,16.0,16.4,16.7,17.0,17.1,17.1,17.1,17.0,16.8,16.6,16.3,16.1,15.8,15.5,15.2,15.0,14.8,14.8,14.8,15.0,15.3,15.6,16.0,16.5,16.9,17.3,17.6,17.8,17.9,17.9,17.9,17.7],"soil_temperature_54cm":[13.7,13.7,13.8,13.8,13.8,13.8,13.7,13.7,13.7,13.7,13.7,13.7,13.7,13.7,13.7,13.7,13.7,13.7,13.7,13.7,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,13.9,14.1,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.2,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.1,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,13.8,13.8,13.8,13.8,13.8,13.8,13.8,13.9,13.9,13.9,13.9,13.9,13.9,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.0,14.1,14.1,14.1,14.1,14.1],"soil_moisture_0_1cm":[0.300,0.301,0.302,0.303,0.307,0.308,0.308,0.309,0.308,0.307,0.306,0.305,0.303,0.301,0.299,0.297,0.296,0.295,0.295,0.296,0.297,0.298,0.299,0.300,0.302,0.303,0.304,0.305,0.305,0.306,0.307,0.307,0.307,0.306,0.305,0.304,0.302,0.300,0.298,0.296,0.294,0.293,0.293,0.293,0.294,0.295,0.297,0.298,0.299,0.300,0.301,0.302,0.303,0.303,0.304,0.304,0.304,0.303,0.301,0.299,0.296,0.294,0.291,0.290,0.289,0.288,0.288,0.289,0.291,0.293,0.294,0.295,0.297,0.298,0.300,0.301,0.302,0.303,0.304,0.304,0.304,0.303,0.301,0.300,0.298,0.296,0.293,0.291,0.289,0.288,0.287,0.288,0.289,0.290,0.291,0.292,0.294,0.295,0.296,0.298,0.299,0.300,0.301,0.301,0.301,0.300,0.299,0.298,0.296,0.294,0.292,0.291,0.290,0.290,0.290,0.291,0.292,0.292,0.293,0.294,0.295,0.296,0.297,0.298,0.299,0.300,0.301,0.301,0.301,0.300,0.298,0.294,0.289,0.285,0.281,0.277,0.274,0.272,0.272,0.272,0.273,0.274,0.276,0.277,0.279,0.280,0.282,0.283,0.285,0.286,0.288,0.288,0.288,0.287,0.285,0.282,0.279,0.275,0.271,0.267,0.264,0.263,0.262,0.263,0.264,0.267,0.269,0.271],"soil_moisture_1_3cm":[0.302,0.303,0.304,0.304,0.309,0.309,0.310,0.310,0.310,0.310,0.310,0.309,0.308,0.307,0.305,0.304,0.303,0.302,0.301,0.301,0.301,0.301,0.302,0.303,0.304,0.304,0.305,0.306,0.306,0.307,0.308,0.308,0.308,0.308,0.308,0.307,0.306,0.305,0.304,0.302,0.301,0.300,0.299,0.298,0.298,0.299,0.299,0.300,0.301,0.302,0.302,0.303,0.304,0.304,0.305,0.306,0.306,0.306,0.305,0.304,0.302,0.300,0.299,0.297,0.296,0.295,0.295,0.294,0.294,0.294,0.295,0.296,0.297,0.298,0.299,0.299,0.300,0.301,0.302,0.302,0.303,0.303,0.302,0.301,0.300,0.299,0.298,0.296,0.295,0.294,0.293,0.292,0.292,0.292,0.292,0.292,0.293,0.294,0.295,0.296,0.297,0.298,0.299,0.299,0.299,0.299,0.299,0.299,0.298,0.297,0.296,0.295,0.294,0.293,0.292,0.292,0.292,0.292,0.293,0.294,0.294,0.295,0.296,0.296,0.297,0.298,0.298,0.299,0.299,0.300,0.299,0.297,0.295,0.293,0.291,0.288,0.286,0.284,0.282,0.281,0.280,0.280,0.280,0.280,0.281,0.282,0.283,0.283,0.284,0.285,0.286,0.287,0.287,0.287,0.287,0.286,0.285,0.283,0.281,0.279,0.277,0.275,0.274,0.273,0.273,0.273,0.273,0.274],"soil_moisture_3_9cm":[0.311,0.311,0.312,0.312,0.316,0.316,0.316,0.316,0.316,0.316,0.316,0.316,0.316,0.315,0.315,0.315,0.314,0.314,0.313,0.313,0.313,0.312,0.312,0.312,0.312,0.312,0.312,0.312,0.312,0.313,0.313,0.313,0.313,0.313,0.313,0.313,0.313,0.313,0.313,0.312,0.312,0.311,0.311,0.311,0.310,0.310,0.310,0.310,0.310,0.310,0.310,0.310,0.310,0.310,0.310,0.311,0.311,0.311,0.311,0.311,0.311,0.310,0.310,0.309,0.309,0.308,0.308,0.308,0.307,0.307,0.307,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.306,0.305,0.305,0.304,0.304,0.304,0.304,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.303,0.302,0.302,0.302,0.301,0.301,0.301,0.301,0.301,0.301,0.301,0.301,0.301,0.301,0.301,0.301,0.302,0.302,0.302,0.302,0.302,0.302,0.301,0.301,0.300,0.300,0.299,0.299,0.298,0.298,0.298,0.297,0.297,0.297,0.296,0.296,0.296,0.295,0.295,0.295,0.295,0.295,0.295,0.295,0.295,0.295,0.294,0.294,0.294,0.293,0.293,0.293,0.292,0.292,0.292,0.291,0.291,0.291],"soil_moisture_9_27cm":[0.334,0.334,0.334,0.334,0.335,0.335,0.335,0.335,0.335,0.335,0.335,0.335,0.335,0.334,0.334,0.334,0.334,0.334,0.334,0.334,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.333,0.332,0.332,0.332,0.332,0.332,0.332,0.332,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.331,0.330,0.330,0.330,0.330,0.330,0.330,0.330,0.329,0.329,0.329,0.329,0.329,0.329,0.329,0.329,0.329,0.329,0.328,0.328,0.328,0.328,0.328,0.328,0.328,0.328,0.328,0.327,0.327,0.327,0.327,0.327,0.327,0.327,0.326,0.326,0.326,0.326,0.326,0.326,0.326,0.326,0.326,0.326,0.326,0.326,0.325,0.325,0.325,0.325,0.325,0.325,0.325,0.325,0.325,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.324,0.323,0.323,0.323,0.323,0.323,0.323,0.323,0.323,0.323,0.323,0.323,0.323,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.322,0.321,0.321,0.321,0.320,0.320,0.320,0.320,0.320,0.320,0.320,0.320,0.320],"soil_moisture_27_81cm":[0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.373,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.372,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.371,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.370,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.369,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.368,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.367,0.366,0.366,0.366,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.365,0.364,0.364,0.364,0.364,0.364,0.364,0.364,0.364,0.364,0.364,0.364,0.364],"uv_index":[0.00,0.00,0.00,0.00,0.00,0.00,0.15,0.75,1.75,2.95,4.30,5.45,6.35,6.80,6.80,6.30,5.40,4.20,2.90,1.65,0.70,0.15,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.15,0.75,1.75,3.00,4.30,5.45,6.35,6.80,6.80,6.30,5.40,4.20,2.90,1.65,0.70,0.15,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.15,0.70,1.50,2.50,3.70,4.90,5.65,4.75,3.60,3.00,1.45,0.80,0.80,0.45,0.20,0.05,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.15,0.80,1.75,2.95,4.25,5.25,6.25,6.45,6.45,6.10,5.10,4.15,2.65,1.60,0.75,0.15,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.20,0.80,1.75,3.00,4.30,5.50,6.35,6.85,6.80,6.35,5.40,4.25,2.90,1.70,0.75,0.15,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.20,0.80,1.40,2.15,3.00,4.10,5.35,6.20,6.55,6.55,6.20,5.30,4.05,2.90,1.90,0.95,0.30,0.00,0.00,0.00,0.00,0.00,0.00,0.00,0.05,0.35,1.05,2.05,3.05,4.20,5.35,6.20,6.55,6.55,6.20,5.35,4.15,3.00,1.95,1.00,0.30,0.00],"is_day":[0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0]},"daily_units":{"time":"unixtime","weathercode":"wmo code","temperature_2m_max":"°C","temperature_2m_min":"°C","apparent_temperature_max":"°C","apparent_temperature_min":"°C","sunrise":"unixtime","sunset":"unixtime","uv_index_max":"","uv_index_clear_sky_max":"","precipitation_sum":"mm","rain_sum":"mm","showers_sum":"mm","snowfall_sum":"cm","precipitation_hours":"h","precipitation_probability_max":"%","precipitation_probability_mean":"%","windspeed_10m_max":"km/h","windgusts_10m_max":"km/h","winddirection_10m_dominant":"°","shortwave_radiation_sum":"MJ/m²","et0_fao_evapotranspiration":"mm"},"daily":{"time":[1685055600,1685142000,1685228400,1685314800,1685401200,1685487600,1685574000],"weathercode":[2,3,3,3,3,3,2],"temperature_2m_max":[18.4,19.5,21.5,17.3,17.0,21.6,22.0],"temperature_2m_min":[8.5,8.2,9.1,7.9,8.2,10.4,10.6],"apparent_temperature_max":[15.8,18.0,19.4,15.0,14.7,18.9,21.1],"apparent_temperature_min":[5.9,6.4,7.9,4.9,5.4,8.6,8.8],"sunrise":[1685073154,1685159494,1685245835,1685332175,1685418515,1685504865,1685591218],"sunset":[1685131347,1685217821,1685304295,1685390770,1685477245,1685563712,1685650177],"uv_index_max":[6.80,6.80,5.65,6.45,6.85,6.55,6.55],"uv_index_clear_sky_max":[6.80,6.80,6.70,6.80,6.85,6.55,6.60],"precipitation_sum":[0.00,0.00,0.00,0.00,0.00,0.00,0.00],"rain_sum":[0.00,0.00,0.00,0.00,0.00,0.00,0.00],"showers_sum":[0.00,0.00,0.00,0.00,0.00,0.00,0.00],"snowfall_sum":[0.00,0.00,0.00,0.00,0.00,0.00,0.00],"precipitation_hours":[0.0,0.0,0.0,0.0,0.0,0.0,0.0],"precipitation_probability_max":[0,0,16,0,0,0,3],"precipitation_probability_mean":[0,0,16,0,0,0,3],"windspeed_10m_max":[17.8,12.4,20.1,17.7,16.6,19.9,15.6],"windgusts_10m_max":[39.2,28.4,43.9,43.6,40.3,48.6,39.6],"winddirection_10m_dominant":[61,65,51,48,39,58,58],"shortwave_radiation_sum":[27.31,27.67,24.99,25.18,20.19,28.09,28.64],"et0_fao_evapotranspiration":[4.40,4.37,4.73,4.11,3.32,5.17,4.90]}}',
      ) as Map<String, dynamic>;

      final fullWeather =
          FullWeatherModel.fromJson(json, city: city).fullWeather;

      expect(fullWeather.city, city);
      expect(
        fullWeather.currentWeather,
        Weather(
          unitSystem: UnitSystem.metric,
          date: date_time_utils.fromUtcUnixTime(location, 1685134800),
          temperature: 11.9,
          tempFeel: 8.7,
          pressure: 1027,
          humidity: 0.62,
          clouds: 0,
          uvIndex: 0,
          windSpeed: 14,
          windDirection: WindDirection.northeast,
          weatherCode: 0,
          pop: 0,
        ),
      );

      expect(
        fullWeather.hourlyForecasts[0],
        HourlyForecast(
          unitSystem: UnitSystem.metric,
          temperature: 11.9,
          weatherCode: 0,
          date: date_time_utils.fromUtcUnixTime(location, 1685134800),
          pop: 0,
          isDay: false,
        ),
      );

      final dailyForecast = DailyForecast(
        unitSystem: UnitSystem.metric,
        date: date_time_utils.fromUtcUnixTime(location, 1685055600),
        sunrise: date_time_utils.fromUtcUnixTime(location, 1685073154),
        sunset: date_time_utils.fromUtcUnixTime(location, 1685131347),
        weatherCode: 2,
        minTemperature: 8.5,
        maxTemperature: 18.4,
        pop: 0,
      );

      expect(fullWeather.dailyForecasts[0], dailyForecast);
      expect(fullWeather.currentDayForecast, dailyForecast);
    });

    test('handles nulls gravefully', () {
      final json = jsonDecode(
        // Weather data based on data fetched from Open-Meteo (https://open-meteo.com/)
        // under the CC BY 4.0. See https://open-meteo.com/en/license for data
        // sources.
        //
        // It's all null baby
        '{"latitude":51.5,"longitude":-0.120000124,"generationtime_ms":3.939986228942871,"utc_offset_seconds":3600,"timezone":"Europe/London","timezone_abbreviation":"BST","elevation":null,"current_weather":{"temperature":null,"windspeed":null,"winddirection":null,"weathercode":null,"is_day":null,"time":1685134800},"hourly_units":{"time":"unixtime","temperature_2m":"°C","relativehumidity_2m":"%","dewpoint_2m":"°C","apparent_temperature":"°C","precipitation_probability":"%","precipitation":"mm","rain":"mm","showers":"mm","snowfall":"cm","snow_depth":"m","weathercode":"wmo code","pressure_msl":"hPa","surface_pressure":"hPa","cloudcover":"%","cloudcover_low":"%","cloudcover_mid":"%","cloudcover_high":"%","visibility":"m","evapotranspiration":"mm","et0_fao_evapotranspiration":"mm","vapor_pressure_deficit":"kPa","windspeed_10m":"km/h","windspeed_80m":"km/h","windspeed_120m":"km/h","windspeed_180m":"km/h","winddirection_10m":"°","winddirection_80m":"°","winddirection_120m":"°","winddirection_180m":"°","windgusts_10m":"km/h","temperature_80m":"°C","temperature_120m":"°C","temperature_180m":"°C","soil_temperature_0cm":"°C","soil_temperature_6cm":"°C","soil_temperature_18cm":"°C","soil_temperature_54cm":"°C","soil_moisture_0_1cm":"m³/m³","soil_moisture_1_3cm":"m³/m³","soil_moisture_3_9cm":"m³/m³","soil_moisture_9_27cm":"m³/m³","soil_moisture_27_81cm":"m³/m³","uv_index":"","is_day":""},"hourly":{"time":[1685055600,1685059200,1685062800,1685066400,1685070000,1685073600,1685077200,1685080800,1685084400,1685088000,1685091600,1685095200,1685098800,1685102400,1685106000,1685109600,1685113200,1685116800,1685120400,1685124000,1685127600,1685131200,1685134800,1685138400,1685142000,1685145600,1685149200,1685152800,1685156400,1685160000,1685163600,1685167200,1685170800,1685174400,1685178000,1685181600,1685185200,1685188800,1685192400,1685196000,1685199600,1685203200,1685206800,1685210400,1685214000,1685217600,1685221200,1685224800,1685228400,1685232000,1685235600,1685239200,1685242800,1685246400,1685250000,1685253600,1685257200,1685260800,1685264400,1685268000,1685271600,1685275200,1685278800,1685282400,1685286000,1685289600,1685293200,1685296800,1685300400,1685304000,1685307600,1685311200,1685314800,1685318400,1685322000,1685325600,1685329200,1685332800,1685336400,1685340000,1685343600,1685347200,1685350800,1685354400,1685358000,1685361600,1685365200,1685368800,1685372400,1685376000,1685379600,1685383200,1685386800,1685390400,1685394000,1685397600,1685401200,1685404800,1685408400,1685412000,1685415600,1685419200,1685422800,1685426400,1685430000,1685433600,1685437200,1685440800,1685444400,1685448000,1685451600,1685455200,1685458800,1685462400,1685466000,1685469600,1685473200,1685476800,1685480400,1685484000,1685487600,1685491200,1685494800,1685498400,1685502000,1685505600,1685509200,1685512800,1685516400,1685520000,1685523600,1685527200,1685530800,1685534400,1685538000,1685541600,1685545200,1685548800,1685552400,1685556000,1685559600,1685563200,1685566800,1685570400,1685574000,1685577600,1685581200,1685584800,1685588400,1685592000,1685595600,1685599200,1685602800,1685606400,1685610000,1685613600,1685617200,1685620800,1685624400,1685628000,1685631600,1685635200,1685638800,1685642400,1685646000,1685649600,1685653200,1685656800],"temperature_2m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"relativehumidity_2m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"dewpoint_2m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"apparent_temperature":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"precipitation_probability":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"precipitation":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"rain":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"showers":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"snowfall":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"snow_depth":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"weathercode":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"pressure_msl":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"surface_pressure":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"cloudcover":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"cloudcover_low":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"cloudcover_mid":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"cloudcover_high":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"visibility":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"evapotranspiration":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"et0_fao_evapotranspiration":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"vapor_pressure_deficit":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"windspeed_10m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"windspeed_80m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"windspeed_120m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"windspeed_180m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"winddirection_10m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"winddirection_80m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"winddirection_120m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"winddirection_180m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"windgusts_10m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"temperature_80m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"temperature_120m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"temperature_180m":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_temperature_0cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_temperature_6cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_temperature_18cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_temperature_54cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_moisture_0_1cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_moisture_1_3cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_moisture_3_9cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_moisture_9_27cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"soil_moisture_27_81cm":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"uv_index":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null],"is_day":[null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null]},"daily_units":{"time":"unixtime","weathercode":"wmo code","temperature_2m_max":"°C","temperature_2m_min":"°C","apparent_temperature_max":"°C","apparent_temperature_min":"°C","sunrise":"unixtime","sunset":"unixtime","uv_index_max":"","uv_index_clear_sky_max":"","precipitation_sum":"mm","rain_sum":"mm","showers_sum":"mm","snowfall_sum":"cm","precipitation_hours":"h","precipitation_probability_max":"%","precipitation_probability_mean":"%","windspeed_10m_max":"km/h","windgusts_10m_max":"km/h","winddirection_10m_dominant":"°","shortwave_radiation_sum":"MJ/m²","et0_fao_evapotranspiration":"mm"},"daily":{"time":[1685055600,1685142000,1685228400,1685314800,1685401200,1685487600,1685574000],"weathercode":[null,null,null,null,null,null,null],"temperature_2m_max":[null,null,null,null,null,null,null],"temperature_2m_min":[null,null,null,null,null,null,null],"apparent_temperature_max":[null,null,null,null,null,null,null],"apparent_temperature_min":[null,null,null,null,null,null,null],"sunrise":[null,null,null,null,null,null,null],"sunset":[null,null,null,null,null,null,null],"uv_index_max":[null,null,null,null,null,null,null],"uv_index_clear_sky_max":[null,null,null,null,null,null,null],"precipitation_sum":[null,null,null,null,null,null,null],"rain_sum":[null,null,null,null,null,null,null],"showers_sum":[null,null,null,null,null,null,null],"snowfall_sum":[null,null,null,null,null,null,null],"precipitation_hours":[null,null,null,null,null,null,null],"precipitation_probability_max":[null,null,null,null,null,null,null],"precipitation_probability_mean":[null,null,null,null,null,null,null],"windspeed_10m_max":[null,null,null,null,null,null,null],"windgusts_10m_max":[null,null,null,null,null,null,null],"winddirection_10m_dominant":[null,null,null,null,null,null,null],"shortwave_radiation_sum":[null,null,null,null,null,null,null],"et0_fao_evapotranspiration":[null,null,null,null,null,null,null]}}',
      ) as Map<String, dynamic>;

      final fullWeather =
          FullWeatherModel.fromJson(json, city: city).fullWeather;

      expect(fullWeather.city, city);

      expect(
        fullWeather.currentWeather,
        Weather(
          unitSystem: UnitSystem.metric,
          date: date_time_utils.fromUtcUnixTime(location, 1685134800),
          temperature: null,
          tempFeel: null,
          pressure: null,
          humidity: null,
          clouds: null,
          uvIndex: null,
          windSpeed: null,
          windDirection: null,
          weatherCode: null,
          pop: null,
        ),
      );

      expect(
        fullWeather.hourlyForecasts[0],
        HourlyForecast(
          date: date_time_utils.fromUtcUnixTime(location, 1685134800),
          unitSystem: UnitSystem.metric,
          temperature: null,
          weatherCode: null,
          pop: null,
          isDay: null,
        ),
      );

      final dailyForecast = DailyForecast(
        unitSystem: UnitSystem.metric,
        date: date_time_utils.fromUtcUnixTime(location, 1685055600),
        sunrise: null,
        sunset: null,
        weatherCode: null,
        minTemperature: null,
        maxTemperature: null,
        pop: null,
      );

      expect(fullWeather.dailyForecasts[0], dailyForecast);
      expect(fullWeather.currentDayForecast, dailyForecast);
    });
  });
}
