/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'package:clima/core/object.dart';
import 'package:clima/l10n/app_localizations.dart';
import 'package:clima/ui/state_notifiers/full_weather_state_notifier.dart' as w;
import 'package:clima/ui/utilities/constants.dart';
import 'package:clima/ui/utilities/weather_description.dart';
import 'package:flutter/material.dart';
import 'package:hooks_riverpod/hooks_riverpod.dart';
import 'package:sizer/sizer.dart';

class MainInfoWidget extends ConsumerWidget {
  const MainInfoWidget({super.key});

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final appLocalizations = AppLocalizations.of(context);

    final currentWeather = ref.watch(
      w.fullWeatherStateNotifierProvider.select(
        (state) => state.fullWeather!.currentWeather,
      ),
    );
    final currentDayForecast = ref.watch(
      w.fullWeatherStateNotifierProvider.select(
        (state) => state.fullWeather!.currentDayForecast,
      ),
    );
    final city = ref.watch(
      w.fullWeatherStateNotifierProvider.select(
        (state) => state.fullWeather!.city,
      ),
    );

    return Padding(
      padding: EdgeInsetsDirectional.only(
        top: MediaQuery.of(context).size.shortestSide > kTabletBreakpoint
            ? 8.h
            : 4.h,
        bottom: 4.h,
      ),
      child: Column(
        children: [
          Padding(
            padding: EdgeInsetsDirectional.only(bottom: 1.h),
            child: Text(
              // TODO: `toUpperCase` does not take into account the locale, so
              // it might be a good idea to stop uppercasing the city name
              // entirely.
              city.name.toUpperCase(),
              style: kSubtitle1TextStyle(context).copyWith(
                fontWeight: FontWeight.w900,
                letterSpacing: 5,
                fontSize:
                    MediaQuery.of(context).size.shortestSide > kTabletBreakpoint
                        ? 14.sp
                        : 20.sp,
              ),
            ),
          ),
          Padding(
            padding: EdgeInsetsDirectional.only(bottom: 1.h),
            child: Text(
              currentWeather.temperature?.let(
                    appLocalizations.weather_temperature_value,
                  ) ??
                  appLocalizations.weather_placeholder,
              maxLines: 1,
              style: kSubtitle1TextStyle(context).copyWith(
                fontSize:
                    MediaQuery.of(context).size.shortestSide < kTabletBreakpoint
                        ? 40.sp
                        : 30.sp,
                fontWeight: FontWeight.w100,
              ),
            ),
          ),
          Padding(
            padding: EdgeInsetsDirectional.only(bottom: 1.h),
            child: Row(
              mainAxisAlignment: MainAxisAlignment.center,
              children: [
                Icon(
                  Icons.keyboard_arrow_up,
                  size: kIconSize(context),
                  color: Theme.of(context).textTheme.titleMedium!.color,
                ),
                Padding(
                  padding: EdgeInsetsDirectional.only(end: 1.w),
                  child: Text(
                    currentDayForecast.maxTemperature
                            ?.let(appLocalizations.weather_temperature_value) ??
                        appLocalizations.weather_placeholder,
                    style: kSubtitle1TextStyle(context),
                  ),
                ),
                Padding(
                  padding: EdgeInsetsDirectional.only(start: 1.w),
                  child: Icon(
                    Icons.keyboard_arrow_down,
                    size: kIconSize(context),
                    color: Theme.of(context).textTheme.titleMedium!.color,
                  ),
                ),
                Text(
                  currentDayForecast.minTemperature
                          ?.let(appLocalizations.weather_temperature_value) ??
                      appLocalizations.weather_placeholder,
                  style: kSubtitle1TextStyle(context),
                ),
              ],
            ),
          ),
          Padding(
            padding: EdgeInsetsDirectional.only(top: 1.h),
            child: Text(
              currentWeather.weatherCode
                      ?.let((code) => getWeatherDescription(context, code)) ??
                  appLocalizations.weather_placeholder,
              textAlign: TextAlign.center,
              style: kSubtitle1TextStyle(context).copyWith(
                fontWeight: FontWeight.w300,
                letterSpacing: 5,
                fontSize:
                    MediaQuery.of(context).size.shortestSide < kTabletBreakpoint
                        ? 15.sp
                        : 10.sp,
              ),
            ),
          ),
        ],
      ),
    );
  }
}
