/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'package:clima/core/failure.dart';
import 'package:clima/core/use_case.dart';
import 'package:clima/domain/entities/city.dart';
import 'package:clima/domain/use_cases/get_city.dart';
import 'package:clima/domain/use_cases/set_city.dart';
import 'package:equatable/equatable.dart';
import 'package:meta/meta.dart';
import 'package:riverpod/riverpod.dart';

@immutable
sealed class CityState extends Equatable {
  const CityState();

  City? get city;
}

class Empty extends CityState {
  const Empty();

  @override
  City? get city => null;

  @override
  List<Object?> get props => const [];
}

class Loading extends CityState {
  const Loading({this.city});

  @override
  final City? city;

  @override
  List<Object?> get props => [city];
}

class Loaded extends CityState {
  const Loaded(this.city);

  @override
  final City? city;

  @override
  List<Object?> get props => [city];
}

class Error extends CityState {
  const Error(this.failure, {this.city});

  final Failure failure;

  @override
  final City? city;

  @override
  List<Object?> get props => [failure, city];
}

class CityStateNotifier extends StateNotifier<CityState> {
  CityStateNotifier(this.getCity, this.setCityUseCase) : super(const Empty());

  final GetCity getCity;

  final SetCity setCityUseCase;

  Future<void> loadCity() async {
    state = Loading(city: state.city);
    final data = await getCity(const NoParams());
    state = data.fold(Error.new, Loaded.new);
  }

  Future<void> setCity(String cityName) async {
    (await setCityUseCase(SetCityParams(cityName))).fold(
      (failure) {
        state = Error(failure, city: state.city);
      },
      (_) => loadCity(),
    );
  }
}

final cityStateNotifierProvider =
    StateNotifierProvider<CityStateNotifier, CityState>(
  (ref) =>
      CityStateNotifier(ref.watch(getCityProvider), ref.watch(setCityProvider)),
);
