/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'package:clima/data/providers.dart';
import 'package:clima/domain/entities/unit_system.dart';
import 'package:clima/l10n/app_localizations.dart';
import 'package:clima/ui/state_notifiers/city_state_notifier.dart' as c;
import 'package:clima/ui/state_notifiers/full_weather_state_notifier.dart' as w;
import 'package:clima/ui/state_notifiers/unit_system_state_notifier.dart' as u;
import 'package:clima/ui/utilities/constants.dart';
import 'package:clima/ui/utilities/hooks.dart';
import 'package:clima/ui/utilities/snack_bars.dart';
import 'package:clima/ui/utilities/styled_text.dart';
import 'package:clima/ui/widgets/others/failure_banner.dart';
import 'package:clima/ui/widgets/others/overflow_menu_button.dart';
import 'package:clima/ui/widgets/weather/additional_info_widget.dart';
import 'package:clima/ui/widgets/weather/daily_forecasts_widget.dart';
import 'package:clima/ui/widgets/weather/hourly_forecasts_widget.dart';
import 'package:clima/ui/widgets/weather/main_info_widget.dart';
import 'package:flutter/material.dart';
import 'package:flutter_hooks/flutter_hooks.dart';
import 'package:hooks_riverpod/hooks_riverpod.dart';
import 'package:intl/intl.dart';
import 'package:material_floating_search_bar_2/material_floating_search_bar_2.dart';
import 'package:sizer/sizer.dart';

class WeatherPage extends HookConsumerWidget {
  const WeatherPage({super.key});

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final appLocalizations = AppLocalizations.of(context);

    final fullWeatherState = ref.watch(w.fullWeatherStateNotifierProvider);
    final fullWeather = fullWeatherState.fullWeather;

    final fullWeatherStateNotifier =
        ref.watch(w.fullWeatherStateNotifierProvider.notifier);

    final controller = useFloatingSearchBarController();

    final cityStateNotifier = ref.watch(c.cityStateNotifierProvider.notifier);
    final cityState = ref.watch(c.cityStateNotifierProvider);

    final refreshIndicatorKey = useGlobalKey<RefreshIndicatorState>();

    final unitSystemStateNotifier =
        ref.watch(u.unitSystemStateNotifierProvider.notifier);

    final unitSystem = ref.watch(
      u.unitSystemStateNotifierProvider.select((state) => state.unitSystem),
    );

    useEffect(
      () {
        Future.microtask(
          () => Future.wait([
            cityStateNotifier.loadCity(),
            unitSystemStateNotifier.loadUnitSystem(),
            fullWeatherStateNotifier.loadFullWeather(),
          ]),
        );

        return null;
      },
      [cityStateNotifier, fullWeatherStateNotifier, unitSystemStateNotifier],
    );

    ref.listen(localeProvider, (_, __) {
      cityStateNotifier.loadCity();
    });

    final (child, :showSnackBarsForFailures) =
        switch ((fullWeatherState, cityState)) {
      (w.Error(fullWeather: null, :final failure), _) => (
          FailureBanner(
            failure: failure,
            onRetry: fullWeatherStateNotifier.loadFullWeather,
          ),
          showSnackBarsForFailures: false,
        ),
      (_, c.Loaded(city: null) || c.Error(city: null)) => (
          Center(
            child: Text(
              appLocalizations.weather_welcome,
              style: TextStyle(
                fontSize:
                    MediaQuery.of(context).size.shortestSide < kTabletBreakpoint
                        ? 13.sp
                        : 7.sp,
              ),
            ),
          ),
          showSnackBarsForFailures: true,
        ),
      ((w.FullWeatherState(fullWeather: _?)), _) => (
          Container(
            constraints: const BoxConstraints.expand(),
            child: SingleChildScrollView(
              physics: const BouncingScrollPhysics(),
              child: Column(
                children: [
                  const MainInfoWidget(),
                  Divider(
                    color: Theme.of(context)
                        .textTheme
                        .titleMedium!
                        .color!
                        .withAlpha(65),
                  ),
                  SizedBox(
                    height: () {
                      if (MediaQuery.of(context).size.shortestSide >
                          kTabletBreakpoint) {
                        return 18.h;
                      } else if (MediaQuery.of(context).size.aspectRatio ==
                          1.0) {
                        return 24.h;
                      } else {
                        return 17.h;
                      }
                    }(),
                    child: const HourlyForecastsWidget(),
                  ),
                  Divider(
                    color: Theme.of(context)
                        .textTheme
                        .titleMedium!
                        .color!
                        .withAlpha(65),
                  ),
                  const DailyForecastsWidget(),
                  Divider(
                    color: Theme.of(context)
                        .textTheme
                        .titleMedium!
                        .color!
                        .withAlpha(65),
                  ),
                  const AdditionalInfoWidget(),
                  Divider(
                    color: Theme.of(context)
                        .textTheme
                        .titleMedium!
                        .color!
                        .withAlpha(65),
                  ),
                  Padding(
                    padding: EdgeInsetsDirectional.symmetric(
                      vertical: 2.h,
                      horizontal: 5.w,
                    ),
                    child: StyledText(
                      appLocalizations.weather_bottomCredit,
                      textAlign: TextAlign.center,
                      style: kSubtitle2TextStyle(context),
                      urls: {
                        'open-meteo': appLocalizations.urls_openMeteo,
                        'open-meteo-data-sources':
                            appLocalizations.urls_openMeteoLicense,
                      },
                    ),
                  ),
                ],
              ),
            ),
          ),
          showSnackBarsForFailures: true,
        ),
      _ => (null, showSnackBarsForFailures: false),
    };

    if (showSnackBarsForFailures) {
      ref.listen(c.cityStateNotifierProvider, (_, state) {
        if (state is c.Error) {
          showFailureSnackBar(context, failure: state.failure, duration: 2);
        }
      });

      ref.listen(w.fullWeatherStateNotifierProvider, (_, state) {
        if (state is w.Error) {
          showFailureSnackBar(context, failure: state.failure, duration: 2);
        }
      });
    }

    return Scaffold(
      resizeToAvoidBottomInset: false,
      body: FloatingSearchAppBar(
        liftOnScrollElevation: 0.0,
        elevation: fullWeather == null ? 2.0 : 0.0,
        systemOverlayStyle: Theme.of(context).appBarTheme.systemOverlayStyle,
        automaticallyImplyBackButton: false,
        controller: controller,
        progress: fullWeatherState is w.Loading || cityState is c.Loading,
        accentColor: Theme.of(context).colorScheme.secondary,
        onSubmitted: (newCityName) async {
          controller.close();

          final trimmedCityName = newCityName.trim();
          if (trimmedCityName.isEmpty) {
            return;
          }

          await cityStateNotifier.setCity(trimmedCityName);
          if (ref.read(c.cityStateNotifierProvider) is! c.Error) {
            await fullWeatherStateNotifier.loadFullWeather();
          }
        },
        title: fullWeather == null
            // If we use `null` here, it would show the hint. We want it to
            // show nothing.
            ? const SizedBox.shrink()
            : Row(
                children: [
                  Text(
                    '${appLocalizations.weather_updated(
                      DateFormat.Md(
                        Localizations.localeOf(context).toLanguageTag(),
                      )
                          .addPattern(
                            MediaQuery.of(context).alwaysUse24HourFormat
                                ? 'Hm'
                                : 'h:mm a',
                          )
                          .format(
                            // We can't just use `toLocal` here, because by default
                            // `timezone` sets the local timezone to UTC, so
                            // `toLocal` basically acts the same as `toUtc` in
                            // this case. There _are_ ways to get the local
                            // timezone and set `timezone's` local timezone to that,
                            // but the workaround here is simpler.
                            DateTime.fromMicrosecondsSinceEpoch(
                              fullWeather
                                  .currentWeather.date.microsecondsSinceEpoch,
                            ),
                          ),
                    )} · ',
                    style: TextStyle(
                      color: Theme.of(context).textTheme.titleSmall!.color,
                      fontSize: MediaQuery.of(context).size.shortestSide <
                              kTabletBreakpoint
                          ? 11.sp
                          : 5.sp,
                    ),
                  ),
                  Text(
                    switch (unitSystem) {
                      UnitSystem.metric => appLocalizations.weather_metric,
                      UnitSystem.imperial => appLocalizations.weather_imperial,
                      null => ''
                    },
                    style: TextStyle(
                      color: Theme.of(context).textTheme.titleMedium!.color,
                      fontSize: MediaQuery.of(context).size.shortestSide <
                              kTabletBreakpoint
                          ? 11.sp
                          : 5.sp,
                    ),
                  ),
                ],
              ),
        hint: appLocalizations.weather_enterCityName,
        color: Theme.of(context).appBarTheme.backgroundColor,
        transitionCurve: Curves.easeInOut,
        leadingActions: [
          FloatingSearchBarAction.back(
            color: Theme.of(context).appBarTheme.iconTheme!.color,
          ),
        ],
        actions: [
          FloatingSearchBarAction(
            child: CircularButton(
              icon: Icon(
                Icons.search,
                color: Theme.of(context).appBarTheme.iconTheme!.color,
              ),
              tooltip: appLocalizations.weather_searchButtonTooltip,
              onPressed: () {
                controller.open();
              },
            ),
          ),
          const FloatingSearchBarAction(child: OverflowMenuButton()),
          FloatingSearchBarAction.searchToClear(
            color: Theme.of(context).appBarTheme.iconTheme!.color,
            showIfClosed: false,
            searchButtonSemanticLabel:
                appLocalizations.weather_searchButtonTooltip,
            clearButtonSemanticLabel:
                appLocalizations.weather_clearButtonTooltip,
          ),
        ],
        body: SafeArea(
          child: RefreshIndicator(
            key: refreshIndicatorKey,
            onRefresh: () async {
              await cityStateNotifier.loadCity();
              await fullWeatherStateNotifier.loadFullWeather();
            },
            color: Theme.of(context).textTheme.titleMedium!.color,
            child: Padding(
              padding: EdgeInsetsDirectional.symmetric(
                horizontal: () {
                  if (MediaQuery.of(context).orientation ==
                          Orientation.landscape &&
                      MediaQuery.of(context).size.shortestSide >
                          kTabletBreakpoint) {
                    return 10.0.w;
                  } else if (MediaQuery.of(context).orientation ==
                          Orientation.landscape &&
                      MediaQuery.of(context).size.shortestSide <
                          kTabletBreakpoint) {
                    return 30.0.w;
                  } else {
                    return 5.0.w;
                  }
                }(),
              ),
              child: child,
            ),
          ),
        ),
      ),
    );
  }
}
