/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'package:clima/data/models/dark_theme_model.dart';
import 'package:clima/data/models/theme_model.dart';
import 'package:clima/data/providers.dart';
import 'package:clima/data/repos/city_repo_impl.dart';
import 'package:clima/data/repos/full_weather_repo.dart';
import 'package:clima/data/repos/unit_system_repo_impl.dart';
import 'package:clima/domain/repos/city_repo.dart';
import 'package:clima/domain/repos/full_weather_repo.dart';
import 'package:clima/domain/repos/unit_system_repo.dart';
import 'package:clima/l10n/app_localizations.dart';
import 'package:clima/ui/pages/weather_page.dart';
import 'package:clima/ui/state_notifiers/theme_state_notifier.dart'
    show themeStateNotifierProvider;
import 'package:clima/ui/state_notifiers/theme_state_notifier.dart' as t;
import 'package:clima/ui/themes/black_theme.dart';
import 'package:clima/ui/themes/dark_theme.dart';
import 'package:clima/ui/themes/light_theme.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:flutter_hooks/flutter_hooks.dart';
import 'package:hooks_riverpod/hooks_riverpod.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:sizer/sizer.dart';
import 'package:timezone/timezone.dart' as tz;

import 'ui/build_flavor.dart';
import 'ui/themes/clima_theme.dart';

Future<void> main({
  TransitionBuilder? builder,
  Widget Function(Widget widget)? topLevelBuilder,
  Locale? Function(BuildContext)? getLocale,
}) async {
  // Unless you do this, using method channels (like `SharedPreferences` does)
  // before running `runApp` throws an error.
  WidgetsFlutterBinding.ensureInitialized();

  tz.initializeDatabase(
    Uint8List.sublistView(
      await rootBundle.load('packages/timezone/data/latest_all.tzf'),
    ),
  );

  final sharedPreferences = await SharedPreferences.getInstance();

  final widget = ProviderScope(
    overrides: [
      sharedPreferencesProvider.overrideWithValue(sharedPreferences),
      cityRepoProvider.overrideWith((ref) => ref.watch(cityRepoImplProvider)),
      unitSystemRepoProvider
          .overrideWith((ref) => ref.watch(unitSystemRepoImplProvider)),
      fullWeatherRepoProvider
          .overrideWith((ref) => ref.watch(fullWeatherRepoImplProvider)),
    ],
    child: _App(builder: builder, getLocale: getLocale),
  );

  runApp(topLevelBuilder?.call(widget) ?? widget);
}

class _App extends HookConsumerWidget {
  const _App({this.builder, this.getLocale});

  final TransitionBuilder? builder;

  final Locale? Function(BuildContext)? getLocale;

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    validateBuildFlavor();

    final themeStateNotifier = ref.watch(themeStateNotifierProvider.notifier);

    useEffect(
      () {
        Future.microtask(themeStateNotifier.loadTheme);

        return null;
      },
      [themeStateNotifier],
    );

    final themeState = ref.watch(themeStateNotifierProvider);

    if (themeState is t.EmptyState || themeState is t.Loading) {
      return const SizedBox.shrink();
    }

    return Sizer(
      builder: (context, orientation, screenType) => MaterialApp(
        locale: getLocale?.call(context),
        localizationsDelegates: AppLocalizations.localizationsDelegates,
        // The app is translated to Arabic but it's still got some rough edges,
        // so leave it out for now.
        //
        // Make sure to keep English as the first locale, because the first
        // locale is used as the fallback one in case no other locale matches
        // the user's language.
        supportedLocales: const [
          Locale('en'),
          Locale('de'),
          Locale('nl'),
          Locale('fr'),
        ],
        onGenerateTitle: (context) => AppLocalizations.of(context).appName,
        builder: (context, child) {
          final climaTheme =
              switch ((Theme.of(context).brightness, themeState.darkTheme!)) {
            (Brightness.light, _) => lightClimaTheme,
            (Brightness.dark, DarkThemeModel.black) => blackClimaTheme,
            (Brightness.dark, DarkThemeModel.darkGrey) => darkGreyClimaTheme,
          };

          child = ClimaTheme(data: climaTheme, child: child!);

          return builder?.call(context, child) ?? child;
        },
        home: HookConsumer(
          builder: (context, ref, child) {
            final locale = Localizations.localeOf(context);

            useEffect(
              () {
                Future.microtask(
                  () => ref.watch(localeProvider.notifier).state = locale,
                );

                return null;
              },
              [locale],
            );

            if (ref.watch(localeProvider) == null) {
              return const SizedBox.shrink();
            } else {
              return child!;
            }
          },
          child: const WeatherPage(),
        ),
        theme: lightTheme,
        darkTheme: switch (themeState.darkTheme!) {
          DarkThemeModel.black => blackTheme,
          DarkThemeModel.darkGrey => darkGreyTheme,
        },
        themeMode: switch (themeState.theme!) {
          ThemeModel.systemDefault => ThemeMode.system,
          ThemeModel.light => ThemeMode.light,
          ThemeModel.dark => ThemeMode.dark,
        },
      ),
    );
  }
}
