/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'package:clima/core/object.dart';
import 'package:clima/domain/entities/unit_system.dart';
import 'package:clima/domain/utils/unit_conversion.dart';
import 'package:equatable/equatable.dart';

class HourlyForecast extends Equatable {
  const HourlyForecast({
    required this.date,
    required this.temperature,
    required this.pop,
    required this.weatherCode,
    required this.unitSystem,
    required this.isDay,
  });

  final DateTime date;

  final double? temperature;

  /// Probability of precipitation.
  final double? pop;

  final int? weatherCode;

  final UnitSystem unitSystem;

  final bool? isDay;

  @override
  List<Object?> get props =>
      [date, temperature, pop, weatherCode, unitSystem, isDay];

  HourlyForecast changeUnitSystem(UnitSystem newUnitSystem) {
    if (unitSystem == newUnitSystem) {
      return this;
    }

    final newTemperature = switch (unitSystem) {
      UnitSystem.imperial => temperature?.let(convertFahrenheitToCelsius),
      UnitSystem.metric => temperature?.let(convertCelsiusToFahrenheit),
    };

    return HourlyForecast(
      pop: pop,
      date: date,
      temperature: newTemperature,
      weatherCode: weatherCode,
      unitSystem: newUnitSystem,
      isDay: isDay,
    );
  }
}
