/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'package:clima/core/either.dart';
import 'package:clima/core/failure.dart';
import 'package:clima/data/data_sources/city_local_data_source.dart';
import 'package:clima/data/data_sources/city_remote_data_source.dart';
import 'package:clima/data/providers.dart';
import 'package:clima/domain/entities/city.dart';
import 'package:clima/domain/repos/city_repo.dart';
import 'package:riverpod/riverpod.dart';

class CityRepoImpl implements CityRepo {
  CityRepoImpl(
    this.ref, {
    required this.localDataSource,
    required this.remoteDataSource,
  });

  final ProviderRef<Object?> ref;

  final CityLocalDataSource localDataSource;

  final CityRemoteDataSource remoteDataSource;

  @override
  Future<Either<Failure, City?>> getCity() async {
    final cityEither =
        (await localDataSource.getCity()).map((cityModel) => cityModel?.city);

    if (cityEither is Left) {
      return cityEither;
    }

    final city = (cityEither as Right<Failure, City?>).value;

    if (city == null) {
      return cityEither;
    }

    final locale = ref.read(localeProvider)!;

    if (city.language != locale.toLanguageTag()) {
      final newCityEither =
          (await remoteDataSource.getCityById(city.id, locale: locale))
              .map((cityModel) => cityModel.city);

      if (newCityEither is Left) {
        // Return the city in the previous locale at least.
        return cityEither;
      }

      final newCity = (newCityEither as Right<Failure, City>).value;

      await localDataSource.setCity(newCity);

      return newCityEither;
    }

    return cityEither;
  }

  @override
  Future<Either<Failure, void>> setCity(String cityName) async {
    final cityModel = await remoteDataSource.getCityByName(
      cityName,
      locale: ref.read(localeProvider)!,
    );

    return cityModel.fold(
      (failure) async => Left(failure),
      (model) => localDataSource.setCity(model.city),
    );
  }
}

final cityRepoImplProvider = Provider(
  (ref) => CityRepoImpl(
    ref,
    localDataSource: ref.watch(cityLocalDataSourceProvider),
    remoteDataSource: ref.watch(cityRemoteDataSourceProvider),
  ),
);
