/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'dart:convert';
import 'dart:ui';

import 'package:clima/core/either.dart';
import 'package:clima/core/failure.dart';
import 'package:clima/data/models/city_model.dart';
import 'package:http/http.dart' as http;
import 'package:riverpod/riverpod.dart';

class CityRemoteDataSource {
  Future<Either<Failure, CityModel>> getCityById(
    int cityId, {
    required Locale locale,
  }) async {
    final response = await http.get(
      Uri.https(
        'geocoding-api.open-meteo.com',
        'v1/get',
        {'id': '$cityId', 'language': locale.toLanguageTag()},
      ),
    );

    Object? body;

    try {
      body = jsonDecode(response.body);
    } on FormatException {
      return const Left(FailedToParseResponse());
    }

    if (body is! Map<String, dynamic>) {
      return const Left(FailedToParseResponse());
    }

    if (body['error'] == true) {
      return Left(ServerError(reason: body['reason'] as String?));
    }

    try {
      return Right(CityModel.fromJson(body, language: locale.toLanguageTag()));
    } catch (_) {
      return const Left(FailedToParseResponse());
    }
  }

  Future<Either<Failure, CityModel>> getCityByName(
    String cityName, {
    required Locale locale,
  }) async {
    final response = await http.get(
      Uri.https(
        'geocoding-api.open-meteo.com',
        'v1/search',
        {
          'name': cityName,
          'language': locale.toLanguageTag(),
          'count': '1',
        },
      ),
    );

    Object? body;

    try {
      body = jsonDecode(response.body);
    } on FormatException {
      return const Left(FailedToParseResponse());
    }

    if (body is! Map<String, dynamic>) {
      return const Left(FailedToParseResponse());
    }

    if (body['error'] == true) {
      return Left(ServerError(reason: body['reason'] as String?));
    } else if (body['results'] == null) {
      // XXX: API returns 200 in this case, so we have to handle this separately.
      return Left(InvalidCityName(cityName));
    }

    try {
      return Right(
        CityModel.fromJson(
          // ignore: avoid_dynamic_calls
          body['results'][0] as Map<String, dynamic>,
          language: locale.toLanguageTag(),
        ),
      );
    } catch (_) {
      return const Left(FailedToParseResponse());
    }
  }
}

final cityRemoteDataSourceProvider = Provider((ref) => CityRemoteDataSource());
