/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

import 'package:clima/core/either.dart';
import 'package:clima/core/failure.dart';
import 'package:clima/data/models/city_model.dart';
import 'package:clima/data/providers.dart';
import 'package:clima/domain/entities/city.dart';
import 'package:riverpod/riverpod.dart';
import 'package:shared_preferences/shared_preferences.dart';

const _cityNameKey = 'city_name';
const _cityIdKey = 'city_id';
const _cityLanguageKey = 'city_language';
const _cityLongKey = 'city_long';
const _cityLatKey = 'city_lat';

class CityLocalDataSource {
  CityLocalDataSource(this._prefs);

  final SharedPreferences _prefs;

  Future<Either<Failure, CityModel?>> getCity() async {
    final name = _prefs.getString(_cityNameKey);
    final id = _prefs.getInt(_cityIdKey);
    final language = _prefs.getString(_cityLanguageKey);
    final long = _prefs.getDouble(_cityLongKey);
    final lat = _prefs.getDouble(_cityLatKey);

    if (name == null ||
        id == null ||
        language == null ||
        long == null ||
        lat == null) {
      return const Right(null);
    }

    return Right(
      CityModel(
        City(
          id: id,
          name: name,
          language: language,
          long: long,
          lat: lat,
        ),
      ),
    );
  }

  Future<Either<Failure, void>> setCity(City city) async {
    await Future.wait([
      _prefs.setString(_cityNameKey, city.name),
      _prefs.setInt(_cityIdKey, city.id),
      _prefs.setString(_cityLanguageKey, city.language),
      _prefs.setDouble(_cityLongKey, city.long),
      _prefs.setDouble(_cityLatKey, city.lat),
    ]);

    return const Right(null);
  }
}

final cityLocalDataSourceProvider = Provider(
  (ref) => CityLocalDataSource(ref.watch(sharedPreferencesProvider)),
);
