package ca.chancehorizon.paseo


import android.Manifest
import android.app.Activity
import android.app.Activity.RESULT_OK
import android.content.Context
import android.content.Intent
import android.content.SharedPreferences
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.os.Environment
import android.os.PowerManager
import android.provider.Settings
import android.speech.tts.TextToSpeech
import android.speech.tts.Voice
import android.util.Log
import android.view.View
import android.widget.FrameLayout
import android.widget.TextView
import android.widget.Toast
import androidx.activity.result.ActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.core.net.toUri
import androidx.preference.DropDownPreference
import androidx.preference.Preference
import androidx.preference.PreferenceCategory
import androidx.preference.PreferenceFragmentCompat
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetDialog
import java.io.*
import java.text.SimpleDateFormat
import java.util.*


class PrefsFragment : PreferenceFragmentCompat(), TextToSpeech.OnInitListener {

    private var tts: TextToSpeech? = null
    private var ttsAvailable = false

    private val WRITEREQUESTCODE = 102
    private val READREQUESTCODE = 103
    private val FOLDERSELECTREQUESTCODE = 104


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        tts = TextToSpeech(context, this)

        val paseoPrefs = context?.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)
        ttsAvailable = paseoPrefs!!.getBoolean("prefTTSAvailable", false)

        if (ttsAvailable) {
            val result = tts?.setLanguage(Locale.US)
            if (result == TextToSpeech.LANG_MISSING_DATA ||
                    result == TextToSpeech.LANG_NOT_SUPPORTED) {
                Log.e("TTS", "Language not supported")
            }
            val voiceLanguages : DropDownPreference? = findPreference("prefVoiceLanguage")

            // only populate the list of voices if tts is available on the device
            if (tts != null && voiceLanguages != null) {
                setListPreferenceData(voiceLanguages)
            }
        }
        else {
            Log.e("TTS", "Initialization failed")

            // disable all the settings for text to speech
            val ttsItems : PreferenceCategory? = findPreference("textToSpeech")
            ttsItems?.isEnabled = false

            ttsItems?.summary = getString(R.string.no_TTS)
        }

        setBatteryOptimizationText(requireContext())
    }



    override fun onCreatePreferences(savedInstanceState: Bundle?, rootKey: String?) {
        setPreferencesFromResource(R.xml.paseo_settings, rootKey)
    }



    override fun onInit(status: Int) {

        if (status == TextToSpeech.SUCCESS) {
            val result = tts?.setLanguage(Locale.US)
            if (result == TextToSpeech.LANG_MISSING_DATA ||
                    result == TextToSpeech.LANG_NOT_SUPPORTED) {
                Log.e("TTS", "Language not supported")
            }
            val voiceLanguages : DropDownPreference? = findPreference("prefVoiceLanguage")

            // only populate the list of voices if tts is available on the device
            if (tts != null) {
                setListPreferenceData(voiceLanguages!!)
            }

            // display the backup folder if set
            if (true) {
                val paseoPrefs = context?.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

                // retrieve the user set backup folder from the shared preferences
                val theBackupFolder = paseoPrefs!!.getString("theBackupFolder", "/storage/emulated/0/Download/")

                // show the backup folder
                val backupFolder: Preference? = findPreference("prefBackupFolder")
                backupFolder?.summary = theBackupFolder
            }
        }
    }



    override fun onResume() {
        super.onResume()

        // need this here in order to change the text that shows th state of battery optimization after the user has set it
        setBatteryOptimizationText(requireContext())
    }



    override fun onDestroy() {

        // Shutdown TTS
        if (tts != null) {
            tts!!.stop()
            tts!!.shutdown()
        }

        super.onDestroy()
    }



    @RequiresApi(Build.VERSION_CODES.R)
    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)

        if (requestCode == FOLDERSELECTREQUESTCODE && resultCode == RESULT_OK) {

            val dbFile: File = requireContext().getDatabasePath("paseoDB.db")

            // only do the backup if the backup location exists and the database file exists
            if (data != null && dbFile.exists()) {
                // set default name of exported file
                val theDate = SimpleDateFormat("yyyyMMdd-HHmmss", Locale.getDefault()).format(Date())

                val file = data?.data?.getPath() //create path from uri
                var backupFolder = file?.replace("/tree/", "/storage/")
                backupFolder = backupFolder?.replace(":", "/")

                // save the user selected backup folder in the app's preferences
                val paseoPrefs = context?.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)
                val edit: SharedPreferences.Editor = paseoPrefs!!.edit()
                edit.putString("theBackupFolder", backupFolder)
                edit.apply()

                // show the backup folder in the settings screen
                val backupFolderPref: Preference? = findPreference("prefBackupFolder")
                backupFolderPref?.summary = backupFolder

                var canSelectFolder = false

                // need to check for write to external storage permission on Android > 10 (api 29+)
                if (Build.VERSION.SDK_INT > 29) {
                    if (Environment.isExternalStorageManager()) {
                        canSelectFolder = true
                    }
                }
                else {
                    canSelectFolder = true
                }

                if (canSelectFolder) {
                    // The app has been granted the MANAGE_EXTERNAL_STORAGE permission
                    try {
                        dbFile.copyTo(File(backupFolder, "paseoDB$theDate.db"), overwrite = true)
                    }
                    catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
                else {
                    // The app has not been granted the MANAGE_EXTERNAL_STORAGE permission
                    // Request the permission from the user
                    val intent = Intent(Settings.ACTION_MANAGE_APP_ALL_FILES_ACCESS_PERMISSION)
                    intent.data = Uri.parse("package:" + requireContext().packageName)
                    startActivity(intent)
                }
            }
        }
    }



    // populate the list of available languages for text to speech
    fun setListPreferenceData(ttsLanguages : DropDownPreference) {

        // only populate the list of voice if tts is available on the device
        if (tts == null || tts!!.voice == null) {
            return
        }

        val availableVoices = tts!!.voices

        val availableLocales:List<Locale> = Locale.getAvailableLocales().toList()

        val voiceList = mutableListOf<String>()

        // only continue if voices have been found.
        if (!availableVoices.isNullOrEmpty()) {
            // loop through all the voices and create a list of them
            for (v: Voice in availableVoices) {
                if (v.locale.language == Locale.getDefault().language
                        && availableLocales.contains(v.locale)
                        && !v.isNetworkConnectionRequired
                        && tts?.isLanguageAvailable(v.locale) != TextToSpeech.LANG_MISSING_DATA
                        && tts?.isLanguageAvailable(v.locale) != TextToSpeech.LANG_NOT_SUPPORTED
                        && !(v.features.contains(TextToSpeech.Engine.KEY_FEATURE_NOT_INSTALLED))) {
                    voiceList.add(v.name)
                }
            }

            if (!voiceList.isEmpty()) {
                // sort the voices list so that similar voices (by country) are listed together
                voiceList.sort()

                val voiceArray = voiceList.toTypedArray()

                ttsLanguages.entries = voiceArray

                val paseoPrefs = context?.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

                // if no voice has been set by user (store in shared prefs) then just use the first one from the list
                val ttsVoice = paseoPrefs!!.getString("prefVoiceLanguage", voiceList[0])
                val savedVoice = voiceArray.indexOf(ttsVoice)

                // populate the voices popup list
                ttsLanguages.entryValues = voiceArray

                // if the saved voice is not in the list, just use the first one
                //  this can happen if the locale on the device has been changed since the user set the voice in Paseo
                if (savedVoice < 0) {
                    ttsLanguages.setValueIndex(0)
                }
                else {
                    ttsLanguages.setValueIndex(savedVoice)
                }
            }
        }
    }



    // respond to the user tapping on specific preferences
    override fun onPreferenceTreeClick(preference: Preference): Boolean {

        val key = preference.key

        // play a test message to test if voice settings are suitable
        if (key == "prefTestVoiceButton") {

            if (!ttsAvailable) {
                Toast.makeText(context,getString(R.string.NoTTSWarning), Toast.LENGTH_LONG).show()
            }
            else {
                val theText = getString(R.string.SampleTTSText)
                val paseoPrefs = context?.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

                val ttsPitch = paseoPrefs!!.getFloat("prefVoicePitch", 100F)
                val ttsRate = paseoPrefs.getFloat("prefVoiceRate", 100F)

                // set the voice to use to speak with
                val ttsVoice = paseoPrefs.getString("prefVoiceLanguage", "en_US - en-US-language")
                val ttsLocale1 = ttsVoice!!.substring(0, 2)
                val ttsLocale2 = ttsVoice.substring(3)
                val voiceobj = Voice(ttsVoice, Locale(ttsLocale1, ttsLocale2), 1, 1, false, null)
                tts?.voice = voiceobj

                tts?.setPitch(ttsPitch / 100)
                tts?.setSpeechRate(ttsRate / 100)

                var attemptSpeech = tts?.speak(theText, TextToSpeech.QUEUE_FLUSH, null, "")

                if (attemptSpeech == -1) {
                    tts = TextToSpeech(context, this)

                    attemptSpeech = tts?.speak(theText, TextToSpeech.QUEUE_FLUSH, null, "")
                }
            }
        }

        // Export and Backup stuffs
        if (key == "prefExport" || key == "prefAutoBackup" /*|| key == "prefBackupFolder"*/) {
            val permission = ContextCompat.checkSelfPermission(requireContext(),
                    Manifest.permission.WRITE_EXTERNAL_STORAGE)
            if (android.os.Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.TIRAMISU &&
                permission != PackageManager.PERMISSION_GRANTED) {
                Log.i("TAG", "Permission to write denied")

                if (ActivityCompat.shouldShowRequestPermissionRationale(requireActivity(), Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
                    val builder = AlertDialog.Builder(requireActivity())
                    builder.setMessage(R.string.write_permission_last_warning)
                            .setTitle(R.string.permission_warning_title)

                    builder.setPositiveButton("OK") { dialog, id ->
                        ActivityCompat.requestPermissions(this.requireActivity(), arrayOf(Manifest.permission.WRITE_EXTERNAL_STORAGE), WRITEREQUESTCODE)
                    }

                    val dialog = builder.create()
                    dialog.show()
                } else {
                    ActivityCompat.requestPermissions(this.requireActivity(), arrayOf(Manifest.permission.WRITE_EXTERNAL_STORAGE), WRITEREQUESTCODE)
                }
            }

            // save export file only when user selects export in settings screen (do not continue here for autobackup)
            else if (key == "prefExport") {

                // need to use storage access framework for Android >= 10
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {

                    val intent = Intent(Intent.ACTION_CREATE_DOCUMENT)

                    // the type of file we want to pick; for now it's all
                    intent.type = "application/db"

                    // set default name of exported file
                    val theDate = SimpleDateFormat("yyyyMMdd-HHmmss", Locale.getDefault()).format(Date())
                    val exportFile = "paseoDB$theDate.db"
                    intent.putExtra(Intent.EXTRA_TITLE, exportFile)

                    // perform the exporting (copying of paseo database to shared storage)
                    startExportDBForResult.launch(intent)
                } else {
                    val dbFile = File("/data/data/ca.chancehorizon.paseo/databases/paseoDB.db")
                    if (dbFile.exists()) {
                        File("/storage/emulated/0/Download/paseoDB.db").delete()
                        dbFile.copyTo(File("/storage/emulated/0/Download/paseoDB.db"), overwrite = true)
                        Toast.makeText(context, getText(R.string.exportedDetails), Toast.LENGTH_LONG).show()
                    }
                }
            }
        }
        // set the export/backup folder
        else if (key == "prefBackupFolder") {
            val intent = Intent(Intent.ACTION_OPEN_DOCUMENT_TREE).apply {
                addFlags(
                        Intent.FLAG_GRANT_PERSISTABLE_URI_PERMISSION
                                or Intent.FLAG_GRANT_WRITE_URI_PERMISSION
                                or Intent.FLAG_GRANT_READ_URI_PERMISSION
                                or Intent.FLAG_GRANT_PREFIX_URI_PERMISSION)
                putExtra("android.content.extra.SHOW_ADVANCED", true)
                putExtra("android.content.extra.FANCY", true)
                putExtra("android.content.extra.SHOW_FILESIZE", true)
            }

            startActivityForResult(Intent.createChooser(intent, "Select a file"), FOLDERSELECTREQUESTCODE)

        }

        // Import
        if (key == "prefImport") {
            val permission = ContextCompat.checkSelfPermission(requireContext(),
                    Manifest.permission.READ_EXTERNAL_STORAGE)
            if (android.os.Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.TIRAMISU &&
                permission != PackageManager.PERMISSION_GRANTED) {
                Log.i("TAG", "Permission to write denied")

                if (ActivityCompat.shouldShowRequestPermissionRationale(requireActivity(), Manifest.permission.READ_EXTERNAL_STORAGE)) {
                    val builder = AlertDialog.Builder(requireActivity())
                    builder.setMessage(R.string.read_permission_last_warning)
                        .setTitle(R.string.permission_warning_title)

                    builder.setPositiveButton("OK") { dialog, id ->
                        ActivityCompat.requestPermissions(this.requireActivity(), arrayOf(Manifest.permission.READ_EXTERNAL_STORAGE), READREQUESTCODE)
                    }

                    val dialog = builder.create()
                    dialog.show()
                } else {
                    ActivityCompat.requestPermissions(this.requireActivity(), arrayOf(Manifest.permission.READ_EXTERNAL_STORAGE), READREQUESTCODE)
                }
            }
            else {
                // set up the path to this application's files

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {

                    val intent = Intent(Intent.ACTION_OPEN_DOCUMENT)
                    intent.addCategory(Intent.CATEGORY_OPENABLE)
                    // the type of file we want to pick; for now it's all
                    intent.type = "*/*"

                    startImportDBForResult.launch(intent)
                }
                else {
                    val dbFile = File("/storage/emulated/0/Download/paseoDB.db")
                    if (dbFile.exists()) {
                        try {
                            dbFile.copyTo(File("/data/data/ca.chancehorizon.paseo/databases/paseoDB.db"), overwrite = true)
                        } catch (e: Error) {
                            Toast.makeText(context, e.message, Toast.LENGTH_LONG).show()
                        }

                        Toast.makeText(context, getText(R.string.importedDetails), Toast.LENGTH_LONG).show()
                    } else {
                        Toast.makeText(context, getText(R.string.importedFailed), Toast.LENGTH_LONG).show()
                    }
                }
            }
        }

        // let user set battery optimization for Paseo
        if (key == "prefBatteryOptimization") {
            setBatteryOptimization(requireActivity())
            setBatteryOptimizationText(requireActivity())
        }

        // show the paseo about bottomsheet (which is also shown on "first run"
        if (key == "prefAbout") { // do your work
            val view: View = layoutInflater.inflate(R.layout.paseo_welcome_bottomsheet, null)

            val aboutPaseoBottomSheet = BottomSheetDialog(requireContext())
            aboutPaseoBottomSheet.setContentView(view)
            aboutPaseoBottomSheet.show()
            val bottomSheet = aboutPaseoBottomSheet.findViewById<View>(R.id.design_bottom_sheet) as FrameLayout
            val bottomSheetBehavior = BottomSheetBehavior.from(bottomSheet)
            bottomSheetBehavior.state = BottomSheetBehavior.STATE_EXPANDED

            // show the app version of Paseo in the about bottom sheet
            try {
                val pInfo = requireContext().packageManager.getPackageInfo(requireContext().packageName, 0)
                val paseoVersion = "v" + pInfo.versionName
                val versionTextView = view.findViewById<TextView>(R.id.version)
                versionTextView.text = paseoVersion
            } catch (e: PackageManager.NameNotFoundException) {
                e.printStackTrace()
            }

        }

        // show the paseo license bottomsheet
        if (key == "prefLicense") {
            val view: View = layoutInflater.inflate(R.layout.paseo_license_bottomsheet, null)

            val aboutPaseoBottomSheet = BottomSheetDialog(requireContext())
            aboutPaseoBottomSheet.setContentView(view)
            aboutPaseoBottomSheet.show()
            val bottomSheet = aboutPaseoBottomSheet.findViewById<View>(R.id.design_bottom_sheet) as FrameLayout
            val bottomSheetBehavior = BottomSheetBehavior.from(bottomSheet)
            bottomSheetBehavior.state = BottomSheetBehavior.STATE_EXPANDED
        }

        return super.onPreferenceTreeClick(preference)
    }



    fun setBatteryOptimizationText(mContext: Context)
    {
        val powerManager =
                mContext.getSystemService(AppCompatActivity.POWER_SERVICE) as PowerManager
        val packageName = mContext.packageName

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            if (!powerManager.isIgnoringBatteryOptimizations(packageName)) {
                // show that battery optimization is ON for Paseo
                val isBatteryOptimized : Preference? = findPreference("isBatteryOptimized")
                isBatteryOptimized?.summary = getString(R.string.battery_opt_summary_on)
                val battOptItem2 : Preference? = findPreference("prefBatteryOptimization")
                battOptItem2?.isVisible = true
                battOptItem2?.summary = getString(R.string.battery_opt_summary_2)
            }
            else {
                // show that battery optimization is OFF for Paseo
                val isBatteryOptimized: Preference? = findPreference("isBatteryOptimized")
                isBatteryOptimized?.summary = getString(R.string.battery_opt_summary_off)
                val battOptItem2: Preference? = findPreference("prefBatteryOptimization")
                battOptItem2?.isVisible = false
            }
        }
    }



    // let user set Android's battery optimization setting for Paseo
    private fun setBatteryOptimization(mContext: Context) {

        val packageName = mContext.packageName
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            val batterySettingIntent = Intent()

                batterySettingIntent.action = Settings.ACTION_REQUEST_IGNORE_BATTERY_OPTIMIZATIONS

            batterySettingIntent.data = Uri.parse("package:$packageName")
            batterySettingIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)

            try {
                mContext.startActivity(batterySettingIntent)
            }
            catch (e: Error) {
                Toast.makeText(context, e.message, Toast.LENGTH_LONG).show()
            }
            finally {
                setBatteryOptimizationText(mContext)
            }
        }
    }



    // copy paseo's steps database to a shared storage location
    val startExportDBForResult = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) {
        result: ActivityResult ->
        if (result.resultCode == Activity.RESULT_OK) {
            exportDBFile(result)
        }
    }



    private fun exportDBFile(result: ActivityResult) {
        val intent = result.data

        // get the steps database
        val dbFile: File = requireActivity().getDatabasePath("paseoDB.db")

        // only export the paseo database if it actually exists
        if (dbFile.exists()) {
            File(intent.toString()).delete()
            try {
                try {
                    val uri: Uri = intent?.getData()!!
                    val outputStream: OutputStream? = getActivity()?.getContentResolver()?.openOutputStream(uri)
                    val inputStream: InputStream? = getActivity()?.getContentResolver()?.openInputStream(dbFile.toUri())

                    val buffer = ByteArray(1024)
                    var read: Int

                    while (inputStream?.read(buffer).also { read = it!! } !== -1) {
                        outputStream?.write(buffer, 0, read)
                    }
                    inputStream?.close()
                    outputStream?.flush()
                    outputStream?.close()

                    Toast.makeText(context, "Paseo database exported successfully", Toast.LENGTH_SHORT).show()
                } catch (e: IOException) {
                    Toast.makeText(context, "Export failed", Toast.LENGTH_SHORT).show()
                }
            } catch (e: Error) {
                Toast.makeText(context, e.message, Toast.LENGTH_LONG).show()
            }
        }
    }



    // copy an exported paseo database to paseo's private storage (overwriting the currently used one)
    val startImportDBForResult = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) {
        result: ActivityResult ->
        if (result.resultCode == Activity.RESULT_OK) {
            val intent = result.data

            // get the steps database
            val dbFile: File = requireActivity().getDatabasePath("paseoDB.db")

//            val dbFile = File("/data/data/ca.chancehorizon.paseo/databases/paseoDB.db")
            if (dbFile.exists()) {
                try {
                    try {
                        val uri: Uri = intent?.getData()!!
                        val inputStream: InputStream? = activity?.contentResolver?.openInputStream(uri)
                        val outputStream: OutputStream? = activity?.contentResolver?.openOutputStream(dbFile.toUri())

                        val buffer = ByteArray(1024)
                        var read: Int

                        while (inputStream?.read(buffer).also { read = it!! } !== -1) {
                            outputStream?.write(buffer, 0, read)
                        }
                        inputStream?.close()
                        outputStream?.flush()
                        outputStream?.close()

                        Toast.makeText(context, "Import of Paseo database succeeded", Toast.LENGTH_SHORT).show()
                    } catch (e: IOException) {
                        Toast.makeText(context, "Failed to import database", Toast.LENGTH_SHORT).show()
                    }
                }
                catch (e: Error) {
                    Toast.makeText(context, e.message, Toast.LENGTH_LONG).show()
                }
            }
        }
    }



    override fun onDisplayPreferenceDialog(preference: Preference) {
        val theDialog = preference as? OptionDialogPreference

        if (theDialog != null) {
            val dialogFragment = DialogPrefCompat.newInstance(theDialog.key)

            dialogFragment.setTargetFragment(this, 0)

            dialogFragment.positiveResult = {}

            val theFragmentManager = getParentFragmentManager()
            dialogFragment.show(theFragmentManager, null)
        }
        else
        {
            super.onDisplayPreferenceDialog(preference)
        }
    }
}