/*
 * This file is part of Shosetsu.
 *
 * Shosetsu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Shosetsu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Shosetsu.  If not, see <https://www.gnu.org/licenses/>.
 *
 */

package app.shosetsu.android.view.compose

import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.LocalMinimumTouchTargetEnforcement
import androidx.compose.ui.Modifier
import androidx.compose.ui.composed
import androidx.compose.ui.layout.LayoutModifier
import androidx.compose.ui.layout.Measurable
import androidx.compose.ui.layout.MeasureResult
import androidx.compose.ui.layout.MeasureScope
import androidx.compose.ui.platform.LocalViewConfiguration
import androidx.compose.ui.platform.debugInspectorInfo
import androidx.compose.ui.unit.Constraints
import androidx.compose.ui.unit.DpSize
import kotlin.math.roundToInt

@OptIn(ExperimentalMaterial3Api::class)
@Suppress("ModifierInspectorInfo")
fun Modifier.minimumTouchTargetSize(): Modifier = composed(
	inspectorInfo = debugInspectorInfo {
		name = "minimumTouchTargetSize"
		properties["README"] = "Adds outer padding to measure at least 48.dp (default) in " +
				"size to disambiguate touch interactions if the element would measure smaller"
	},
) {
	if (LocalMinimumTouchTargetEnforcement.current) {
		val size = LocalViewConfiguration.current.minimumTouchTargetSize
		MinimumTouchTargetModifier(size)
	} else {
		Modifier
	}
}

private class MinimumTouchTargetModifier(val size: DpSize) : LayoutModifier {
	override fun MeasureScope.measure(
		measurable: Measurable,
		constraints: Constraints,
	): MeasureResult {
		val placeable = measurable.measure(constraints)

		// Be at least as big as the minimum dimension in both dimensions
		val width = maxOf(placeable.width, size.width.roundToPx())
		val height = maxOf(placeable.height, size.height.roundToPx())

		return layout(width, height) {
			val centerX = ((width - placeable.width) / 2f).roundToInt()
			val centerY = ((height - placeable.height) / 2f).roundToInt()
			placeable.place(centerX, centerY)
		}
	}

	override fun equals(other: Any?): Boolean {
		val otherModifier = other as? MinimumTouchTargetModifier ?: return false
		return size == otherModifier.size
	}

	override fun hashCode(): Int {
		return size.hashCode()
	}
}