package agrigolo.opendrummer

import agrigolo.opendrummer.databinding.ActivityMainBinding
import agrigolo.opendrummer.ui.home.HomeFragment
import android.os.Bundle
import android.view.Menu
import android.view.MenuItem
import android.view.View
import android.widget.Button
import android.widget.SeekBar
import android.widget.TextView
import androidx.activity.OnBackPressedCallback
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.core.view.GravityCompat
import androidx.drawerlayout.widget.DrawerLayout
import androidx.navigation.findNavController
import androidx.navigation.ui.AppBarConfiguration
import androidx.navigation.ui.navigateUp
import androidx.navigation.ui.setupActionBarWithNavController
import androidx.navigation.ui.setupWithNavController
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.google.android.material.navigation.NavigationView
import kotlin.math.roundToInt

class MainActivity : AppCompatActivity() {

    private lateinit var backPressedCallback: OnBackPressedCallback
    private lateinit var appBarConfiguration: AppBarConfiguration
    private lateinit var binding: ActivityMainBinding
    private lateinit var recyclerView: RecyclerView
    private lateinit var adapter: RecyclerViewAdapter
    private var player: LoopingPlayer? = null
    private val stylesHelper = StylesHelper()
    private var currentBpm = 0.0f

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        binding = ActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)
        setSupportActionBar(binding.appBarMain.toolbar)
        val playButton = findViewById<Button>(R.id.playButton)
        val bpmLabel = findViewById<TextView>(R.id.temposlider_label)
        val bpmSlider = findViewById<SeekBar>(R.id.tempo_slider)
        val bpmHelp = findViewById<TextView>(R.id.temposlider_help)
        playButton.setEnabled(false)
        playButton.setText(R.string.message_no_style_selected)
        bpmSlider.setEnabled(false)
        bpmSlider.visibility = View.INVISIBLE
        bpmHelp.visibility = View.INVISIBLE

        //The swipe gesture to open the drawer is disabled to avoid opening it while using the playback speed slider
        binding.drawerLayout.setDrawerLockMode(
            DrawerLayout.LOCK_MODE_LOCKED_CLOSED,
            GravityCompat.START
        )

        player = LoopingPlayer(this)

        // Start/Stop button press
        binding.appBarMain.playButton.setOnClickListener { _ ->
            if (player?.isPlaying() == true) {
                player?.stop()
                playButton.setText(R.string.playbutton_play)

            } else {
                player?.play()
                playButton.setText(R.string.playbutton_stop)
            }
        }

        // Set playback speed. Range: 0.9x-1.1x
        binding.appBarMain.tempoSlider.setOnSeekBarChangeListener(object :
            SeekBar.OnSeekBarChangeListener {
            override fun onProgressChanged(seekBar: SeekBar, progress: Int, fromUser: Boolean) {
                val seekProgress = progress - 10
                val seekBarRange = 20 // Range of the SeekBar is -10 to 10, so the total range is 20
                val floatRange = 0.2f // The desired range for the playback rate is 1.1 - 0.9 = 0.2
                val floatValue = 0.9f + (seekProgress + 10) * (floatRange / seekBarRange)
                val newPlaybackRate =
                    (floatValue * 100).roundToInt() / 100.0f // Round to second decimal
                //Log.w("openDrummer","New playback rate: $newPlaybackRate")
                player?.setPlaybackRate(newPlaybackRate)
                val newBpm = (currentBpm * newPlaybackRate).roundToInt()
                bpmLabel.text = "$newBpm"
            }

            override fun onStartTrackingTouch(seekBar: SeekBar?) {
            }

            override fun onStopTrackingTouch(seekBar: SeekBar?) {
            }
        })

        val drawerLayout: DrawerLayout = binding.drawerLayout
        val navView: NavigationView = binding.navView
        val navController = findNavController(R.id.nav_host_fragment_content_main)
        // Passing each menu ID as a set of Ids because each
        // menu should be considered as top level destinations.
        appBarConfiguration = AppBarConfiguration(
            setOf(R.id.nav_home), drawerLayout
        )
        setupActionBarWithNavController(navController, appBarConfiguration)
        navView.setupWithNavController(navController)

        recyclerView = findViewById(R.id.recycler_view)

        // Set up the adapter and layout manager
        val styleFileNames = stylesHelper.getStyleFileNames()

        adapter = RecyclerViewAdapter(styleFileNames) { item ->
            // Handle the item click and navigate to the fragment
            // If a sound is playing, stop it first
            if (player?.isPlaying() == true) {
                player?.stop()
            }
            navigateToFragment(item)
            drawerLayout.closeDrawer(GravityCompat.START)
            val styleId = this.getResources().getIdentifier(item, "raw", this.packageName)
            stylesHelper.getStyleInfo(item)["bpm"].also { bpmLabel.text = it }
            // Disable the play button and the slider until the sound is loaded in memory
            bpmSlider.setEnabled(false)
            bpmSlider.visibility = View.VISIBLE
            bpmHelp.visibility = View.VISIBLE
            playButton.setText(R.string.playbutton_loading)
            playButton.setEnabled(false)
            currentBpm = stylesHelper.getStyleInfo(item)["bpm"].toString().toFloat()
            bpmSlider.progress = 10
            player?.changeSound(styleId,
                onLoadComplete = {
                    // Sound loaded, enable play button
                    playButton.setText(R.string.playbutton_play)
                    bpmSlider.setEnabled(true)
                    playButton.setEnabled(true)
                },
                onLoadError = {
                }
            )
            playButton.setEnabled(true)
            playButton.setText(R.string.playbutton_play)
        }
        recyclerView.layoutManager = LinearLayoutManager(this)
        recyclerView.adapter = adapter

        // Disable Back press to avoid uncontrolled behaviour
        backPressedCallback = object : OnBackPressedCallback(true) {
            override fun handleOnBackPressed() {
                // Do nothing
            }
        }
        onBackPressedDispatcher.addCallback(this, backPressedCallback)
    }

    override fun onDestroy() {
        super.onDestroy()
        player?.release()
    }

    override fun onCreateOptionsMenu(menu: Menu): Boolean {
        menuInflater.inflate(R.menu.main, menu)

        return true
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return when (item.itemId) {
            R.id.action_about -> {
                showAboutDialog()
                true
            }

            else -> super.onOptionsItemSelected(item)
        }
    }

    override fun onSupportNavigateUp(): Boolean {
        val navController = findNavController(R.id.nav_host_fragment_content_main)
        return navController.navigateUp(appBarConfiguration) || super.onSupportNavigateUp()
    }

    private fun navigateToFragment(style: String) {
        val title = stylesHelper.getStyleInfo(style)["name"]
        val bpm = stylesHelper.getStyleInfo(style)["bpm"]
        val fragment = HomeFragment.newInstance(title!!, bpm!!)
        supportFragmentManager.beginTransaction()
            .replace(R.id.nav_host_fragment_content_main, fragment)
            .addToBackStack(null)
            .commit()
    }

    private fun showAboutDialog() {
        val builder = AlertDialog.Builder(this)
        builder.setTitle("OpenDrummer v1.0")
            .setMessage(
                "Copyright © 2025 Alessandro Grigolo" +
                        "\n\nThis program is free software: you can redistribute it and/or modify" +
                        " it under the terms of the GNU General Public License as published by" +
                        " the Free Software Foundation, either version 3 of the License, or" +
                        " (at your option) any later version." +
                        "\n\n" +
                        "This program is distributed in the hope that it will be useful," +
                        "but WITHOUT ANY WARRANTY; without even the implied warranty of" +
                        " MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the" +
                        " GNU General Public License for more details." +
                        "\n\n" +
                        "You should have received a copy of the GNU General Public License" +
                        " along with this program. If not, see <http://www.gnu.org/licenses/>."
            )
            .setPositiveButton("OK") { dialog, _ ->
                dialog.dismiss()
            }
        val dialog = builder.create()
        dialog.show()
    }
}