package agrigolo.opendrummer

import android.content.Context
import android.media.AudioAttributes
import android.media.SoundPool
import android.util.Log
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class LoopingPlayer(private val context: Context) : SoundPool.OnLoadCompleteListener {
    private var soundPool: SoundPool? = null
    private var soundId: Int? = null
    private var isPlaying = false
    private var currentStreamId: Int? = null
    private var currentPlaybackRate: Float = 1.0f

    init {
        createSoundPool()
    }

    private fun createSoundPool() {
        val audioAttributes = AudioAttributes.Builder()
            .setUsage(AudioAttributes.USAGE_MEDIA)
            .setContentType(AudioAttributes.CONTENT_TYPE_MUSIC)
            .build()

        soundPool = SoundPool.Builder()
            .setMaxStreams(1)
            .setAudioAttributes(audioAttributes)
            .build()

        soundPool?.setOnLoadCompleteListener(this)
    }

    private fun loadSound(resourceId: Int) {
        CoroutineScope(Dispatchers.IO).launch {
            val loadedSoundId = soundPool?.load(context, resourceId, 1)
            withContext(Dispatchers.Main) {
                if (loadedSoundId != null) {
                    soundId = loadedSoundId // Set the soundId
                } else {
                    Log.e("OpenDrummer", "Error loading sound")
                }
            }
        }
    }

    fun changeSound(newResourceId: Int, onLoadComplete: (Int) -> Unit, onLoadError: (Int) -> Unit) {
        stop()
        loadSound(newResourceId)
        // Set the callback functions for load completion
        setLoadCompleteListener(onLoadComplete, onLoadError)
    }

    private var loadCompleteListener: ((Int) -> Unit)? = null
    private var loadErrorListener: ((Int) -> Unit)? = null

    private fun setLoadCompleteListener(onLoadComplete: (Int) -> Unit, onLoadError: (Int) -> Unit) {
        loadCompleteListener = onLoadComplete
        loadErrorListener = onLoadError
    }

    fun play() {
        if (isPlaying) {
            return
        }
        soundId?.let { id ->
            currentStreamId = soundPool?.play(
                id,
                1.0f, // leftVolume
                1.0f, // rightVolume
                1, // priority
                -1, // loop (-1 for infinite loop)
                currentPlaybackRate // rate
            )?.also {
                isPlaying = true
            }
        } ?: run {
            Log.w("OpenDrummer", "soundId is not ready, cannot play sound")
        }
    }

    fun setPlaybackRate(newRate: Float) {
        currentPlaybackRate = newRate
        Log.w("OpenDrummer","New playback rate $currentPlaybackRate")
        CoroutineScope(Dispatchers.IO).launch {
            currentStreamId?.let { streamId ->
                withContext(Dispatchers.Main) {
                    soundPool?.setRate(streamId, newRate)
                }
            }
        }
    }

    fun stop() {
        CoroutineScope(Dispatchers.IO).launch {
            currentStreamId?.let { streamId ->
                soundPool?.stop(streamId)
            }
            isPlaying = false
        }
    }

    fun isPlaying(): Boolean {
        return isPlaying
    }

    fun release() {
        stop()
        soundPool?.release()
        soundPool = null
        soundId = null
        currentStreamId = null
    }

    // Implement the OnLoadCompleteListener interface
    override fun onLoadComplete(soundPool: SoundPool?, sampleId: Int, status: Int) {
        if (status == 0) { // 0 indicates success
            Log.d("OpenDrummer", "Sound loaded successfully: $sampleId")
            loadCompleteListener?.invoke(sampleId) // Notify success
        } else {
            Log.e("OpenDrummer", "Error loading sound: $status")
            loadErrorListener?.invoke(status) // Notify error
        }
    }
}