import { NativeModules, DeviceEventEmitter, NativeEventEmitter, AppState, Platform } from 'react-native';

/**
 * Supported Request types
 * @todo PUT
 */
export let RequestMethod;
/**
 * Supported Body Payloads for the respective RequestMethod
 */

(function (RequestMethod) {
  RequestMethod["GET"] = "get";
  RequestMethod["POST"] = "post";
  RequestMethod["DELETE"] = "delete";
})(RequestMethod || (RequestMethod = {}));

/**
 * /**
 * Factory function to create a persistent TcpStream connection to a target
 * Wraps the native side emitter and subscribes to the targets data messages (string).
 * The TcpStream currently emits per line of data received . That is it reads data from the socket until a new line is reached, at which time
 * it will emit the data read (by calling onData(data,null). If an error is received or the connection is dropped it onData will be called
 * with the second parameter containing the error string (ie onData(null,'some error');
 * Note: Receiving an 'EOF' error from the target we're connected to signifies the end of a stream or the target dropped the connection.
 *       This will cause the module to drop the TcpConnection and remove all data event listeners.
 *       Should you wish to reconnect to the target you must initiate a new connection by calling createTcpConnection again.
 * @param param {target: String, writeTimeout: Number} :
 *        `target` onion to connect to (ex: kciybn4d4vuqvobdl2kdp3r2rudqbqvsymqwg4jomzft6m6gaibaf6yd.onion:50001)
 *        'writeTimeout' in seconds to wait before timing out on writing to the socket (Defaults to 7)
 * @param onData TcpConnDatahandler node style callback called when data or an error is received for this connection
 * @returns TcpStream
 */
const createTcpConnection = async (param, onData) => {
  const {
    target
  } = param;
  await NativeModules.TorBridge.startTcpConn(target);
  let lsnr_handle = [];
  /**
   * Handles errors from Tcp Connection
   * Mainly check for EOF (connection closed/end of stream) and removes lnsers
   */

  const onError = async event => {
    if (event.toLowerCase() === 'eof') {
      console.warn("Got to end of stream on TcpStream to ".concat(target, ". Removing listners"));
      await close();
    }
  };

  if (Platform.OS === 'android') {
    lsnr_handle.push(DeviceEventEmitter.addListener("".concat(target, "-data"), event => {
      onData(event);
    }));
    lsnr_handle.push(DeviceEventEmitter.addListener("".concat(target, "-error"), async event => {
      await onError(event);
      await onData(undefined, event);
    }));
  } else if (Platform.OS === 'ios') {
    const emitter = new NativeEventEmitter(NativeModules.TorBridge);
    lsnr_handle.push(emitter.addListener("torTcpStreamData", event => {
      onData(event);
    }));
    lsnr_handle.push(emitter.addListener("torTcpStreamError", async event => {
      await onError(event);
      await onData(undefined, event);
    }));
  }

  const writeTimeout = param.writeTimeout || 7;

  const write = msg => NativeModules.TorBridge.sendTcpConnMsg(target, msg, writeTimeout);

  const close = () => {
    lsnr_handle.map(e => e.remove());
    return NativeModules.TorBridge.stopTcpConn(target);
  };

  return {
    close,
    write
  };
};

const TorBridge = NativeModules.TorBridge;
/**
 * Tor module factory function
 * @param stopDaemonOnBackground
 * @default true
 * When set to true will shutdown the Tor daemon when the application is backgrounded preventing pre-emitive shutdowns by the OS
 * @param startDaemonOnActive
 * @default false
 * When set to true will automatically start/restart the Tor daemon when the application is bought back to the foreground (from the background)
 * @param os The OS the module is running on (Set automatically and is provided as an injectable for testing purposes)
 * @default The os the module is running on.
 */

export default (({
  stopDaemonOnBackground = true,
  startDaemonOnActive = false,
  os = Platform.OS
} = {}) => {
  let bootstrapPromise;
  let lastAppState = 'active';
  let _appStateLsnerSet = false;

  const _handleAppStateChange = async nextAppState => {
    if (startDaemonOnActive && lastAppState.match(/background/) && nextAppState === 'active') {
      const status = NativeModules.TorBridge.getDaemonStatus(); // Daemon should be in NOTINIT status if coming from background and this is enabled, so if not shutodwn and start again

      if (status !== 'NOTINIT') {
        await stopIfRunning();
      }

      startIfNotStarted();
    }

    if (stopDaemonOnBackground && lastAppState.match(/active/) && nextAppState === 'background') {
      const status = NativeModules.TorBridge.getDaemonStatus();

      if (status !== 'NOTINIT') {
        await stopIfRunning();
      }
    }

    lastAppState = nextAppState;
  };

  const startIfNotStarted = () => {
    if (!bootstrapPromise) {
      bootstrapPromise = NativeModules.TorBridge.startDaemon();
    }

    return bootstrapPromise;
  };

  const stopIfRunning = async () => {
    console.warn('Stopping Tor daemon.');
    bootstrapPromise = undefined;
    await NativeModules.TorBridge.stopDaemon();
  };
  /**
   * Post process request result
   */


  const onAfterRequest = async (res) => {
    if (os === 'android') {
      // Mapping JSONObject to ReadableMap for the bridge is a bit of a manual shitshow
      // so android JSON will be returned as string from the other side and we parse it here
      //
      if (res !== null && res !== void 0 && res.json) {
        const json = JSON.parse(res.json);
        return { ...res,
          json
        };
      }
    }

    return res;
  }; // Register app state lsner only once


  if (!_appStateLsnerSet) {
    AppState.addEventListener('change', _handleAppStateChange);
  }

  return {
    async get(url, headers, trustSSL = true) {
      await startIfNotStarted();
      return await onAfterRequest(await TorBridge.request(url, RequestMethod.GET, '', headers || {}, trustSSL));
    },

    async post(url, body, headers, trustSSL = true) {
      await startIfNotStarted();
      return await onAfterRequest(await TorBridge.request(url, RequestMethod.POST, body, headers || {}, trustSSL));
    },

    async delete(url, body, headers, trustSSL = true) {
      await startIfNotStarted();
      return await onAfterRequest(await TorBridge.request(url, RequestMethod.DELETE, body || '', headers || {}, trustSSL));
    },

    startIfNotStarted,
    stopIfRunning,
    request: TorBridge.request,
    getDaemonStatus: TorBridge.getDaemonStatus,
    createTcpConnection
  };
});
//# sourceMappingURL=index.js.map