import * as React from 'react';
import { Animated, StyleProp, ViewStyle } from 'react-native';
import { IconSource } from './Icon';
import { Theme } from '../types';
declare type Route = {
    key: string;
    title?: string;
    icon?: IconSource;
    badge?: string | number | boolean;
    color?: string;
    accessibilityLabel?: string;
    testID?: string;
};
declare type NavigationState = {
    index: number;
    routes: Route[];
};
declare type Props = {
    /**
     * Whether the shifting style is used, the active tab appears wider and the inactive tabs won't have a label.
     * By default, this is `true` when you have more than 3 tabs.
     */
    shifting?: boolean;
    /**
     * Whether to show labels in tabs. When `false`, only icons will be displayed.
     */
    labeled?: boolean;
    /**
     * State for the bottom navigation. The state should contain the following properties:
     *
     * - `index`: a number reprsenting the index of the active route in the `routes` array
     * - `routes`: an array containing a list of route objects used for rendering the tabs
     *
     * Each route object should contain the following properties:
     *
     * - `key`: a unique key to identify the route (required)
     * - `title`: title of the route to use as the tab label
     * - `icon`: icon to use as the tab icon, can be a string, an image source or a react component
     * - `color`: color to use as background color for shifting bottom navigation
     * - `badge`: badge to show on the tab icon, can be `true` to show a dot, `string` or `number` to show text.
     * - `accessibilityLabel`: accessibility label for the tab button
     * - `testID`: test id for the tab button
     *
     * Example:
     *
     * ```js
     * {
     *   index: 1,
     *   routes: [
     *     { key: 'music', title: 'Music', icon: 'queue-music', color: '#3F51B5' },
     *     { key: 'albums', title: 'Albums', icon: 'album', color: '#009688' },
     *     { key: 'recents', title: 'Recents', icon: 'history', color: '#795548' },
     *     { key: 'purchased', title: 'Purchased', icon: 'shopping-cart', color: '#607D8B' },
     *   ]
     * }
     * ```
     *
     * `BottomNavigation` is a controlled component, which means the `index` needs to be updated via the `onIndexChange` callback.
     */
    navigationState: NavigationState;
    /**
     * Callback which is called on tab change, receives the index of the new tab as argument.
     * The navigation state needs to be updated when it's called, otherwise the change is dropped.
     */
    onIndexChange: (index: number) => void;
    /**
     * Callback which returns a react element to render as the page for the tab. Receives an object containing the route as the argument:
     *
     * ```js
     * renderScene = ({ route, jumpTo }) => {
     *   switch (route.key) {
     *     case 'music':
     *       return <MusicRoute jumpTo={jumpTo} />;
     *     case 'albums':
     *       return <AlbumsRoute jumpTo={jumpTo} />;
     *   }
     * }
     * ```
     *
     * Pages are lazily rendered, which means that a page will be rendered the first time you navigate to it.
     * After initial render, all the pages stay rendered to preserve their state.
     *
     * You need to make sure that your individual routes implement a `shouldComponentUpdate` to improve the performance.
     * To make it easier to specify the components, you can use the `SceneMap` helper:
     *
     * ```js
     * renderScene = BottomNavigation.SceneMap({
     *   music: MusicRoute,
     *   albums: AlbumsRoute,
     * });
     * ```
     *
     * Specifying the components this way is easier and takes care of implementing a `shouldComponentUpdate` method.
     * Each component will receive the current route and a `jumpTo` method as it's props.
     * The `jumpTo` method can be used to navigate to other tabs programmatically:
     *
     * ```js
     * this.props.jumpTo('albums')
     * ```
     */
    renderScene: (props: {
        route: Route;
        jumpTo: (key: string) => void;
    }) => React.ReactNode | null;
    /**
     * Callback which returns a React Element to be used as tab icon.
     */
    renderIcon?: (props: {
        route: Route;
        focused: boolean;
        color: string;
    }) => React.ReactNode;
    /**
     * Callback which React Element to be used as tab label.
     */
    renderLabel?: (props: {
        route: Route;
        focused: boolean;
        color: string;
    }) => React.ReactNode;
    /**
     * Get label text for the tab, uses `route.title` by default. Use `renderLabel` to replace label component.
     */
    getLabelText?: (props: {
        route: Route;
    }) => string;
    /**
     * Get accessibility label for the tab button. This is read by the screen reader when the user taps the tab.
     * Uses `route.accessibilityLabel` by default.
     */
    getAccessibilityLabel?: (props: {
        route: Route;
    }) => string | undefined;
    /**
     * Get the id to locate this tab button in tests, uses `route.testID` by default.
     */
    getTestID?: (props: {
        route: Route;
    }) => string | undefined;
    /**
     * Get badge for the tab, uses `route.badge` by default.
     */
    getBadge?: (props: {
        route: Route;
    }) => boolean | number | string | undefined;
    /**
     * Get color for the tab, uses `route.color` by default.
     */
    getColor?: (props: {
        route: Route;
    }) => string | undefined;
    /**
     * Function to execute on tab press. It receives the route for the pressed tab, useful for things like scroll to top.
     */
    onTabPress?: (props: {
        route: Route;
    }) => void;
    /**
     * Custom color for icon and label in the active tab.
     */
    activeColor?: string;
    /**
     * Custom color for icon and label in the inactive tab.
     */
    inactiveColor?: string;
    /**
     * Whether animation is enabled for scenes transitions in `shifting` mode.
     * By default, the scenes cross-fade during tab change when `shifting` is enabled.
     * Specify `sceneAnimationEnabled` as `false` to disable the animation.
     */
    sceneAnimationEnabled?: boolean;
    /**
     * Whether the bottom navigation bar is hidden when keyboard is shown.
     * On Android, this works best when [`windowSoftInputMode`](https://developer.android.com/guide/topics/manifest/activity-element#wsoft) is set to `adjustResize`.
     */
    keyboardHidesNavigationBar?: boolean;
    /**
     * Style for the bottom navigation bar.
     * You can set a bottom padding here if you have a translucent navigation bar on Android:
     *
     * ```js
     * barStyle={{ paddingBottom: 48 }}
     * ```
     */
    barStyle?: StyleProp<ViewStyle>;
    style?: StyleProp<ViewStyle>;
    /**
     * @optional
     */
    theme: Theme;
};
declare type State = {
    /**
     * Visibility of the navigation bar, visible state is 1 and invisible is 0.
     */
    visible: Animated.Value;
    /**
     * Active state of individual tab items, active state is 1 and inactve state is 0.
     */
    tabs: Animated.Value[];
    /**
     * The top offset for each tab item to position it offscreen.
     * Placing items offscreen helps to save memory usage for inactive screens with removeClippedSubviews.
     * We use animated values for this to prevent unnecesary re-renders.
     */
    offsets: Animated.Value[];
    /**
     * Index of the currently active tab. Used for setting the background color.
     * Use don't use the color as an animated value directly, because `setValue` seems to be buggy with colors.
     */
    index: Animated.Value;
    /**
     * Animation for the touch, used to determine it's scale and opacity.
     */
    touch: Animated.Value;
    /**
     * Animation for the background color ripple, used to determine it's scale and opacity.
     */
    ripple: Animated.Value;
    /**
     * Layout of the navigation bar. The width is used to determine the size and position of the ripple.
     */
    layout: {
        height: number;
        width: number;
        measured: boolean;
    };
    /**
     * Currently active index. Used only for getDerivedStateFromProps.
     */
    current: number;
    /**
     * Previously active index. Used to determine the position of the ripple.
     */
    previous: number;
    /**
     * List of loaded tabs, tabs will be loaded when navigated to.
     */
    loaded: number[];
    /**
     * Trak whether the keyboard is visible to show and hide the navigation bar.
     */
    keyboard: boolean;
};
/**
 * Bottom navigation provides quick navigation between top-level views of an app with a bottom navigation bar.
 * It is primarily designed for use on mobile.
 *
 * For integration with React Navigation, you can use [react-navigation-material-bottom-tab-navigator](https://github.com/react-navigation/react-navigation-material-bottom-tab-navigator).
 *
 * By default Bottom navigation uses primary color as a background, in dark theme with `adaptive` mode it will use surface colour instead.
 * See [Dark Theme](https://callstack.github.io/react-native-paper/theming.html#dark-theme) for more informations
 *
 * <div class="screenshots">
 *   <img class="medium" src="screenshots/bottom-navigation.gif" />
 * </div>
 *
 * ## Usage
 * ```js
 * import * as React from 'react';
 * import { BottomNavigation, Text } from 'react-native-paper';
 *
 * const MusicRoute = () => <Text>Music</Text>;
 *
 * const AlbumsRoute = () => <Text>Albums</Text>;
 *
 * const RecentsRoute = () => <Text>Recents</Text>;
 *
 * export default class MyComponent extends React.Component {
 *   state = {
 *     index: 0,
 *     routes: [
 *       { key: 'music', title: 'Music', icon: 'queue-music' },
 *       { key: 'albums', title: 'Albums', icon: 'album' },
 *       { key: 'recents', title: 'Recents', icon: 'history' },
 *     ],
 *   };
 *
 *   _handleIndexChange = index => this.setState({ index });
 *
 *   _renderScene = BottomNavigation.SceneMap({
 *     music: MusicRoute,
 *     albums: AlbumsRoute,
 *     recents: RecentsRoute,
 *   });
 *
 *   render() {
 *     return (
 *       <BottomNavigation
 *         navigationState={this.state}
 *         onIndexChange={this._handleIndexChange}
 *         renderScene={this._renderScene}
 *       />
 *     );
 *   }
 * }
 * ```
 */
declare class BottomNavigation extends React.Component<Props, State> {
    /**
     * Function which takes a map of route keys to components.
     * Pure components are used to minmize re-rendering of the pages.
     * This drastically improves the animation performance.
     */
    static SceneMap(scenes: {
        [key: string]: React.ComponentType<{
            route: Route;
            jumpTo: (key: string) => void;
        }>;
    }): ({ route, jumpTo, }: {
        route: Route;
        jumpTo: (key: string) => void;
    }) => JSX.Element;
    static defaultProps: {
        labeled: boolean;
        keyboardHidesNavigationBar: boolean;
    };
    static getDerivedStateFromProps(nextProps: any, prevState: State): {
        tabs: any;
        offsets: any;
    };
    constructor(props: Props);
    componentDidMount(): void;
    componentDidUpdate(prevProps: Props): void;
    componentWillUnmount(): void;
    private handleKeyboardShow;
    private handleKeyboardHide;
    private animateToCurrentIndex;
    private handleLayout;
    private handleTabPress;
    private jumpTo;
    private isShifting;
    render(): JSX.Element;
}
declare const _default: (React.ComponentClass<Pick<Props, "style" | "shifting" | "labeled" | "navigationState" | "onIndexChange" | "renderScene" | "renderIcon" | "renderLabel" | "getLabelText" | "getAccessibilityLabel" | "getTestID" | "getBadge" | "getColor" | "onTabPress" | "activeColor" | "inactiveColor" | "sceneAnimationEnabled" | "keyboardHidesNavigationBar" | "barStyle"> & {
    theme?: import("@callstack/react-theme-provider").$DeepPartial<Theme> | undefined;
}, any> & import("@callstack/react-theme-provider/typings/hoist-non-react-statics").NonReactStatics<(React.ComponentClass<Props, any> & typeof BottomNavigation) | (React.FunctionComponent<Props> & typeof BottomNavigation), {}>) | (React.FunctionComponent<Pick<Props, "style" | "shifting" | "labeled" | "navigationState" | "onIndexChange" | "renderScene" | "renderIcon" | "renderLabel" | "getLabelText" | "getAccessibilityLabel" | "getTestID" | "getBadge" | "getColor" | "onTabPress" | "activeColor" | "inactiveColor" | "sceneAnimationEnabled" | "keyboardHidesNavigationBar" | "barStyle"> & {
    theme?: import("@callstack/react-theme-provider").$DeepPartial<Theme> | undefined;
}> & import("@callstack/react-theme-provider/typings/hoist-non-react-statics").NonReactStatics<(React.ComponentClass<Props, any> & typeof BottomNavigation) | (React.FunctionComponent<Props> & typeof BottomNavigation), {}>);
export default _default;
