var EventEmitter = require('events').EventEmitter
var inherits     = require('util').inherits
var ranges       = require('string-range')
var fixRange     = require('level-fix-range')
var xtend        = require('xtend')
var Batch        = require('./batch')

inherits(SubDB, EventEmitter)

function SubDB (db, prefix, options) {
  if('string' === typeof options) {
    console.error('db.sublevel(name, seperator<string>) is depreciated')
    console.error('use db.sublevel(name, {sep: separator})) if you must')
    options = {sep: options}
  }
  if(!(this instanceof SubDB)) return new SubDB(db, prefix, options)
  if(!db)     throw new Error('must provide db')
  if(!prefix) throw new Error('must provide prefix')

  options = options || {}
  options.sep = options.sep || '\xff'

  this._parent = db
  this._options = options
  this.options = options
  this._prefix = prefix
  this._root = root(this)
  db.sublevels[prefix] = this
  this.sublevels = {}
  this.methods = {}
  var self = this
  this.hooks = {
    pre: function () {
      return self.pre.apply(self, arguments)
    },
    post: function () {
      return self.post.apply(self, arguments)
    }
  }
}

var SDB = SubDB.prototype

SDB._key = function (key) {
  var sep = this._options.sep
  return sep
    + this._prefix
    + sep
    + key
}

SDB._getOptsAndCb = function (opts, cb) {
  if (typeof opts == 'function') {
    cb = opts
    opts = {}
  }
  return { opts: xtend(opts, this._options), cb: cb }
}

SDB.sublevel = function (prefix, options) {
  if(this.sublevels[prefix])
    return this.sublevels[prefix]
  return new SubDB(this, prefix, options || this._options)
}

SDB.put = function (key, value, opts, cb) {
  var res = this._getOptsAndCb(opts, cb)
  this._root.put(this.prefix(key), value, res.opts, res.cb)
}

SDB.get = function (key, opts, cb) {
  var res = this._getOptsAndCb(opts, cb)
  this._root.get(this.prefix(key), res.opts, res.cb)
}

SDB.del = function (key, opts, cb) {
  var res = this._getOptsAndCb(opts, cb)
  this._root.del(this.prefix(key), res.opts, res.cb)
}

SDB.batch = function (changes, opts, cb) {
  if(!Array.isArray(changes))
    return new Batch(this)
  var self = this,
      res = this._getOptsAndCb(opts, cb)
  changes.forEach(function (ch) {

    //OH YEAH, WE NEED TO VALIDATE THAT UPDATING THIS KEY/PREFIX IS ALLOWED
    if('string' === typeof ch.prefix)
      ch.key = ch.prefix + ch.key
    else
      ch.key = (ch.prefix || self).prefix(ch.key)

    if(ch.prefix) ch.prefix = null
  })
  this._root.batch(changes, res.opts, res.cb)
}

SDB._getKeyEncoding = function () {
  if(this.options.keyEncoding)
    return this.options.keyEncoding
  if(this._parent && this._parent._getKeyEncoding)
    return this._parent._getKeyEncoding()
}

SDB._getValueEncoding = function () {
  if(this.options.valueEncoding)
    return this.options.valueEncoding
  if(this._parent && this._parent._getValueEncoding)
    return this._parent._getValueEncoding()
}

SDB.prefix = function (key) {
  var sep = this._options.sep
  return this._parent.prefix() + sep + this._prefix + sep + (key || '')
}

SDB.keyStream =
SDB.createKeyStream = function (opts) {
  opts = opts || {}
  opts.keys = true
  opts.values = false
  return this.createReadStream(opts)
}

SDB.valueStream =
SDB.createValueStream = function (opts) {
  opts = opts || {}
  opts.keys = false
  opts.values = true
  opts.keys = false
  return this.createReadStream(opts)
}

function selectivelyMerge(_opts, opts) {
  [ 'valueEncoding'
  , 'encoding'
  , 'keyEncoding'
  , 'reverse'
  , 'values'
  , 'keys'
  , 'limit'
  , 'fillCache'
  ]
  .forEach(function (k) {
    if (opts.hasOwnProperty(k)) _opts[k] = opts[k]
  })
}

SDB.readStream =
SDB.createReadStream = function (opts) {
  opts = opts || {}
  var r = root(this)
  var p = this.prefix()

  var _opts = ranges.prefix(opts, p)
  selectivelyMerge(_opts, xtend(opts, this._options))

  var s = r.createReadStream(_opts)

  if(_opts.values === false) {
    var read = s.read
    if (read) {
      s.read = function (size) {
        var val = read.call(this, size)
        if (val) val = val.substring(p.length)
        return val
      }
    } else {
      var emit = s.emit
      s.emit = function (event, val) {
        if(event === 'data') {
          emit.call(this, 'data', val.substring(p.length))
        } else
          emit.call(this, event, val)
      }
    }
    return s
  } else if(_opts.keys === false)
    return s
  else {
    var read = s.read
    if (read) {
      s.read = function (size) {
        var d = read.call(this, size)
        if (d) d.key = d.key.substring(p.length)
        return d
      }
    } else {
      s.on('data', function (d) {
        //mutate the prefix!
        //this doesn't work for createKeyStream admittedly.
        d.key = d.key.substring(p.length)
      })
    }
    return s
  }
}


SDB.writeStream =
SDB.createWriteStream = function () {
  var r = root(this)
  var p = this.prefix()
  var ws = r.createWriteStream.apply(r, arguments)
  var write = ws.write

  var encoding = this._options.encoding
  var valueEncoding = this._options.valueEncoding
  var keyEncoding = this._options.keyEncoding

  // slight optimization, if no encoding was specified at all,
  // which will be the case most times, make write not check at all
  var nocheck = !encoding && !valueEncoding && !keyEncoding

  ws.write = nocheck
    ? function (data) {
        data.key = p + data.key
        return write.call(ws, data)
      }
    : function (data) {
        data.key = p + data.key

        // not merging all options here since this happens on every write and things could get slowed down
        // at this point we only consider encoding important to propagate
        if (encoding && typeof data.encoding === 'undefined')
          data.encoding = encoding
        if (valueEncoding && typeof data.valueEncoding === 'undefined')
          data.valueEncoding = valueEncoding
        if (keyEncoding && typeof data.keyEncoding === 'undefined')
          data.keyEncoding = keyEncoding

        return write.call(ws, data)
      }
  return ws
}

SDB.approximateSize = function () {
  var r = root(db)
  return r.approximateSize.apply(r, arguments)
}

function root(db) {
  if(!db._parent) return db
  return root(db._parent)
}

SDB.pre = function (range, hook) {
  if(!hook) hook = range, range = null
  range = ranges.prefix(range, this.prefix(), this._options.sep)
  var r = root(this._parent)
  var p = this.prefix()
  return r.hooks.pre(fixRange(range), function (ch, add, batch) {
    hook({
      key: ch.key.substring(p.length),
      value: ch.value,
      type: ch.type
    }, function (ch, _p) {
      //maybe remove the second add arg now
      //that op can have prefix?
      add(ch, ch.prefix ? _p : (_p || p))
    }, batch)
  })
}

SDB.post = function (range, hook) {
  if(!hook) hook = range, range = null
  var r = root(this._parent)
  var p = this.prefix()
  range = ranges.prefix(range, p, this._options.sep)
  return r.hooks.post(fixRange(range), function (data) {
    hook({key: data.key.substring(p.length), value: data.value, type: data.type})
  })
}

var exports = module.exports = SubDB

