import HTMLElement from '../html-element/HTMLElement.js';
import * as PropertySymbol from '../../PropertySymbol.js';
import Event from '../../event/Event.js';
import WindowErrorUtility from '../../window/WindowErrorUtility.js';
import WindowBrowserContext from '../../window/WindowBrowserContext.js';
import BrowserErrorCaptureEnum from '../../browser/enums/BrowserErrorCaptureEnum.js';
import DOMExceptionNameEnum from '../../exception/DOMExceptionNameEnum.js';
import ResourceFetch from '../../fetch/ResourceFetch.js';
/**
 * HTML Script Element.
 *
 * Reference:
 * https://developer.mozilla.org/en-US/docs/Web/API/HTMLScriptElement.
 */
export default class HTMLScriptElement extends HTMLElement {
    // Events
    onerror = null;
    onload = null;
    // Internal properties
    [PropertySymbol.evaluateScript] = true;
    // Private properties
    #loadedScriptURL = null;
    /**
     * Returns type.
     *
     * @returns Type.
     */
    get type() {
        return this.getAttribute('type') || '';
    }
    /**
     * Sets type.
     *
     * @param type Type.
     */
    set type(type) {
        this.setAttribute('type', type);
    }
    /**
     * Returns source.
     *
     * @returns Source.
     */
    get src() {
        if (!this.hasAttribute('src')) {
            return '';
        }
        try {
            return new URL(this.getAttribute('src'), this[PropertySymbol.ownerDocument].location.href)
                .href;
        }
        catch (e) {
            return this.getAttribute('src');
        }
    }
    /**
     * Sets source.
     *
     * @param src Source.
     */
    set src(src) {
        this.setAttribute('src', src);
    }
    /**
     * Returns charset.
     *
     * @returns Charset.
     */
    get charset() {
        return this.getAttribute('charset') || '';
    }
    /**
     * Sets charset.
     *
     * @param charset Charset.
     */
    set charset(charset) {
        this.setAttribute('charset', charset);
    }
    /**
     * Returns lang.
     *
     * @returns Lang.
     */
    get lang() {
        return this.getAttribute('lang') || '';
    }
    /**
     * Sets lang.
     *
     * @param lang Lang.
     */
    set lang(lang) {
        this.setAttribute('lang', lang);
    }
    /**
     * Returns async.
     *
     * @returns Async.
     */
    get async() {
        return this.getAttribute('async') !== null;
    }
    /**
     * Sets async.
     *
     * @param async Async.
     */
    set async(async) {
        if (!async) {
            this.removeAttribute('async');
        }
        else {
            this.setAttribute('async', '');
        }
    }
    /**
     * Returns defer.
     *
     * @returns Defer.
     */
    get defer() {
        return this.getAttribute('defer') !== null;
    }
    /**
     * Sets defer.
     *
     * @param defer Defer.
     */
    set defer(defer) {
        if (!defer) {
            this.removeAttribute('defer');
        }
        else {
            this.setAttribute('defer', '');
        }
    }
    /**
     * Returns text.
     *
     * @returns Text.
     */
    get text() {
        return this.textContent;
    }
    /**
     * Sets text.
     *
     * @param text Text.
     */
    set text(text) {
        this.textContent = text;
    }
    /**
     * @override
     */
    [PropertySymbol.cloneNode](deep = false) {
        return super[PropertySymbol.cloneNode](deep);
    }
    /**
     * @override
     */
    [PropertySymbol.connectedToDocument]() {
        const browserSettings = new WindowBrowserContext(this[PropertySymbol.window]).getSettings();
        super[PropertySymbol.connectedToDocument]();
        if (this[PropertySymbol.evaluateScript]) {
            const src = this.getAttribute('src');
            if (src !== null) {
                this.#loadScript(src);
            }
            else if (browserSettings && !browserSettings.disableJavaScriptEvaluation) {
                const textContent = this.textContent;
                const type = this.getAttribute('type');
                if (textContent &&
                    (type === null ||
                        type === 'application/x-ecmascript' ||
                        type === 'application/x-javascript' ||
                        type.startsWith('text/javascript'))) {
                    this[PropertySymbol.ownerDocument][PropertySymbol.currentScript] = this;
                    const code = `//# sourceURL=${this[PropertySymbol.window].location.href}\n` + textContent;
                    if (browserSettings.disableErrorCapturing ||
                        browserSettings.errorCapture !== BrowserErrorCaptureEnum.tryAndCatch) {
                        this[PropertySymbol.window].eval(code);
                    }
                    else {
                        WindowErrorUtility.captureError(this[PropertySymbol.window], () => this[PropertySymbol.window].eval(code));
                    }
                    this[PropertySymbol.ownerDocument][PropertySymbol.currentScript] = null;
                }
            }
        }
    }
    /**
     * @override
     */
    [PropertySymbol.onSetAttribute](attribute, replacedAttribute) {
        super[PropertySymbol.onSetAttribute](attribute, replacedAttribute);
        if (attribute[PropertySymbol.name] === 'src' &&
            attribute[PropertySymbol.value] !== null &&
            this[PropertySymbol.isConnected]) {
            this.#loadScript(attribute[PropertySymbol.value]);
        }
    }
    /**
     * Returns a URL relative to the given Location object.
     *
     * @param url URL.
     */
    async #loadScript(url) {
        const window = this[PropertySymbol.window];
        const browserFrame = new WindowBrowserContext(window).getBrowserFrame();
        const async = this.getAttribute('async') !== null;
        if (!browserFrame) {
            return;
        }
        const browserSettings = browserFrame.page?.context?.browser?.settings;
        if (!url || !this[PropertySymbol.isConnected]) {
            return;
        }
        let absoluteURL;
        try {
            absoluteURL = new URL(url, this[PropertySymbol.window].location.href).href;
        }
        catch (error) {
            return;
        }
        if (this.#loadedScriptURL === absoluteURL) {
            return;
        }
        if (browserSettings &&
            (browserSettings.disableJavaScriptFileLoading || browserSettings.disableJavaScriptEvaluation)) {
            if (browserSettings.handleDisabledFileLoadingAsSuccess) {
                this.dispatchEvent(new Event('load'));
            }
            else {
                WindowErrorUtility.dispatchError(this, new window.DOMException(`Failed to load external script "${absoluteURL}". JavaScript file loading is disabled.`, DOMExceptionNameEnum.notSupportedError));
            }
            return;
        }
        const resourceFetch = new ResourceFetch({
            browserFrame,
            window: this[PropertySymbol.window]
        });
        let code = null;
        let error = null;
        this.#loadedScriptURL = absoluteURL;
        if (async) {
            const readyStateManager = this[PropertySymbol.window][PropertySymbol.readyStateManager];
            readyStateManager.startTask();
            try {
                code = await resourceFetch.fetch(absoluteURL);
            }
            catch (e) {
                error = e;
            }
            readyStateManager.endTask();
        }
        else {
            try {
                code = resourceFetch.fetchSync(absoluteURL);
            }
            catch (e) {
                error = e;
            }
        }
        if (error) {
            WindowErrorUtility.dispatchError(this, error);
        }
        else {
            this[PropertySymbol.ownerDocument][PropertySymbol.currentScript] = this;
            code = '//# sourceURL=' + absoluteURL + '\n' + code;
            if (browserSettings.disableErrorCapturing ||
                browserSettings.errorCapture !== BrowserErrorCaptureEnum.tryAndCatch) {
                this[PropertySymbol.window].eval(code);
            }
            else {
                WindowErrorUtility.captureError(this[PropertySymbol.window], () => this[PropertySymbol.window].eval(code));
            }
            this[PropertySymbol.ownerDocument][PropertySymbol.currentScript] = null;
            this.dispatchEvent(new Event('load'));
        }
    }
}
//# sourceMappingURL=HTMLScriptElement.js.map