'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const reportAllErrors = require('../utils/report-all-errors.js')
const shouldPartition = require('../utils/should-partition.js')
const getSourceCode = require('../utils/get-source-code.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const reportErrors = require('../utils/report-errors.js')
const getSettings = require('../utils/get-settings.js')
const sortNodes = require('../utils/sort-nodes.js')
const complete = require('../utils/complete.js')
let defaultOptions = {
  fallbackSort: { type: 'unsorted' },
  specialCharacters: 'keep',
  partitionByComment: false,
  partitionByNewLine: false,
  type: 'alphabetical',
  groupKind: 'mixed',
  ignoreCase: true,
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
}
const sortExports = createEslintRule.createEslintRule({
  create: context => {
    let settings = getSettings.getSettings(context.settings)
    let options = complete.complete(
      context.options.at(0),
      settings,
      defaultOptions,
    )
    validateCustomSortConfiguration.validateCustomSortConfiguration(options)
    let sourceCode = getSourceCode.getSourceCode(context)
    let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
      ruleName: context.id,
      sourceCode,
    })
    let parts = [[]]
    let registerNode = node => {
      var _a
      let sortingNode = {
        isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
          node,
          eslintDisabledLines,
        ),
        groupKind: node.exportKind === 'value' ? 'value' : 'type',
        size: rangeToDiff.rangeToDiff(node, sourceCode),
        addSafetySemicolonWhenInline: true,
        name: node.source.value,
        node,
      }
      let lastNode = (_a = parts.at(-1)) == null ? void 0 : _a.at(-1)
      if (
        shouldPartition.shouldPartition({
          lastSortingNode: lastNode,
          sortingNode,
          sourceCode,
          options,
        })
      ) {
        parts.push([])
      }
      parts.at(-1).push(sortingNode)
    }
    return {
      'Program:exit': () => {
        let groupKindOrder
        if (options.groupKind === 'values-first') {
          groupKindOrder = ['value', 'type']
        } else if (options.groupKind === 'types-first') {
          groupKindOrder = ['type', 'value']
        } else {
          groupKindOrder = ['any']
        }
        for (let nodes of parts) {
          let filteredGroupKindNodes = groupKindOrder.map(groupKind =>
            nodes.filter(
              currentNode =>
                groupKind === 'any' || currentNode.groupKind === groupKind,
            ),
          )
          let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
            filteredGroupKindNodes.flatMap(groupedNodes =>
              sortNodes.sortNodes(groupedNodes, options, {
                ignoreEslintDisabledNodes,
              }),
            )
          reportAllErrors.reportAllErrors({
            availableMessageIds: {
              unexpectedOrder: 'unexpectedExportsOrder',
            },
            sortNodesExcludingEslintDisabled,
            sourceCode,
            options,
            context,
            nodes,
          })
        }
      },
      ExportNamedDeclaration: node => {
        if (node.source !== null) {
          registerNode(node)
        }
      },
      ExportAllDeclaration: registerNode,
    }
  },
  meta: {
    schema: [
      {
        properties: {
          ...commonJsonSchemas.commonJsonSchemas,
          groupKind: {
            enum: ['mixed', 'values-first', 'types-first'],
            description: 'Specifies top-level groups.',
            type: 'string',
          },
          partitionByComment: commonJsonSchemas.partitionByCommentJsonSchema,
          partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    docs: {
      url: 'https://perfectionist.dev/rules/sort-exports',
      description: 'Enforce sorted exports.',
      recommended: true,
    },
    messages: {
      unexpectedExportsOrder: reportErrors.ORDER_ERROR,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-exports',
})
module.exports = sortExports
