/**
 * Subscript and Superscript plugin for Remark.
 */
import { visit } from "unist-util-visit";
const SUPERSCRIPT_REGEX = /\^([^^ ]+)\^/g;
const SUBSCRIPT_REGEX = /~([^~ ]+)~/g;
function handleSuperSub(node, regex, nodeName, hName) {
    const { value } = node;
    let match;
    let lastIndex = 0;
    const children = [];
    while ((match = regex.exec(value)) !== null) {
        const [fullMatch, text] = match;
        const index = match.index;
        const textNode = {
            type: "text",
            value: value.substring(lastIndex, index),
        };
        // Add the preceding text as a text node
        children.push(textNode);
        const intermediateTextNode = {
            type: "text",
            value: text,
        };
        // Add the superscript/subscript node
        children.push({
            type: nodeName,
            data: {
                hName,
            },
            children: [intermediateTextNode],
        });
        lastIndex = index + fullMatch.length;
    }
    const lastTextNode = {
        type: "text",
        value: value.substring(lastIndex),
    };
    // Add the remaining text as a text node
    children.push(lastTextNode);
    return children;
}
export default function supersub() {
    return (tree) => {
        // Superscript
        visit(tree, ["text"], (node, i, parent) => {
            if (node.type !== "text") {
                return;
            }
            const children = handleSuperSub(node, SUPERSCRIPT_REGEX, "superscript", "sup");
            parent.children.splice(i, 1, ...children);
        });
        // Subscript
        visit(tree, ["text"], (node, i, parent) => {
            if (node.type !== "text") {
                return;
            }
            const children = handleSuperSub(node, SUBSCRIPT_REGEX, "subscript", "sub");
            parent.children.splice(i, 1, ...children);
        });
    };
}
