"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const slash_1 = __importDefault(require("slash"));
const mime_types_1 = require("mime-types");
const util_1 = require("util");
const constants_1 = __importDefault(require("../config/constants"));
const getExtension = (file) => {
    return path_1.default.extname(file).replace('.', '');
};
const isImageFile = (file) => {
    return [
        'apng',
        'bmp',
        'gif',
        'ico',
        'cur',
        'jpg',
        'jpeg',
        'jfif',
        'pjpeg',
        'pjp',
        'png',
        'svg',
        'webp',
    ].includes(getExtension(file));
};
const isHtmlFile = (file) => {
    return ['html', 'htm'].includes(getExtension(file));
};
const convertBackslashPathToSlashPath = (backSlashPath) => {
    return (0, slash_1.default)(backSlashPath);
};
const getAppDir = () => {
    let appPath;
    try {
        appPath = require.resolve('pwa-asset-generator');
    }
    catch (e) {
        appPath = require.main.filename;
    }
    return path_1.default.join(path_1.default.dirname(appPath), '..');
};
const getShellHtmlFilePath = () => {
    return `${getAppDir()}/static/shell.html`;
};
const getImageSavePath = (imageName, outputFolder, ext, maskable, isMaskableIcon) => {
    const getMaskablePrefix = () => {
        if (!isMaskableIcon) {
            return '';
        }
        if (maskable) {
            return '.maskable';
        }
        return '';
    };
    return convertBackslashPathToSlashPath(path_1.default.join(outputFolder, `${imageName}${getMaskablePrefix()}.${ext}`));
};
const fileUrl = (filePath) => {
    let pathName = filePath;
    pathName = pathName.replace(/\\/g, '/');
    // Windows drive letter must be prefixed with a slash
    if (pathName[0] !== '/') {
        pathName = `/${pathName}`;
    }
    return encodeURI(`file://${pathName}`).replace(/[?#]/g, encodeURIComponent);
};
const getFileUrlOfPath = (source) => {
    return fileUrl(path_1.default.resolve(source));
};
const getRelativeFilePath = (referenceFilePath, filePath) => {
    return path_1.default.relative(path_1.default.dirname(path_1.default.resolve(referenceFilePath)), path_1.default.resolve(filePath));
};
const getRelativeImagePath = (outputFilePath, imagePath) => {
    if (outputFilePath) {
        return convertBackslashPathToSlashPath(getRelativeFilePath(outputFilePath, imagePath));
    }
    return convertBackslashPathToSlashPath(imagePath);
};
const getImageBase64Url = (imagePath) => {
    return `data:${(0, mime_types_1.lookup)(imagePath)};base64,${fs_1.default.readFileSync(imagePath, {
        encoding: 'base64',
    })}`;
};
const getHtmlShell = (imagePath, options, isUrl) => {
    const imageUrl = isUrl ? imagePath : getImageBase64Url(imagePath);
    return `${constants_1.default.SHELL_HTML_FOR_LOGO(imageUrl, options.padding, options.background)}`;
};
const isPathAccessible = (filePath, mode) => (0, util_1.promisify)(fs_1.default.access)(filePath, mode).then(() => true);
const makeDirRecursiveSync = (dirPath) => {
    fs_1.default.mkdirSync(dirPath, { recursive: true });
    return dirPath;
};
exports.default = {
    convertBackslashPathToSlashPath,
    getRelativeImagePath,
    getHtmlShell,
    isHtmlFile,
    isImageFile,
    getImageBase64Url,
    getShellHtmlFilePath,
    getImageSavePath,
    getFileUrlOfPath,
    isPathAccessible,
    getRelativeFilePath,
    getAppDir,
    getExtension,
    getFilesInDir: (0, util_1.promisify)(fs_1.default.readdir),
    readFile: (0, util_1.promisify)(fs_1.default.readFile),
    readFileSync: fs_1.default.readFileSync,
    writeFile: (0, util_1.promisify)(fs_1.default.writeFile),
    writeFileSync: fs_1.default.writeFileSync,
    makeDir: (0, util_1.promisify)(fs_1.default.mkdir),
    makeDirSync: fs_1.default.mkdirSync,
    copyFileSync: fs_1.default.copyFileSync,
    exists: (0, util_1.promisify)(fs_1.default.exists),
    makeDirRecursiveSync,
    READ_ACCESS: fs_1.default.constants.R_OK,
    WRITE_ACCESS: fs_1.default.constants.W_OK,
};
//# sourceMappingURL=file.js.map