'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateNewlinesAndPartitionConfiguration = require('../utils/validate-newlines-and-partition-configuration.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const validateGroupsConfiguration = require('../utils/validate-groups-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const sortNodesByGroups = require('../utils/sort-nodes-by-groups.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const makeNewlinesFixes = require('../utils/make-newlines-fixes.js')
const getNewlinesErrors = require('../utils/get-newlines-errors.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getGroupNumber = require('../utils/get-group-number.js')
const getSourceCode = require('../utils/get-source-code.js')
const toSingleLine = require('../utils/to-single-line.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const useGroups = require('../utils/use-groups.js')
const makeFixes = require('../utils/make-fixes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
let defaultOptions = {
  specialCharacters: 'keep',
  newlinesBetween: 'ignore',
  partitionByNewLine: false,
  partitionByComment: false,
  type: 'alphabetical',
  ignoreCase: true,
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
  groups: [],
}
const sortUnionTypes = createEslintRule.createEslintRule({
  create: context => ({
    TSUnionType: node => {
      let settings = getSettings.getSettings(context.settings)
      let options = complete.complete(
        context.options.at(0),
        settings,
        defaultOptions,
      )
      validateCustomSortConfiguration.validateCustomSortConfiguration(options)
      validateGroupsConfiguration.validateGroupsConfiguration(
        options.groups,
        [
          'intersection',
          'conditional',
          'function',
          'operator',
          'keyword',
          'literal',
          'nullish',
          'unknown',
          'import',
          'object',
          'named',
          'tuple',
          'union',
        ],
        [],
      )
      validateNewlinesAndPartitionConfiguration.validateNewlinesAndPartitionConfiguration(
        options,
      )
      let sourceCode = getSourceCode.getSourceCode(context)
      let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
        ruleName: context.id,
        sourceCode,
      })
      let formattedMembers = node.types.reduce(
        (accumulator, type) => {
          let { defineGroup, getGroup } = useGroups.useGroups(options)
          switch (type.type) {
            case 'TSTemplateLiteralType':
            case 'TSLiteralType':
              defineGroup('literal')
              break
            case 'TSIndexedAccessType':
            case 'TSTypeReference':
            case 'TSQualifiedName':
            case 'TSArrayType':
            case 'TSInferType':
              defineGroup('named')
              break
            case 'TSIntersectionType':
              defineGroup('intersection')
              break
            case 'TSUndefinedKeyword':
            case 'TSNullKeyword':
            case 'TSVoidKeyword':
              defineGroup('nullish')
              break
            case 'TSConditionalType':
              defineGroup('conditional')
              break
            case 'TSConstructorType':
            case 'TSFunctionType':
              defineGroup('function')
              break
            case 'TSBooleanKeyword':
            case 'TSUnknownKeyword':
            case 'TSBigIntKeyword':
            case 'TSNumberKeyword':
            case 'TSObjectKeyword':
            case 'TSStringKeyword':
            case 'TSSymbolKeyword':
            case 'TSNeverKeyword':
            case 'TSAnyKeyword':
            case 'TSThisType':
              defineGroup('keyword')
              break
            case 'TSTypeOperator':
            case 'TSTypeQuery':
              defineGroup('operator')
              break
            case 'TSTypeLiteral':
            case 'TSMappedType':
              defineGroup('object')
              break
            case 'TSImportType':
              defineGroup('import')
              break
            case 'TSTupleType':
              defineGroup('tuple')
              break
            case 'TSUnionType':
              defineGroup('union')
              break
          }
          let lastGroup = accumulator.at(-1)
          let lastSortingNode = lastGroup == null ? void 0 : lastGroup.at(-1)
          let sortingNode = {
            isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
              type,
              eslintDisabledLines,
            ),
            size: rangeToDiff.rangeToDiff(type, sourceCode),
            name: sourceCode.getText(type),
            group: getGroup(),
            node: type,
          }
          if (
            hasPartitionComment.hasPartitionComment({
              comments: getCommentsBefore.getCommentsBefore({
                tokenValueToIgnoreBefore: '|',
                node: type,
                sourceCode,
              }),
              partitionByComment: options.partitionByComment,
            }) ||
            (options.partitionByNewLine &&
              lastSortingNode &&
              getLinesBetween.getLinesBetween(
                sourceCode,
                lastSortingNode,
                sortingNode,
              ))
          ) {
            lastGroup = []
            accumulator.push(lastGroup)
          }
          lastGroup == null ? void 0 : lastGroup.push(sortingNode)
          return accumulator
        },
        [[]],
      )
      for (let nodes of formattedMembers) {
        let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
          sortNodesByGroups.sortNodesByGroups(nodes, options, {
            ignoreEslintDisabledNodes,
          })
        let sortedNodes = sortNodesExcludingEslintDisabled(false)
        let sortedNodesExcludingEslintDisabled =
          sortNodesExcludingEslintDisabled(true)
        let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
        pairwise.pairwise(nodes, (left, right) => {
          let leftNumber = getGroupNumber.getGroupNumber(options.groups, left)
          let rightNumber = getGroupNumber.getGroupNumber(options.groups, right)
          let leftIndex = nodeIndexMap.get(left)
          let rightIndex = nodeIndexMap.get(right)
          let indexOfRightExcludingEslintDisabled =
            sortedNodesExcludingEslintDisabled.indexOf(right)
          let messageIds = []
          if (
            leftIndex > rightIndex ||
            leftIndex >= indexOfRightExcludingEslintDisabled
          ) {
            messageIds.push(
              leftNumber === rightNumber
                ? 'unexpectedUnionTypesOrder'
                : 'unexpectedUnionTypesGroupOrder',
            )
          }
          messageIds = [
            ...messageIds,
            ...getNewlinesErrors.getNewlinesErrors({
              missedSpacingError: 'missedSpacingBetweenUnionTypes',
              extraSpacingError: 'extraSpacingBetweenUnionTypes',
              rightNum: rightNumber,
              leftNum: leftNumber,
              sourceCode,
              options,
              right,
              left,
            }),
          ]
          for (let messageId of messageIds) {
            context.report({
              fix: fixer => [
                ...makeFixes.makeFixes({
                  sortedNodes: sortedNodesExcludingEslintDisabled,
                  sourceCode,
                  options,
                  fixer,
                  nodes,
                }),
                ...makeNewlinesFixes.makeNewlinesFixes({
                  sortedNodes: sortedNodesExcludingEslintDisabled,
                  sourceCode,
                  options,
                  fixer,
                  nodes,
                }),
              ],
              data: {
                right: toSingleLine.toSingleLine(right.name),
                left: toSingleLine.toSingleLine(left.name),
                rightGroup: right.group,
                leftGroup: left.group,
              },
              node: right.node,
              messageId,
            })
          }
        })
      }
    },
  }),
  meta: {
    schema: [
      {
        properties: {
          partitionByComment: {
            ...commonJsonSchemas.partitionByCommentJsonSchema,
            description:
              'Allows you to use comments to separate the union types into logical groups.',
          },
          partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          newlinesBetween: commonJsonSchemas.newlinesBetweenJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          groups: commonJsonSchemas.groupsJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    messages: {
      unexpectedUnionTypesGroupOrder:
        'Expected "{{right}}" ({{rightGroup}}) to come before "{{left}}" ({{leftGroup}}).',
      missedSpacingBetweenUnionTypes:
        'Missed spacing between "{{left}}" and "{{right}}" types.',
      extraSpacingBetweenUnionTypes:
        'Extra spacing between "{{left}}" and "{{right}}" types.',
      unexpectedUnionTypesOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-union-types',
      description: 'Enforce sorted union types.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-union-types',
})
module.exports = sortUnionTypes
