'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const makeCommentAfterFixes = require('../utils/make-comment-after-fixes.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getSourceCode = require('../utils/get-source-code.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const makeFixes = require('../utils/make-fixes.js')
const sortNodes = require('../utils/sort-nodes.js')
const pairwise = require('../utils/pairwise.js')
const complete = require('../utils/complete.js')
const compare = require('../utils/compare.js')
let defaultOptions = {
  specialCharacters: 'keep',
  type: 'alphabetical',
  ignoreCase: true,
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
}
const sortSwitchCase = createEslintRule.createEslintRule({
  create: context => ({
    SwitchStatement: switchNode => {
      if (!isSortable.isSortable(switchNode.cases)) {
        return
      }
      let settings = getSettings.getSettings(context.settings)
      let options = complete.complete(
        context.options.at(0),
        settings,
        defaultOptions,
      )
      validateCustomSortConfiguration.validateCustomSortConfiguration(options)
      let sourceCode = getSourceCode.getSourceCode(context)
      let isDiscriminantTrue =
        switchNode.discriminant.type === 'Literal' &&
        switchNode.discriminant.value === true
      if (isDiscriminantTrue) {
        return
      }
      let caseNameSortingNodeGroups = switchNode.cases.reduce(
        (accumulator, caseNode, index) => {
          if (caseNode.test) {
            accumulator.at(-1).push({
              size: rangeToDiff.rangeToDiff(caseNode.test, sourceCode),
              name: getCaseName(sourceCode, caseNode),
              isEslintDisabled: false,
              node: caseNode.test,
            })
          }
          if (
            caseNode.consequent.length &&
            index !== switchNode.cases.length - 1
          ) {
            accumulator.push([])
          }
          return accumulator
        },
        [[]],
      )
      let hasUnsortedNodes = false
      for (let caseNodesSortingNodeGroup of caseNameSortingNodeGroups) {
        let sortedCaseNameSortingNodes = sortNodes.sortNodes(
          caseNodesSortingNodeGroup,
          options,
        )
        hasUnsortedNodes ||
          (hasUnsortedNodes = sortedCaseNameSortingNodes.some(
            (node, index) => node !== caseNodesSortingNodeGroup[index],
          ))
        let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(
          sortedCaseNameSortingNodes,
        )
        pairwise.pairwise(caseNodesSortingNodeGroup, (left, right) => {
          let leftIndex = nodeIndexMap.get(left)
          let rightIndex = nodeIndexMap.get(right)
          if (leftIndex < rightIndex) {
            return
          }
          context.report({
            fix: fixer =>
              makeFixes.makeFixes({
                sortedNodes: sortedCaseNameSortingNodes,
                nodes: caseNodesSortingNodeGroup,
                sourceCode,
                fixer,
              }),
            data: {
              right: right.name,
              left: left.name,
            },
            messageId: 'unexpectedSwitchCaseOrder',
            node: right.node,
          })
        })
      }
      let sortingNodes = switchNode.cases.map(caseNode => ({
        size: caseNode.test
          ? rangeToDiff.rangeToDiff(caseNode.test, sourceCode)
          : 'default'.length,
        name: getCaseName(sourceCode, caseNode),
        addSafetySemicolonWhenInline: true,
        isDefaultClause: !caseNode.test,
        isEslintDisabled: false,
        node: caseNode,
      }))
      let sortingNodeGroupsForDefaultSort = reduceCaseSortingNodes(
        sortingNodes,
        caseNode => !!caseNode.node.consequent.length,
      )
      let sortingNodesGroupWithDefault = sortingNodeGroupsForDefaultSort.find(
        caseNodeGroup => caseNodeGroup.some(node => node.isDefaultClause),
      )
      if (
        sortingNodesGroupWithDefault &&
        !sortingNodesGroupWithDefault.at(-1).isDefaultClause
      ) {
        let defaultCase = sortingNodesGroupWithDefault.find(
          node => node.isDefaultClause,
        )
        let lastCase = sortingNodesGroupWithDefault.at(-1)
        context.report({
          fix: fixer => {
            let punctuatorAfterLastCase = sourceCode.getTokenAfter(
              lastCase.node.test,
            )
            let lastCaseRange = [
              lastCase.node.range[0],
              punctuatorAfterLastCase.range[1],
            ]
            return [
              fixer.replaceText(
                defaultCase.node,
                sourceCode.text.slice(...lastCaseRange),
              ),
              fixer.replaceTextRange(
                lastCaseRange,
                sourceCode.getText(defaultCase.node),
              ),
              ...makeCommentAfterFixes.makeCommentAfterFixes({
                sortedNode: punctuatorAfterLastCase,
                node: defaultCase.node,
                sourceCode,
                fixer,
              }),
              ...makeCommentAfterFixes.makeCommentAfterFixes({
                node: punctuatorAfterLastCase,
                sortedNode: defaultCase.node,
                sourceCode,
                fixer,
              }),
            ]
          },
          data: {
            left: defaultCase.name,
            right: lastCase.name,
          },
          messageId: 'unexpectedSwitchCaseOrder',
          node: defaultCase.node,
        })
      }
      let sortingNodeGroupsForBlockSort = reduceCaseSortingNodes(
        sortingNodes,
        caseNode => caseHasBreakOrReturn(caseNode.node),
      )
      let lastNodeGroup = sortingNodeGroupsForBlockSort.at(-1)
      let lastBlockCaseShouldStayInPlace = !caseHasBreakOrReturn(
        lastNodeGroup.at(-1).node,
      )
      let sortedSortingNodeGroupsForBlockSort = [
        ...sortingNodeGroupsForBlockSort,
      ]
        .sort((a, b) => {
          if (lastBlockCaseShouldStayInPlace) {
            if (a === lastNodeGroup) {
              return 1
            }
            if (b === lastNodeGroup) {
              return -1
            }
          }
          if (a.some(node => node.isDefaultClause)) {
            return 1
          }
          if (b.some(node => node.isDefaultClause)) {
            return -1
          }
          return compare.compare(a.at(0), b.at(0), options)
        })
        .flat()
      let sortingNodeGroupsForBlockSortFlat =
        sortingNodeGroupsForBlockSort.flat()
      pairwise.pairwise(sortingNodeGroupsForBlockSortFlat, (left, right) => {
        let indexOfLeft = sortedSortingNodeGroupsForBlockSort.indexOf(left)
        let indexOfRight = sortedSortingNodeGroupsForBlockSort.indexOf(right)
        if (indexOfLeft < indexOfRight) {
          return
        }
        context.report({
          fix: fixer =>
            hasUnsortedNodes
              ? []
              : makeFixes.makeFixes({
                  sortedNodes: sortedSortingNodeGroupsForBlockSort,
                  nodes: sortingNodeGroupsForBlockSortFlat,
                  sourceCode,
                  fixer,
                }),
          data: {
            right: right.name,
            left: left.name,
          },
          messageId: 'unexpectedSwitchCaseOrder',
          node: right.node,
        })
      })
    },
  }),
  meta: {
    schema: [
      {
        properties: {
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    docs: {
      url: 'https://perfectionist.dev/rules/sort-switch-case',
      description: 'Enforce sorted switch cases.',
      recommended: true,
    },
    messages: {
      unexpectedSwitchCaseOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-switch-case',
})
let getCaseName = (sourceCode, caseNode) => {
  var _a
  if (((_a = caseNode.test) == null ? void 0 : _a.type) === 'Literal') {
    return `${caseNode.test.value}`
  } else if (caseNode.test === null) {
    return 'default'
  }
  return sourceCode.getText(caseNode.test)
}
let reduceCaseSortingNodes = (caseNodes, endsBlock) =>
  caseNodes.reduce(
    (accumulator, caseNode, index) => {
      accumulator.at(-1).push(caseNode)
      if (endsBlock(caseNode) && index !== caseNodes.length - 1) {
        accumulator.push([])
      }
      return accumulator
    },
    [[]],
  )
let caseHasBreakOrReturn = caseNode => {
  var _a
  let statements =
    ((_a = caseNode.consequent[0]) == null ? void 0 : _a.type) ===
    'BlockStatement'
      ? caseNode.consequent[0].body
      : caseNode.consequent
  return statements.some(statementIsBreakOrReturn)
}
let statementIsBreakOrReturn = statement =>
  statement.type === 'BreakStatement' || statement.type === 'ReturnStatement'
module.exports = sortSwitchCase
