'use strict'
const types = require('@typescript-eslint/types')
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const sortNodesByDependencies = require('../utils/sort-nodes-by-dependencies.js')
const validateNewlinesAndPartitionConfiguration = require('../utils/validate-newlines-and-partition-configuration.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const getFirstNodeParentWithType = require('./sort-objects/get-first-node-parent-with-type.js')
const validateGroupsConfiguration = require('../utils/validate-groups-configuration.js')
const getMatchingContextOptions = require('../utils/get-matching-context-options.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const sortNodesByGroups = require('../utils/sort-nodes-by-groups.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const makeNewlinesFixes = require('../utils/make-newlines-fixes.js')
const getNewlinesErrors = require('../utils/get-newlines-errors.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getGroupNumber = require('../utils/get-group-number.js')
const getSourceCode = require('../utils/get-source-code.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const useGroups = require('../utils/use-groups.js')
const makeFixes = require('../utils/make-fixes.js')
const sortNodes = require('../utils/sort-nodes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
const matches = require('../utils/matches.js')
let defaultOptions = {
  partitionByNewLine: false,
  partitionByComment: false,
  newlinesBetween: 'ignore',
  specialCharacters: 'keep',
  destructuredObjects: true,
  objectDeclarations: true,
  styledComponents: true,
  destructureOnly: false,
  useConfigurationIf: {},
  type: 'alphabetical',
  ignorePattern: [],
  ignoreCase: true,
  customGroups: {},
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
  groups: [],
}
const sortObjects = createEslintRule.createEslintRule({
  create: context => {
    let settings = getSettings.getSettings(context.settings)
    let sourceCode = getSourceCode.getSourceCode(context)
    let sortObject = nodeObject => {
      if (!isSortable.isSortable(nodeObject.properties)) {
        return
      }
      let objectParent = getObjectParent({
        onlyFirstParent: true,
        node: nodeObject,
      })
      let matchedContextOptions = getMatchingContextOptions
        .getMatchingContextOptions({
          nodeNames: nodeObject.properties
            .map(property => getNodeName({ sourceCode, property }))
            .filter(nodeName => nodeName !== null),
          contextOptions: context.options,
        })
        .find(options2 => {
          var _a
          if (
            !((_a = options2.useConfigurationIf) == null
              ? void 0
              : _a.callingFunctionNamePattern)
          ) {
            return true
          }
          if (
            (objectParent == null ? void 0 : objectParent.type) ===
              'VariableDeclarator' ||
            !(objectParent == null ? void 0 : objectParent.name)
          ) {
            return false
          }
          return matches.matches(
            objectParent.name,
            options2.useConfigurationIf.callingFunctionNamePattern,
          )
        })
      let completeOptions = complete.complete(
        matchedContextOptions,
        settings,
        defaultOptions,
      )
      let { type } = completeOptions
      if (type === 'unsorted') {
        return
      }
      let options = {
        ...completeOptions,
        type,
      }
      validateCustomSortConfiguration.validateCustomSortConfiguration(options)
      validateGroupsConfiguration.validateGroupsConfiguration(
        options.groups,
        ['multiline', 'method', 'unknown'],
        Object.keys(options.customGroups),
      )
      validateNewlinesAndPartitionConfiguration.validateNewlinesAndPartitionConfiguration(
        options,
      )
      let isDestructuredObject = nodeObject.type === 'ObjectPattern'
      if (isDestructuredObject) {
        if (!options.destructuredObjects) {
          return
        }
      } else if (options.destructureOnly || !options.objectDeclarations) {
        return
      }
      let objectParentForIgnorePattern = getObjectParent({
        onlyFirstParent: false,
        node: nodeObject,
      })
      if (
        (objectParentForIgnorePattern == null
          ? void 0
          : objectParentForIgnorePattern.name) &&
        options.ignorePattern.some(pattern =>
          matches.matches(objectParentForIgnorePattern.name, pattern),
        )
      ) {
        return
      }
      let isStyledCallExpression = identifier =>
        identifier.type === 'Identifier' && identifier.name === 'styled'
      let isCssCallExpression = identifier =>
        identifier.type === 'Identifier' && identifier.name === 'css'
      let isStyledComponents = styledNode =>
        !!styledNode &&
        ((styledNode.type === 'CallExpression' &&
          (isCssCallExpression(styledNode.callee) ||
            (styledNode.callee.type === 'MemberExpression' &&
              isStyledCallExpression(styledNode.callee.object)) ||
            (styledNode.callee.type === 'CallExpression' &&
              isStyledCallExpression(styledNode.callee.callee)))) ||
          (styledNode.type === 'JSXExpressionContainer' &&
            styledNode.parent.type === 'JSXAttribute' &&
            styledNode.parent.name.name === 'style'))
      if (
        !options.styledComponents &&
        (isStyledComponents(nodeObject.parent) ||
          (nodeObject.parent.type === 'ArrowFunctionExpression' &&
            isStyledComponents(nodeObject.parent.parent)))
      ) {
        return
      }
      let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
        ruleName: context.id,
        sourceCode,
      })
      let extractDependencies = init => {
        let dependencies = []
        let checkNode = nodeValue => {
          if (
            nodeValue.type === 'ArrowFunctionExpression' ||
            nodeValue.type === 'FunctionExpression'
          ) {
            return
          }
          if (nodeValue.type === 'Identifier') {
            dependencies.push(nodeValue.name)
          }
          if (nodeValue.type === 'Property') {
            traverseNode(nodeValue.key)
            traverseNode(nodeValue.value)
          }
          if (nodeValue.type === 'ConditionalExpression') {
            traverseNode(nodeValue.test)
            traverseNode(nodeValue.consequent)
            traverseNode(nodeValue.alternate)
          }
          if (
            'expression' in nodeValue &&
            typeof nodeValue.expression !== 'boolean'
          ) {
            traverseNode(nodeValue.expression)
          }
          if ('object' in nodeValue) {
            traverseNode(nodeValue.object)
          }
          if ('callee' in nodeValue) {
            traverseNode(nodeValue.callee)
          }
          if ('left' in nodeValue) {
            traverseNode(nodeValue.left)
          }
          if ('right' in nodeValue) {
            traverseNode(nodeValue.right)
          }
          if ('elements' in nodeValue) {
            let elements = nodeValue.elements.filter(
              currentNode => currentNode !== null,
            )
            for (let element of elements) {
              traverseNode(element)
            }
          }
          if ('argument' in nodeValue && nodeValue.argument) {
            traverseNode(nodeValue.argument)
          }
          if ('arguments' in nodeValue) {
            for (let argument of nodeValue.arguments) {
              traverseNode(argument)
            }
          }
          if ('properties' in nodeValue) {
            for (let property of nodeValue.properties) {
              traverseNode(property)
            }
          }
          if ('expressions' in nodeValue) {
            for (let nodeExpression of nodeValue.expressions) {
              traverseNode(nodeExpression)
            }
          }
        }
        let traverseNode = nodeValue => {
          checkNode(nodeValue)
        }
        traverseNode(init)
        return dependencies
      }
      let formatProperties = props =>
        props.reduce(
          (accumulator, property) => {
            var _a
            if (
              property.type === 'SpreadElement' ||
              property.type === 'RestElement'
            ) {
              accumulator.push([])
              return accumulator
            }
            let lastProperty =
              (_a = accumulator.at(-1)) == null ? void 0 : _a.at(-1)
            let name
            let dependencies = []
            let { setCustomGroups, defineGroup, getGroup } =
              useGroups.useGroups(options)
            if (property.key.type === 'Identifier') {
              ;({ name } = property.key)
            } else if (property.key.type === 'Literal') {
              name = `${property.key.value}`
            } else {
              name = sourceCode.getText(property.key)
            }
            if (property.value.type === 'AssignmentPattern') {
              dependencies = extractDependencies(property.value)
            }
            setCustomGroups(options.customGroups, name)
            if (
              property.value.type === 'ArrowFunctionExpression' ||
              property.value.type === 'FunctionExpression'
            ) {
              defineGroup('method')
            }
            if (property.loc.start.line !== property.loc.end.line) {
              defineGroup('multiline')
            }
            let propertySortingNode = {
              isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
                property,
                eslintDisabledLines,
              ),
              size: rangeToDiff.rangeToDiff(property, sourceCode),
              group: getGroup(),
              node: property,
              dependencies,
              name,
            }
            if (
              (options.partitionByNewLine &&
                lastProperty &&
                getLinesBetween.getLinesBetween(
                  sourceCode,
                  lastProperty,
                  propertySortingNode,
                )) ||
              hasPartitionComment.hasPartitionComment({
                comments: getCommentsBefore.getCommentsBefore({
                  node: property,
                  sourceCode,
                }),
                partitionByComment: options.partitionByComment,
              })
            ) {
              accumulator.push([])
            }
            accumulator.at(-1).push(propertySortingNode)
            return accumulator
          },
          [[]],
        )
      let formattedMembers = formatProperties(nodeObject.properties)
      let nodesSortingFunction =
        isDestructuredObject &&
        typeof options.destructuredObjects === 'object' &&
        !options.destructuredObjects.groups
          ? sortNodes.sortNodes
          : sortNodesByGroups.sortNodesByGroups
      let sortNodesIgnoringEslintDisabledNodes = ignoreEslintDisabledNodes =>
        sortNodesByDependencies.sortNodesByDependencies(
          formattedMembers.flatMap(nodes2 =>
            nodesSortingFunction(nodes2, options, {
              ignoreEslintDisabledNodes,
            }),
          ),
          {
            ignoreEslintDisabledNodes,
          },
        )
      let sortedNodes = sortNodesIgnoringEslintDisabledNodes(false)
      let sortedNodesExcludingEslintDisabled =
        sortNodesIgnoringEslintDisabledNodes(true)
      let nodes = formattedMembers.flat()
      let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
      pairwise.pairwise(nodes, (left, right) => {
        let leftNumber = getGroupNumber.getGroupNumber(options.groups, left)
        let rightNumber = getGroupNumber.getGroupNumber(options.groups, right)
        let leftIndex = nodeIndexMap.get(left)
        let rightIndex = nodeIndexMap.get(right)
        let indexOfRightExcludingEslintDisabled =
          sortedNodesExcludingEslintDisabled.indexOf(right)
        let messageIds = []
        let firstUnorderedNodeDependentOnRight
        if (
          leftIndex > rightIndex ||
          leftIndex >= indexOfRightExcludingEslintDisabled
        ) {
          firstUnorderedNodeDependentOnRight =
            sortNodesByDependencies.getFirstUnorderedNodeDependentOn(
              right,
              nodes,
            )
          if (firstUnorderedNodeDependentOnRight) {
            messageIds.push('unexpectedObjectsDependencyOrder')
          } else {
            messageIds.push(
              leftNumber === rightNumber
                ? 'unexpectedObjectsOrder'
                : 'unexpectedObjectsGroupOrder',
            )
          }
        }
        messageIds = [
          ...messageIds,
          ...getNewlinesErrors.getNewlinesErrors({
            missedSpacingError: 'missedSpacingBetweenObjectMembers',
            extraSpacingError: 'extraSpacingBetweenObjectMembers',
            rightNum: rightNumber,
            leftNum: leftNumber,
            sourceCode,
            options,
            right,
            left,
          }),
        ]
        for (let messageId of messageIds) {
          context.report({
            fix: fixer => [
              ...makeFixes.makeFixes({
                sortedNodes: sortedNodesExcludingEslintDisabled,
                sourceCode,
                options,
                fixer,
                nodes,
              }),
              ...makeNewlinesFixes.makeNewlinesFixes({
                sortedNodes: sortedNodesExcludingEslintDisabled,
                sourceCode,
                options,
                fixer,
                nodes,
              }),
            ],
            data: {
              nodeDependentOnRight:
                firstUnorderedNodeDependentOnRight == null
                  ? void 0
                  : firstUnorderedNodeDependentOnRight.name,
              rightGroup: right.group,
              leftGroup: left.group,
              right: right.name,
              left: left.name,
            },
            node: right.node,
            messageId,
          })
        }
      })
    }
    return {
      ObjectExpression: sortObject,
      ObjectPattern: sortObject,
    }
  },
  meta: {
    schema: {
      items: {
        properties: {
          destructuredObjects: {
            oneOf: [
              {
                type: 'boolean',
              },
              {
                properties: {
                  groups: {
                    description:
                      'Controls whether to use groups to sort destructured objects.',
                    type: 'boolean',
                  },
                },
                additionalProperties: false,
                type: 'object',
              },
            ],
            description: 'Controls whether to sort destructured objects.',
          },
          ignorePattern: {
            description:
              'Specifies names or patterns for nodes that should be ignored by rule.',
            items: {
              type: 'string',
            },
            type: 'array',
          },
          useConfigurationIf:
            commonJsonSchemas.buildUseConfigurationIfJsonSchema({
              additionalProperties: {
                callingFunctionNamePattern: {
                  type: 'string',
                },
              },
            }),
          partitionByComment: {
            ...commonJsonSchemas.partitionByCommentJsonSchema,
            description:
              'Allows you to use comments to separate the keys of objects into logical groups.',
          },
          destructureOnly: {
            description: 'Controls whether to sort only destructured objects.',
            type: 'boolean',
          },
          objectDeclarations: {
            description: 'Controls whether to sort object declarations.',
            type: 'boolean',
          },
          styledComponents: {
            description: 'Controls whether to sort styled components.',
            type: 'boolean',
          },
          type: commonJsonSchemas.buildTypeJsonSchema({ withUnsorted: true }),
          partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          newlinesBetween: commonJsonSchemas.newlinesBetweenJsonSchema,
          customGroups: commonJsonSchemas.customGroupsJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          locales: commonJsonSchemas.localesJsonSchema,
          groups: commonJsonSchemas.groupsJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
      uniqueItems: true,
      type: 'array',
    },
    messages: {
      unexpectedObjectsGroupOrder:
        'Expected "{{right}}" ({{rightGroup}}) to come before "{{left}}" ({{leftGroup}}).',
      unexpectedObjectsDependencyOrder:
        'Expected dependency "{{right}}" to come before "{{nodeDependentOnRight}}".',
      missedSpacingBetweenObjectMembers:
        'Missed spacing between "{{left}}" and "{{right}}" objects.',
      extraSpacingBetweenObjectMembers:
        'Extra spacing between "{{left}}" and "{{right}}" objects.',
      unexpectedObjectsOrder: 'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-objects',
      description: 'Enforce sorted objects.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-objects',
})
let getNodeName = ({ sourceCode, property }) => {
  if (property.type === 'SpreadElement' || property.type === 'RestElement') {
    return null
  }
  if (property.key.type === 'Identifier') {
    return property.key.name
  } else if (property.key.type === 'Literal') {
    return `${property.key.value}`
  }
  return sourceCode.getText(property.key)
}
let getObjectParent = ({ onlyFirstParent, node }) => {
  let variableParentName = getVariableParentName({ onlyFirstParent, node })
  if (variableParentName) {
    return {
      type: 'VariableDeclarator',
      name: variableParentName,
    }
  }
  let callParentName = getCallExpressionParentName({
    onlyFirstParent,
    node,
  })
  if (callParentName) {
    return {
      type: 'CallExpression',
      name: callParentName,
    }
  }
  return null
}
let getVariableParentName = ({ onlyFirstParent, node }) => {
  let variableParent = getFirstNodeParentWithType.getFirstNodeParentWithType({
    allowedTypes: [
      types.TSESTree.AST_NODE_TYPES.VariableDeclarator,
      types.TSESTree.AST_NODE_TYPES.Property,
    ],
    onlyFirstParent,
    node,
  })
  if (!variableParent) {
    return null
  }
  let parentId
  if (variableParent.type === 'VariableDeclarator') {
    parentId = variableParent.id
  } else if ('key' in variableParent) {
    parentId = variableParent.key
  } else {
    return null
  }
  return parentId.type === 'Identifier' ? parentId.name : null
}
let getCallExpressionParentName = ({ onlyFirstParent, node }) => {
  let callParent = getFirstNodeParentWithType.getFirstNodeParentWithType({
    allowedTypes: [types.TSESTree.AST_NODE_TYPES.CallExpression],
    onlyFirstParent,
    node,
  })
  if (!callParent) {
    return null
  }
  return callParent.callee.type === 'Identifier' ? callParent.callee.name : null
}
module.exports = sortObjects
