'use strict'
Object.defineProperties(exports, {
  __esModule: { value: true },
  [Symbol.toStringTag]: { value: 'Module' },
})
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateNewlinesAndPartitionConfiguration = require('../utils/validate-newlines-and-partition-configuration.js')
const types = require('./sort-object-types/types.js')
const validateGeneratedGroupsConfiguration = require('../utils/validate-generated-groups-configuration.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const getCustomGroupsCompareOptions = require('../utils/get-custom-groups-compare-options.js')
const doesCustomGroupMatch = require('./sort-object-types/does-custom-group-match.js')
const getMatchingContextOptions = require('../utils/get-matching-context-options.js')
const generatePredefinedGroups = require('../utils/generate-predefined-groups.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isMemberOptional = require('./sort-object-types/is-member-optional.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const isNodeFunctionType = require('../utils/is-node-function-type.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const sortNodesByGroups = require('../utils/sort-nodes-by-groups.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const makeNewlinesFixes = require('../utils/make-newlines-fixes.js')
const getNewlinesErrors = require('../utils/get-newlines-errors.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getGroupNumber = require('../utils/get-group-number.js')
const getSourceCode = require('../utils/get-source-code.js')
const toSingleLine = require('../utils/to-single-line.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const makeFixes = require('../utils/make-fixes.js')
const useGroups = require('../utils/use-groups.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
const matches = require('../utils/matches.js')
let cachedGroupsByModifiersAndSelectors = /* @__PURE__ */ new Map()
let defaultOptions = {
  partitionByComment: false,
  partitionByNewLine: false,
  newlinesBetween: 'ignore',
  specialCharacters: 'keep',
  useConfigurationIf: {},
  type: 'alphabetical',
  groupKind: 'mixed',
  ignorePattern: [],
  ignoreCase: true,
  customGroups: {},
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
  groups: [],
}
let jsonSchema = {
  items: {
    properties: {
      ignorePattern: {
        description:
          'Specifies names or patterns for nodes that should be ignored by rule.',
        items: {
          type: 'string',
        },
        type: 'array',
      },
      useConfigurationIf: commonJsonSchemas.buildUseConfigurationIfJsonSchema({
        additionalProperties: {
          declarationMatchesPattern: {
            type: 'string',
          },
        },
      }),
      partitionByComment: {
        ...commonJsonSchemas.partitionByCommentJsonSchema,
        description:
          'Allows you to use comments to separate members into logical groups.',
      },
      customGroups: {
        oneOf: [
          commonJsonSchemas.customGroupsJsonSchema,
          commonJsonSchemas.buildCustomGroupsArrayJsonSchema({
            singleCustomGroupJsonSchema: types.singleCustomGroupJsonSchema,
          }),
        ],
      },
      groupKind: {
        enum: ['mixed', 'required-first', 'optional-first'],
        description: 'Specifies top-level groups.',
        type: 'string',
      },
      type: commonJsonSchemas.buildTypeJsonSchema({ withUnsorted: true }),
      partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
      specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
      newlinesBetween: commonJsonSchemas.newlinesBetweenJsonSchema,
      ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
      alphabet: commonJsonSchemas.alphabetJsonSchema,
      locales: commonJsonSchemas.localesJsonSchema,
      groups: commonJsonSchemas.groupsJsonSchema,
      order: commonJsonSchemas.orderJsonSchema,
    },
    additionalProperties: false,
    type: 'object',
  },
  uniqueItems: true,
  type: 'array',
}
const sortObjectTypes = createEslintRule.createEslintRule({
  meta: {
    messages: {
      unexpectedObjectTypesGroupOrder:
        'Expected "{{right}}" ({{rightGroup}}) to come before "{{left}}" ({{leftGroup}}).',
      missedSpacingBetweenObjectTypeMembers:
        'Missed spacing between "{{left}}" and "{{right}}" types.',
      extraSpacingBetweenObjectTypeMembers:
        'Extra spacing between "{{left}}" and "{{right}}" types.',
      unexpectedObjectTypesOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-object-types',
      description: 'Enforce sorted object types.',
      recommended: true,
    },
    schema: jsonSchema,
    type: 'suggestion',
    fixable: 'code',
  },
  create: context => ({
    TSTypeLiteral: node =>
      sortObjectTypeElements({
        availableMessageIds: {
          missedSpacingBetweenMembers: 'missedSpacingBetweenObjectTypeMembers',
          extraSpacingBetweenMembers: 'extraSpacingBetweenObjectTypeMembers',
          unexpectedGroupOrder: 'unexpectedObjectTypesGroupOrder',
          unexpectedOrder: 'unexpectedObjectTypesOrder',
        },
        parentNodeName:
          node.parent.type === 'TSTypeAliasDeclaration'
            ? node.parent.id.name
            : null,
        elements: node.members,
        context,
      }),
  }),
  defaultOptions: [defaultOptions],
  name: 'sort-object-types',
})
let sortObjectTypeElements = ({
  availableMessageIds,
  parentNodeName,
  elements,
  context,
}) => {
  if (!isSortable.isSortable(elements)) {
    return
  }
  let settings = getSettings.getSettings(context.settings)
  let sourceCode = getSourceCode.getSourceCode(context)
  let matchedContextOptions = getMatchingContextOptions
    .getMatchingContextOptions({
      nodeNames: elements.map(node =>
        getNodeName({ typeElement: node, sourceCode }),
      ),
      contextOptions: context.options,
    })
    .find(options2 => {
      var _a
      if (
        !((_a = options2.useConfigurationIf) == null
          ? void 0
          : _a.declarationMatchesPattern)
      ) {
        return true
      }
      if (!parentNodeName) {
        return false
      }
      return matches.matches(
        parentNodeName,
        options2.useConfigurationIf.declarationMatchesPattern,
      )
    })
  let completeOptions = complete.complete(
    matchedContextOptions,
    settings,
    defaultOptions,
  )
  let { type } = completeOptions
  if (type === 'unsorted') {
    return
  }
  let options = {
    ...completeOptions,
    type,
  }
  validateCustomSortConfiguration.validateCustomSortConfiguration(options)
  validateGeneratedGroupsConfiguration.validateGeneratedGroupsConfiguration({
    customGroups: options.customGroups,
    selectors: types.allSelectors,
    modifiers: types.allModifiers,
    groups: options.groups,
  })
  validateNewlinesAndPartitionConfiguration.validateNewlinesAndPartitionConfiguration(
    options,
  )
  if (
    options.ignorePattern.some(
      pattern => parentNodeName && matches.matches(parentNodeName, pattern),
    )
  ) {
    return
  }
  let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
    ruleName: context.id,
    sourceCode,
  })
  let formattedMembers = elements.reduce(
    (accumulator, typeElement) => {
      if (
        typeElement.type === 'TSCallSignatureDeclaration' ||
        typeElement.type === 'TSConstructSignatureDeclaration'
      ) {
        accumulator.push([])
        return accumulator
      }
      let lastGroup = accumulator.at(-1)
      let lastSortingNode = lastGroup == null ? void 0 : lastGroup.at(-1)
      let { setCustomGroups, defineGroup, getGroup } =
        useGroups.useGroups(options)
      let selectors = []
      let modifiers = []
      if (typeElement.type === 'TSIndexSignature') {
        selectors.push('index-signature')
      }
      if (isNodeFunctionType.isNodeFunctionType(typeElement)) {
        selectors.push('method')
      }
      if (typeElement.loc.start.line !== typeElement.loc.end.line) {
        modifiers.push('multiline')
        selectors.push('multiline')
      }
      if (
        !['index-signature', 'method'].some(selector =>
          selectors.includes(selector),
        )
      ) {
        selectors.push('property')
      }
      selectors.push('member')
      if (isMemberOptional.isMemberOptional(typeElement)) {
        modifiers.push('optional')
      } else {
        modifiers.push('required')
      }
      let predefinedGroups = generatePredefinedGroups.generatePredefinedGroups({
        cache: cachedGroupsByModifiersAndSelectors,
        selectors,
        modifiers,
      })
      for (let predefinedGroup of predefinedGroups) {
        defineGroup(predefinedGroup)
      }
      let name = getNodeName({ typeElement, sourceCode })
      if (Array.isArray(options.customGroups)) {
        for (let customGroup of options.customGroups) {
          if (
            doesCustomGroupMatch.doesCustomGroupMatch({
              elementName: name,
              customGroup,
              selectors,
              modifiers,
            })
          ) {
            defineGroup(customGroup.groupName, true)
            if (getGroup() === customGroup.groupName) {
              break
            }
          }
        }
      } else {
        setCustomGroups(options.customGroups, name, {
          override: true,
        })
      }
      let sortingNode = {
        isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
          typeElement,
          eslintDisabledLines,
        ),
        groupKind: isMemberOptional.isMemberOptional(typeElement)
          ? 'optional'
          : 'required',
        size: rangeToDiff.rangeToDiff(typeElement, sourceCode),
        addSafetySemicolonWhenInline: true,
        group: getGroup(),
        node: typeElement,
        name,
      }
      if (
        (options.partitionByComment &&
          hasPartitionComment.hasPartitionComment({
            comments: getCommentsBefore.getCommentsBefore({
              node: typeElement,
              sourceCode,
            }),
            partitionByComment: options.partitionByComment,
          })) ||
        (options.partitionByNewLine &&
          lastSortingNode &&
          getLinesBetween.getLinesBetween(
            sourceCode,
            lastSortingNode,
            sortingNode,
          ))
      ) {
        lastGroup = []
        accumulator.push(lastGroup)
      }
      lastGroup == null ? void 0 : lastGroup.push(sortingNode)
      return accumulator
    },
    [[]],
  )
  let groupKindOrder
  if (options.groupKind === 'required-first') {
    groupKindOrder = ['required', 'optional']
  } else if (options.groupKind === 'optional-first') {
    groupKindOrder = ['optional', 'required']
  } else {
    groupKindOrder = ['any']
  }
  for (let nodes of formattedMembers) {
    let filteredGroupKindNodes = groupKindOrder.map(groupKind =>
      nodes.filter(
        currentNode =>
          groupKind === 'any' || currentNode.groupKind === groupKind,
      ),
    )
    let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
      filteredGroupKindNodes.flatMap(groupedNodes =>
        sortNodesByGroups.sortNodesByGroups(groupedNodes, options, {
          getGroupCompareOptions: groupNumber =>
            getCustomGroupsCompareOptions.getCustomGroupsCompareOptions(
              options,
              groupNumber,
            ),
          ignoreEslintDisabledNodes,
        }),
      )
    let sortedNodes = sortNodesExcludingEslintDisabled(false)
    let sortedNodesExcludingEslintDisabled =
      sortNodesExcludingEslintDisabled(true)
    let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
    pairwise.pairwise(nodes, (left, right) => {
      let leftNumber = getGroupNumber.getGroupNumber(options.groups, left)
      let rightNumber = getGroupNumber.getGroupNumber(options.groups, right)
      let leftIndex = nodeIndexMap.get(left)
      let rightIndex = nodeIndexMap.get(right)
      let indexOfRightExcludingEslintDisabled =
        sortedNodesExcludingEslintDisabled.indexOf(right)
      let messageIds = []
      if (
        leftIndex > rightIndex ||
        leftIndex >= indexOfRightExcludingEslintDisabled
      ) {
        messageIds.push(
          leftNumber === rightNumber
            ? availableMessageIds.unexpectedOrder
            : availableMessageIds.unexpectedGroupOrder,
        )
      }
      messageIds = [
        ...messageIds,
        ...getNewlinesErrors.getNewlinesErrors({
          missedSpacingError: availableMessageIds.missedSpacingBetweenMembers,
          extraSpacingError: availableMessageIds.extraSpacingBetweenMembers,
          rightNum: rightNumber,
          leftNum: leftNumber,
          sourceCode,
          options,
          right,
          left,
        }),
      ]
      for (let messageId of messageIds) {
        context.report({
          fix: fixer => [
            ...makeFixes.makeFixes({
              sortedNodes: sortedNodesExcludingEslintDisabled,
              sourceCode,
              options,
              fixer,
              nodes,
            }),
            ...makeNewlinesFixes.makeNewlinesFixes({
              sortedNodes: sortedNodesExcludingEslintDisabled,
              sourceCode,
              options,
              fixer,
              nodes,
            }),
          ],
          data: {
            right: toSingleLine.toSingleLine(right.name),
            left: toSingleLine.toSingleLine(left.name),
            rightGroup: right.group,
            leftGroup: left.group,
          },
          node: right.node,
          messageId,
        })
      }
    })
  }
}
let getNodeName = ({ typeElement, sourceCode }) => {
  var _a, _b
  let name
  let formatName = value => value.replace(/[,;]$/u, '')
  if (typeElement.type === 'TSPropertySignature') {
    if (typeElement.key.type === 'Identifier') {
      ;({ name } = typeElement.key)
    } else if (typeElement.key.type === 'Literal') {
      name = `${typeElement.key.value}`
    } else {
      let end =
        ((_a = typeElement.typeAnnotation) == null ? void 0 : _a.range.at(0)) ??
        typeElement.range.at(1) - (typeElement.optional ? '?'.length : 0)
      name = sourceCode.text.slice(typeElement.range.at(0), end)
    }
  } else if (typeElement.type === 'TSIndexSignature') {
    let endIndex =
      ((_b = typeElement.typeAnnotation) == null ? void 0 : _b.range.at(0)) ??
      typeElement.range.at(1)
    name = formatName(sourceCode.text.slice(typeElement.range.at(0), endIndex))
  } else if (
    typeElement.type === 'TSMethodSignature' &&
    'name' in typeElement.key
  ) {
    ;({ name } = typeElement.key)
  } else {
    name = formatName(
      sourceCode.text.slice(typeElement.range.at(0), typeElement.range.at(1)),
    )
  }
  return name
}
exports.default = sortObjectTypes
exports.jsonSchema = jsonSchema
exports.sortObjectTypeElements = sortObjectTypeElements
