'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getSourceCode = require('../utils/get-source-code.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const sortNodes = require('../utils/sort-nodes.js')
const makeFixes = require('../utils/make-fixes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
let defaultOptions = {
  specialCharacters: 'keep',
  partitionByNewLine: false,
  partitionByComment: false,
  type: 'alphabetical',
  groupKind: 'mixed',
  ignoreCase: true,
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
}
const sortNamedExports = createEslintRule.createEslintRule({
  create: context => ({
    ExportNamedDeclaration: node => {
      var _a
      if (!isSortable.isSortable(node.specifiers)) {
        return
      }
      let settings = getSettings.getSettings(context.settings)
      let options = complete.complete(
        context.options.at(0),
        settings,
        defaultOptions,
      )
      validateCustomSortConfiguration.validateCustomSortConfiguration(options)
      let sourceCode = getSourceCode.getSourceCode(context)
      let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
        ruleName: context.id,
        sourceCode,
      })
      let formattedMembers = [[]]
      for (let specifier of node.specifiers) {
        let groupKind = specifier.exportKind === 'type' ? 'type' : 'value'
        let name
        if (specifier.exported.type === 'Identifier') {
          ;({ name } = specifier.exported)
        } else {
          name = specifier.exported.value
        }
        let lastSortingNode =
          (_a = formattedMembers.at(-1)) == null ? void 0 : _a.at(-1)
        let sortingNode = {
          isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
            specifier,
            eslintDisabledLines,
          ),
          size: rangeToDiff.rangeToDiff(specifier, sourceCode),
          node: specifier,
          groupKind,
          name,
        }
        if (
          hasPartitionComment.hasPartitionComment({
            comments: getCommentsBefore.getCommentsBefore({
              node: specifier,
              sourceCode,
            }),
            partitionByComment: options.partitionByComment,
          }) ||
          (options.partitionByNewLine &&
            lastSortingNode &&
            getLinesBetween.getLinesBetween(
              sourceCode,
              lastSortingNode,
              sortingNode,
            ))
        ) {
          formattedMembers.push([])
        }
        formattedMembers.at(-1).push(sortingNode)
      }
      let groupKindOrder
      if (options.groupKind === 'values-first') {
        groupKindOrder = ['value', 'type']
      } else if (options.groupKind === 'types-first') {
        groupKindOrder = ['type', 'value']
      } else {
        groupKindOrder = ['any']
      }
      for (let nodes of formattedMembers) {
        let filteredGroupKindNodes = groupKindOrder.map(groupKind =>
          nodes.filter(
            currentNode =>
              groupKind === 'any' || currentNode.groupKind === groupKind,
          ),
        )
        let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
          filteredGroupKindNodes.flatMap(groupedNodes =>
            sortNodes.sortNodes(groupedNodes, options, {
              ignoreEslintDisabledNodes,
            }),
          )
        let sortedNodes = sortNodesExcludingEslintDisabled(false)
        let sortedNodesExcludingEslintDisabled =
          sortNodesExcludingEslintDisabled(true)
        let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
        pairwise.pairwise(nodes, (left, right) => {
          let leftIndex = nodeIndexMap.get(left)
          let rightIndex = nodeIndexMap.get(right)
          let indexOfRightExcludingEslintDisabled =
            sortedNodesExcludingEslintDisabled.indexOf(right)
          if (
            leftIndex < rightIndex &&
            leftIndex < indexOfRightExcludingEslintDisabled
          ) {
            return
          }
          context.report({
            fix: fixer =>
              makeFixes.makeFixes({
                sortedNodes: sortedNodesExcludingEslintDisabled,
                sourceCode,
                options,
                fixer,
                nodes,
              }),
            data: {
              right: right.name,
              left: left.name,
            },
            messageId: 'unexpectedNamedExportsOrder',
            node: right.node,
          })
        })
      }
    },
  }),
  meta: {
    schema: [
      {
        properties: {
          partitionByComment: {
            ...commonJsonSchemas.partitionByCommentJsonSchema,
            description:
              'Allows you to use comments to separate the named exports members into logical groups.',
          },
          groupKind: {
            enum: ['mixed', 'values-first', 'types-first'],
            description: 'Specifies top-level groups.',
            type: 'string',
          },
          partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    docs: {
      url: 'https://perfectionist.dev/rules/sort-named-exports',
      description: 'Enforce sorted named exports.',
      recommended: true,
    },
    messages: {
      unexpectedNamedExportsOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-named-exports',
})
module.exports = sortNamedExports
