'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getSourceCode = require('../utils/get-source-code.js')
const toSingleLine = require('../utils/to-single-line.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const sortNodes = require('../utils/sort-nodes.js')
const makeFixes = require('../utils/make-fixes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
let defaultOptions = {
  specialCharacters: 'keep',
  partitionByComment: false,
  partitionByNewLine: false,
  type: 'alphabetical',
  ignoreCase: true,
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
}
const sortMaps = createEslintRule.createEslintRule({
  create: context => ({
    NewExpression: node => {
      var _a, _b
      if (
        node.callee.type !== 'Identifier' ||
        node.callee.name !== 'Map' ||
        !node.arguments.length ||
        ((_a = node.arguments[0]) == null ? void 0 : _a.type) !==
          'ArrayExpression'
      ) {
        return
      }
      let [{ elements }] = node.arguments
      if (!isSortable.isSortable(elements)) {
        return
      }
      let settings = getSettings.getSettings(context.settings)
      let options = complete.complete(
        context.options.at(0),
        settings,
        defaultOptions,
      )
      validateCustomSortConfiguration.validateCustomSortConfiguration(options)
      let sourceCode = getSourceCode.getSourceCode(context)
      let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
        ruleName: context.id,
        sourceCode,
      })
      let parts = elements.reduce(
        (accumulator, element) => {
          if (element === null || element.type === 'SpreadElement') {
            accumulator.push([])
          } else {
            accumulator.at(-1).push(element)
          }
          return accumulator
        },
        [[]],
      )
      for (let part of parts) {
        let formattedMembers = [[]]
        for (let element of part) {
          let name
          if (element.type === 'ArrayExpression') {
            let [left] = element.elements
            if (!left) {
              name = `${left}`
            } else if (left.type === 'Literal') {
              name = left.raw
            } else {
              name = sourceCode.getText(left)
            }
          } else {
            name = sourceCode.getText(element)
          }
          let lastSortingNode =
            (_b = formattedMembers.at(-1)) == null ? void 0 : _b.at(-1)
          let sortingNode = {
            isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
              element,
              eslintDisabledLines,
            ),
            size: rangeToDiff.rangeToDiff(element, sourceCode),
            node: element,
            name,
          }
          if (
            hasPartitionComment.hasPartitionComment({
              comments: getCommentsBefore.getCommentsBefore({
                node: element,
                sourceCode,
              }),
              partitionByComment: options.partitionByComment,
            }) ||
            (options.partitionByNewLine &&
              lastSortingNode &&
              getLinesBetween.getLinesBetween(
                sourceCode,
                lastSortingNode,
                sortingNode,
              ))
          ) {
            formattedMembers.push([])
          }
          formattedMembers.at(-1).push(sortingNode)
        }
        for (let nodes of formattedMembers) {
          let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
            sortNodes.sortNodes(nodes, options, { ignoreEslintDisabledNodes })
          let sortedNodes = sortNodesExcludingEslintDisabled(false)
          let sortedNodesExcludingEslintDisabled =
            sortNodesExcludingEslintDisabled(true)
          let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
          pairwise.pairwise(nodes, (left, right) => {
            let leftIndex = nodeIndexMap.get(left)
            let rightIndex = nodeIndexMap.get(right)
            let indexOfRightExcludingEslintDisabled =
              sortedNodesExcludingEslintDisabled.indexOf(right)
            if (
              leftIndex < rightIndex &&
              leftIndex < indexOfRightExcludingEslintDisabled
            ) {
              return
            }
            context.report({
              fix: fixer =>
                makeFixes.makeFixes({
                  sortedNodes: sortedNodesExcludingEslintDisabled,
                  sourceCode,
                  options,
                  fixer,
                  nodes,
                }),
              data: {
                right: toSingleLine.toSingleLine(right.name),
                left: toSingleLine.toSingleLine(left.name),
              },
              messageId: 'unexpectedMapElementsOrder',
              node: right.node,
            })
          })
        }
      }
    },
  }),
  meta: {
    schema: [
      {
        properties: {
          partitionByComment: {
            ...commonJsonSchemas.partitionByCommentJsonSchema,
            description:
              'Allows you to use comments to separate the maps members into logical groups.',
          },
          partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    docs: {
      url: 'https://perfectionist.dev/rules/sort-maps',
      description: 'Enforce sorted Map elements.',
      recommended: true,
    },
    messages: {
      unexpectedMapElementsOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-maps',
})
module.exports = sortMaps
