'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const validateGroupsConfiguration = require('../utils/validate-groups-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const sortNodesByGroups = require('../utils/sort-nodes-by-groups.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getGroupNumber = require('../utils/get-group-number.js')
const getSourceCode = require('../utils/get-source-code.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const useGroups = require('../utils/use-groups.js')
const makeFixes = require('../utils/make-fixes.js')
const pairwise = require('../utils/pairwise.js')
const complete = require('../utils/complete.js')
const matches = require('../utils/matches.js')
let defaultOptions = {
  specialCharacters: 'keep',
  type: 'alphabetical',
  ignorePattern: [],
  ignoreCase: true,
  customGroups: {},
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
  groups: [],
}
const sortJsxProps = createEslintRule.createEslintRule({
  create: context => ({
    JSXElement: node => {
      if (!isSortable.isSortable(node.openingElement.attributes)) {
        return
      }
      let settings = getSettings.getSettings(context.settings)
      let options = complete.complete(
        context.options.at(0),
        settings,
        defaultOptions,
      )
      validateCustomSortConfiguration.validateCustomSortConfiguration(options)
      validateGroupsConfiguration.validateGroupsConfiguration(
        options.groups,
        ['multiline', 'shorthand', 'unknown'],
        Object.keys(options.customGroups),
      )
      let sourceCode = getSourceCode.getSourceCode(context)
      let shouldIgnore = false
      if (options.ignorePattern.length) {
        let tagName = sourceCode.getText(node.openingElement.name)
        shouldIgnore = options.ignorePattern.some(pattern =>
          matches.matches(tagName, pattern),
        )
      }
      if (
        shouldIgnore ||
        !isSortable.isSortable(node.openingElement.attributes)
      ) {
        return
      }
      let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
        ruleName: context.id,
        sourceCode,
      })
      let parts = node.openingElement.attributes.reduce(
        (accumulator, attribute) => {
          if (attribute.type === 'JSXSpreadAttribute') {
            accumulator.push([])
            return accumulator
          }
          let name =
            attribute.name.type === 'JSXNamespacedName'
              ? `${attribute.name.namespace.name}:${attribute.name.name.name}`
              : attribute.name.name
          let { setCustomGroups, defineGroup, getGroup } =
            useGroups.useGroups(options)
          setCustomGroups(options.customGroups, name)
          if (attribute.value === null) {
            defineGroup('shorthand')
          } else if (attribute.loc.start.line !== attribute.loc.end.line) {
            defineGroup('multiline')
          }
          let jsxNode = {
            isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
              attribute,
              eslintDisabledLines,
            ),
            size: rangeToDiff.rangeToDiff(attribute, sourceCode),
            group: getGroup(),
            node: attribute,
            name,
          }
          accumulator.at(-1).push(jsxNode)
          return accumulator
        },
        [[]],
      )
      for (let nodes of parts) {
        let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
          sortNodesByGroups.sortNodesByGroups(nodes, options, {
            ignoreEslintDisabledNodes,
          })
        let sortedNodes = sortNodesExcludingEslintDisabled(false)
        let sortedNodesExcludingEslintDisabled =
          sortNodesExcludingEslintDisabled(true)
        let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
        pairwise.pairwise(nodes, (left, right) => {
          let leftIndex = nodeIndexMap.get(left)
          let rightIndex = nodeIndexMap.get(right)
          let indexOfRightExcludingEslintDisabled =
            sortedNodesExcludingEslintDisabled.indexOf(right)
          if (
            leftIndex < rightIndex &&
            leftIndex < indexOfRightExcludingEslintDisabled
          ) {
            return
          }
          let leftNumber = getGroupNumber.getGroupNumber(options.groups, left)
          let rightNumber = getGroupNumber.getGroupNumber(options.groups, right)
          context.report({
            fix: fixer =>
              makeFixes.makeFixes({
                sortedNodes: sortedNodesExcludingEslintDisabled,
                sourceCode,
                fixer,
                nodes,
              }),
            data: {
              rightGroup: right.group,
              leftGroup: left.group,
              right: right.name,
              left: left.name,
            },
            messageId:
              leftNumber === rightNumber
                ? 'unexpectedJSXPropsOrder'
                : 'unexpectedJSXPropsGroupOrder',
            node: right.node,
          })
        })
      }
    },
  }),
  meta: {
    schema: [
      {
        properties: {
          ignorePattern: {
            description:
              'Specifies names or patterns for nodes that should be ignored by rule.',
            items: {
              type: 'string',
            },
            type: 'array',
          },
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          customGroups: commonJsonSchemas.customGroupsJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          groups: commonJsonSchemas.groupsJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    messages: {
      unexpectedJSXPropsGroupOrder:
        'Expected "{{right}}" ({{rightGroup}}) to come before "{{left}}" ({{leftGroup}}).',
      unexpectedJSXPropsOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-jsx-props',
      description: 'Enforce sorted JSX props.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-jsx-props',
})
module.exports = sortJsxProps
