'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const validateGroupsConfiguration = require('../utils/validate-groups-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const sortNodesByGroups = require('../utils/sort-nodes-by-groups.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getGroupNumber = require('../utils/get-group-number.js')
const getSourceCode = require('../utils/get-source-code.js')
const toSingleLine = require('../utils/to-single-line.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const useGroups = require('../utils/use-groups.js')
const makeFixes = require('../utils/make-fixes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
let defaultOptions = {
  specialCharacters: 'keep',
  type: 'alphabetical',
  ignoreCase: true,
  customGroups: {},
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
  groups: [],
}
const sortHeritageClauses = createEslintRule.createEslintRule({
  meta: {
    schema: [
      {
        properties: {
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          customGroups: commonJsonSchemas.customGroupsJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          groups: commonJsonSchemas.groupsJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    messages: {
      unexpectedHeritageClausesGroupOrder:
        'Expected "{{right}}" ({{rightGroup}}) to come before "{{left}}" ({{leftGroup}}).',
      unexpectedHeritageClausesOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-heritage-clauses',
      description: 'Enforce sorted heritage clauses.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  create: context => {
    let settings = getSettings.getSettings(context.settings)
    let options = complete.complete(
      context.options.at(0),
      settings,
      defaultOptions,
    )
    validateCustomSortConfiguration.validateCustomSortConfiguration(options)
    validateGroupsConfiguration.validateGroupsConfiguration(
      options.groups,
      ['unknown'],
      Object.keys(options.customGroups),
    )
    return {
      TSInterfaceDeclaration: declaration =>
        sortHeritageClauses$1(context, options, declaration.extends),
      ClassDeclaration: declaration =>
        sortHeritageClauses$1(context, options, declaration.implements),
    }
  },
  defaultOptions: [defaultOptions],
  name: 'sort-heritage-clauses',
})
let sortHeritageClauses$1 = (context, options, heritageClauses) => {
  if (!isSortable.isSortable(heritageClauses)) {
    return
  }
  let sourceCode = getSourceCode.getSourceCode(context)
  let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
    ruleName: context.id,
    sourceCode,
  })
  let nodes = heritageClauses.map(heritageClause => {
    let name = getHeritageClauseExpressionName(heritageClause.expression)
    let { setCustomGroups, getGroup } = useGroups.useGroups(options)
    setCustomGroups(options.customGroups, name)
    return {
      isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
        heritageClause,
        eslintDisabledLines,
      ),
      size: rangeToDiff.rangeToDiff(heritageClause, sourceCode),
      node: heritageClause,
      group: getGroup(),
      name,
    }
  })
  let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
    sortNodesByGroups.sortNodesByGroups(nodes, options, {
      ignoreEslintDisabledNodes,
    })
  let sortedNodes = sortNodesExcludingEslintDisabled(false)
  let sortedNodesExcludingEslintDisabled =
    sortNodesExcludingEslintDisabled(true)
  let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
  pairwise.pairwise(nodes, (left, right) => {
    let leftIndex = nodeIndexMap.get(left)
    let rightIndex = nodeIndexMap.get(right)
    let indexOfRightExcludingEslintDisabled =
      sortedNodesExcludingEslintDisabled.indexOf(right)
    if (
      leftIndex < rightIndex &&
      leftIndex < indexOfRightExcludingEslintDisabled
    ) {
      return
    }
    let leftNumber = getGroupNumber.getGroupNumber(options.groups, left)
    let rightNumber = getGroupNumber.getGroupNumber(options.groups, right)
    context.report({
      data: {
        right: toSingleLine.toSingleLine(right.name),
        left: toSingleLine.toSingleLine(left.name),
        rightGroup: right.group,
        leftGroup: left.group,
      },
      fix: fixer =>
        makeFixes.makeFixes({
          sortedNodes: sortedNodesExcludingEslintDisabled,
          sourceCode,
          fixer,
          nodes,
        }),
      messageId:
        leftNumber === rightNumber
          ? 'unexpectedHeritageClausesOrder'
          : 'unexpectedHeritageClausesGroupOrder',
      node: right.node,
    })
  })
}
let getHeritageClauseExpressionName = expression => {
  if (expression.type === 'Identifier') {
    return expression.name
  }
  if ('property' in expression) {
    return getHeritageClauseExpressionName(expression.property)
  }
  throw new Error(
    'Unexpected heritage clause expression. Please report this issue here: https://github.com/azat-io/eslint-plugin-perfectionist/issues',
  )
}
module.exports = sortHeritageClauses
