'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getSourceCode = require('../utils/get-source-code.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const sortNodes = require('../utils/sort-nodes.js')
const makeFixes = require('../utils/make-fixes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
let defaultOptions = {
  specialCharacters: 'keep',
  partitionByComment: false,
  partitionByNewLine: false,
  type: 'alphabetical',
  groupKind: 'mixed',
  ignoreCase: true,
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
}
const sortExports = createEslintRule.createEslintRule({
  create: context => {
    let settings = getSettings.getSettings(context.settings)
    let options = complete.complete(
      context.options.at(0),
      settings,
      defaultOptions,
    )
    validateCustomSortConfiguration.validateCustomSortConfiguration(options)
    let sourceCode = getSourceCode.getSourceCode(context)
    let partitionComment = options.partitionByComment
    let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
      ruleName: context.id,
      sourceCode,
    })
    let parts = [[]]
    let registerNode = node => {
      var _a
      let sortingNode = {
        isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
          node,
          eslintDisabledLines,
        ),
        groupKind: node.exportKind === 'value' ? 'value' : 'type',
        size: rangeToDiff.rangeToDiff(node, sourceCode),
        addSafetySemicolonWhenInline: true,
        name: node.source.value,
        node,
      }
      let lastNode = (_a = parts.at(-1)) == null ? void 0 : _a.at(-1)
      if (
        (partitionComment &&
          hasPartitionComment.hasPartitionComment({
            comments: getCommentsBefore.getCommentsBefore({
              sourceCode,
              node,
            }),
            partitionByComment: options.partitionByComment,
          })) ||
        (options.partitionByNewLine &&
          lastNode &&
          getLinesBetween.getLinesBetween(sourceCode, lastNode, sortingNode))
      ) {
        parts.push([])
      }
      parts.at(-1).push(sortingNode)
    }
    return {
      'Program:exit': () => {
        let groupKindOrder
        if (options.groupKind === 'values-first') {
          groupKindOrder = ['value', 'type']
        } else if (options.groupKind === 'types-first') {
          groupKindOrder = ['type', 'value']
        } else {
          groupKindOrder = ['any']
        }
        for (let nodes of parts) {
          let filteredGroupKindNodes = groupKindOrder.map(groupKind =>
            nodes.filter(
              currentNode =>
                groupKind === 'any' || currentNode.groupKind === groupKind,
            ),
          )
          let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
            filteredGroupKindNodes.flatMap(groupedNodes =>
              sortNodes.sortNodes(groupedNodes, options, {
                ignoreEslintDisabledNodes,
              }),
            )
          let sortedNodes = sortNodesExcludingEslintDisabled(false)
          let sortedNodesExcludingEslintDisabled =
            sortNodesExcludingEslintDisabled(true)
          let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
          pairwise.pairwise(nodes, (left, right) => {
            let leftIndex = nodeIndexMap.get(left)
            let rightIndex = nodeIndexMap.get(right)
            let indexOfRightExcludingEslintDisabled =
              sortedNodesExcludingEslintDisabled.indexOf(right)
            if (
              leftIndex < rightIndex &&
              leftIndex < indexOfRightExcludingEslintDisabled
            ) {
              return
            }
            context.report({
              fix: fixer =>
                makeFixes.makeFixes({
                  sortedNodes: sortedNodesExcludingEslintDisabled,
                  sourceCode,
                  options,
                  fixer,
                  nodes,
                }),
              data: {
                right: right.name,
                left: left.name,
              },
              messageId: 'unexpectedExportsOrder',
              node: right.node,
            })
          })
        }
      },
      ExportNamedDeclaration: node => {
        if (node.source !== null) {
          registerNode(node)
        }
      },
      ExportAllDeclaration: registerNode,
    }
  },
  meta: {
    schema: [
      {
        properties: {
          partitionByComment: {
            ...commonJsonSchemas.partitionByCommentJsonSchema,
            description:
              'Allows you to use comments to separate the exports into logical groups.',
          },
          groupKind: {
            enum: ['mixed', 'values-first', 'types-first'],
            description: 'Specifies top-level groups.',
            type: 'string',
          },
          partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    docs: {
      url: 'https://perfectionist.dev/rules/sort-exports',
      description: 'Enforce sorted exports.',
      recommended: true,
    },
    messages: {
      unexpectedExportsOrder: 'Expected "{{right}}" to come before "{{left}}".',
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-exports',
})
module.exports = sortExports
