'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const sortNodesByDependencies = require('../utils/sort-nodes-by-dependencies.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getEnumMembers = require('../utils/get-enum-members.js')
const getSourceCode = require('../utils/get-source-code.js')
const toSingleLine = require('../utils/to-single-line.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const sortNodes = require('../utils/sort-nodes.js')
const makeFixes = require('../utils/make-fixes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
let defaultOptions = {
  partitionByComment: false,
  partitionByNewLine: false,
  specialCharacters: 'keep',
  forceNumericSort: false,
  type: 'alphabetical',
  sortByValue: false,
  ignoreCase: true,
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
}
const sortEnums = createEslintRule.createEslintRule({
  create: context => ({
    TSEnumDeclaration: enumDeclaration => {
      let members = getEnumMembers.getEnumMembers(enumDeclaration)
      if (
        !isSortable.isSortable(members) ||
        !members.every(({ initializer }) => initializer)
      ) {
        return
      }
      let settings = getSettings.getSettings(context.settings)
      let options = complete.complete(
        context.options.at(0),
        settings,
        defaultOptions,
      )
      validateCustomSortConfiguration.validateCustomSortConfiguration(options)
      let sourceCode = getSourceCode.getSourceCode(context)
      let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
        ruleName: context.id,
        sourceCode,
      })
      let extractDependencies = (expression, enumName) => {
        let dependencies = []
        let stack = [expression]
        while (stack.length) {
          let node = stack.pop()
          if (
            node.type === 'MemberExpression' &&
            node.object.type === 'Identifier' &&
            node.object.name === enumName &&
            node.property.type === 'Identifier'
          ) {
            dependencies.push(node.property.name)
          } else if (node.type === 'Identifier') {
            dependencies.push(node.name)
          }
          if ('left' in node) {
            stack.push(node.left)
          }
          if ('right' in node) {
            stack.push(node.right)
          }
          if ('expressions' in node) {
            stack.push(...node.expressions)
          }
        }
        return dependencies
      }
      let formattedMembers = members.reduce(
        (accumulator, member) => {
          var _a
          let dependencies = []
          if (member.initializer) {
            dependencies = extractDependencies(
              member.initializer,
              enumDeclaration.id.name,
            )
          }
          let lastSortingNode =
            (_a = accumulator.at(-1)) == null ? void 0 : _a.at(-1)
          let sortingNode = {
            numericValue: member.initializer
              ? getExpressionNumberValue(member.initializer)
              : null,
            name:
              member.id.type === 'Literal'
                ? `${member.id.value}`
                : sourceCode.getText(member.id),
            isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
              member,
              eslintDisabledLines,
            ),
            size: rangeToDiff.rangeToDiff(member, sourceCode),
            node: member,
            dependencies,
          }
          if (
            hasPartitionComment.hasPartitionComment({
              comments: getCommentsBefore.getCommentsBefore({
                node: member,
                sourceCode,
              }),
              partitionByComment: options.partitionByComment,
            }) ||
            (options.partitionByNewLine &&
              lastSortingNode &&
              getLinesBetween.getLinesBetween(
                sourceCode,
                lastSortingNode,
                sortingNode,
              ))
          ) {
            accumulator.push([])
          }
          accumulator.at(-1).push(sortingNode)
          return accumulator
        },
        [[]],
      )
      let sortingNodes = formattedMembers.flat()
      let isNumericEnum = sortingNodes.every(
        sortingNode =>
          sortingNode.numericValue !== null &&
          !Number.isNaN(sortingNode.numericValue),
      )
      let compareOptions = {
        // Get the enum value rather than the name if needed.
        nodeValueGetter:
          options.sortByValue || (isNumericEnum && options.forceNumericSort)
            ? sortingNode => {
                var _a, _b
                if (isNumericEnum) {
                  return sortingNode.numericValue.toString()
                }
                if (
                  ((_a = sortingNode.node.initializer) == null
                    ? void 0
                    : _a.type) === 'Literal'
                ) {
                  return (
                    ((_b = sortingNode.node.initializer.value) == null
                      ? void 0
                      : _b.toString()) ?? ''
                  )
                }
                return ''
              }
            : null,
        /**
         * If the enum is numeric, and we sort by value, always use the
         * `natural` sort type, which will correctly sort them.
         */
        type:
          isNumericEnum && (options.forceNumericSort || options.sortByValue)
            ? 'natural'
            : options.type,
        specialCharacters: options.specialCharacters,
        ignoreCase: options.ignoreCase,
        alphabet: options.alphabet,
        locales: options.locales,
        order: options.order,
      }
      let sortNodesIgnoringEslintDisabledNodes = ignoreEslintDisabledNodes =>
        sortNodesByDependencies.sortNodesByDependencies(
          formattedMembers.flatMap(nodes =>
            sortNodes.sortNodes(nodes, compareOptions, {
              ignoreEslintDisabledNodes,
            }),
          ),
          {
            ignoreEslintDisabledNodes,
          },
        )
      let sortedNodes = sortNodesIgnoringEslintDisabledNodes(false)
      let sortedNodesExcludingEslintDisabled =
        sortNodesIgnoringEslintDisabledNodes(true)
      let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
      pairwise.pairwise(sortingNodes, (left, right) => {
        let leftIndex = nodeIndexMap.get(left)
        let rightIndex = nodeIndexMap.get(right)
        let indexOfRightExcludingEslintDisabled =
          sortedNodesExcludingEslintDisabled.indexOf(right)
        if (
          leftIndex < rightIndex &&
          leftIndex < indexOfRightExcludingEslintDisabled
        ) {
          return
        }
        let firstUnorderedNodeDependentOnRight =
          sortNodesByDependencies.getFirstUnorderedNodeDependentOn(
            right,
            sortingNodes,
          )
        context.report({
          fix: fixer =>
            makeFixes.makeFixes({
              sortedNodes: sortedNodesExcludingEslintDisabled,
              nodes: sortingNodes,
              sourceCode,
              options,
              fixer,
            }),
          data: {
            nodeDependentOnRight:
              firstUnorderedNodeDependentOnRight == null
                ? void 0
                : firstUnorderedNodeDependentOnRight.name,
            right: toSingleLine.toSingleLine(right.name),
            left: toSingleLine.toSingleLine(left.name),
          },
          messageId: firstUnorderedNodeDependentOnRight
            ? 'unexpectedEnumsDependencyOrder'
            : 'unexpectedEnumsOrder',
          node: right.node,
        })
      })
    },
  }),
  meta: {
    schema: [
      {
        properties: {
          partitionByComment: {
            ...commonJsonSchemas.partitionByCommentJsonSchema,
            description:
              'Allows you to use comments to separate the members of enums into logical groups.',
          },
          forceNumericSort: {
            description:
              'Will always sort numeric enums by their value regardless of the sort type specified.',
            type: 'boolean',
          },
          sortByValue: {
            description: 'Compare enum values instead of names.',
            type: 'boolean',
          },
          partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    messages: {
      unexpectedEnumsDependencyOrder:
        'Expected dependency "{{right}}" to come before "{{nodeDependentOnRight}}".',
      unexpectedEnumsOrder: 'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-enums',
      description: 'Enforce sorted TypeScript enums.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-enums',
})
let getExpressionNumberValue = expression => {
  switch (expression.type) {
    case 'BinaryExpression':
      return getBinaryExpressionNumberValue(
        expression.left,
        expression.right,
        expression.operator,
      )
    case 'UnaryExpression':
      return getUnaryExpressionNumberValue(
        expression.argument,
        expression.operator,
      )
    case 'Literal':
      return typeof expression.value === 'number'
        ? expression.value
        : Number.NaN
    default:
      return Number.NaN
  }
}
let getUnaryExpressionNumberValue = (argumentExpression, operator) => {
  let argument = getExpressionNumberValue(argumentExpression)
  switch (operator) {
    case '+':
      return argument
    case '-':
      return -argument
    case '~':
      return ~argument
    /* v8 ignore next 2 - Unsure if we can reach it */
    default:
      return Number.NaN
  }
}
let getBinaryExpressionNumberValue = (
  leftExpression,
  rightExpression,
  operator,
) => {
  let left = getExpressionNumberValue(leftExpression)
  let right = getExpressionNumberValue(rightExpression)
  switch (operator) {
    case '**':
      return left ** right
    case '>>':
      return left >> right
    case '<<':
      return left << right
    case '+':
      return left + right
    case '-':
      return left - right
    case '*':
      return left * right
    case '/':
      return left / right
    case '%':
      return left % right
    case '|':
      return left | right
    case '&':
      return left & right
    case '^':
      return left ^ right
    /* v8 ignore next 2 - Unsure if we can reach it */
    default:
      return Number.NaN
  }
}
module.exports = sortEnums
