'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const validateGroupsConfiguration = require('../utils/validate-groups-configuration.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const getDecoratorName = require('./sort-decorators/get-decorator-name.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const sortNodesByGroups = require('../utils/sort-nodes-by-groups.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const getNodeDecorators = require('../utils/get-node-decorators.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getGroupNumber = require('../utils/get-group-number.js')
const getSourceCode = require('../utils/get-source-code.js')
const toSingleLine = require('../utils/to-single-line.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const useGroups = require('../utils/use-groups.js')
const makeFixes = require('../utils/make-fixes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
let defaultOptions = {
  specialCharacters: 'keep',
  partitionByComment: false,
  sortOnProperties: true,
  sortOnParameters: true,
  sortOnAccessors: true,
  type: 'alphabetical',
  sortOnClasses: true,
  sortOnMethods: true,
  ignoreCase: true,
  customGroups: {},
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
  groups: [],
}
const sortDecorators = createEslintRule.createEslintRule({
  meta: {
    schema: [
      {
        properties: {
          partitionByComment: {
            ...commonJsonSchemas.partitionByCommentJsonSchema,
            description:
              'Allows you to use comments to separate the decorators into logical groups.',
          },
          sortOnParameters: {
            description:
              'Controls whether sorting should be enabled for method parameter decorators.',
            type: 'boolean',
          },
          sortOnProperties: {
            description:
              'Controls whether sorting should be enabled for class property decorators.',
            type: 'boolean',
          },
          sortOnAccessors: {
            description:
              'Controls whether sorting should be enabled for class accessor decorators.',
            type: 'boolean',
          },
          sortOnMethods: {
            description:
              'Controls whether sorting should be enabled for class method decorators.',
            type: 'boolean',
          },
          sortOnClasses: {
            description:
              'Controls whether sorting should be enabled for class decorators.',
            type: 'boolean',
          },
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          customGroups: commonJsonSchemas.customGroupsJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          groups: commonJsonSchemas.groupsJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    messages: {
      unexpectedDecoratorsGroupOrder:
        'Expected "{{right}}" ({{rightGroup}}) to come before "{{left}}" ({{leftGroup}}).',
      unexpectedDecoratorsOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-decorators',
      description: 'Enforce sorted decorators.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  create: context => {
    let settings = getSettings.getSettings(context.settings)
    let options = complete.complete(
      context.options.at(0),
      settings,
      defaultOptions,
    )
    validateCustomSortConfiguration.validateCustomSortConfiguration(options)
    validateGroupsConfiguration.validateGroupsConfiguration(
      options.groups,
      ['unknown'],
      Object.keys(options.customGroups),
    )
    return {
      Decorator: decorator => {
        if (!options.sortOnParameters) {
          return
        }
        if (
          'decorators' in decorator.parent &&
          decorator.parent.type === 'Identifier' &&
          decorator.parent.parent.type === 'FunctionExpression'
        ) {
          let { decorators } = decorator.parent
          if (decorator !== decorators[0]) {
            return
          }
          sortDecorators$1(context, options, decorators)
        }
      },
      PropertyDefinition: propertyDefinition =>
        options.sortOnProperties
          ? sortDecorators$1(
              context,
              options,
              getNodeDecorators.getNodeDecorators(propertyDefinition),
            )
          : null,
      AccessorProperty: accessorDefinition =>
        options.sortOnAccessors
          ? sortDecorators$1(
              context,
              options,
              getNodeDecorators.getNodeDecorators(accessorDefinition),
            )
          : null,
      MethodDefinition: methodDefinition =>
        options.sortOnMethods
          ? sortDecorators$1(
              context,
              options,
              getNodeDecorators.getNodeDecorators(methodDefinition),
            )
          : null,
      ClassDeclaration: declaration =>
        options.sortOnClasses
          ? sortDecorators$1(
              context,
              options,
              getNodeDecorators.getNodeDecorators(declaration),
            )
          : null,
    }
  },
  defaultOptions: [defaultOptions],
  name: 'sort-decorators',
})
let sortDecorators$1 = (context, options, decorators) => {
  if (!isSortable.isSortable(decorators)) {
    return
  }
  let sourceCode = getSourceCode.getSourceCode(context)
  let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
    ruleName: context.id,
    sourceCode,
  })
  let formattedMembers = decorators.reduce(
    (accumulator, decorator) => {
      if (
        hasPartitionComment.hasPartitionComment({
          comments: getCommentsBefore.getCommentsBefore({
            node: decorator,
            sourceCode,
          }),
          partitionByComment: options.partitionByComment,
        })
      ) {
        accumulator.push([])
      }
      let { setCustomGroups, getGroup } = useGroups.useGroups(options)
      let name = getDecoratorName.getDecoratorName(decorator)
      setCustomGroups(options.customGroups, name)
      let sortingNode = {
        isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
          decorator,
          eslintDisabledLines,
        ),
        size: rangeToDiff.rangeToDiff(decorator, sourceCode),
        group: getGroup(),
        node: decorator,
        name,
      }
      accumulator.at(-1).push(sortingNode)
      return accumulator
    },
    [[]],
  )
  let sortNodesExcludingEslintDisabled = ignoreEslintDisabledNodes =>
    formattedMembers.flatMap(nodes2 =>
      sortNodesByGroups.sortNodesByGroups(nodes2, options, {
        ignoreEslintDisabledNodes,
      }),
    )
  let sortedNodes = sortNodesExcludingEslintDisabled(false)
  let sortedNodesExcludingEslintDisabled =
    sortNodesExcludingEslintDisabled(true)
  let nodes = formattedMembers.flat()
  let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
  pairwise.pairwise(nodes, (left, right) => {
    let leftIndex = nodeIndexMap.get(left)
    let rightIndex = nodeIndexMap.get(right)
    let indexOfRightExcludingEslintDisabled =
      sortedNodesExcludingEslintDisabled.indexOf(right)
    if (
      leftIndex < rightIndex &&
      leftIndex < indexOfRightExcludingEslintDisabled
    ) {
      return
    }
    let leftNumber = getGroupNumber.getGroupNumber(options.groups, left)
    let rightNumber = getGroupNumber.getGroupNumber(options.groups, right)
    context.report({
      fix: fixer =>
        makeFixes.makeFixes({
          sortedNodes: sortedNodesExcludingEslintDisabled,
          ignoreFirstNodeHighestBlockComment: true,
          sourceCode,
          options,
          fixer,
          nodes,
        }),
      data: {
        right: toSingleLine.toSingleLine(right.name),
        left: toSingleLine.toSingleLine(left.name),
        rightGroup: right.group,
        leftGroup: left.group,
      },
      messageId:
        leftNumber === rightNumber
          ? 'unexpectedDecoratorsOrder'
          : 'unexpectedDecoratorsGroupOrder',
      node: right.node,
    })
  })
}
module.exports = sortDecorators
