'use strict'
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const sortNodesByDependencies = require('../utils/sort-nodes-by-dependencies.js')
const validateNewlinesAndPartitionConfiguration = require('../utils/validate-newlines-and-partition-configuration.js')
const validateGeneratedGroupsConfiguration = require('../utils/validate-generated-groups-configuration.js')
const types = require('./sort-classes/types.js')
const validateCustomSortConfiguration = require('../utils/validate-custom-sort-configuration.js')
const getCustomGroupsCompareOptions = require('../utils/get-custom-groups-compare-options.js')
const getOverloadSignatureGroups = require('./sort-classes/get-overload-signature-groups.js')
const generatePredefinedGroups = require('../utils/generate-predefined-groups.js')
const doesCustomGroupMatch = require('./sort-classes/does-custom-group-match.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const sortNodesByGroups = require('../utils/sort-nodes-by-groups.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const makeNewlinesFixes = require('../utils/make-newlines-fixes.js')
const getNewlinesErrors = require('../utils/get-newlines-errors.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getGroupNumber = require('../utils/get-group-number.js')
const getSourceCode = require('../utils/get-source-code.js')
const toSingleLine = require('../utils/to-single-line.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const useGroups = require('../utils/use-groups.js')
const makeFixes = require('../utils/make-fixes.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
const matches = require('../utils/matches.js')
let cachedGroupsByModifiersAndSelectors = /* @__PURE__ */ new Map()
let defaultOptions = {
  groups: [
    'index-signature',
    ['static-property', 'static-accessor-property'],
    ['static-get-method', 'static-set-method'],
    ['protected-static-property', 'protected-static-accessor-property'],
    ['protected-static-get-method', 'protected-static-set-method'],
    ['private-static-property', 'private-static-accessor-property'],
    ['private-static-get-method', 'private-static-set-method'],
    'static-block',
    ['property', 'accessor-property'],
    ['get-method', 'set-method'],
    ['protected-property', 'protected-accessor-property'],
    ['protected-get-method', 'protected-set-method'],
    ['private-property', 'private-accessor-property'],
    ['private-get-method', 'private-set-method'],
    'constructor',
    ['static-method', 'static-function-property'],
    ['protected-static-method', 'protected-static-function-property'],
    ['private-static-method', 'private-static-function-property'],
    ['method', 'function-property'],
    ['protected-method', 'protected-function-property'],
    ['private-method', 'private-function-property'],
    'unknown',
  ],
  ignoreCallbackDependenciesPatterns: [],
  partitionByComment: false,
  partitionByNewLine: false,
  newlinesBetween: 'ignore',
  specialCharacters: 'keep',
  type: 'alphabetical',
  ignoreCase: true,
  customGroups: [],
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
}
const sortClasses = createEslintRule.createEslintRule({
  create: context => ({
    ClassBody: node => {
      var _a
      if (!isSortable.isSortable(node.body)) {
        return
      }
      let settings = getSettings.getSettings(context.settings)
      let options = complete.complete(
        context.options.at(0),
        settings,
        defaultOptions,
      )
      validateCustomSortConfiguration.validateCustomSortConfiguration(options)
      validateGeneratedGroupsConfiguration.validateGeneratedGroupsConfiguration(
        {
          customGroups: options.customGroups,
          modifiers: types.allModifiers,
          selectors: types.allSelectors,
          groups: options.groups,
        },
      )
      validateNewlinesAndPartitionConfiguration.validateNewlinesAndPartitionConfiguration(
        options,
      )
      let sourceCode = getSourceCode.getSourceCode(context)
      let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
        ruleName: context.id,
        sourceCode,
      })
      let className = (_a = node.parent.id) == null ? void 0 : _a.name
      let getDependencyName = props =>
        `${props.isStatic ? 'static ' : ''}${props.isPrivateHash ? '#' : ''}${props.nodeNameWithoutStartingHash}`
      let classMethodsDependencyNames = new Set(
        node.body
          .map(member => {
            if (
              (member.type === 'MethodDefinition' ||
                member.type === 'TSAbstractMethodDefinition') &&
              'name' in member.key
            ) {
              return getDependencyName({
                isPrivateHash: member.key.type === 'PrivateIdentifier',
                nodeNameWithoutStartingHash: member.key.name,
                isStatic: member.static,
              })
            }
            return null
          })
          .filter(Boolean),
      )
      let extractDependencies = (expression, isMemberStatic) => {
        let dependencies = []
        let checkNode = nodeValue => {
          if (
            nodeValue.type === 'MemberExpression' &&
            (nodeValue.object.type === 'ThisExpression' ||
              (nodeValue.object.type === 'Identifier' &&
                nodeValue.object.name === className)) &&
            (nodeValue.property.type === 'Identifier' ||
              nodeValue.property.type === 'PrivateIdentifier')
          ) {
            let isStaticDependency =
              isMemberStatic || nodeValue.object.type === 'Identifier'
            let dependencyName = getDependencyName({
              isPrivateHash: nodeValue.property.type === 'PrivateIdentifier',
              nodeNameWithoutStartingHash: nodeValue.property.name,
              isStatic: isStaticDependency,
            })
            if (!classMethodsDependencyNames.has(dependencyName)) {
              dependencies.push(dependencyName)
            }
          }
          if (nodeValue.type === 'Property') {
            traverseNode(nodeValue.key)
            traverseNode(nodeValue.value)
          }
          if (nodeValue.type === 'ConditionalExpression') {
            traverseNode(nodeValue.test)
            traverseNode(nodeValue.consequent)
            traverseNode(nodeValue.alternate)
          }
          if (
            'expression' in nodeValue &&
            typeof nodeValue.expression !== 'boolean'
          ) {
            traverseNode(nodeValue.expression)
          }
          if ('object' in nodeValue) {
            traverseNode(nodeValue.object)
          }
          if ('callee' in nodeValue) {
            traverseNode(nodeValue.callee)
          }
          if ('init' in nodeValue && nodeValue.init) {
            traverseNode(nodeValue.init)
          }
          if ('body' in nodeValue && nodeValue.body) {
            traverseNode(nodeValue.body)
          }
          if ('left' in nodeValue) {
            traverseNode(nodeValue.left)
          }
          if ('right' in nodeValue) {
            traverseNode(nodeValue.right)
          }
          if ('elements' in nodeValue) {
            let elements = nodeValue.elements.filter(
              currentNode => currentNode !== null,
            )
            for (let element of elements) {
              traverseNode(element)
            }
          }
          if ('argument' in nodeValue && nodeValue.argument) {
            traverseNode(nodeValue.argument)
          }
          if ('arguments' in nodeValue) {
            let shouldIgnore = false
            if (nodeValue.type === 'CallExpression') {
              let functionName =
                'name' in nodeValue.callee ? nodeValue.callee.name : null
              shouldIgnore =
                functionName !== null &&
                options.ignoreCallbackDependenciesPatterns.some(pattern =>
                  matches.matches(functionName, pattern),
                )
            }
            if (!shouldIgnore) {
              for (let argument of nodeValue.arguments) {
                traverseNode(argument)
              }
            }
          }
          if ('declarations' in nodeValue) {
            for (let declaration of nodeValue.declarations) {
              traverseNode(declaration)
            }
          }
          if ('properties' in nodeValue) {
            for (let property of nodeValue.properties) {
              traverseNode(property)
            }
          }
          if ('expressions' in nodeValue) {
            for (let nodeExpression of nodeValue.expressions) {
              traverseNode(nodeExpression)
            }
          }
        }
        let traverseNode = nodeValue => {
          if (Array.isArray(nodeValue)) {
            for (let nodeItem of nodeValue) {
              traverseNode(nodeItem)
            }
          } else {
            checkNode(nodeValue)
          }
        }
        traverseNode(expression)
        return dependencies
      }
      let overloadSignatureGroups =
        getOverloadSignatureGroups.getOverloadSignatureGroups(node.body)
      let formattedNodes = node.body.reduce(
        (accumulator, member) => {
          var _a2, _b, _c, _d, _e
          let name
          let dependencies = []
          let { defineGroup, getGroup } = useGroups.useGroups(options)
          if (member.type === 'StaticBlock') {
            name = 'static'
          } else if (member.type === 'TSIndexSignature') {
            name = sourceCode.text.slice(
              member.range.at(0),
              ((_a2 = member.typeAnnotation) == null
                ? void 0
                : _a2.range.at(0)) ?? member.range.at(1),
            )
          } else if (member.key.type === 'Identifier') {
            ;({ name } = member.key)
          } else {
            name = sourceCode.getText(member.key)
          }
          let isPrivateHash =
            'key' in member && member.key.type === 'PrivateIdentifier'
          let decorated = false
          let decorators = []
          if ('decorators' in member) {
            decorated = member.decorators.length > 0
            for (let decorator of member.decorators) {
              if (decorator.expression.type === 'Identifier') {
                decorators.push(decorator.expression.name)
              } else if (
                decorator.expression.type === 'CallExpression' &&
                decorator.expression.callee.type === 'Identifier'
              ) {
                decorators.push(decorator.expression.callee.name)
              }
            }
          }
          let memberValue
          let modifiers = []
          let selectors = []
          let addSafetySemicolonWhenInline = true
          if (
            member.type === 'MethodDefinition' ||
            member.type === 'TSAbstractMethodDefinition'
          ) {
            if (member.static) {
              modifiers.push('static')
            }
            if (member.type === 'TSAbstractMethodDefinition') {
              modifiers.push('abstract')
            } else if (!node.parent.declare) {
              addSafetySemicolonWhenInline = false
            }
            if (decorated) {
              modifiers.push('decorated')
            }
            if (member.override) {
              modifiers.push('override')
            }
            if (member.accessibility === 'protected') {
              modifiers.push('protected')
            } else if (member.accessibility === 'private' || isPrivateHash) {
              modifiers.push('private')
            } else {
              modifiers.push('public')
            }
            if (member.optional) {
              modifiers.push('optional')
            }
            if (member.value.async) {
              modifiers.push('async')
            }
            if (member.kind === 'constructor') {
              selectors.push('constructor')
            }
            if (member.kind === 'get') {
              selectors.push('get-method')
            }
            if (member.kind === 'set') {
              selectors.push('set-method')
            }
            selectors.push('method')
          } else if (member.type === 'TSIndexSignature') {
            if (member.static) {
              modifiers.push('static')
            }
            if (member.readonly) {
              modifiers.push('readonly')
            }
            selectors.push('index-signature')
          } else if (member.type === 'StaticBlock') {
            addSafetySemicolonWhenInline = false
            selectors.push('static-block')
            dependencies = extractDependencies(member, true)
          } else if (
            member.type === 'AccessorProperty' ||
            member.type === 'TSAbstractAccessorProperty'
          ) {
            if (member.static) {
              modifiers.push('static')
            }
            if (member.type === 'TSAbstractAccessorProperty') {
              modifiers.push('abstract')
            }
            if (decorated) {
              modifiers.push('decorated')
            }
            if (member.override) {
              modifiers.push('override')
            }
            if (member.accessibility === 'protected') {
              modifiers.push('protected')
            } else if (member.accessibility === 'private' || isPrivateHash) {
              modifiers.push('private')
            } else {
              modifiers.push('public')
            }
            selectors.push('accessor-property')
          } else {
            if (member.static) {
              modifiers.push('static')
            }
            if (member.declare) {
              modifiers.push('declare')
            }
            if (member.type === 'TSAbstractPropertyDefinition') {
              modifiers.push('abstract')
            }
            if (decorated) {
              modifiers.push('decorated')
            }
            if (member.override) {
              modifiers.push('override')
            }
            if (member.readonly) {
              modifiers.push('readonly')
            }
            if (member.accessibility === 'protected') {
              modifiers.push('protected')
            } else if (member.accessibility === 'private' || isPrivateHash) {
              modifiers.push('private')
            } else {
              modifiers.push('public')
            }
            if (member.optional) {
              modifiers.push('optional')
            }
            if (
              ((_b = member.value) == null ? void 0 : _b.type) ===
                'ArrowFunctionExpression' ||
              ((_c = member.value) == null ? void 0 : _c.type) ===
                'FunctionExpression'
            ) {
              if (member.value.async) {
                modifiers.push('async')
              }
              selectors.push('function-property')
            } else if (member.value) {
              memberValue = sourceCode.getText(member.value)
              dependencies = extractDependencies(member.value, member.static)
            }
            selectors.push('property')
          }
          let predefinedGroups =
            generatePredefinedGroups.generatePredefinedGroups({
              cache: cachedGroupsByModifiersAndSelectors,
              selectors,
              modifiers,
            })
          for (let predefinedGroup of predefinedGroups) {
            defineGroup(predefinedGroup)
          }
          for (let customGroup of options.customGroups) {
            if (
              doesCustomGroupMatch.doesCustomGroupMatch({
                elementValue: memberValue,
                elementName: name,
                customGroup,
                decorators,
                modifiers,
                selectors,
              })
            ) {
              defineGroup(customGroup.groupName, true)
              if (getGroup() === customGroup.groupName) {
                break
              }
            }
          }
          let overloadSignatureGroupMember =
            (_d = overloadSignatureGroups.find(overloadSignatures =>
              overloadSignatures.includes(member),
            )) == null
              ? void 0
              : _d.at(-1)
          let sortingNode = {
            dependencyName: getDependencyName({
              nodeNameWithoutStartingHash: name.startsWith('#')
                ? name.slice(1)
                : name,
              isStatic: modifiers.includes('static'),
              isPrivateHash,
            }),
            size: overloadSignatureGroupMember
              ? rangeToDiff.rangeToDiff(
                  overloadSignatureGroupMember,
                  sourceCode,
                )
              : rangeToDiff.rangeToDiff(member, sourceCode),
            isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
              member,
              eslintDisabledLines,
            ),
            addSafetySemicolonWhenInline,
            group: getGroup(),
            node: member,
            dependencies,
            name,
          }
          let lastMember =
            (_e = accumulator.at(-1)) == null ? void 0 : _e.at(-1)
          if (
            (options.partitionByNewLine &&
              lastMember &&
              getLinesBetween.getLinesBetween(
                sourceCode,
                lastMember,
                sortingNode,
              )) ||
            hasPartitionComment.hasPartitionComment({
              comments: getCommentsBefore.getCommentsBefore({
                node: member,
                sourceCode,
              }),
              partitionByComment: options.partitionByComment,
            })
          ) {
            accumulator.push([])
          }
          accumulator.at(-1).push(sortingNode)
          return accumulator
        },
        [[]],
      )
      let sortNodesIgnoringEslintDisabledNodes = ignoreEslintDisabledNodes =>
        sortNodesByDependencies.sortNodesByDependencies(
          formattedNodes.flatMap(nodes2 =>
            sortNodesByGroups.sortNodesByGroups(nodes2, options, {
              isNodeIgnored: sortingNode =>
                getGroupNumber.getGroupNumber(options.groups, sortingNode) ===
                options.groups.length,
              getGroupCompareOptions: groupNumber =>
                getCustomGroupsCompareOptions.getCustomGroupsCompareOptions(
                  options,
                  groupNumber,
                ),
              ignoreEslintDisabledNodes,
            }),
          ),
          {
            ignoreEslintDisabledNodes,
          },
        )
      let sortedNodes = sortNodesIgnoringEslintDisabledNodes(false)
      let sortedNodesExcludingEslintDisabled =
        sortNodesIgnoringEslintDisabledNodes(true)
      let nodes = formattedNodes.flat()
      pairwise.pairwise(nodes, (left, right) => {
        let leftNumber = getGroupNumber.getGroupNumber(options.groups, left)
        let rightNumber = getGroupNumber.getGroupNumber(options.groups, right)
        let indexOfLeft = sortedNodes.indexOf(left)
        let indexOfRight = sortedNodes.indexOf(right)
        let indexOfRightExcludingEslintDisabled =
          sortedNodesExcludingEslintDisabled.indexOf(right)
        let messageIds = []
        let firstUnorderedNodeDependentOnRight =
          sortNodesByDependencies.getFirstUnorderedNodeDependentOn(right, nodes)
        if (
          firstUnorderedNodeDependentOnRight ||
          indexOfLeft > indexOfRight ||
          indexOfLeft >= indexOfRightExcludingEslintDisabled
        ) {
          if (firstUnorderedNodeDependentOnRight) {
            messageIds.push('unexpectedClassesDependencyOrder')
          } else {
            messageIds.push(
              leftNumber === rightNumber
                ? 'unexpectedClassesOrder'
                : 'unexpectedClassesGroupOrder',
            )
          }
        }
        messageIds = [
          ...messageIds,
          ...getNewlinesErrors.getNewlinesErrors({
            missedSpacingError: 'missedSpacingBetweenClassMembers',
            extraSpacingError: 'extraSpacingBetweenClassMembers',
            rightNum: rightNumber,
            leftNum: leftNumber,
            sourceCode,
            options,
            right,
            left,
          }),
        ]
        for (let messageId of messageIds) {
          context.report({
            fix: fixer => [
              ...makeFixes.makeFixes({
                sortedNodes: sortedNodesExcludingEslintDisabled,
                sourceCode,
                options,
                fixer,
                nodes,
              }),
              ...makeNewlinesFixes.makeNewlinesFixes({
                sortedNodes: sortedNodesExcludingEslintDisabled,
                sourceCode,
                options,
                fixer,
                nodes,
              }),
            ],
            data: {
              nodeDependentOnRight:
                firstUnorderedNodeDependentOnRight == null
                  ? void 0
                  : firstUnorderedNodeDependentOnRight.name,
              right: toSingleLine.toSingleLine(right.name),
              left: toSingleLine.toSingleLine(left.name),
              rightGroup: right.group,
              leftGroup: left.group,
            },
            node: right.node,
            messageId,
          })
        }
      })
    },
  }),
  meta: {
    schema: [
      {
        properties: {
          ignoreCallbackDependenciesPatterns: {
            description:
              'Patterns that should be ignored when detecting dependencies in method callbacks.',
            items: {
              type: 'string',
            },
            type: 'array',
          },
          partitionByComment: {
            ...commonJsonSchemas.partitionByCommentJsonSchema,
            description:
              'Allows to use comments to separate the class members into logical groups.',
          },
          customGroups: commonJsonSchemas.buildCustomGroupsArrayJsonSchema({
            singleCustomGroupJsonSchema: types.singleCustomGroupJsonSchema,
          }),
          partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
          specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
          newlinesBetween: commonJsonSchemas.newlinesBetweenJsonSchema,
          ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
          alphabet: commonJsonSchemas.alphabetJsonSchema,
          type: commonJsonSchemas.buildTypeJsonSchema(),
          locales: commonJsonSchemas.localesJsonSchema,
          groups: commonJsonSchemas.groupsJsonSchema,
          order: commonJsonSchemas.orderJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    messages: {
      unexpectedClassesGroupOrder:
        'Expected "{{right}}" ({{rightGroup}}) to come before "{{left}}" ({{leftGroup}}).',
      unexpectedClassesDependencyOrder:
        'Expected dependency "{{right}}" to come before "{{nodeDependentOnRight}}".',
      missedSpacingBetweenClassMembers:
        'Missed spacing between "{{left}}" and "{{right}}" objects.',
      extraSpacingBetweenClassMembers:
        'Extra spacing between "{{left}}" and "{{right}}" objects.',
      unexpectedClassesOrder: 'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-classes',
      description: 'Enforce sorted classes.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-classes',
})
module.exports = sortClasses
