'use strict'
Object.defineProperties(exports, {
  __esModule: { value: true },
  [Symbol.toStringTag]: { value: 'Module' },
})
const commonJsonSchemas = require('../utils/common-json-schemas.js')
const validateGeneratedGroupsConfiguration = require('../utils/validate-generated-groups-configuration.js')
const types = require('./sort-array-includes/types.js')
const getCustomGroupsCompareOptions = require('../utils/get-custom-groups-compare-options.js')
const doesCustomGroupMatch = require('./sort-array-includes/does-custom-group-match.js')
const getMatchingContextOptions = require('../utils/get-matching-context-options.js')
const generatePredefinedGroups = require('../utils/generate-predefined-groups.js')
const getEslintDisabledLines = require('../utils/get-eslint-disabled-lines.js')
const isNodeEslintDisabled = require('../utils/is-node-eslint-disabled.js')
const hasPartitionComment = require('../utils/has-partition-comment.js')
const createNodeIndexMap = require('../utils/create-node-index-map.js')
const sortNodesByGroups = require('../utils/sort-nodes-by-groups.js')
const getCommentsBefore = require('../utils/get-comments-before.js')
const createEslintRule = require('../utils/create-eslint-rule.js')
const getLinesBetween = require('../utils/get-lines-between.js')
const getGroupNumber = require('../utils/get-group-number.js')
const getSourceCode = require('../utils/get-source-code.js')
const toSingleLine = require('../utils/to-single-line.js')
const rangeToDiff = require('../utils/range-to-diff.js')
const getSettings = require('../utils/get-settings.js')
const isSortable = require('../utils/is-sortable.js')
const makeFixes = require('../utils/make-fixes.js')
const useGroups = require('../utils/use-groups.js')
const complete = require('../utils/complete.js')
const pairwise = require('../utils/pairwise.js')
let cachedGroupsByModifiersAndSelectors = /* @__PURE__ */ new Map()
let defaultOptions = {
  groupKind: 'literals-first',
  specialCharacters: 'keep',
  partitionByComment: false,
  partitionByNewLine: false,
  useConfigurationIf: {},
  type: 'alphabetical',
  ignoreCase: true,
  locales: 'en-US',
  customGroups: [],
  alphabet: '',
  order: 'asc',
  groups: [],
}
let jsonSchema = {
  items: {
    properties: {
      partitionByComment: {
        ...commonJsonSchemas.partitionByCommentJsonSchema,
        description:
          'Allows you to use comments to separate the array members into logical groups.',
      },
      groupKind: {
        enum: ['mixed', 'literals-first', 'spreads-first'],
        description: 'Specifies top-level groups.',
        type: 'string',
      },
      customGroups: commonJsonSchemas.buildCustomGroupsArrayJsonSchema({
        singleCustomGroupJsonSchema: types.singleCustomGroupJsonSchema,
      }),
      useConfigurationIf: commonJsonSchemas.buildUseConfigurationIfJsonSchema(),
      type: commonJsonSchemas.buildTypeJsonSchema({ withUnsorted: true }),
      partitionByNewLine: commonJsonSchemas.partitionByNewLineJsonSchema,
      specialCharacters: commonJsonSchemas.specialCharactersJsonSchema,
      ignoreCase: commonJsonSchemas.ignoreCaseJsonSchema,
      alphabet: commonJsonSchemas.alphabetJsonSchema,
      locales: commonJsonSchemas.localesJsonSchema,
      groups: commonJsonSchemas.groupsJsonSchema,
      order: commonJsonSchemas.orderJsonSchema,
    },
    additionalProperties: false,
    type: 'object',
  },
  uniqueItems: true,
  type: 'array',
}
const sortArrayIncludes = createEslintRule.createEslintRule({
  create: context => ({
    MemberExpression: node => {
      if (
        (node.object.type === 'ArrayExpression' ||
          node.object.type === 'NewExpression') &&
        node.property.type === 'Identifier' &&
        node.property.name === 'includes'
      ) {
        let elements =
          node.object.type === 'ArrayExpression'
            ? node.object.elements
            : node.object.arguments
        sortArray({
          availableMessageIds: {
            unexpectedGroupOrder: 'unexpectedArrayIncludesGroupOrder',
            unexpectedOrder: 'unexpectedArrayIncludesOrder',
          },
          elements,
          context,
        })
      }
    },
  }),
  meta: {
    messages: {
      unexpectedArrayIncludesGroupOrder:
        'Expected "{{right}}" ({{rightGroup}}) to come before "{{left}}" ({{leftGroup}}).',
      unexpectedArrayIncludesOrder:
        'Expected "{{right}}" to come before "{{left}}".',
    },
    docs: {
      description: 'Enforce sorted arrays before include method.',
      url: 'https://perfectionist.dev/rules/sort-array-includes',
      recommended: true,
    },
    schema: jsonSchema,
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-array-includes',
})
let sortArray = ({ availableMessageIds, elements, context }) => {
  if (!isSortable.isSortable(elements)) {
    return
  }
  let sourceCode = getSourceCode.getSourceCode(context)
  let settings = getSettings.getSettings(context.settings)
  let matchedContextOptions =
    getMatchingContextOptions.getMatchingContextOptions({
      nodeNames: elements
        .filter(element => element !== null)
        .map(element => getNodeName({ sourceCode, element })),
      contextOptions: context.options,
    })
  let completeOptions = complete.complete(
    matchedContextOptions[0],
    settings,
    defaultOptions,
  )
  let { type } = completeOptions
  if (type === 'unsorted') {
    return
  }
  let options = {
    ...completeOptions,
    type,
  }
  validateGeneratedGroupsConfiguration.validateGeneratedGroupsConfiguration({
    customGroups: options.customGroups,
    selectors: types.allSelectors,
    groups: options.groups,
    modifiers: [],
  })
  let eslintDisabledLines = getEslintDisabledLines.getEslintDisabledLines({
    ruleName: context.id,
    sourceCode,
  })
  let formattedMembers = elements.reduce(
    (accumulator, element) => {
      var _a
      if (element === null) {
        return accumulator
      }
      let { defineGroup, getGroup } = useGroups.useGroups(options)
      let groupKind
      let selector
      if (element.type === 'SpreadElement') {
        groupKind = 'spread'
        selector = 'spread'
      } else {
        groupKind = 'literal'
        selector = 'literal'
      }
      let predefinedGroups = generatePredefinedGroups.generatePredefinedGroups({
        cache: cachedGroupsByModifiersAndSelectors,
        selectors: [selector],
        modifiers: [],
      })
      for (let predefinedGroup of predefinedGroups) {
        defineGroup(predefinedGroup)
      }
      let name = getNodeName({ sourceCode, element })
      for (let customGroup of options.customGroups) {
        if (
          doesCustomGroupMatch.doesCustomGroupMatch({
            selectors: [selector],
            elementName: name,
            customGroup,
          })
        ) {
          defineGroup(customGroup.groupName, true)
          if (getGroup() === customGroup.groupName) {
            break
          }
        }
      }
      let sortingNode = {
        isEslintDisabled: isNodeEslintDisabled.isNodeEslintDisabled(
          element,
          eslintDisabledLines,
        ),
        size: rangeToDiff.rangeToDiff(element, sourceCode),
        group: getGroup(),
        node: element,
        groupKind,
        name,
      }
      let lastSortingNode =
        (_a = accumulator.at(-1)) == null ? void 0 : _a.at(-1)
      if (
        hasPartitionComment.hasPartitionComment({
          comments: getCommentsBefore.getCommentsBefore({
            node: element,
            sourceCode,
          }),
          partitionByComment: options.partitionByComment,
        }) ||
        (options.partitionByNewLine &&
          lastSortingNode &&
          getLinesBetween.getLinesBetween(
            sourceCode,
            lastSortingNode,
            sortingNode,
          ))
      ) {
        accumulator.push([])
      }
      accumulator.at(-1).push(sortingNode)
      return accumulator
    },
    [[]],
  )
  let groupKindOrder
  if (options.groupKind === 'literals-first') {
    groupKindOrder = ['literal', 'spread']
  } else if (options.groupKind === 'spreads-first') {
    groupKindOrder = ['spread', 'literal']
  } else {
    groupKindOrder = ['any']
  }
  for (let nodes of formattedMembers) {
    let filteredGroupKindNodes = groupKindOrder.map(groupKind =>
      nodes.filter(
        currentNode =>
          groupKind === 'any' || currentNode.groupKind === groupKind,
      ),
    )
    let sortNodesIgnoringEslintDisabledNodes = ignoreEslintDisabledNodes =>
      filteredGroupKindNodes.flatMap(groupedNodes =>
        sortNodesByGroups.sortNodesByGroups(groupedNodes, options, {
          getGroupCompareOptions: groupNumber =>
            getCustomGroupsCompareOptions.getCustomGroupsCompareOptions(
              options,
              groupNumber,
            ),
          ignoreEslintDisabledNodes,
        }),
      )
    let sortedNodes = sortNodesIgnoringEslintDisabledNodes(false)
    let sortedNodesExcludingEslintDisabled =
      sortNodesIgnoringEslintDisabledNodes(true)
    let nodeIndexMap = createNodeIndexMap.createNodeIndexMap(sortedNodes)
    pairwise.pairwise(nodes, (left, right) => {
      let leftIndex = nodeIndexMap.get(left)
      let rightIndex = nodeIndexMap.get(right)
      let leftNumber = getGroupNumber.getGroupNumber(options.groups, left)
      let rightNumber = getGroupNumber.getGroupNumber(options.groups, right)
      let indexOfRightExcludingEslintDisabled =
        sortedNodesExcludingEslintDisabled.indexOf(right)
      if (
        leftIndex < rightIndex &&
        leftIndex < indexOfRightExcludingEslintDisabled
      ) {
        return
      }
      context.report({
        fix: fixer =>
          makeFixes.makeFixes({
            sortedNodes: sortedNodesExcludingEslintDisabled,
            sourceCode,
            options,
            fixer,
            nodes,
          }),
        data: {
          right: toSingleLine.toSingleLine(right.name),
          left: toSingleLine.toSingleLine(left.name),
          rightGroup: right.group,
          leftGroup: left.group,
        },
        messageId:
          leftNumber === rightNumber
            ? availableMessageIds.unexpectedOrder
            : availableMessageIds.unexpectedGroupOrder,
        node: right.node,
      })
    })
  }
}
let getNodeName = ({ sourceCode, element }) =>
  element.type === 'Literal' ? `${element.value}` : sourceCode.getText(element)
exports.default = sortArrayIncludes
exports.defaultOptions = defaultOptions
exports.jsonSchema = jsonSchema
exports.sortArray = sortArray
