// src/index.ts
import path, { posix } from "path";
import { fdir } from "fdir";
import picomatch from "picomatch";
function normalizePattern(pattern, expandDirectories, cwd, properties, isIgnore) {
  var _a;
  let result = pattern;
  if (pattern.endsWith("/")) {
    result = pattern.slice(0, -1);
  }
  if (!result.endsWith("*") && expandDirectories) {
    result += "/**";
  }
  if (path.isAbsolute(result)) {
    result = posix.relative(cwd, result);
  } else {
    result = posix.normalize(result);
  }
  const parentDirectoryMatch = /^(\/?\.\.)+/.exec(result);
  if (parentDirectoryMatch == null ? void 0 : parentDirectoryMatch[0]) {
    const potentialRoot = posix.join(cwd, parentDirectoryMatch[0]);
    if (properties.root.length > potentialRoot.length) {
      properties.root = potentialRoot;
      properties.depthOffset = -(parentDirectoryMatch[0].length + 1) / 3;
    }
  } else if (!isIgnore && properties.depthOffset >= 0) {
    const current = result.split("/");
    (_a = properties.commonPath) != null ? _a : properties.commonPath = current;
    const newCommonPath = [];
    for (let i = 0; i < Math.min(properties.commonPath.length, current.length); i++) {
      const part = current[i];
      if (properties.commonPath[i] === part && !/[\!\*\{\}\(\)]/.test(part)) {
        newCommonPath.push(part);
      } else {
        break;
      }
    }
    newCommonPath.pop();
    properties.depthOffset = newCommonPath.length;
    properties.commonPath = newCommonPath;
    properties.root = newCommonPath.length > 0 ? `${cwd}/${newCommonPath.join("/")}` : cwd;
  }
  return result;
}
function processPatterns({ patterns, ignore = [], expandDirectories = true }, cwd, properties) {
  const matchPatterns = [];
  const ignorePatterns = ignore.map((p) => normalizePattern(p, expandDirectories, cwd, properties, true));
  if (!patterns) {
    return { match: ["**/*"], ignore: ignorePatterns };
  }
  for (let pattern of patterns) {
    pattern = normalizePattern(pattern, expandDirectories, cwd, properties, false);
    if (pattern.startsWith("!") && pattern[1] !== "(") {
      ignorePatterns.push(pattern.slice(1));
    } else {
      matchPatterns.push(pattern);
    }
  }
  return { match: matchPatterns, ignore: ignorePatterns };
}
function getRelativePath(path2, cwd, root) {
  return posix.relative(cwd, `${root}/${path2}`);
}
function processPath(path2, cwd, root, isDirectory, absolute) {
  const relativePath = (absolute ? path2.slice(root.length + 1) : path2) || "./";
  if (root === cwd) {
    return isDirectory ? relativePath.slice(0, -1) : relativePath;
  }
  return getRelativePath(relativePath, cwd, root);
}
function crawl(options, cwd, sync) {
  const properties = {
    root: cwd,
    commonPath: null,
    depthOffset: 0
  };
  const processed = processPatterns(options, cwd, properties);
  const matcher = picomatch(processed.match, {
    dot: options.dot,
    nocase: options.caseSensitiveMatch === false,
    ignore: processed.ignore
  });
  const exclude = picomatch(processed.ignore, {
    dot: options.dot,
    nocase: options.caseSensitiveMatch === false
  });
  const fdirOptions = {
    // use relative paths in the matcher
    filters: [(p, isDirectory) => matcher(processPath(p, cwd, properties.root, isDirectory, options.absolute))],
    exclude: (_, p) => exclude(processPath(p, cwd, properties.root, true, true)),
    pathSeparator: "/",
    relativePaths: true
  };
  if (options.deep) {
    fdirOptions.maxDepth = Math.round(options.deep - properties.depthOffset);
  }
  if (options.absolute) {
    fdirOptions.relativePaths = false;
    fdirOptions.resolvePaths = true;
    fdirOptions.includeBasePath = true;
  }
  if (options.onlyDirectories) {
    fdirOptions.excludeFiles = true;
    fdirOptions.includeDirs = true;
  } else if (options.onlyFiles === false) {
    fdirOptions.includeDirs = true;
  }
  const api = new fdir(fdirOptions).crawl(properties.root);
  if (cwd === properties.root || options.absolute) {
    return sync ? api.sync() : api.withPromise();
  }
  return sync ? api.sync().map((p) => getRelativePath(p, cwd, properties.root) + (!p || p.endsWith("/") ? "/" : "")) : api.withPromise().then((paths) => paths.map((p) => getRelativePath(p, cwd, properties.root) + (!p || p.endsWith("/") ? "/" : "")));
}
async function glob(patternsOrOptions, options) {
  if (patternsOrOptions && (options == null ? void 0 : options.patterns)) {
    throw new Error("Cannot pass patterns as both an argument and an option");
  }
  const opts = Array.isArray(patternsOrOptions) ? { ...options, patterns: patternsOrOptions } : patternsOrOptions;
  const cwd = opts.cwd ? path.resolve(opts.cwd).replace(/\\/g, "/") : process.cwd().replace(/\\/g, "/");
  return crawl(opts, cwd, false);
}
function globSync(patternsOrOptions, options) {
  if (patternsOrOptions && (options == null ? void 0 : options.patterns)) {
    throw new Error("Cannot pass patterns as both an argument and an option");
  }
  const opts = Array.isArray(patternsOrOptions) ? { ...options, patterns: patternsOrOptions } : patternsOrOptions;
  const cwd = opts.cwd ? path.resolve(opts.cwd).replace(/\\/g, "/") : process.cwd().replace(/\\/g, "/");
  return crawl(opts, cwd, true);
}
export {
  glob,
  globSync
};
//# sourceMappingURL=index.mjs.map