"use strict";
/**
 * This file contains a Vite loader for wyw-in-js.
 * It uses the transform.ts function to generate class names from source code,
 * returns transformed code without template literals and attaches generated source maps
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = require("fs");
const path_1 = __importDefault(require("path"));
const vite_1 = require("vite");
const shared_1 = require("@wyw-in-js/shared");
const transform_1 = require("@wyw-in-js/transform");
function wywInJS({ debug, include, exclude, sourceMap, preprocessor, ...rest } = {}) {
    const filter = (0, vite_1.createFilter)(include, exclude);
    const cssLookup = {};
    const cssFileLookup = {};
    let config;
    let devServer;
    const { emitter, onDone } = (0, transform_1.createFileReporter)(debug ?? false);
    // <dependency id, targets>
    const targets = [];
    const cache = new transform_1.TransformCacheCollection();
    return {
        name: 'wyw-in-js',
        enforce: 'post',
        buildEnd() {
            onDone(process.cwd());
        },
        configResolved(resolvedConfig) {
            config = resolvedConfig;
        },
        configureServer(_server) {
            devServer = _server;
        },
        load(url) {
            const [id] = url.split('?', 1);
            return cssLookup[id];
        },
        /* eslint-disable-next-line consistent-return */
        resolveId(importeeUrl) {
            const [id] = importeeUrl.split('?', 1);
            if (cssLookup[id])
                return id;
            return cssFileLookup[id];
        },
        handleHotUpdate(ctx) {
            // it's module, so just transform it
            if (ctx.modules.length)
                return ctx.modules;
            // Select affected modules of changed dependency
            const affected = targets.filter((x) => 
            // file is dependency of any target
            x.dependencies.some((dep) => dep === ctx.file) ||
                // or changed module is a dependency of any target
                x.dependencies.some((dep) => ctx.modules.some((m) => m.file === dep)));
            const deps = affected.flatMap((target) => target.dependencies);
            // eslint-disable-next-line no-restricted-syntax
            for (const depId of deps) {
                cache.invalidateForFile(depId);
            }
            return affected
                .map((target) => devServer.moduleGraph.getModuleById(target.id))
                .concat(ctx.modules)
                .filter((m) => !!m);
        },
        async transform(code, url) {
            const [id] = url.split('?', 1);
            // Do not transform ignored and generated files
            if (url.includes('node_modules') || !filter(url) || id in cssLookup)
                return;
            const log = shared_1.logger.extend('vite').extend((0, transform_1.getFileIdx)(id));
            log('transform %s', id);
            const asyncResolve = async (what, importer, stack) => {
                const resolved = await this.resolve(what, importer);
                if (resolved) {
                    if (resolved.external) {
                        // If module is marked as external, Rollup will not resolve it,
                        // so we need to resolve it ourselves with default resolver
                        const resolvedId = (0, shared_1.syncResolve)(what, importer, stack);
                        log("resolve ✅ '%s'@'%s -> %O\n%s", what, importer, resolved);
                        return resolvedId;
                    }
                    log("resolve ✅ '%s'@'%s -> %O\n%s", what, importer, resolved);
                    // Vite adds param like `?v=667939b3` to cached modules
                    const resolvedId = resolved.id.split('?', 1)[0];
                    if (resolvedId.startsWith('\0')) {
                        // \0 is a special character in Rollup that tells Rollup to not include this in the bundle
                        // https://rollupjs.org/guide/en/#outputexports
                        return null;
                    }
                    if (!(0, fs_1.existsSync)(resolvedId)) {
                        await (0, vite_1.optimizeDeps)(config);
                    }
                    return resolvedId;
                }
                log("resolve ❌ '%s'@'%s", what, importer);
                throw new Error(`Could not resolve ${what}`);
            };
            const transformServices = {
                options: {
                    filename: id,
                    root: process.cwd(),
                    preprocessor,
                    pluginOptions: rest,
                },
                cache,
                eventEmitter: emitter,
            };
            const result = await (0, transform_1.transform)(transformServices, code, asyncResolve);
            let { cssText, dependencies } = result;
            // Heads up, there are three cases:
            // 1. cssText is undefined, it means that file was not transformed
            // 2. cssText is empty, it means that file was transformed, but it does not contain any styles
            // 3. cssText is not empty, it means that file was transformed and it contains styles
            if (typeof cssText === 'undefined') {
                return;
            }
            if (cssText === '') {
                /* eslint-disable-next-line consistent-return */
                return {
                    code: result.code,
                    map: result.sourceMap,
                };
            }
            dependencies ??= [];
            const cssFilename = path_1.default
                .normalize(`${id.replace(/\.[jt]sx?$/, '')}.wyw-in-js.css`)
                .replace(/\\/g, path_1.default.posix.sep);
            const cssRelativePath = path_1.default
                .relative(config.root, cssFilename)
                .replace(/\\/g, path_1.default.posix.sep);
            const cssId = `/${cssRelativePath}`;
            if (sourceMap && result.cssSourceMapText) {
                const map = Buffer.from(result.cssSourceMapText).toString('base64');
                cssText += `/*# sourceMappingURL=data:application/json;base64,${map}*/`;
            }
            cssLookup[cssFilename] = cssText;
            cssFileLookup[cssId] = cssFilename;
            result.code += `\nimport ${JSON.stringify(cssFilename)};\n`;
            for (let i = 0, end = dependencies.length; i < end; i++) {
                // eslint-disable-next-line no-await-in-loop
                const depModule = await this.resolve(dependencies[i], url, {
                    isEntry: false,
                });
                if (depModule)
                    dependencies[i] = depModule.id;
            }
            const target = targets.find((t) => t.id === id);
            if (!target)
                targets.push({ id, dependencies });
            else
                target.dependencies = dependencies;
            if (devServer?.moduleGraph) {
                const module = devServer.moduleGraph.getModuleById(cssFilename);
                if (module) {
                    devServer.reloadModule(module);
                }
            }
            /* eslint-disable-next-line consistent-return */
            return { code: result.code, map: result.sourceMap };
        },
    };
}
exports.default = wywInJS;
