"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const PropertySymbol = __importStar(require("../../PropertySymbol.cjs"));
const WindowErrorUtility_js_1 = __importDefault(require("../../window/WindowErrorUtility.cjs"));
const AbortController_js_1 = __importDefault(require("../../fetch/AbortController.cjs"));
const BrowserFrameFactory_js_1 = __importDefault(require("./BrowserFrameFactory.cjs"));
const BrowserFrameURL_js_1 = __importDefault(require("./BrowserFrameURL.cjs"));
const BrowserFrameValidator_js_1 = __importDefault(require("./BrowserFrameValidator.cjs"));
const AsyncTaskManager_js_1 = __importDefault(require("../../async-task-manager/AsyncTaskManager.cjs"));
const BrowserErrorCaptureEnum_js_1 = __importDefault(require("../enums/BrowserErrorCaptureEnum.cjs"));
/**
 * Browser frame navigation utility.
 */
class BrowserFrameNavigator {
    /**
     * Navigates to a page.
     *
     * @throws Error if the request can't be resolved (because of SSL error or similar). It will not throw if the response is not ok.
     * @param options Options.
     * @param options.windowClass Window class.
     * @param options.frame Frame.
     * @param options.url URL.
     * @param [options.formData] Form data.
     * @param [options.method] Method.
     * @param [options.goToOptions] Go to options.
     * @returns Response.
     */
    static async navigate(options) {
        const { windowClass, frame, url, formData, method, goToOptions } = options;
        const referrer = goToOptions?.referrer || frame.window.location.origin;
        const targetURL = BrowserFrameURL_js_1.default.getRelativeURL(frame, url);
        if (!frame.window) {
            throw new Error('The frame has been destroyed, the "window" property is not set.');
        }
        if (targetURL.protocol === 'javascript:') {
            if (frame && !frame.page.context.browser.settings.disableJavaScriptEvaluation) {
                const readyStateManager = frame.window[PropertySymbol.readyStateManager];
                readyStateManager.startTask();
                // The browser will wait for the next tick before executing the script.
                await new Promise((resolve) => frame.page.mainFrame.window.setTimeout(resolve));
                const code = '//# sourceURL=' + frame.url + '\n' + targetURL.href.replace('javascript:', '');
                if (frame.page.context.browser.settings.disableErrorCapturing ||
                    frame.page.context.browser.settings.errorCapture !== BrowserErrorCaptureEnum_js_1.default.tryAndCatch) {
                    frame.window.eval(code);
                }
                else {
                    WindowErrorUtility_js_1.default.captureError(frame.window, () => frame.window.eval(code));
                }
                readyStateManager.endTask();
            }
            return null;
        }
        if (!BrowserFrameValidator_js_1.default.validateCrossOriginPolicy(frame, targetURL)) {
            return null;
        }
        if (!BrowserFrameValidator_js_1.default.validateFrameNavigation(frame)) {
            if (!frame.page.context.browser.settings.navigation.disableFallbackToSetURL) {
                frame.window.location[PropertySymbol.setURL](frame, targetURL.href);
            }
            return null;
        }
        const width = frame.window.innerWidth;
        const height = frame.window.innerHeight;
        const devicePixelRatio = frame.window.devicePixelRatio;
        const parentWindow = frame.window.parent !== frame.window ? frame.window.parent : null;
        const topWindow = frame.window.top !== frame.window ? frame.window.top : null;
        for (const childFrame of frame.childFrames) {
            BrowserFrameFactory_js_1.default.destroyFrame(childFrame);
        }
        frame.childFrames = [];
        frame.window[PropertySymbol.destroy]();
        frame[PropertySymbol.asyncTaskManager].destroy();
        frame[PropertySymbol.asyncTaskManager] = new AsyncTaskManager_js_1.default();
        frame.window = new windowClass(frame, { url: targetURL.href, width, height });
        frame.window.parent = parentWindow;
        frame.window.top = topWindow;
        frame.window.devicePixelRatio = devicePixelRatio;
        if (referrer) {
            frame.window.document[PropertySymbol.referrer] = referrer;
        }
        if (targetURL.protocol === 'about:') {
            return null;
        }
        const readyStateManager = frame.window[PropertySymbol.readyStateManager];
        readyStateManager.startTask();
        const abortController = new AbortController_js_1.default();
        let response;
        let responseText;
        const timeout = frame.window.setTimeout(() => abortController.abort(new Error('Request timed out.')), goToOptions?.timeout ?? 30000);
        const finalize = () => {
            frame.window.clearTimeout(timeout);
            readyStateManager.endTask();
            const listeners = frame[PropertySymbol.listeners].navigation;
            frame[PropertySymbol.listeners].navigation = [];
            for (const listener of listeners) {
                listener();
            }
        };
        try {
            response = await frame.window.fetch(targetURL.href, {
                referrer,
                referrerPolicy: goToOptions?.referrerPolicy || 'origin',
                signal: abortController.signal,
                method: method || (formData ? 'POST' : 'GET'),
                headers: goToOptions?.hard ? { 'Cache-Control': 'no-cache' } : undefined,
                body: formData
            });
            // Handles the "X-Frame-Options" header for child frames.
            if (frame.parentFrame) {
                const originURL = frame.parentFrame.window.location;
                const xFrameOptions = response.headers?.get('X-Frame-Options')?.toLowerCase();
                const isSameOrigin = originURL.origin === targetURL.origin || targetURL.origin === 'null';
                if (xFrameOptions === 'deny' || (xFrameOptions === 'sameorigin' && !isSameOrigin)) {
                    throw new Error(`Refused to display '${url}' in a frame because it set 'X-Frame-Options' to '${xFrameOptions}'.`);
                }
            }
            responseText = await response.text();
        }
        catch (error) {
            finalize();
            throw error;
        }
        if (!response.ok) {
            frame.page.console.error(`GET ${targetURL.href} ${response.status} (${response.statusText})`);
        }
        // Fixes issue where evaluating the response can throw an error.
        // By using requestAnimationFrame() the error will not reject the promise.
        // The error will be caught by process error level listener or a try and catch in the requestAnimationFrame().
        frame.window.requestAnimationFrame(() => (frame.content = responseText));
        await new Promise((resolve) => frame.window.requestAnimationFrame(() => {
            finalize();
            resolve(null);
        }));
        return response;
    }
}
exports.default = BrowserFrameNavigator;
//# sourceMappingURL=BrowserFrameNavigator.cjs.map