import { Config } from './config';
import { PrettyPrintableError } from './errors';
import { BooleanFlagProps, CompletableFlag, Deprecation, Arg as IArg, ArgInput, FlagInput, FlagOutput, Input, ArgProps, OptionFlagProps, ParserOutput, ArgOutput } from './interfaces/parser';
import { Plugin } from './interfaces/plugin';
import { LoadOptions } from './interfaces/config';
import { CommandError } from './interfaces/errors';
/**
 * An abstract class which acts as the base for each command
 * in your project.
 */
export declare abstract class Command {
    argv: string[];
    config: Config;
    private static readonly _base;
    /** A command ID, used mostly in error or verbose reporting. */
    static id: string;
    /**
     * The tweet-sized description for your class, used in a parent-commands
     * sub-command listing and as the header for the command help.
     */
    static summary?: string;
    /**
     * A full description of how to use the command.
     *
     * If no summary, the first line of the description will be used as the summary.
     */
    static description: string | undefined;
    /** Hide the command from help */
    static hidden: boolean;
    /** An array of aliases for this command that are hidden from help. */
    static hiddenAliases: string[];
    /** Mark the command as a given state (e.g. beta or deprecated) in help */
    static state?: 'beta' | 'deprecated' | string;
    static deprecationOptions?: Deprecation;
    /**
     * Emit deprecation warning when a command alias is used
     */
    static deprecateAliases?: boolean;
    /**
     * An override string (or strings) for the default usage documentation.
     */
    static usage: string | string[] | undefined;
    static help: string | undefined;
    /** An array of aliases for this command. */
    static aliases: string[];
    /** When set to false, allows a variable amount of arguments */
    static strict: boolean;
    /** An order-dependent object of arguments for the command */
    static args: ArgInput;
    static plugin: Plugin | undefined;
    static readonly pluginName?: string;
    static readonly pluginType?: string;
    static readonly pluginAlias?: string;
    /**
     * An array of examples to show at the end of the command's help.
     *
     * IF only a string is provided, it will try to look for a line that starts
     * with the cmd.bin as the example command and the rest as the description.
     * If found, the command will be formatted appropriately.
     *
     * ```
     * EXAMPLES:
     *   A description of a particular use case.
     *
     *     $ <%= config.bin => command flags
     * ```
     */
    static examples: Command.Example[];
    static hasDynamicHelp: boolean;
    protected static '_--': boolean;
    protected static _enableJsonFlag: boolean;
    static get enableJsonFlag(): boolean;
    static set enableJsonFlag(value: boolean);
    static get '--'(): boolean;
    static set '--'(value: boolean);
    get passThroughEnabled(): boolean;
    /**
     * instantiate and run the command
     *
     * @param {Command.Class} this - the command class
     * @param {string[]} argv argv
     * @param {LoadOptions} opts options
     * @returns {Promise<unknown>} result
     */
    static run<T extends Command>(this: new (argv: string[], config: Config) => T, argv?: string[], opts?: LoadOptions): Promise<ReturnType<T['run']>>;
    protected static _baseFlags: FlagInput;
    static get baseFlags(): FlagInput;
    static set baseFlags(flags: FlagInput);
    /** A hash of flags for the command */
    protected static _flags: FlagInput;
    static get flags(): FlagInput;
    static set flags(flags: FlagInput);
    id: string | undefined;
    protected debug: (...args: any[]) => void;
    constructor(argv: string[], config: Config);
    protected get ctor(): typeof Command;
    protected _run<T>(): Promise<T>;
    exit(code?: number): never;
    warn(input: string | Error): string | Error;
    error(input: string | Error, options: {
        code?: string;
        exit: false;
    } & PrettyPrintableError): void;
    error(input: string | Error, options?: {
        code?: string;
        exit?: number;
    } & PrettyPrintableError): never;
    log(message?: string, ...args: any[]): void;
    logToStderr(message?: string, ...args: any[]): void;
    /**
     * Determine if the command is being run with the --json flag in a command that supports it.
     *
     * @returns {boolean} true if the command supports json and the --json flag is present
     */
    jsonEnabled(): boolean;
    /**
     * actual command run code goes here
     */
    abstract run(): Promise<any>;
    protected init(): Promise<any>;
    protected warnIfFlagDeprecated(flags: Record<string, unknown>): void;
    protected warnIfCommandDeprecated(): void;
    protected parse<F extends FlagOutput, B extends FlagOutput, A extends ArgOutput>(options?: Input<F, B, A>, argv?: string[]): Promise<ParserOutput<F, B, A>>;
    protected catch(err: CommandError): Promise<any>;
    protected finally(_: Error | undefined): Promise<any>;
    protected toSuccessJson(result: unknown): any;
    protected toErrorJson(err: unknown): any;
    protected logJson(json: unknown): void;
    private removeEnvVar;
}
export declare namespace Command {
    type Class = typeof Command & {
        id: string;
        run(argv?: string[], config?: LoadOptions): Promise<any>;
    };
    interface Loadable extends Cached {
        load(): Promise<Command.Class>;
    }
    type Cached = {
        [key: string]: unknown;
        id: string;
        hidden: boolean;
        hiddenAliases: string[];
        state?: 'beta' | 'deprecated' | string;
        deprecationOptions?: Deprecation;
        aliases: string[];
        summary?: string;
        description?: string;
        usage?: string | string[];
        examples?: Example[];
        strict?: boolean;
        type?: string;
        pluginName?: string;
        pluginType?: string;
        pluginAlias?: string;
        flags: {
            [name: string]: Flag.Cached;
        };
        args: {
            [name: string]: Arg.Cached;
        };
        hasDynamicHelp?: boolean;
    };
    type Flag = CompletableFlag<any>;
    namespace Flag {
        type Cached = Omit<Flag, 'parse' | 'input'> & (BooleanFlagProps | OptionFlagProps);
        type Any = Flag | Cached;
    }
    type Arg = IArg<any>;
    namespace Arg {
        type Cached = Omit<Arg, 'parse' | 'input'> & ArgProps;
        type Any = Arg | Cached;
    }
    type Example = string | {
        description: string;
        command: string;
    };
}
