/*!
 * (C) Ionic http://ionicframework.com - MIT License
 */
import { Host, h } from "@stencil/core";
import { getElementRoot, raf } from "../../utils/helpers";
import { hapticSelectionChanged, hapticSelectionEnd, hapticSelectionStart } from "../../utils/native/haptic";
import { isPlatform } from "../../utils/platform";
import { createColorClasses } from "../../utils/theme";
import { getIonMode } from "../../global/ionic-global";
/**
 * @virtualProp {"ios" | "md"} mode - The mode determines which platform styles to use.
 * @internal
 */
export class PickerColumnInternal {
    constructor() {
        this.isScrolling = false;
        this.isColumnVisible = false;
        this.canExitInputMode = true;
        this.centerPickerItemInView = (target, smooth = true, canExitInputMode = true) => {
            const { el, isColumnVisible } = this;
            if (isColumnVisible) {
                // (Vertical offset from parent) - (three empty picker rows) + (half the height of the target to ensure the scroll triggers)
                const top = target.offsetTop - 3 * target.clientHeight + target.clientHeight / 2;
                if (el.scrollTop !== top) {
                    /**
                     * Setting this flag prevents input
                     * mode from exiting in the picker column's
                     * scroll callback. This is useful when the user manually
                     * taps an item or types on the keyboard as both
                     * of these can cause a scroll to occur.
                     */
                    this.canExitInputMode = canExitInputMode;
                    el.scroll({
                        top,
                        left: 0,
                        behavior: smooth ? 'smooth' : undefined,
                    });
                }
            }
        };
        this.setPickerItemActiveState = (item, isActive) => {
            if (isActive) {
                item.classList.add(PICKER_ITEM_ACTIVE_CLASS);
                item.part.add(PICKER_ITEM_ACTIVE_PART);
            }
            else {
                item.classList.remove(PICKER_ITEM_ACTIVE_CLASS);
                item.part.remove(PICKER_ITEM_ACTIVE_PART);
            }
        };
        /**
         * When ionInputModeChange is emitted, each column
         * needs to check if it is the one being made available
         * for text entry.
         */
        this.inputModeChange = (ev) => {
            if (!this.numericInput) {
                return;
            }
            const { useInputMode, inputModeColumn } = ev.detail;
            /**
             * If inputModeColumn is undefined then this means
             * all numericInput columns are being selected.
             */
            const isColumnActive = inputModeColumn === undefined || inputModeColumn === this.el;
            if (!useInputMode || !isColumnActive) {
                this.setInputModeActive(false);
                return;
            }
            this.setInputModeActive(true);
        };
        /**
         * Setting isActive will cause a re-render.
         * As a result, we do not want to cause the
         * re-render mid scroll as this will cause
         * the picker column to jump back to
         * whatever value was selected at the
         * start of the scroll interaction.
         */
        this.setInputModeActive = (state) => {
            if (this.isScrolling) {
                this.scrollEndCallback = () => {
                    this.isActive = state;
                };
                return;
            }
            this.isActive = state;
        };
        /**
         * When the column scrolls, the component
         * needs to determine which item is centered
         * in the view and will emit an ionChange with
         * the item object.
         */
        this.initializeScrollListener = () => {
            /**
             * The haptics for the wheel picker are
             * an iOS-only feature. As a result, they should
             * be disabled on Android.
             */
            const enableHaptics = isPlatform('ios');
            const { el } = this;
            let timeout;
            let activeEl = this.activeItem;
            const scrollCallback = () => {
                raf(() => {
                    if (timeout) {
                        clearTimeout(timeout);
                        timeout = undefined;
                    }
                    if (!this.isScrolling) {
                        enableHaptics && hapticSelectionStart();
                        this.isScrolling = true;
                    }
                    /**
                     * Select item in the center of the column
                     * which is the month/year that we want to select
                     */
                    const bbox = el.getBoundingClientRect();
                    const centerX = bbox.x + bbox.width / 2;
                    const centerY = bbox.y + bbox.height / 2;
                    const activeElement = el.shadowRoot.elementFromPoint(centerX, centerY);
                    if (activeEl !== null) {
                        this.setPickerItemActiveState(activeEl, false);
                    }
                    if (activeElement === null || activeElement.disabled) {
                        return;
                    }
                    /**
                     * If we are selecting a new value,
                     * we need to run haptics again.
                     */
                    if (activeElement !== activeEl) {
                        enableHaptics && hapticSelectionChanged();
                        if (this.canExitInputMode) {
                            /**
                             * The native iOS wheel picker
                             * only dismisses the keyboard
                             * once the selected item has changed
                             * as a result of a swipe
                             * from the user. If `canExitInputMode` is
                             * `false` then this means that the
                             * scroll is happening as a result of
                             * the `value` property programmatically changing
                             * either by an application or by the user via the keyboard.
                             */
                            this.exitInputMode();
                        }
                    }
                    activeEl = activeElement;
                    this.setPickerItemActiveState(activeElement, true);
                    timeout = setTimeout(() => {
                        this.isScrolling = false;
                        enableHaptics && hapticSelectionEnd();
                        /**
                         * Certain tasks (such as those that
                         * cause re-renders) should only be done
                         * once scrolling has finished, otherwise
                         * flickering may occur.
                         */
                        const { scrollEndCallback } = this;
                        if (scrollEndCallback) {
                            scrollEndCallback();
                            this.scrollEndCallback = undefined;
                        }
                        /**
                         * Reset this flag as the
                         * next scroll interaction could
                         * be a scroll from the user. In this
                         * case, we should exit input mode.
                         */
                        this.canExitInputMode = true;
                        const dataIndex = activeElement.getAttribute('data-index');
                        /**
                         * If no value it is
                         * possible we hit one of the
                         * empty padding columns.
                         */
                        if (dataIndex === null) {
                            return;
                        }
                        const index = parseInt(dataIndex, 10);
                        const selectedItem = this.items[index];
                        if (selectedItem.value !== this.value) {
                            this.setValue(selectedItem.value);
                        }
                    }, 250);
                });
            };
            /**
             * Wrap this in an raf so that the scroll callback
             * does not fire when component is initially shown.
             */
            raf(() => {
                el.addEventListener('scroll', scrollCallback);
                this.destroyScrollListener = () => {
                    el.removeEventListener('scroll', scrollCallback);
                };
            });
        };
        /**
         * Tells the parent picker to
         * exit text entry mode. This is only called
         * when the selected item changes during scroll, so
         * we know that the user likely wants to scroll
         * instead of type.
         */
        this.exitInputMode = () => {
            const { parentEl } = this;
            if (parentEl == null)
                return;
            parentEl.exitInputMode();
            /**
             * setInputModeActive only takes
             * effect once scrolling stops to avoid
             * a component re-render while scrolling.
             * However, we want the visual active
             * indicator to go away immediately, so
             * we call classList.remove here.
             */
            this.el.classList.remove('picker-column-active');
        };
        this.isActive = false;
        this.disabled = false;
        this.items = [];
        this.value = undefined;
        this.color = 'primary';
        this.numericInput = false;
    }
    valueChange() {
        if (this.isColumnVisible) {
            /**
             * Only scroll the active item into view when the picker column
             * is actively visible to the user.
             */
            this.scrollActiveItemIntoView();
        }
    }
    /**
     * Only setup scroll listeners
     * when the picker is visible, otherwise
     * the container will have a scroll
     * height of 0px.
     */
    componentWillLoad() {
        const visibleCallback = (entries) => {
            const ev = entries[0];
            if (ev.isIntersecting) {
                const { activeItem, el } = this;
                this.isColumnVisible = true;
                /**
                 * Because this initial call to scrollActiveItemIntoView has to fire before
                 * the scroll listener is set up, we need to manage the active class manually.
                 */
                const oldActive = getElementRoot(el).querySelector(`.${PICKER_ITEM_ACTIVE_CLASS}`);
                if (oldActive) {
                    this.setPickerItemActiveState(oldActive, false);
                }
                this.scrollActiveItemIntoView();
                if (activeItem) {
                    this.setPickerItemActiveState(activeItem, true);
                }
                this.initializeScrollListener();
            }
            else {
                this.isColumnVisible = false;
                if (this.destroyScrollListener) {
                    this.destroyScrollListener();
                    this.destroyScrollListener = undefined;
                }
            }
        };
        new IntersectionObserver(visibleCallback, { threshold: 0.001 }).observe(this.el);
        const parentEl = (this.parentEl = this.el.closest('ion-picker-internal'));
        if (parentEl !== null) {
            // TODO(FW-2832): type
            parentEl.addEventListener('ionInputModeChange', (ev) => this.inputModeChange(ev));
        }
    }
    componentDidRender() {
        var _a;
        const { activeItem, items, isColumnVisible, value } = this;
        if (isColumnVisible) {
            if (activeItem) {
                this.scrollActiveItemIntoView();
            }
            else if (((_a = items[0]) === null || _a === void 0 ? void 0 : _a.value) !== value) {
                /**
                 * If the picker column does not have an active item and the current value
                 * does not match the first item in the picker column, that means
                 * the value is out of bounds. In this case, we assign the value to the
                 * first item to match the scroll position of the column.
                 *
                 */
                this.setValue(items[0].value);
            }
        }
    }
    /** @internal  */
    async scrollActiveItemIntoView() {
        const activeEl = this.activeItem;
        if (activeEl) {
            this.centerPickerItemInView(activeEl, false, false);
        }
    }
    /**
     * Sets the value prop and fires the ionChange event.
     * This is used when we need to fire ionChange from
     * user-generated events that cannot be caught with normal
     * input/change event listeners.
     * @internal
     */
    async setValue(value) {
        const { items } = this;
        this.value = value;
        const findItem = items.find((item) => item.value === value && item.disabled !== true);
        if (findItem) {
            this.ionChange.emit(findItem);
        }
    }
    get activeItem() {
        // If the whole picker column is disabled, the current value should appear active
        // If the current value item is specifically disabled, it should not appear active
        const selector = `.picker-item[data-value="${this.value}"]${this.disabled ? '' : ':not([disabled])'}`;
        return getElementRoot(this.el).querySelector(selector);
    }
    render() {
        const { items, color, disabled: pickerDisabled, isActive, numericInput } = this;
        const mode = getIonMode(this);
        /**
         * exportparts is needed so ion-datetime can expose the parts
         * from two layers of shadow nesting. If this causes problems,
         * the attribute can be moved to datetime.tsx and set on every
         * instance of ion-picker-column-internal there instead.
         */
        return (h(Host, { key: 'd0e3818aa4ab06aefd988d1b42ae754244727b8f', exportparts: `${PICKER_ITEM_PART}, ${PICKER_ITEM_ACTIVE_PART}`, disabled: pickerDisabled, tabindex: pickerDisabled ? null : 0, class: createColorClasses(color, {
                [mode]: true,
                ['picker-column-active']: isActive,
                ['picker-column-numeric-input']: numericInput,
            }) }, h("div", { key: 'ec3b16415db5f64e3fab37fda023560ebcdc9423', class: "picker-item picker-item-empty", "aria-hidden": "true" }, "\u00A0"), h("div", { key: '40c78c9a1277807572d73e23048c2730cab88f5a', class: "picker-item picker-item-empty", "aria-hidden": "true" }, "\u00A0"), h("div", { key: 'c51fa817f6999356cf3568a46098e954936b45d5', class: "picker-item picker-item-empty", "aria-hidden": "true" }, "\u00A0"), items.map((item, index) => {
            const isItemDisabled = pickerDisabled || item.disabled || false;
            {
                /*
                Users should be able to tab
                between multiple columns. As a result,
                we set tabindex here so that tabbing switches
                between columns instead of buttons. Users
                can still use arrow keys on the keyboard to
                navigate the column up and down.
              */
            }
            return (h("button", { tabindex: "-1", class: {
                    'picker-item': true,
                }, "data-value": item.value, "data-index": index, onClick: (ev) => {
                    this.centerPickerItemInView(ev.target, true);
                }, disabled: isItemDisabled, part: PICKER_ITEM_PART }, item.text));
        }), h("div", { key: '80a7461dfe571f46466219b05007b021711aed32', class: "picker-item picker-item-empty", "aria-hidden": "true" }, "\u00A0"), h("div", { key: 'fed82263295f58c51e2b753d0c06c7d71872b8a7', class: "picker-item picker-item-empty", "aria-hidden": "true" }, "\u00A0"), h("div", { key: 'e805dff4122cc4e5fb4d7dcb96054fee8ad3bd07', class: "picker-item picker-item-empty", "aria-hidden": "true" }, "\u00A0")));
    }
    static get is() { return "ion-picker-column-internal"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "ios": ["picker-column-internal.ios.scss"],
            "md": ["picker-column-internal.md.scss"]
        };
    }
    static get styleUrls() {
        return {
            "ios": ["picker-column-internal.ios.css"],
            "md": ["picker-column-internal.md.css"]
        };
    }
    static get properties() {
        return {
            "disabled": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the user cannot interact with the picker."
                },
                "attribute": "disabled",
                "reflect": false,
                "defaultValue": "false"
            },
            "items": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "PickerColumnItem[]",
                    "resolved": "PickerColumnItem[]",
                    "references": {
                        "PickerColumnItem": {
                            "location": "import",
                            "path": "./picker-column-internal-interfaces",
                            "id": "src/components/picker-column-internal/picker-column-internal-interfaces.ts::PickerColumnItem"
                        }
                    }
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "A list of options to be displayed in the picker"
                },
                "defaultValue": "[]"
            },
            "value": {
                "type": "any",
                "mutable": true,
                "complexType": {
                    "original": "string | number",
                    "resolved": "number | string | undefined",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The selected option in the picker."
                },
                "attribute": "value",
                "reflect": false
            },
            "color": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "Color",
                    "resolved": "\"danger\" | \"dark\" | \"light\" | \"medium\" | \"primary\" | \"secondary\" | \"success\" | \"tertiary\" | \"warning\" | string & Record<never, never> | undefined",
                    "references": {
                        "Color": {
                            "location": "import",
                            "path": "../../interface",
                            "id": "src/interface.d.ts::Color"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The color to use from your application's color palette.\nDefault options are: `\"primary\"`, `\"secondary\"`, `\"tertiary\"`, `\"success\"`, `\"warning\"`, `\"danger\"`, `\"light\"`, `\"medium\"`, and `\"dark\"`.\nFor more information on colors, see [theming](/docs/theming/basics)."
                },
                "attribute": "color",
                "reflect": true,
                "defaultValue": "'primary'"
            },
            "numericInput": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }],
                    "text": "If `true`, tapping the picker will\nreveal a number input keyboard that lets\nthe user type in values for each picker\ncolumn. This is useful when working\nwith time pickers."
                },
                "attribute": "numeric-input",
                "reflect": false,
                "defaultValue": "false"
            }
        };
    }
    static get states() {
        return {
            "isActive": {}
        };
    }
    static get events() {
        return [{
                "method": "ionChange",
                "name": "ionChange",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the value has changed."
                },
                "complexType": {
                    "original": "PickerColumnItem",
                    "resolved": "PickerColumnItem",
                    "references": {
                        "PickerColumnItem": {
                            "location": "import",
                            "path": "./picker-column-internal-interfaces",
                            "id": "src/components/picker-column-internal/picker-column-internal-interfaces.ts::PickerColumnItem"
                        }
                    }
                }
            }];
    }
    static get methods() {
        return {
            "scrollActiveItemIntoView": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }]
                }
            },
            "setValue": {
                "complexType": {
                    "signature": "(value?: string | number) => Promise<void>",
                    "parameters": [{
                            "name": "value",
                            "type": "string | number | undefined",
                            "docs": ""
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Sets the value prop and fires the ionChange event.\nThis is used when we need to fire ionChange from\nuser-generated events that cannot be caught with normal\ninput/change event listeners.",
                    "tags": [{
                            "name": "internal",
                            "text": undefined
                        }]
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "value",
                "methodName": "valueChange"
            }];
    }
}
const PICKER_ITEM_ACTIVE_CLASS = 'picker-item-active';
const PICKER_ITEM_PART = 'wheel-item';
const PICKER_ITEM_ACTIVE_PART = 'active';
