import { Dispatch, SetStateAction, EffectCallback, RefObject, CSSProperties, useLayoutEffect, DependencyList } from 'react';

interface UseBooleanOutput {
    value: boolean;
    setValue: Dispatch<SetStateAction<boolean>>;
    setTrue: () => void;
    setFalse: () => void;
    toggle: () => void;
}
/**
 * Custom hook for handling boolean state with useful utility functions.
 * @param {boolean} [defaultValue] - The initial value for the boolean state (default is `false`).
 * @returns {UseBooleanOutput} An object containing the boolean state value and utility functions to manipulate the state.
 * @property {boolean} value - The current boolean state value.
 * @property {Function} setValue - Function to set the boolean state directly.
 * @property {Function} setTrue - Function to set the boolean state to `true`.
 * @property {Function} setFalse - Function to set the boolean state to `false`.
 * @property {Function} toggle - Function to toggle the boolean state.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-boolean)
 * @example
 * const { value, setTrue, setFalse, toggle } = useBoolean(true);
 *
 * console.log(value); // true
 * setFalse();
 * console.log(value); // false
 * toggle();
 * console.log(value); // true
 */
declare function useBoolean(defaultValue?: boolean): UseBooleanOutput;

/**
 * Custom hook for handling click events anywhere on the document.
 * @param {Function} handler - The function to be called when a click event is detected anywhere on the document.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-click-any-where)
 * @example
 * // Usage of useClickAnywhere hook
 * const handleClick = (event) => {
 *   console.log('Document clicked!', event);
 * };
 *
 * // Attach click event handler to document
 * useClickAnywhere(handleClick);
 */
declare function useClickAnyWhere(handler: (event: MouseEvent) => void): void;

/**
 * The copied text as `string` or `null` if nothing has been copied yet.
 */
type CopiedValue = string | null;
/**
 * Function to copy text to the clipboard.
 * @param text The text to copy to the clipboard.
 * @returns {Promise<boolean>} A promise that resolves to `true` if the text was copied successfully, or `false` otherwise.
 */
type CopyFn = (text: string) => Promise<boolean>;
/**
 * Custom hook for copying text to the clipboard.
 * @returns {[CopiedValue, CopyFn]} An tuple containing the copied text and a function to copy text to the clipboard.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-copy-to-clipboard)
 * @see [MDN Clipboard API](https://developer.mozilla.org/en-US/docs/Web/API/Clipboard_API)
 * @example
 * // Usage of useCopyToClipboard hook
 * const [copiedText, copyToClipboard] = useCopyToClipboard();
 * const textToCopy = 'Hello, world!';
 *
 * // Attempt to copy text to the clipboard
 * copyToClipboard(textToCopy)
 *   .then(success => {
 *     if (success) {
 *       console.log(`Text "${textToCopy}" copied to clipboard successfully.`);
 *     } else {
 *       console.error('Failed to copy text to clipboard.');
 *     }
 *   });
 */
declare function useCopyToClipboard(): [CopiedValue, CopyFn];

interface LegacyCountdownOptions {
    seconds: number;
    interval: number;
    isIncrement?: boolean;
}
interface LegacyCountdownControllers {
    start: () => void;
    stop: () => void;
    reset: () => void;
}
interface CountdownOptions {
    countStart: number;
    intervalMs?: number;
    isIncrement?: boolean;
    countStop?: number;
}
interface CountdownControllers {
    startCountdown: () => void;
    stopCountdown: () => void;
    resetCountdown: () => void;
}
/**
 * A hook to manage countdown - New interface with default value.
 * @overload
 * @param  {CountdownOptions} countdownOptions the countdown's options.
 * @param  {number} countdownOptions.countStart the countdown's starting number, initial value of the returned number.
 * @param  {?number} [countdownOptions.countStop] `0` by default, the countdown's stopping number. Pass `-Infinity` to decrease forever.
 * @param  {?number} [countdownOptions.intervalMs] `1000` by default, the countdown's interval, in milliseconds.
 * @param  {?boolean} [countdownOptions.isIncrement] `false` by default, true if the countdown is increment.
 * @returns {[number, CountdownControllers]} An array containing the countdown's count and its controllers.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-countdown)
 * @example
 * const [counter, { start, stop, reset }] = useCountdown({
 *   countStart: 10,
 *   intervalMs: 1000,
 *   isIncrement: false,
 * });
 */
declare function useCountdown(countdownOptions: CountdownOptions): [number, CountdownControllers];
/**
 * A hook to manage countdown - Legacy interface
 * @overload
 * @param  {LegacyCountdownOptions} countdownOptions the countdown's options.
 * @param  {number} countdownOptions.seconds the countdown's number, generally time seconds.
 * @param  {number} countdownOptions.interval the countdown's interval, milliseconds.
 * @param  {?boolean} [countdownOptions.isIncrement] `false` by default, determine the countdown is increment, otherwise is decrement.
 * @returns {[number, LegacyCountdownControllers]} An array containing the countdown's count and its controllers.
 * @deprecated new useCountdown interface is already available (see [Documentation](https://usehooks-ts.com/react-hook/use-countdown)), the old version will retire on usehooks-ts@3.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-countdown)
 * @example
 * const [counter, { start, stop, reset }] = useCountdown({
 *   seconds: 10,
 *   interval: 1000,
 *   isIncrement: false,
 * });
 */
declare function useCountdown(countdownOptions: LegacyCountdownOptions): [number, LegacyCountdownControllers];

interface UseCounterOutput {
    count: number;
    increment: () => void;
    decrement: () => void;
    reset: () => void;
    setCount: Dispatch<SetStateAction<number>>;
}
/**
 * Custom hook that manages a counter with increment, decrement, reset, and setCount functionalities.
 * @param {number} [initialValue] - The initial value for the counter.
 * @returns {object} An object containing the current count and functions to interact with the counter.
 * @property {number} count - The current count value.
 * @property {Function} increment - Increments the counter by 1.
 * @property {Function} decrement - Decrements the counter by 1.
 * @property {Function} reset - Resets the counter to its initial value.
 * @property {Function} setCount - Function to set a specific value to the counter.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-counter)
 * @example
 * // Usage of useCounter hook
 * const { count, increment, decrement, reset, setCount } = useCounter(5);
 * console.log(`Current count: ${count}`);
 * increment(); // Increases count by 1
 * decrement(); // Decreases count by 1
 * reset(); // Resets count to its initial value
 * setCount(10); // Sets count to 10
 */
declare function useCounter(initialValue?: number): UseCounterOutput;

type DarkModeOptions = {
    defaultValue?: boolean;
    localStorageKey?: string;
    initializeWithValue?: boolean;
};
interface DarkModeOutput {
    isDarkMode: boolean;
    toggle: () => void;
    enable: () => void;
    disable: () => void;
    set: (value: boolean) => void;
}
declare function useDarkMode(options?: DarkModeOptions): DarkModeOutput;
/**
 * Custom hook that returns the current state of the dark mode.
 * @deprecated this useDarkMode's signature is deprecated, it now accepts an options object instead of multiple parameters.
 * @param  {boolean} defaultValue the initial value of the dark mode, default `false`.
 * @param  {string} [localStorageKey] the key to use in the local storage, default `'usehooks-ts-dark-mode'`.
 * @returns {DarkModeOutput} An object containing the dark mode's state and its controllers.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-dark-mode)
 * @example
 * const { isDarkMode, toggle, enable, disable, set } = useDarkMode(false, 'my-key');
 */
declare function useDarkMode(defaultValue: boolean, localStorageKey?: string): DarkModeOutput;

/**
 * Custom hook for debouncing a value.
 * @template T - The type of the value to be debounced.
 * @param {T} value - The value to be debounced.
 * @param {number} [delay] - The delay in milliseconds for debouncing. Defaults to 500 milliseconds.
 * @returns {T} The debounced value.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-debounce)
 * @example
 * const debouncedSearchTerm = useDebounce(searchTerm, 300);
 * @deprecated useDebounce uses a naive setTimeout implementation and will be removed.
 * For a more robust implementation, use useDebounceCallback for functions and useDebounceValue for primitive values instead. The new implementation uses lodash.debounce under the hood.
 */
declare function useDebounce<T>(value: T, delay?: number): T;

/**
 * Configuration options for controlling the behavior of the debounced function.
 */
interface DebounceOptions {
    /**
     * Determines whether the function should be invoked on the leading edge of the timeout.
     */
    leading?: boolean;
    /**
     * Determines whether the function should be invoked on the trailing edge of the timeout.
     */
    trailing?: boolean;
    /**
     * The maximum time the specified function is allowed to be delayed before it is invoked.
     */
    maxWait?: number;
}
/**
 * Functions to manage a debounced callback.
 */
interface ControlFunctions {
    /**
     * Cancels pending function invocations.
     */
    cancel: () => void;
    /**
     * Immediately invokes pending function invocations.
     */
    flush: () => void;
    /**
     * Checks if there are any pending function invocations.
     * @returns `true` if there are pending invocations, otherwise `false`.
     */
    isPending: () => boolean;
}
/**
 * Represents the state and control functions of a debounced callback.
 * Subsequent calls to the debounced function return the result of the last invocation.
 * Note: If there are no previous invocations, the result will be undefined.
 * Ensure proper handling in your code.
 */
interface DebouncedState<T extends (...args: any) => ReturnType<T>> extends ControlFunctions {
    (...args: Parameters<T>): ReturnType<T> | undefined;
}
/**
 * Hook to create a debounced version of a callback function.
 * @template T - Type of the original callback function.
 * @param {T} func - The callback function to be debounced.
 * @param {number} delay - The delay in milliseconds before the callback is invoked (default is `500` milliseconds).
 * @param {DebounceOptions} [options] - Options to control the behavior of the debounced function.
 * @returns {DebouncedState<T>} A debounced version of the original callback along with control functions.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-debounce-callback)
 * @example
 * const debouncedCallback = useDebounceCallback(
 *   (searchTerm) => {
 *     // Perform search after user stops typing for 500 milliseconds
 *     searchApi(searchTerm);
 *   },
 *   500
 * );
 *
 * // Later in the component
 * debouncedCallback('react hooks'); // Will invoke the callback after 500 milliseconds of inactivity.
 */
declare function useDebounceCallback<T extends (...args: any) => ReturnType<T>>(func: T, delay?: number, options?: DebounceOptions): DebouncedState<T>;

/**
 * Returns a debounced version of the provided value, along with a function to update it.
 * @template T - The type of the value.
 * @param {T | (() => T)} initialValue - The value to be debounced.
 * @param {number} delay - The delay in milliseconds before the value is updated (default is 500ms).
 * @param {object} [options] - Optional configurations for the debouncing behavior.
 * @param {?boolean} [options.leading] - Determines if the debounced function should be invoked on the leading edge of the timeout (default to false).
 * @param {?boolean} [options.trailing] - Determines if the debounced function should be invoked on the trailing edge of the timeout (default to false).
 * @param {?(left: T, right: T) => boolean} [options.equalityFn] - A function to determine if the value has changed. Defaults to a function that checks if the value is strictly equal to the previous value.
 * @param {?number} [options.maxWait] - The maximum time the debounced function is allowed to be delayed before it's invoked.
 * @returns {[T, DebouncedState<(value: T) => void>]} An array containing the debounced value and the function to update it.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-debounce-value)
 * @example
 * const [debouncedValue, updateDebouncedValue] = useDebounceValue(inputValue, 500, { leading: true });
 */
declare function useDebounceValue<T>(initialValue: T | (() => T), delay: number, options?: {
    leading?: boolean;
    maxWait?: number;
    trailing?: boolean;
    equalityFn?: (left: T, right: T) => boolean;
}): [T, DebouncedState<(value: T) => void>];

type Options$1 = {
    preserveTitleOnUnmount?: boolean;
};
/**
 * A hook to set the document title.
 * @param {string} title - The title to set.
 * @param {?Options} [options] - The options.
 * @param {?boolean} [options.preserveTitleOnUnmount] - Whether to keep the title after unmounting the component (default is `true`).
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-document-title)
 * @example
 * useDocumentTitle('My new title');
 */
declare function useDocumentTitle(title: string, options?: Options$1): void;

/**
 * @deprecated - Don't use this hook, it'll be removed in the next major version.
 * A hook that runs an effect only once (at mount).
 * @param {EffectCallback} effect - The effect to run.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-effect-once)
 * @example
 * useEffectOnce(() => {
 *   console.log('Hello World');
 * });
 */
declare function useEffectOnce(effect: EffectCallback): void;

type Size$1 = {
    width: number | undefined;
    height: number | undefined;
};
type UseElementSizeOptions = {
    initializeWithValue?: boolean;
};
/** Supports both array and object destructing */
type UseElementSizeResult = [(node: Element | null) => void, Size$1] & (Size$1 & {
    ref: (node: Element | null) => void;
});
/**
 * @deprecated - Use `useResizeObserver` instead.
 * A hook for tracking the size of a DOM element.
 * @template T - The type of the DOM element. Defaults to `HTMLDivElement`.
 * @param {?UseElementSizeOptions} [options] - The options for customizing the behavior of the hook (optional).
 * @param {?boolean} [options.initializeWithValue] - If `true` (default), the hook will initialize reading the element's size. In SSR, you should set it to `false`.
 * @returns The ref-setting function and the size of the element. Either as an tuple [ref, size] or as an object { ref, width, height }.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-element-size)
 * @example
 * const [ref, { width = 0, height = 0 }] = useElementSize();
 * // or
 * const { ref, width = 0, height = 0 } = useElementSize();
 *
 * return (
 *   <div ref={ref}>
 *     My size is {size.width}x{size.height}
 *   </div>
 * );
 */
declare function useElementSize<T extends HTMLElement = HTMLDivElement>(options?: UseElementSizeOptions): UseElementSizeResult;

/**
 * A hook for creating a memoized event callback.
 * @template Args - An array of argument types for the event callback.
 * @template R - The return type of the event callback.
 * @param {(...args: Args) => R} fn - The callback function.
 * @returns {(...args: Args) => R} A memoized event callback function.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-event-callback)
 * @example
 * const handleClick = useEventCallback((event) => {
 *   // Handle the event here
 * });
 */
declare function useEventCallback<Args extends unknown[], R>(fn: (...args: Args) => R): (...args: Args) => R;

declare function useEventListener<K extends keyof MediaQueryListEventMap>(eventName: K, handler: (event: MediaQueryListEventMap[K]) => void, element: RefObject<MediaQueryList>, options?: boolean | AddEventListenerOptions): void;
declare function useEventListener<K extends keyof WindowEventMap>(eventName: K, handler: (event: WindowEventMap[K]) => void, element?: undefined, options?: boolean | AddEventListenerOptions): void;
declare function useEventListener<K extends keyof HTMLElementEventMap, T extends HTMLElement = HTMLDivElement>(eventName: K, handler: (event: HTMLElementEventMap[K]) => void, element: RefObject<T>, options?: boolean | AddEventListenerOptions): void;
declare function useEventListener<K extends keyof DocumentEventMap>(eventName: K, handler: (event: DocumentEventMap[K]) => void, element: RefObject<Document>, options?: boolean | AddEventListenerOptions): void;

/**
 * Represents the state of an HTTP request.
 * @template T - The type of data expected in the response.
 * @interface State
 * @property {T | undefined} data - The data received from the HTTP request.
 * @property {Error | undefined} error - An error object if the request encounters an error.
 */
interface State<T> {
    data?: T;
    error?: Error;
}
/**
 * @deprecated - `useFetch` is deprecated and will be removed in the next major version. See the documentation for more information.
 * Custom hook for making HTTP requests and managing the state of the request.
 * @template T - The type of data expected in the response.
 * @param {string | undefined} url - The URL to make the HTTP request to.
 * @param {RequestInit} [options] - The [options for the HTTP request]() (optional).
 * @returns {State<T>} The state object representing the result of the HTTP request.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-fetch)
 * @see [MDN Fetch API](https://developer.mozilla.org/en-US/docs/Web/API/Fetch_API)
 * @example
 * const { data, error } = useFetch<User>('https://api.example.com/user');
 */
declare function useFetch<T = unknown>(url?: string, options?: RequestInit): State<T>;

/**
 * Custom hook for tracking whether a DOM element is being hovered over.
 * @template T - The type of the DOM element. Defaults to `HTMLElement`.
 * @param {RefObject<T>} elementRef - The ref object for the DOM element to track.
 * @returns {boolean} A boolean value indicating whether the element is being hovered over.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-hover)
 * @example
 * const buttonRef = useRef<HTMLButtonElement>(null);
 * const isHovered = useHover(buttonRef);
 * // Access the isHovered variable to determine if the button is being hovered over.
 */
declare function useHover<T extends HTMLElement = HTMLElement>(elementRef: RefObject<T>): boolean;

interface ImageStyle {
    thumbnail: CSSProperties;
    fullSize: CSSProperties;
}
interface ImageOnLoadType {
    handleImageOnLoad: () => void;
    css: ImageStyle;
}
/**
 * Custom hook for handling image loading events and providing related CSS styles.
 * @deprecated This hook is deprecated and will be removed in a future release.
 * @returns {ImageOnLoadType} An object containing a function to handle image load events and related CSS styles.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-image-on-load)
 * @example
 * const { handleImageOnLoad, css } = useImageOnLoad();
 * // Use handleImageOnLoad as the onLoad handler for the full-size image.
 * // Apply the CSS styles from the `css` object to control visibility and transitions.
 */
declare function useImageOnLoad(): ImageOnLoadType;

type ObserverCallback = (isIntersecting: boolean, entry: IntersectionObserverEntry) => void;
/**
 * Represents the options for configuring the Intersection Observer.
 * @interface IntersectionObserverOptions
 * @property {number | number[]} [threshold=0] - A threshold indicating the percentage of the target's visibility needed to trigger the callback.
 * @property {Element | Document | null} [root=null] - The element that is used as the viewport for checking visibility of the target.
 * @property {string} [rootMargin='0%'] - A margin around the root.
 * @property {boolean} [freezeOnceVisible=false] - If true, freezes the intersection state once the element becomes visible.
 * @property {ObserverCallback} [onChange] - A callback function to be invoked when the intersection state changes.
 * @property {boolean} [initialIsIntersecting=false] - The initial state of the intersection.
 */
interface IntersectionObserverOptions extends IntersectionObserverInit {
    freezeOnceVisible?: boolean;
    onChange?: ObserverCallback;
    initialIsIntersecting?: boolean;
}
/** Supports both array and object destructing */
type IntersectionResult = [
    (node?: Element | null) => void,
    boolean,
    IntersectionObserverEntry | undefined
] & {
    ref: (node?: Element | null) => void;
    isIntersecting: boolean;
    entry?: IntersectionObserverEntry;
};
/**
 * Custom hook for tracking the intersection of a DOM element with its containing element or the viewport.
 * @param {IntersectionObserverOptions} options - The options for the Intersection Observer.
 * @returns {IntersectionResult} The ref callback, a boolean indicating if the element is intersecting, and the intersection observer entry.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-intersection-observer)
 * @see [MDN Intersection Observer API](https://developer.mozilla.org/en-US/docs/Web/API/Intersection_Observer_API)
 * @example
 * // Example 1
 * const [ref, isIntersecting, entry] = useIntersectionObserver({ threshold: 0.5 });
 *
 * // Example 2
 * const { ref, isIntersecting, entry } = useIntersectionObserver({ threshold: 0.5 });
 */
declare function useIntersectionObserver(options: IntersectionObserverOptions): IntersectionResult;
/**
 * @deprecated Use the new signature with an unique option object instead.
 * Custom hook for tracking the intersection of a DOM element with its containing element or the viewport.
 * @param {RefObject<Element>} elementRef - The ref object for the DOM element to observe.
 * @param {IntersectionObserverOptions} [options] - The options for the Intersection Observer (optional).
 * @returns {IntersectionObserverEntry | undefined} The intersection observer entry representing the state of the intersection.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-intersection-observer)
 * @see [MDN Intersection Observer API](https://developer.mozilla.org/en-US/docs/Web/API/Intersection_Observer_API)
 * @example
 * const targetRef = useRef<HTMLDivElement>(null);
 * const options = { threshold: 0.5 };
 * const entry = useIntersectionObserver(targetRef, options);
 * // Access the intersection details from the entry variable.
 */
declare function useIntersectionObserver(elementRef: RefObject<Element>, legacyOptions: IntersectionObserverOptions): IntersectionObserverEntry | undefined;

/**
 * Custom hook for creating an interval that invokes a callback function at a specified delay.
 * @param {() => void} callback - The function to be invoked at each interval.
 * @param {number | null} delay - The time, in milliseconds, between each invocation of the callback. Use `null` to clear the interval.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-interval)
 * @see [MDN setInterval](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/setInterval)
 * @example
 * const handleInterval = () => {
 *   // Code to be executed at each interval
 * };
 * useInterval(handleInterval, 1000);
 */
declare function useInterval(callback: () => void, delay: number | null): void;

/**
 * Custom hook for determining if the code is running on the client side (in the browser).
 * @returns {boolean} A boolean value indicating whether the code is running on the client side.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-is-client)
 * @example
 * const isClient = useIsClient();
 * // Use isClient to conditionally render or execute code specific to the client side.
 */
declare function useIsClient(): boolean;

/**
 * @deprecated - Don't use this hook, it's an anti-pattern.
 * Custom hook for determining if the component is rendering for the first time.
 * @returns {boolean} A boolean value indicating whether the component is rendering for the first time.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-is-first-render)
 * @example
 * const isFirstRender = useIsFirstRender();
 * // Use isFirstRender to conditionally execute code only on the initial render.
 */
declare function useIsFirstRender(): boolean;

/**
 * Custom hook for determining if the component is currently mounted.
 * @returns {() => boolean} A function that returns a boolean value indicating whether the component is mounted.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-is-mounted)
 * @example
 * const isComponentMounted = useIsMounted();
 * // Use isComponentMounted() to check if the component is currently mounted before performing certain actions.
 */
declare function useIsMounted(): () => boolean;

/**
 * Custom hook for using either `useLayoutEffect` or `useEffect` based on the environment (client-side or server-side).
 * @param {Function} effect - The effect function to be executed.
 * @param {Array<any>} [dependencies] - An array of dependencies for the effect (optional).
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-isomorphic-layout-effect)
 * @example
 * useIsomorphicLayoutEffect(() => {
 *   // Code to be executed during the layout phase on the client side
 * }, [dependency1, dependency2]);
 */
declare const useIsomorphicLayoutEffect: typeof useLayoutEffect;

declare global {
    interface WindowEventMap {
        'local-storage': CustomEvent;
    }
}
interface UseLocalStorageOptions<T> {
    serializer?: (value: T) => string;
    deserializer?: (value: string) => T;
    initializeWithValue?: boolean;
}
/**
 * Custom hook for using local storage to persist state across page reloads.
 * @template T - The type of the state to be stored in local storage.
 * @param {string} key - The key under which the value will be stored in local storage.
 * @param {T | (() => T)} initialValue - The initial value of the state or a function that returns the initial value.
 * @param {UseLocalStorageOptions<T>} [options] - Options for customizing the behavior of serialization and deserialization (optional).
 * @param {?boolean} [options.initializeWithValue] - If `true` (default), the hook will initialize reading the local storage. In SSR, you should set it to `false`, returning the initial value initially.
 * @param {?((value: T) => string)} [options.serializer] - A function to serialize the value before storing it.
 * @param {?((value: string) => T)} [options.deserializer] - A function to deserialize the stored value.
 * @returns {[T, Dispatch<SetStateAction<T>>]} A tuple containing the stored value and a function to set the value.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-local-storage)
 * @see [MDN Local Storage](https://developer.mozilla.org/en-US/docs/Web/API/Window/localStorage)
 * @example
 * const [count, setCount] = useLocalStorage('count', 0);
 * // Access the `count` value and the `setCount` function to update it.
 */
declare function useLocalStorage<T>(key: string, initialValue: T | (() => T), options?: UseLocalStorageOptions<T>): [T, Dispatch<SetStateAction<T>>];

/**
 * Custom hook for locking and unlocking the body scroll to prevent scrolling.
 * @param {?boolean} [initialLocked] - The initial state of body scroll lock (default to `false`).
 * @param {?string} [rootId] - The ID of the root element to calculate scrollbar width (default to `___gatsby` to not introduce breaking change).
 * @returns {[boolean, (locked: boolean) => void]} A tuple containing the current state of body scroll lock and a function to set the state.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-locked-body)
 * @example
 * const [isLocked, setLocked] = useLockedBody(true);
 * // Use isLocked to conditionally apply styles or prevent scrolling.
 * // Use setLocked to dynamically control the body scroll lock state.
 */
declare function useLockedBody(initialLocked?: boolean, rootId?: string): [boolean, (locked: boolean) => void];

/**
 * Represents the type for either a Map or an array of key-value pairs.
 * @template K - The type of keys in the map.
 * @template V - The type of values in the map.
 */
type MapOrEntries<K, V> = Map<K, V> | [K, V][];
/**
 * Represents the actions available to interact with the map state.
 * @template K - The type of keys in the map.
 * @template V - The type of values in the map.
 */
interface Actions<K, V> {
    set: (key: K, value: V) => void;
    setAll: (entries: MapOrEntries<K, V>) => void;
    remove: (key: K) => void;
    reset: Map<K, V>['clear'];
}
/**
 * Represents the output of the `useMap` hook.
 * We hide some setters from the returned map to disable autocompletion
 * @template K - The type of keys in the map.
 * @template V - The type of values in the map.
 */
type Return<K, V> = [Omit<Map<K, V>, 'set' | 'clear' | 'delete'>, Actions<K, V>];
/**
 * Custom hook for managing a key-value map state with setter actions.
 * @template K - The type of keys in the map.
 * @template V - The type of values in the map.
 * @param {MapOrEntries<K, V>} [initialState] - The initial state of the map as a Map or an array of key-value pairs (optional).
 * @returns {Return<K, V>} A tuple containing the map state and actions to interact with the map.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-map)
 * @example
 * const [map, mapActions] = useMap();
 * // Access the `map` state and use `mapActions` to set, remove, or reset entries.
 */
declare function useMap<K, V>(initialState?: MapOrEntries<K, V>): Return<K, V>;

type UseMediaQueryOptions = {
    defaultValue?: boolean;
    initializeWithValue?: boolean;
};
declare function useMediaQuery(query: string, options?: UseMediaQueryOptions): boolean;
/**
 * Custom hook for tracking the state of a media query.
 * @deprecated - this useMediaQuery's signature is deprecated, it now accepts an query parameter and an options object.
 * @param {string} query - The media query to track.
 * @param {?boolean} [defaultValue] - The default value to return if the hook is being run on the server (default is `false`).
 * @returns {boolean} The current state of the media query (true if the query matches, false otherwise).
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-media-query)
 * @see [MDN Match Media](https://developer.mozilla.org/en-US/docs/Web/API/Window/matchMedia)
 * @example
 * const isSmallScreen = useMediaQuery('(max-width: 600px)');
 * // Use `isSmallScreen` to conditionally apply styles or logic based on the screen size.
 */
declare function useMediaQuery(query: string, defaultValue: boolean): boolean;

type EventType = 'mousedown' | 'mouseup' | 'touchstart' | 'touchend';
/**
 * Custom hook for handling clicks outside a specified element.
 * @template T - The type of the element's reference.
 * @param {RefObject<T> | RefObject<T>[]} ref - The React ref object(s) representing the element(s) to watch for outside clicks.
 * @param {(event: MouseEvent | TouchEvent) => void} handler - The callback function to be executed when a click outside the element occurs.
 * @param {EventType} [eventType] - The mouse event type to listen for (optional, default is 'mousedown').
 * @returns {void}
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-on-click-outside)
 * @example
 * const containerRef = useRef(null);
 * useOnClickOutside([containerRef], () => {
 *   // Handle clicks outside the container.
 * });
 */
declare function useOnClickOutside<T extends HTMLElement = HTMLElement>(ref: RefObject<T> | RefObject<T>[], handler: (event: MouseEvent | TouchEvent) => void, eventType?: EventType): void;

/**
 * Represents the type for the options available when reading from local storage.
 * @template T - The type of the stored value.
 */
interface Options<T, InitializeWithValue extends boolean | undefined> {
    deserializer?: (value: string) => T;
    initializeWithValue: InitializeWithValue;
}
/** Value could be null if it does not exist in LocalStorage */
type Value<T> = T | null;
declare function useReadLocalStorage<T>(key: string, options: Options<T, false>): Value<T> | undefined;
declare function useReadLocalStorage<T>(key: string, options?: Partial<Options<T, true>>): Value<T>;

type Size = {
    width: number | undefined;
    height: number | undefined;
};
type ResizeHandler = (size: Size) => void;
type BoxOptions = 'border-box' | 'content-box' | 'device-pixel-content-box';
type UseResizeObserverOptions<T extends HTMLElement = HTMLElement> = {
    ref: RefObject<T>;
    onResize?: ResizeHandler;
    box?: BoxOptions;
};
type UseResizeObserverResult = Size;
/**
 * Custom hook for observing the size of an element using the ResizeObserver API.
 *
 * @template T - The type of the element to observe.
 * @param {UseResizeObserverOptions<T>} options - The options for the ResizeObserver. (default is `{}`).
 * @param {RefObject<T>} options.ref - The ref of the element to observe.
 * @param {ResizeHandler} [options.onResize] - When using `onResize`, the hook doesn't re-render on element size changes; it delegates handling to the provided callback . (default is `undefined`).
 * @param {string} [options.box] - The box model to use for the ResizeObserver. (default is `'content-box'`).
 * @returns {UseResizeObserverResult} - The size of the observed element.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-resize-observer)
 * @see [MDN ResizeObserver API](https://developer.mozilla.org/en-US/docs/Web/API/ResizeObserver)
 * @example
 * const myRef = useRef(null);
 * const { width = 0, height = 0 } = useResizeObserver({
 *   ref: myRef,
 *   box: 'content-box',
 * });
 *
 * <div ref={myRef}>Hello, world!</div>
 */
declare function useResizeObserver<T extends HTMLElement = HTMLElement>(options: UseResizeObserverOptions<T>): UseResizeObserverResult;

type UseScreenOptions<InitializeWithValue extends boolean | undefined> = {
    initializeWithValue: InitializeWithValue;
    debounceDelay?: number;
};
declare function useScreen(options: UseScreenOptions<false>): Screen | undefined;
declare function useScreen(options?: Partial<UseScreenOptions<true>>): Screen;

type UseScriptStatus = 'idle' | 'loading' | 'ready' | 'error';
interface UseScriptOptions {
    shouldPreventLoad?: boolean;
    removeOnUnmount?: boolean;
}
/**
 * Custom hook for dynamically loading scripts and tracking their loading status.
 * @param {string | null} src - The source URL of the script to load. Set to `null` or omit to prevent loading (optional).
 * @param {UseScriptOptions} [options] - Additional options for controlling script loading (optional).
 * @param {boolean} [options.shouldPreventLoad] - If `true`, prevents the script from being loaded (optional).
 * @param {boolean} [options.removeOnUnmount] - If `true`, removes the script from the DOM when the component unmounts (optional).
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-script)
 * @returns {UseScriptStatus} The status of the script loading, which can be one of 'idle', 'loading', 'ready', or 'error'.
 * @example
 * const scriptStatus = useScript('https://example.com/script.js', { removeOnUnmount: true });
 * // Access the status of the script loading (e.g., 'loading', 'ready', 'error').
 */
declare function useScript(src: string | null, options?: UseScriptOptions): UseScriptStatus;

declare global {
    interface WindowEventMap {
        'session-storage': CustomEvent;
    }
}
/**
 * Represents the options for customizing the behavior of serialization and deserialization.
 * @template T - The type of the state to be stored in session storage.
 * @interface Options
 * @property {(value: T) => string} [serializer] - A function to serialize the value before storing it.
 * @property {(value: string) => T} [deserializer] - A function to deserialize the stored value.
 */
interface UseSessionStorageOptions<T> {
    serializer?: (value: T) => string;
    deserializer?: (value: string) => T;
    initializeWithValue?: boolean;
}
/**
 * Custom hook for using session storage to persist state across page reloads.
 * @template T - The type of the state to be stored in session storage.
 * @param {string} key - The key under which the value will be stored in session storage.
 * @param {T | (() => T)} initialValue - The initial value of the state or a function that returns the initial value.
 * @param {?UseSessionStorageOptions<T>} [options] - Options for customizing the behavior of serialization and deserialization (optional).
 * @param {?boolean} [options.initializeWithValue] - If `true` (default), the hook will initialize reading the session storage. In SSR, you should set it to `false`, returning the initial value initially.
 * @param {(value: T) => string} [options.serializer] - A function to serialize the value before storing it.
 * @returns {[T, Dispatch<SetStateAction<T>>]} A tuple containing the stored value and a function to set the value.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-session-storage)
 * @see [MDN Session Storage](https://developer.mozilla.org/en-US/docs/Web/API/Window/sessionStorage)
 * @example
 * const [count, setCount] = useSessionStorage('count', 0);
 * // Access the `count` value and the `setCount` function to update it.
 */
declare function useSessionStorage<T>(key: string, initialValue: T | (() => T), options?: UseSessionStorageOptions<T>): [T, Dispatch<SetStateAction<T>>];

/**
 * @deprecated useSsr isn't a valid react hook and will be dropped in the next major release. See #258
 * @returns {{ isBrowser: boolean, isServer: boolean }} An object containing the boolean values `isBrowser` and `isServer`.
 * @example
 * const { isBrowser, isServer } = useSsr();
 */
declare function useSsr(): {
    isBrowser: boolean;
    isServer: boolean;
};

interface Helpers {
    goToNextStep: () => void;
    goToPrevStep: () => void;
    reset: () => void;
    canGoToNextStep: boolean;
    canGoToPrevStep: boolean;
    setStep: Dispatch<SetStateAction<number>>;
}
/**
 * Custom hook for managing and navigating between steps in a multi-step process.
 * @param {number} maxStep - The maximum step in the process.
 * @returns {[number, Helpers]} An tuple containing the current step and helper functions for navigating steps.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-step)
 * @example
 * const [currentStep, { goToNextStep, goToPrevStep, reset, canGoToNextStep, canGoToPrevStep, setStep }] = useStep(3);
 * // Access and use the current step and provided helper functions.
 */
declare function useStep(maxStep: number): [number, Helpers];

type TernaryDarkMode = 'system' | 'dark' | 'light';
type TernaryDarkModeOptions = {
    defaultValue?: TernaryDarkMode;
    localStorageKey?: string;
    initializeWithValue?: boolean;
};
type TernaryDarkModeResult = {
    isDarkMode: boolean;
    ternaryDarkMode: TernaryDarkMode;
    setTernaryDarkMode: Dispatch<SetStateAction<TernaryDarkMode>>;
    toggleTernaryDarkMode: () => void;
};
declare function useTernaryDarkMode(options?: TernaryDarkModeOptions): TernaryDarkModeResult;
/**
 * Custom hook for managing ternary (system, dark, light) dark mode with local storage support.
 * @deprecated this useTernaryDarkMode's signature is deprecated, it now accepts an options object instead of multiple parameters.
 * @param {string} localStorageKey - The key for storing dark mode preference in local storage (default is `'usehooks-ts-ternary-dark-mode'`).
 * @returns {TernaryDarkModeResult} An object containing the dark mode state and helper functions.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-ternary-dark-mode)
 * @example
 * const { isDarkMode, ternaryDarkMode, setTernaryDarkMode, toggleTernaryDarkMode } = useTernaryDarkMode('my-key');
 * // Access and use the dark mode state and provided helper functions.
 */
declare function useTernaryDarkMode(localStorageKey: string): TernaryDarkModeResult;

/**
 * Custom hook for handling timeouts in React components.
 * @param {() => void} callback - The function to be executed when the timeout elapses.
 * @param {number | null} delay - The duration (in milliseconds) for the timeout. Set to `null` to clear the timeout.
 * @returns {void} This hook does not return anything.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-timeout)
 * @see [MDN setTimeout](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/setTimeout)
 * @example
 * // Usage of useTimeout hook
 * useTimeout(() => {
 *   // Code to be executed after the specified delay
 * }, 1000); // Set a timeout of 1000 milliseconds (1 second)
 */
declare function useTimeout(callback: () => void, delay: number | null): void;

/**
 * Custom hook for managing a boolean toggle state in React components.
 * @param {boolean} [defaultValue] - The initial value for the toggle state.
 * @returns {[boolean, () => void, Dispatch<SetStateAction<boolean>>]} A tuple containing the current state,
 * a function to toggle the state, and a function to set the state explicitly.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-toggle)
 * @example
 * // Usage of useToggle hook
 * const [isToggled, toggle, setToggle] = useToggle(); // Initial value is false
 * // OR
 * const [isToggled, toggle, setToggle] = useToggle(true); // Initial value is true
 * // Use isToggled in your component, toggle to switch the state, setToggle to set the state explicitly.
 */
declare function useToggle(defaultValue?: boolean): [boolean, () => void, Dispatch<SetStateAction<boolean>>];

/**
 * Hook that runs a cleanup function when the component is unmounted.
 * @param {() => void} func - The cleanup function to be executed on unmount.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-unmount)
 * @example
 * // Usage in a functional component
 * useUnmount(() => {
 *   // Cleanup logic here
 * });
 */
declare function useUnmount(func: () => void): void;

/**
 * @deprecated - Don't use this hook, it's an anti-pattern.
 * Custom hook that runs an effect only on updates (not on the initial render).
 * @param {EffectCallback} effect - The function to run as the effect.
 * @param {DependencyList} [deps] - An optional array of dependencies for the effect.
 * @see [Documentation](https://usehooks-ts.com/react-hook/use-update-effect)
 * @example
 * // Usage of useUpdateEffect hook
 * useUpdateEffect(() => {
 *   // Effect code to run on updates
 *   console.log('Component updated!');
 * }, [dependency1, dependency2]);
 */
declare function useUpdateEffect(effect: EffectCallback, deps?: DependencyList): void;

interface WindowSize<T extends number | undefined = number | undefined> {
    width: T;
    height: T;
}
type UseWindowSizeOptions<InitializeWithValue extends boolean | undefined> = {
    initializeWithValue: InitializeWithValue;
    debounceDelay?: number;
};
declare function useWindowSize(options: UseWindowSizeOptions<false>): WindowSize;
declare function useWindowSize(options?: Partial<UseWindowSizeOptions<true>>): WindowSize<number>;

export { type Actions, type DebounceOptions, type DebouncedState, type MapOrEntries, type TernaryDarkMode, type UseScriptOptions, type UseScriptStatus, useBoolean, useClickAnyWhere, useCopyToClipboard, useCountdown, useCounter, useDarkMode, useDebounce, useDebounceCallback, useDebounceValue, useDocumentTitle, useEffectOnce, useElementSize, useEventCallback, useEventListener, useFetch, useHover, useImageOnLoad, useIntersectionObserver, useInterval, useIsClient, useIsFirstRender, useIsMounted, useIsomorphicLayoutEffect, useLocalStorage, useLockedBody, useMap, useMediaQuery, useOnClickOutside, useReadLocalStorage, useResizeObserver, useScreen, useScript, useSessionStorage, useSsr, useStep, useTernaryDarkMode, useTimeout, useToggle, useUnmount, useUpdateEffect, useWindowSize };
