'use strict';

var express = require('express');
var fs = require('fs');
var path = require('path');
var pc = require('picocolors');

/******************************************************************************
Copyright (c) Microsoft Corporation.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
***************************************************************************** */

function __awaiter(thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
}

const _State = {
    viteConfig: undefined,
};
function clearState() {
    _State.viteConfig = undefined;
}
const Config = {
    mode: (process.env.NODE_ENV === "production"
        ? "production"
        : "development"),
    inlineViteConfig: undefined,
    ignorePaths: undefined,
    transformer: undefined,
};
function info(msg) {
    const timestamp = new Date().toLocaleString("en-US").split(",")[1].trim();
    console.log(`${pc.dim(timestamp)} ${pc.bold(pc.cyan("[vite-express]"))} ${pc.green(msg)}`);
}
function isStaticFilePath(path) {
    return path.match(/(\.\w+$)|@vite|@id|@react-refresh/);
}
function getTransformedHTML(html, req) {
    return Config.transformer ? Config.transformer(html, req) : html;
}
function getDefaultViteConfig() {
    return {
        root: process.cwd(),
        base: "/",
        build: { outDir: "dist" },
    };
}
function getViteConfigPath() {
    if (fs.existsSync("vite.config.js"))
        return "vite.config.js";
    else if (fs.existsSync("vite.config.ts"))
        return "vite.config.ts";
    throw new Error("Unable to locate Vite config");
}
function resolveConfig() {
    var _a, _b, _c;
    return __awaiter(this, void 0, void 0, function* () {
        if (Config.inlineViteConfig) {
            info(`${pc.yellow("Inline config")} detected, ignoring ${pc.yellow("Vite config file")}`);
            return Object.assign(Object.assign({}, getDefaultViteConfig()), Config.inlineViteConfig);
        }
        try {
            const { resolveConfig } = yield import('vite');
            try {
                const config = yield resolveConfig({}, "build");
                info(`Using ${pc.yellow("Vite")} to resolve the ${pc.yellow("config file")}`);
                return config;
            }
            catch (e) {
                console.error(e);
                info(pc.red(`Unable to use ${pc.yellow("Vite")}, running in ${pc.yellow("viteless")} mode`));
            }
        }
        catch (e) {
        }
        try {
            const config = fs.readFileSync(getViteConfigPath(), "utf8");
            const root = (_a = config.match(/"?root"?\s*:\s*"([^"]+)"/)) === null || _a === void 0 ? void 0 : _a[1];
            const base = (_b = config.match(/"?base"?\s*:\s*"([^"]+)"/)) === null || _b === void 0 ? void 0 : _b[1];
            const outDir = (_c = config.match(/"?outDir"?\s*:\s*"([^"]+)"/)) === null || _c === void 0 ? void 0 : _c[1];
            const defaultConfig = getDefaultViteConfig();
            return {
                root: root !== null && root !== void 0 ? root : defaultConfig.root,
                base: base !== null && base !== void 0 ? base : defaultConfig.base,
                build: { outDir: outDir !== null && outDir !== void 0 ? outDir : defaultConfig.build.outDir },
            };
        }
        catch (e) {
            info(pc.red(`Unable to locate ${pc.yellow("vite.config.*s file")}, using default options`));
            return getDefaultViteConfig();
        }
    });
}
function getViteConfig() {
    return __awaiter(this, void 0, void 0, function* () {
        if (!_State.viteConfig) {
            _State.viteConfig = yield resolveConfig();
        }
        return _State.viteConfig;
    });
}
function getDistPath() {
    return __awaiter(this, void 0, void 0, function* () {
        const config = yield getViteConfig();
        return path.resolve(config.root, config.build.outDir);
    });
}
function serveStatic() {
    return __awaiter(this, void 0, void 0, function* () {
        const distPath = yield getDistPath();
        if (!fs.existsSync(distPath)) {
            info(`${pc.red(`Static files at ${pc.gray(distPath)} not found!`)}`);
            info(`${pc.yellow(`Did you forget to run ${pc.bold(pc.green("vite build"))} command?`)}`);
        }
        else {
            info(`${pc.green(`Serving static files from ${pc.gray(distPath)}`)}`);
        }
        return express.static(distPath, { index: false });
    });
}
const stubMiddleware = (req, res, next) => next();
function injectStaticMiddleware(app, middleware) {
    return __awaiter(this, void 0, void 0, function* () {
        const config = yield getViteConfig();
        app.use(config.base, middleware);
        const stubMiddlewareLayer = app._router.stack.find((layer) => layer.handle === stubMiddleware);
        if (stubMiddlewareLayer !== undefined) {
            const serveStaticLayer = app._router.stack.pop();
            app._router.stack = app._router.stack.map((layer) => {
                return layer === stubMiddlewareLayer ? serveStaticLayer : layer;
            });
        }
    });
}
function isIgnoredPath(path, req) {
    if (Config.ignorePaths === undefined)
        return false;
    return Config.ignorePaths instanceof RegExp
        ? path.match(Config.ignorePaths)
        : Config.ignorePaths(path, req);
}
function findClosestIndexToRoot(reqPath, root) {
    const basePath = reqPath.slice(0, reqPath.lastIndexOf("/"));
    const dirs = basePath.split("/");
    while (dirs.length > 0) {
        const pathToTest = path.join(root, ...dirs, "index.html");
        if (fs.existsSync(pathToTest)) {
            return pathToTest;
        }
        dirs.pop();
    }
    return undefined;
}
function injectViteIndexMiddleware(app, server) {
    return __awaiter(this, void 0, void 0, function* () {
        const config = yield getViteConfig();
        app.use(config.base, (req, res, next) => __awaiter(this, void 0, void 0, function* () {
            if (req.method !== "GET")
                return next();
            if (isIgnoredPath(req.path, req))
                return next();
            if (isStaticFilePath(req.path))
                next();
            else {
                const indexPath = findClosestIndexToRoot(req.path, config.root);
                if (indexPath === undefined)
                    return next();
                const template = fs.readFileSync(indexPath, "utf8");
                const html = yield server.transformIndexHtml(req.originalUrl, template);
                res.send(getTransformedHTML(html, req));
            }
        }));
    });
}
function injectIndexMiddleware(app) {
    return __awaiter(this, void 0, void 0, function* () {
        const distPath = yield getDistPath();
        const config = yield getViteConfig();
        app.use(config.base, (req, res, next) => {
            if (isIgnoredPath(req.path, req))
                return next();
            const indexPath = findClosestIndexToRoot(req.path, distPath);
            if (indexPath === undefined)
                return next();
            const html = fs.readFileSync(indexPath, "utf8");
            res.send(getTransformedHTML(html, req));
        });
    });
}
function startServer(server) {
    return __awaiter(this, void 0, void 0, function* () {
        const { createServer, mergeConfig } = yield import('vite');
        const config = yield getViteConfig();
        const isUsingViteResolvedConfig = Object.entries(config).length > 3;
        const vite = yield createServer(mergeConfig(isUsingViteResolvedConfig ? {} : config, {
            clearScreen: false,
            appType: "custom",
            server: { middlewareMode: true },
        }));
        server.on("close", () => vite.close());
        return vite;
    });
}
function config(config) {
    if (config.mode !== undefined)
        Config.mode = config.mode;
    Config.ignorePaths = config.ignorePaths;
    Config.inlineViteConfig = config.inlineViteConfig;
    Config.transformer = config.transformer;
}
function bind(app, server, callback) {
    return __awaiter(this, void 0, void 0, function* () {
        info(`Running in ${pc.yellow(Config.mode)} mode`);
        clearState();
        if (Config.mode === "development") {
            const vite = yield startServer(server);
            yield injectStaticMiddleware(app, vite.middlewares);
            yield injectViteIndexMiddleware(app, vite);
        }
        else {
            yield injectStaticMiddleware(app, yield serveStatic());
            yield injectIndexMiddleware(app);
        }
        callback === null || callback === void 0 ? void 0 : callback();
    });
}
function listen(app, port, callback) {
    const server = app.listen(port, () => bind(app, server, callback));
    return server;
}
function build() {
    return __awaiter(this, void 0, void 0, function* () {
        const { build } = yield import('vite');
        info("Build starting...");
        yield build();
        info("Build completed!");
    });
}
var main = { config, bind, listen, build, static: () => stubMiddleware };

module.exports = main;
