"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.appleDeviceSpecsForLaunchImages = exports.generateImages = void 0;
const meta_1 = __importDefault(require("./helpers/meta"));
const puppets_1 = __importDefault(require("./helpers/puppets"));
const flags_1 = __importDefault(require("./helpers/flags"));
const logger_1 = __importDefault(require("./helpers/logger"));
const constants_1 = __importDefault(require("./config/constants"));
/**
 Generates PWA assets based on a source input and saves generated images in the output folder provided

 @param source - A local image file, a local HTML file, a remote image or remote HTML file path
 @param outputFolderPath - The path of the folder to save the images in
 @param options - Option flags of the library in an object, keeps default values
 @param loggerFn - An optional logger function to log the output
 @returns A promise of result object that resolves when all images are generated and file updates are finalized

 @example
 ```javascript
 import pwaAssetGenerator = require('pwa-asset-generator');

 (async () => {
        const { savedImages, htmlMeta, manifestJsonContent } = await pwaAssetGenerator.generateImages(
          'https://raw.githubusercontent.com/onderceylan/pwa-asset-generator/HEAD/static/logo.png',
          './temp',
           {
          scrape: false,
          background: "linear-gradient(to right, #fa709a 0%, #fee140 100%)",
          splashOnly: true,
          portraitOnly: true,
          log: false
           });
    })();
 ```
 */
function generateImages(source, outputFolderPath, options, loggerFn) {
    return __awaiter(this, void 0, void 0, function* () {
        let modOptions;
        const logger = loggerFn || (0, logger_1.default)(generateImages.name, options);
        if (!source) {
            throw Error('Please specify a URL or file path as a source');
        }
        if (options) {
            modOptions = Object.assign(Object.assign(Object.assign(Object.assign(Object.assign({}, flags_1.default.getDefaultOptions()), options), flags_1.default.normalizeOnlyFlagPairs('splashOnly', 'iconOnly', options, logger)), flags_1.default.normalizeOnlyFlagPairs('landscapeOnly', 'portraitOnly', options, logger)), flags_1.default.normalizeSandboxOption(options.noSandbox, logger));
        }
        else {
            modOptions = Object.assign({}, flags_1.default.getDefaultOptions());
        }
        const output = flags_1.default.normalizeOutput(outputFolderPath);
        const savedImages = yield puppets_1.default.generateImages(source, output, modOptions);
        const manifestJsonContent = meta_1.default.generateIconsContentForManifest(savedImages, modOptions);
        const htmlMeta = meta_1.default.generateHtmlForIndexPage(savedImages, modOptions);
        if (!modOptions.splashOnly) {
            if (modOptions.manifest) {
                yield meta_1.default.addIconsToManifest(manifestJsonContent, modOptions.manifest);
                logger.success(`Icons are saved to Web App Manifest file ${modOptions.manifest}`);
            }
            else if (!modOptions.splashOnly) {
                logger.warn('Web App Manifest file is not specified, printing out the content to console instead');
                logger.success('Below is the icons content for your manifest.json file. You can copy/paste it manually');
                logger.raw(`\n${JSON.stringify(manifestJsonContent, null, 2)}\n\n`);
            }
        }
        if (modOptions.index) {
            yield meta_1.default.addMetaTagsToIndexPage(htmlMeta, modOptions.index, modOptions.xhtml);
            logger.success(`iOS meta tags are saved to index html file ${modOptions.index}`);
        }
        else {
            logger.warn('Index html file is not specified, printing out the content to console instead');
            logger.success('Below is the iOS meta tags content for your index.html file. You can copy/paste it manually');
            logger.raw(`\n${meta_1.default.formatMetaTags(htmlMeta)}\n`);
        }
        return {
            savedImages,
            htmlMeta,
            manifestJsonContent,
        };
    });
}
exports.generateImages = generateImages;
/**
 Static data for Apple Device specs that are used for generating launch images
 */
const appleDeviceSpecsForLaunchImages = constants_1.default.APPLE_HIG_SPLASH_SCREEN_FALLBACK_DATA;
exports.appleDeviceSpecsForLaunchImages = appleDeviceSpecsForLaunchImages;
//# sourceMappingURL=main.js.map