"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AndroidProject = void 0;
const path_1 = require("path");
const cross_fetch_1 = __importDefault(require("cross-fetch"));
const utils_fs_1 = require("@ionic/utils-fs");
const gradle_file_1 = require("./gradle-file");
const xml_1 = require("../xml");
const properties_1 = require("../properties");
const platform_project_1 = require("../platform-project");
const read_src_1 = require("../read-src");
class AndroidProject extends platform_project_1.PlatformProject {
    constructor(project) {
        super(project);
        this.buildGradle = null;
        this.appBuildGradle = null;
        const manifestPath = this.getAndroidManifestPath();
        this.manifest = new xml_1.XmlFile(manifestPath, project.vfs);
    }
    async load() {
        try {
            await this.manifest.load();
            this.buildGradle = await this.loadGradle('build.gradle');
            this.appBuildGradle = await this.loadGradle('app/build.gradle');
        }
        catch (e) {
            this.setError(e);
        }
    }
    getBuildGradle() {
        return this.buildGradle;
    }
    getAppBuildGradle() {
        return this.appBuildGradle;
    }
    getAndroidManifest() {
        return this.manifest;
    }
    /**
     * Get a project file container for the given path in the project root.
     * This will return an existing file container or create a new one.
     */
    getProjectFile(path, create) {
        var _a;
        const root = (_a = this.project.config.android) === null || _a === void 0 ? void 0 : _a.path;
        if (!root) {
            return null;
        }
        const filename = (0, path_1.join)(root, path);
        const existing = this.project.vfs.get(filename);
        if (existing) {
            return existing.getData();
        }
        return create(filename);
    }
    getResourceXmlFile(resourcePath) {
        return this.getXmlFile((0, path_1.join)(this.getResourcesPath(), resourcePath));
    }
    getXmlFile(path) {
        return this.getProjectFile(path, (filename) => new xml_1.XmlFile(filename, this.project.vfs));
    }
    getPropertiesFile(path) {
        return this.getProjectFile(path, (filename) => new properties_1.PropertiesFile(filename, this.project.vfs));
    }
    async getGradleFile(path) {
        if (path === 'build.gradle') {
            return this.buildGradle;
        }
        else if (path === 'app/build.gradle') {
            return this.appBuildGradle;
        }
        return this.loadGradle(path);
    }
    /**
     * Update the Android package name. This renames the package in `AndroidManifest.xml`,
     * the `applicationId` in `app/build.gradle`, and renames the java
     * package for the `MainActivity.java` file.
     *
     * This action will mutate the project on disk!
     */
    async setPackageName(packageName) {
        var _a, _b, _c, _d;
        const sourceDir = (0, path_1.join)(this.getAppRoot(), 'src', 'main', 'java');
        const oldPackageName = await ((_a = this.manifest
            .getDocumentElement()) === null || _a === void 0 ? void 0 : _a.getAttribute('package'));
        const oldPackageParts = (_b = oldPackageName === null || oldPackageName === void 0 ? void 0 : oldPackageName.split('.')) !== null && _b !== void 0 ? _b : [];
        if (packageName === oldPackageName) {
            return;
        }
        const existingPackage = (0, path_1.join)(sourceDir, ...oldPackageParts);
        if (!(await (0, utils_fs_1.pathExists)(existingPackage))) {
            throw new Error('Current Java package name and directory structure do not match the <manifest> package attribute. Ensure these match before modifying the project package name');
        }
        (_c = this.manifest.getDocumentElement()) === null || _c === void 0 ? void 0 : _c.setAttribute('package', packageName);
        await ((_d = this.appBuildGradle) === null || _d === void 0 ? void 0 : _d.setApplicationId(packageName));
        this.manifest.setAttrs('manifest/application/activity', {
            'android:name': `${packageName}.MainActivity`,
        });
        if (!this.getAppRoot()) {
            return;
        }
        const newPackageParts = packageName.split('.');
        const destDir = (0, path_1.join)(sourceDir, ...newPackageParts);
        // TODO: Don't hard code this
        const mainActivityName = this.getMainActivityFilename();
        let activityFile = (0, path_1.join)(sourceDir, ...oldPackageParts, mainActivityName);
        // Make the new directory tree and any missing parents
        await (0, utils_fs_1.mkdirp)(destDir);
        // Move the old activity file over
        await (0, utils_fs_1.move)(activityFile, (0, path_1.join)(destDir, 'MainActivity.java'));
        // Try to delete the empty directories we left behind, starting
        // from the deepest
        let sourceDirLeaf = (0, path_1.join)(sourceDir, ...oldPackageParts);
        for (const _ of oldPackageParts) {
            try {
                await (0, utils_fs_1.rmdir)(sourceDirLeaf);
            }
            catch (ex) {
                // This will fail if directory is not empty, that's fine, we won't delete those
            }
            sourceDirLeaf = (0, path_1.join)(sourceDirLeaf, '..');
        }
        // Rename the package in the main source file
        activityFile = (0, path_1.join)(sourceDir, ...newPackageParts, this.getMainActivityFilename());
        if (await (0, utils_fs_1.pathExists)(activityFile)) {
            const activitySource = await (0, utils_fs_1.readFile)(activityFile, {
                encoding: 'utf-8',
            });
            const newActivitySource = activitySource === null || activitySource === void 0 ? void 0 : activitySource.replace(/(package\s+)[^;]+/, `$1${packageName}`);
            await (0, utils_fs_1.writeFile)(activityFile, newActivitySource);
        }
    }
    getMainActivityFilename() {
        const activity = this.manifest.find('manifest/application/activity');
        if (!activity) {
            return 'MainActivity.java';
        }
        const activityName = activity[0].getAttribute('android:name');
        const parts = activityName === null || activityName === void 0 ? void 0 : activityName.split('.');
        if (!parts) {
            return '';
        }
        return `${parts[parts.length - 1]}.java`;
    }
    getPackageName() {
        var _a;
        return (_a = this.manifest.getDocumentElement()) === null || _a === void 0 ? void 0 : _a.getAttribute('package');
    }
    setVersionCode(versionCode) {
        var _a;
        return (_a = this.appBuildGradle) === null || _a === void 0 ? void 0 : _a.setVersionCode(versionCode);
    }
    async getVersionCode() {
        var _a, _b;
        return (_b = (await ((_a = this.appBuildGradle) === null || _a === void 0 ? void 0 : _a.getVersionCode()))) !== null && _b !== void 0 ? _b : null;
    }
    incrementVersionCode() {
        var _a, _b;
        return (_b = (_a = this.appBuildGradle) === null || _a === void 0 ? void 0 : _a.incrementVersionCode()) !== null && _b !== void 0 ? _b : Promise.resolve();
    }
    setVersionName(versionName) {
        var _a;
        return (_a = this.appBuildGradle) === null || _a === void 0 ? void 0 : _a.setVersionName(versionName);
    }
    getVersionName() {
        var _a, _b;
        return (_b = (_a = this.appBuildGradle) === null || _a === void 0 ? void 0 : _a.getVersionName()) !== null && _b !== void 0 ? _b : Promise.resolve(null);
    }
    /**
     * Add a new file to the given resources directory with the given contents and
     * given file name
     **/
    getResource(resDir, file, options = { encoding: 'utf-8' }) {
        const root = this.getResourcesRoot();
        if (!root) {
            return;
        }
        const dir = (0, path_1.join)(root, resDir);
        if (!options) {
            return (0, utils_fs_1.readFile)((0, path_1.join)(dir, file));
        }
        return (0, utils_fs_1.readFile)((0, path_1.join)(dir, file), options);
    }
    /**
     * Add a new file to the given resources directory with the given contents and
     * given file name
     **/
    async addResource(resDir, file, contents) {
        const root = this.getResourcesRoot();
        if (!root) {
            return;
        }
        const dir = (0, path_1.join)(root, resDir);
        if (!(await (0, utils_fs_1.pathExists)(dir))) {
            await (0, utils_fs_1.mkdir)(dir);
        }
        return (0, utils_fs_1.writeFile)((0, path_1.join)(dir, file), contents);
    }
    async copyFile(src, dest) {
        var _a, _b, _c;
        if (!((_c = (_b = (_a = this.project) === null || _a === void 0 ? void 0 : _a.config) === null || _b === void 0 ? void 0 : _b.android) === null || _c === void 0 ? void 0 : _c.path)) {
            return Promise.reject();
        }
        const destPath = (0, path_1.join)(this.project.config.android.path, dest);
        if (/^(https?:\/\/)/.test(src)) {
            const res = await (0, cross_fetch_1.default)(src);
            return (0, utils_fs_1.writeFile)(destPath, Buffer.from(await res.arrayBuffer()));
        }
        const srcPath = (0, path_1.join)(this.project.config.android.path, src);
        return (0, utils_fs_1.copy)(srcPath, destPath);
    }
    /**
     * Copy the given source into the given resources directory with the
     * given file name
     **/
    async copyToResources(resDir, file, source) {
        const root = this.getResourcesRoot();
        if (!root) {
            return;
        }
        const dir = (0, path_1.join)(root, resDir);
        if (!(await (0, utils_fs_1.pathExists)(dir))) {
            await (0, utils_fs_1.mkdir)(dir);
        }
        const sourceData = await (0, read_src_1.readSource)(source);
        return (0, utils_fs_1.writeFile)((0, path_1.join)(dir, file), sourceData);
    }
    getAndroidManifestPath() {
        var _a, _b;
        if (!((_a = this.project.config.android) === null || _a === void 0 ? void 0 : _a.path)) {
            return null;
        }
        return (0, path_1.join)((_b = this.project.config.android) === null || _b === void 0 ? void 0 : _b.path, 'app', 'src', 'main', 'AndroidManifest.xml');
    }
    getResourcesPath() {
        return (0, path_1.join)('app', 'src', 'main', 'res');
    }
    getResourcesRoot() {
        var _a, _b;
        if (!((_a = this.project.config.android) === null || _a === void 0 ? void 0 : _a.path)) {
            return null;
        }
        return (0, path_1.join)((_b = this.project.config.android) === null || _b === void 0 ? void 0 : _b.path, this.getResourcesPath());
    }
    getAppRoot() {
        var _a, _b;
        if (!((_a = this.project.config.android) === null || _a === void 0 ? void 0 : _a.path)) {
            return null;
        }
        // TODO: Don't hard-code app
        return (0, path_1.join)((_b = this.project.config.android) === null || _b === void 0 ? void 0 : _b.path, 'app');
    }
    async loadGradle(path) {
        var _a, _b;
        if (!((_a = this.project.config.android) === null || _a === void 0 ? void 0 : _a.path)) {
            return null;
        }
        const filename = (0, path_1.join)((_b = this.project.config.android) === null || _b === void 0 ? void 0 : _b.path, path);
        return new gradle_file_1.GradleFile(filename, this.project.vfs);
    }
}
exports.AndroidProject = AndroidProject;
//# sourceMappingURL=project.js.map