module Twine
  module Formatters
    class Apple < Abstract
      include Twine::Placeholders

      def format_name
        'apple'
      end

      def extension
        '.strings'
      end

      def can_handle_directory?(path)
        Dir.entries(path).any? { |item| /^.+\.lproj$/.match(item) }
      end

      def default_file_name
        'Localizable.strings'
      end

      def determine_language_given_path(path)
        path_arr = path.split(File::SEPARATOR)
        path_arr.each do |segment|
          match = /^(.+)\.lproj$/.match(segment)
          if match
            if match[1] == "Base"
              return @options[:developer_language]
            else
              return match[1]
            end
          end
        end

        return super
      end

      def output_path_for_language(lang)
        "#{lang}.lproj"
      end

      def read(io, lang)
        last_comment = nil
        while line = io.gets
          # matches a `key = "value"` line, where key may be quoted or unquoted. The former may also contain escaped characters
          match = /^\s*((?:"(?:[^"\\]|\\.)+")|(?:[^"\s=]+))\s*=\s*"((?:[^"\\]|\\.)*)"/.match(line)
          if match
            key = match[1]
            key = key[1..-2] if key[0] == '"' and key[-1] == '"'
            key.gsub!('\\"', '"')
            value = match[2]
            value.gsub!('\\"', '"')
            set_translation_for_key(key, lang, value)
            if last_comment
              set_comment_for_key(key, last_comment)
            end
          end

          match = /\/\* (.*) \*\//.match(line)
          if match
            last_comment = match[1]
          else
            last_comment = nil
          end
        end
      end

      def format_header(lang)
        "/**\n * Apple Strings File\n * Generated by Twine\n * Language: #{lang}\n */"
      end

      def format_section_header(section)
        "/********** #{section.name} **********/\n"
      end

      def key_value_pattern
        "\"%{key}\" = \"%{value}\";\n"
      end

      def format_comment(definition, lang)
        "/* #{definition.comment.gsub('*/', '* /')} */\n" if definition.comment
      end

      def format_key(key)
        escape_quotes(key)
      end

      def format_value(value)
        # Replace Android's %s with iOS %@
        value = convert_placeholders_from_android_to_twine(value)
        escape_quotes(value)
      end

      def should_include_definition(definition, lang)
        return !definition.is_plural? && super
      end
    end
  end
end

Twine::Formatters.formatters << Twine::Formatters::Apple.new
