#!/usr/bin/env bash
# Please run this script to configure the repository after cloning it.

# When configuring with private repository, the following override hierarchy is used:
# - commandline parameters - most specific, always wins.
# - stdin parameters.
# - saved repository - least specific, if present.
# - fallback to opensource mode.

# Stop on the first error.
set -e -u

BASE_PATH=$(cd "$(dirname "$0")"; pwd)

DEFAULT_PRIVATE_HEADER="$BASE_PATH/private_default.h"
PRIVATE_HEADER="private.h"
PRIVATE_PROPERTIES="android/secure.properties"
PRIVATE_NETWORK_CONFIG="android/res/xml/network_security_config.xml"
PRIVATE_GOOGLE_SERVICES="android/google-services.json"

SAVED_PRIVATE_REPO_FILE="$BASE_PATH/.private_repository_url"
SAVED_PRIVATE_BRANCH_FILE="$BASE_PATH/.private_repository_branch"
TMP_REPO_DIR="$BASE_PATH/.tmp.private.repo"

usage() {
  echo "This tool configures Organic Maps for an opensource build by default"
  echo "and bootstraps the Boost submodule after that."
  echo
  echo "To e.g. publish in app stores populate following configs with your own private keys etc."
  echo "  $PRIVATE_HEADER"
  echo "  $PRIVATE_PROPERTIES"
  echo "  $PRIVATE_NETWORK_CONFIG"
  echo "  $PRIVATE_GOOGLE_SERVICES"
  echo "The tool can copy over the necessary configs from a given private repo and a branch"
  echo "(it copies all files except README.md)."
  echo "It remembers the repo and the branch to pull the config changes automatically on next run."
  echo
  echo "Usage:"
  echo "  $0 private_repo_url [private_repo_branch]  - copy configs from a private repo (master is the default branch)"
  echo "  echo 'private_repo_url [private_repo_branch]' | $0  - alternate invocation for private repo configuration"
  echo "  $0  - use a saved repo and a branch if present or default to an opensource build configs"
  echo ""
}

setup_opensource() {
  echo "Initializing repository with default values in Open-Source mode."
  cat "$DEFAULT_PRIVATE_HEADER" > "$BASE_PATH/$PRIVATE_HEADER"
  echo 'ext {
  spropStoreFile = "debug.keystore"
  spropStorePassword = "12345678"
  spropKeyAlias = "debug"
  spropKeyPassword = "12345678"
}
' > "$BASE_PATH/$PRIVATE_PROPERTIES"
  echo '<?xml version="1.0" encoding="utf-8"?>
<network-security-config/>
' > "$BASE_PATH/$PRIVATE_NETWORK_CONFIG"
  echo '{"client": [{"api_key": [{"current_key": ""}], "client_info": {"mobilesdk_app_id": "omaps", "android_client_info": {"package_name": "app.organicmaps.debug"}}}], "configuration_version": "1", "project_info": {"storage_bucket": "", "project_id": "", "project_number": ""}}' > "$BASE_PATH/$PRIVATE_GOOGLE_SERVICES"
}

# Clone the private repo and copy all of its files (except README.md) into the main repo
setup_private() {
  echo "Copying private configuration files from the repo '$PRIVATE_REPO', branch '$PRIVATE_BRANCH'"
  rm -rf "$TMP_REPO_DIR"
  git clone --branch "$PRIVATE_BRANCH" --depth 1 "$PRIVATE_REPO" "$TMP_REPO_DIR"
  echo "$PRIVATE_REPO" > "$SAVED_PRIVATE_REPO_FILE"
  echo "$PRIVATE_BRANCH" > "$SAVED_PRIVATE_BRANCH_FILE"
  echo "Saved private repository url '$PRIVATE_REPO' to '$SAVED_PRIVATE_REPO_FILE'"
  echo "Saved private branch '$PRIVATE_BRANCH' to '$SAVED_PRIVATE_BRANCH_FILE'"
  cd $TMP_REPO_DIR
  rm -rf "$TMP_REPO_DIR/.git" "$TMP_REPO_DIR/README.md"
  cp -Rv "$TMP_REPO_DIR"/* "$BASE_PATH"
  rm -rf "$TMP_REPO_DIR"
  echo "Private files have been updated."
}

if [ "${1-}" = "-h" -o "${1-}" = "--help" ]; then
  usage
  exit 1
fi

ARGS_PRIVATE_REPO=${1-}
ARGS_PRIVATE_BRANCH=${2-}

if [ -n "$ARGS_PRIVATE_REPO" ]; then
  PRIVATE_REPO=$ARGS_PRIVATE_REPO
  if [ -n "$ARGS_PRIVATE_BRANCH" ]; then
    PRIVATE_BRANCH=$ARGS_PRIVATE_BRANCH
  else
    PRIVATE_BRANCH=master
  fi
else
  read -t 1 READ_PRIVATE_REPO READ_PRIVATE_BRANCH || true
  if [ -n "${READ_PRIVATE_REPO-}" ]; then
    PRIVATE_REPO=$READ_PRIVATE_REPO
    if [ -n "${READ_PRIVATE_BRANCH-}" ]; then
      PRIVATE_BRANCH=$READ_PRIVATE_BRANCH
    else
      PRIVATE_BRANCH=master
    fi
  elif [ -f "$SAVED_PRIVATE_REPO_FILE" ]; then
    PRIVATE_REPO=`cat "$SAVED_PRIVATE_REPO_FILE"`
    echo "Using stored private repository URL: $PRIVATE_REPO"
    if [ -f "$SAVED_PRIVATE_BRANCH_FILE" ]; then
      PRIVATE_BRANCH=`cat "$SAVED_PRIVATE_BRANCH_FILE"`
      echo "Using stored private branch: $PRIVATE_BRANCH"
    else
      PRIVATE_BRANCH=master
    fi
  else
    PRIVATE_REPO=""
  fi
fi

if [ -n "$PRIVATE_REPO" ]; then
  setup_private
else
  setup_opensource
fi

# TODO: Remove these lines when XCode project is finally generated by CMake.
if [ ! -d "$BASE_PATH/3party/boost/" ]; then
  echo "You need to have Boost submodule present to run bootstrap.sh"
  echo "Try 'git submodule update --init --recursive'"
  exit 1
fi
if [ ! -d "$BASE_PATH/3party/boost/tools" ]; then
  echo "Boost's submodule 'tools' is missing, it is required for bootstrap"
  echo "Try 'git submodule update --init --recursive'"
  exit 1
fi

cd "$BASE_PATH/3party/boost/"
if [[ "$OSTYPE" == msys ]]; then
  echo "For Windows please run:"
  echo "cd 3party\\boost"
  echo "bootstrap.bat"
  echo "b2 headers"
  echo "cd ..\\.."
else
  ./bootstrap.sh
  ./b2 headers
fi
cd "$BASE_PATH"
