#ifndef OSMIUM_IO_OUTPUT_ITERATOR_HPP
#define OSMIUM_IO_OUTPUT_ITERATOR_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2015 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <cstddef>
#include <iterator>
#include <memory>
#include <utility>

#include <osmium/memory/buffer.hpp>
#include <osmium/osm/diff_object.hpp>

namespace osmium {

    namespace memory {
        class Item;
    } // namespace memory

    namespace io {

        template <class TDest>
        class OutputIterator : public std::iterator<std::output_iterator_tag, osmium::memory::Item> {

            struct buffer_wrapper {

                osmium::memory::Buffer buffer;

                buffer_wrapper(size_t buffer_size) :
                    buffer(buffer_size, osmium::memory::Buffer::auto_grow::no) {
                }

            }; // struct buffer_wrapper

            static constexpr size_t default_buffer_size = 10 * 1024 * 1024;

            TDest* m_destination;

            std::shared_ptr<buffer_wrapper> m_buffer_wrapper;

        public:

            explicit OutputIterator(TDest& destination, const size_t buffer_size = default_buffer_size) :
                m_destination(&destination),
                m_buffer_wrapper(std::make_shared<buffer_wrapper>(buffer_size)) {
            }

            OutputIterator(const OutputIterator&) = default;
            OutputIterator(OutputIterator&&) = default;

            OutputIterator& operator=(const OutputIterator&) = default;
            OutputIterator& operator=(OutputIterator&&) = default;

            ~OutputIterator() = default;

            void flush() {
                osmium::memory::Buffer buffer(m_buffer_wrapper->buffer.capacity(), osmium::memory::Buffer::auto_grow::no);
                std::swap(m_buffer_wrapper->buffer, buffer);
                (*m_destination)(std::move(buffer));
            }

            OutputIterator& operator=(const osmium::memory::Item& item) {
                try {
                    m_buffer_wrapper->buffer.push_back(item);
                } catch (osmium::buffer_is_full&) {
                    flush();
                    m_buffer_wrapper->buffer.push_back(item);
                }
                return *this;
            }

            OutputIterator& operator=(const osmium::DiffObject& diff) {
                return this->operator=(diff.curr());
            }

            OutputIterator& operator*() {
                return *this;
            }

            OutputIterator& operator++() {
                return *this;
            }

            OutputIterator& operator++(int) {
                return *this;
            }

        }; // class OutputIterator

    } // namespace io

} // namespace osmium

#endif // OSMIUM_IO_OUTPUT_ITERATOR_HPP
