package app.michaelwuensch.bitbanana.channelManagement.listItems;

import androidx.annotation.Nullable;

import com.google.protobuf.ByteString;

import app.michaelwuensch.bitbanana.util.AliasManager;

public abstract class ChannelListItem implements Comparable<ChannelListItem> {

    public static final int TYPE_OPEN_CHANNEL = 0;
    public static final int TYPE_PENDING_OPEN_CHANNEL = 1;
    public static final int TYPE_PENDING_CLOSING_CHANNEL = 2;
    public static final int TYPE_PENDING_FORCE_CLOSING_CHANNEL = 3;
    public static final int TYPE_WAITING_CLOSE_CHANNEL = 4;
    public static final int TYPE_CLOSED_CHANNEL = 5;

    abstract public int getType();

    abstract public ByteString getChannelByteString();

    @Override
    public int compareTo(ChannelListItem channelListItem) {
        ChannelListItem other = channelListItem;

        String ownPubkey = "";
        switch (this.getType()) {
            case TYPE_OPEN_CHANNEL:
                ownPubkey = ((OpenChannelItem) this).getChannel().getRemotePubkey();
                break;
            case TYPE_PENDING_OPEN_CHANNEL:
                ownPubkey = ((PendingOpenChannelItem) this).getChannel().getChannel().getRemoteNodePub();
                break;
            case TYPE_PENDING_CLOSING_CHANNEL:
                ownPubkey = ((PendingClosingChannelItem) this).getChannel().getChannel().getRemoteNodePub();
                break;
            case TYPE_PENDING_FORCE_CLOSING_CHANNEL:
                ownPubkey = ((PendingForceClosingChannelItem) this).getChannel().getChannel().getRemoteNodePub();
                break;
            case TYPE_WAITING_CLOSE_CHANNEL:
                ownPubkey = ((WaitingCloseChannelItem) this).getChannel().getChannel().getRemoteNodePub();
                break;
            case TYPE_CLOSED_CHANNEL:
                ownPubkey = ((ClosedChannelItem) this).getChannel().getRemotePubkey();
        }

        String otherPubkey = "";
        switch (other.getType()) {
            case TYPE_OPEN_CHANNEL:
                otherPubkey = ((OpenChannelItem) other).getChannel().getRemotePubkey();
                break;
            case TYPE_PENDING_OPEN_CHANNEL:
                otherPubkey = ((PendingOpenChannelItem) other).getChannel().getChannel().getRemoteNodePub();
                break;
            case TYPE_PENDING_CLOSING_CHANNEL:
                otherPubkey = ((PendingClosingChannelItem) other).getChannel().getChannel().getRemoteNodePub();
                break;
            case TYPE_PENDING_FORCE_CLOSING_CHANNEL:
                otherPubkey = ((PendingForceClosingChannelItem) other).getChannel().getChannel().getRemoteNodePub();
                break;
            case TYPE_WAITING_CLOSE_CHANNEL:
                otherPubkey = ((WaitingCloseChannelItem) other).getChannel().getChannel().getRemoteNodePub();
                break;
            case TYPE_CLOSED_CHANNEL:
                otherPubkey = ((ClosedChannelItem) other).getChannel().getRemotePubkey();
        }

        String ownAlias = AliasManager.getInstance().getAlias(ownPubkey).toLowerCase();
        String otherAlias = AliasManager.getInstance().getAlias(otherPubkey).toLowerCase();

        return ownAlias.compareTo(otherAlias);
    }

    public boolean equalsWithSameContent(@Nullable Object obj) {
        if (!equals(obj)) {
            return false;
        }

        ChannelListItem that = (ChannelListItem) obj;

        switch (this.getType()) {
            case TYPE_OPEN_CHANNEL:
                return ((OpenChannelItem) this).getChannel().getNumUpdates() == ((OpenChannelItem) that).getChannel().getNumUpdates();
            case TYPE_PENDING_OPEN_CHANNEL:
                return ((PendingOpenChannelItem) this).getChannel().getChannel().getLocalBalance() == ((PendingOpenChannelItem) that).getChannel().getChannel().getLocalBalance();
            case TYPE_PENDING_CLOSING_CHANNEL:
                return ((PendingClosingChannelItem) this).getChannel().getChannel().getLocalBalance() == ((PendingClosingChannelItem) that).getChannel().getChannel().getLocalBalance();
            case TYPE_PENDING_FORCE_CLOSING_CHANNEL:
                return ((PendingForceClosingChannelItem) this).getChannel().getChannel().getLocalBalance() == ((PendingForceClosingChannelItem) that).getChannel().getChannel().getLocalBalance();
            case TYPE_WAITING_CLOSE_CHANNEL:
                return ((WaitingCloseChannelItem) this).getChannel().getChannel().getLocalBalance() == ((WaitingCloseChannelItem) that).getChannel().getChannel().getLocalBalance();
            case TYPE_CLOSED_CHANNEL:
                return ((ClosedChannelItem) this).getChannel().getSettledBalance() == ((ClosedChannelItem) that).getChannel().getSettledBalance();
        }
        return false;
    }

    @Override
    public boolean equals(@Nullable Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj == null || obj.getClass() != this.getClass()) {
            return false;
        }
        ChannelListItem that = (ChannelListItem) obj;
        if (that.getType() != this.getType()) {
            return false;
        }

        switch (this.getType()) {
            case TYPE_OPEN_CHANNEL:
                return ((OpenChannelItem) this).getChannel().getChanId() == ((OpenChannelItem) that).getChannel().getChanId();
            case TYPE_PENDING_OPEN_CHANNEL:
                return ((PendingOpenChannelItem) this).getChannel().getChannel().getChannelPoint().equals(((PendingOpenChannelItem) that).getChannel().getChannel().getChannelPoint());
            case TYPE_PENDING_CLOSING_CHANNEL:
                return ((PendingClosingChannelItem) this).getChannel().getChannel().getChannelPoint().equals(((PendingClosingChannelItem) that).getChannel().getChannel().getChannelPoint());
            case TYPE_PENDING_FORCE_CLOSING_CHANNEL:
                return ((PendingForceClosingChannelItem) this).getChannel().getChannel().getChannelPoint().equals(((PendingForceClosingChannelItem) that).getChannel().getChannel().getChannelPoint());
            case TYPE_WAITING_CLOSE_CHANNEL:
                return ((WaitingCloseChannelItem) this).getChannel().getChannel().getChannelPoint().equals(((WaitingCloseChannelItem) that).getChannel().getChannel().getChannelPoint());
            case TYPE_CLOSED_CHANNEL:
                return ((ClosedChannelItem) this).getChannel().getChannelPoint().equals(((ClosedChannelItem) that).getChannel().getChannelPoint());
            default:
                return false;
        }
    }

    @Override
    public int hashCode() {
        switch (this.getType()) {
            case TYPE_OPEN_CHANNEL:
                return Long.valueOf(((OpenChannelItem) this).getChannel().getChanId()).hashCode();
            case TYPE_PENDING_OPEN_CHANNEL:
                return ((PendingOpenChannelItem) this).getChannel().getChannel().getChannelPoint().hashCode();
            case TYPE_PENDING_CLOSING_CHANNEL:
                return ((PendingClosingChannelItem) this).getChannel().getChannel().getChannelPoint().hashCode();
            case TYPE_PENDING_FORCE_CLOSING_CHANNEL:
                return ((PendingForceClosingChannelItem) this).getChannel().getChannel().getChannelPoint().hashCode();
            case TYPE_WAITING_CLOSE_CHANNEL:
                return ((WaitingCloseChannelItem) this).getChannel().getChannel().getChannelPoint().hashCode();
            case TYPE_CLOSED_CHANNEL:
                return ((ClosedChannelItem) this).getChannel().getChannelPoint().hashCode();
            default:
                return 0;
        }
    }
}
