package app.crossword.yourealwaysbe.forkyz.util;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import android.content.Context;
import android.net.Uri;

import app.crossword.yourealwaysbe.forkyz.R;
import app.crossword.yourealwaysbe.forkyz.util.files.DirHandle;
import app.crossword.yourealwaysbe.forkyz.util.files.FileHandle;
import app.crossword.yourealwaysbe.forkyz.util.files.FileHandler;
import app.crossword.yourealwaysbe.forkyz.util.files.PuzHandle;
import app.crossword.yourealwaysbe.puz.Puzzle;
import app.crossword.yourealwaysbe.puz.io.PuzzleStreamReader;

/**
 * Takes an arbitrary URI and tries to convert to a puzzle and import
 */
public class PuzzleImporter {
    private static final Logger LOGGER
        = Logger.getLogger(PuzzleImporter.class.getCanonicalName());

    /** Import from a URI supported by resolver
     *
     * Currently does not use file extension or MIME type. Instead, use puzlib
     * that tries each known format in turn until one succeeds. Clunky, but
     * hopefully robust.
     *
     * @return new puz handle if succeeded (will return null if failed
     * or uri is null)
     */
    public static PuzHandle importUri(
        Context context, FileHandler fileHandler, Uri uri
    ) {
        if (uri == null)
            return null;

        try (
            InputStream is = new BufferedInputStream(
                context.getContentResolver().openInputStream(uri)
            )
        ) {
            return importInputStream(
                context, fileHandler, is, getFileName(uri)
            );
        } catch (IOException e) {
            return null;
        }
    }

    public static List<PuzHandle> processToImportDirectory(
        Context context, FileHandler fileHandler
    ) {
        List<PuzHandle> importedFiles = new ArrayList<>();

        DirHandle toImportDir
            = fileHandler.getToImportDirectory();

        for (FileHandle fh : fileHandler.listFiles(toImportDir)) {
            try (InputStream is = fileHandler.getBufferedInputStream(fh)) {
                PuzHandle ph = PuzzleImporter.importInputStream(
                    context, fileHandler, is, fileHandler.getName(fh)
                );
                if (ph == null) {
                    // deliberately only get once needed to avoid
                    // unneeded dir creation in java file mode
                    // (currently SAF creates all dirs upfront)
                    DirHandle toImportFailedDir
                        = fileHandler.getToImportFailedDirectory();
                    fileHandler.moveTo(fh, toImportDir, toImportFailedDir);
                } else {
                    DirHandle toImportDoneDir
                        = fileHandler.getToImportDoneDirectory();
                    importedFiles.add(ph);
                    fileHandler.moveTo(fh, toImportDir, toImportDoneDir);
                }
            } catch (IOException e) {
                DirHandle toImportFailedDir
                    = fileHandler.getToImportFailedDirectory();
                fileHandler.moveTo(fh, toImportDir, toImportFailedDir);
            }
        }

        return importedFiles;
    }

    /**
     * Try best to make sure there is some source
     *
     * Fall back to author, title, fallback
     */
    private static void ensurePuzSource(
        Context context, Puzzle puz, String fileName
    ) {
        String source = AppPuzzleUtils.deriveSource(
            puz.getSource(),
            fileName,
            puz.getAuthor(),
            puz.getTitle()
        );

        if (source == null || source.isEmpty())
            source = context.getString(R.string.import_fallback_source);

        puz.setSource(source);
    }

    private static void ensurePuzDate(Puzzle puz, String fileName) {
        LocalDate date = AppPuzzleUtils.deriveDate(puz.getDate(), fileName);
        if (date == null)
            date = LocalDate.now();
        puz.setDate(date);
    }

    private static String getFileName(Uri uri) {
        if (uri == null)
            return null;

        String path = uri.getPath();
        if (path != null)
            return new File(path).getName();

        return null;
    }

    private static PuzHandle importInputStream(
        Context context,
        FileHandler fileHandler,
        InputStream is,
        String fileName
    ) throws IOException {
        Puzzle puz = puz = PuzzleStreamReader.parseInputStatic(is);
        if (puz == null)
            return null;

        ensurePuzDate(puz, fileName);
        ensurePuzSource(context, puz, fileName);

        try {
            return fileHandler.saveNewPuzzle(
                puz, AppPuzzleUtils.generateFileName(puz)
            );
        } catch (IOException e) {
            LOGGER.severe("Failed to save imported puzzle: " + e);
            return null;
        }

    }
}
