package app.crossword.yourealwaysbe.view;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint.Align;
import android.graphics.Paint.Style;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.text.StaticLayout;
import android.text.TextPaint;

import androidx.core.content.ContextCompat;
import androidx.core.text.HtmlCompat;

import app.crossword.yourealwaysbe.forkyz.ForkyzApplication;
import app.crossword.yourealwaysbe.forkyz.R;
import app.crossword.yourealwaysbe.puz.Box;
import app.crossword.yourealwaysbe.puz.Clue;
import app.crossword.yourealwaysbe.puz.ClueID;
import app.crossword.yourealwaysbe.puz.Note;
import app.crossword.yourealwaysbe.puz.Playboard.Word;
import app.crossword.yourealwaysbe.puz.Playboard;
import app.crossword.yourealwaysbe.puz.Position;
import app.crossword.yourealwaysbe.puz.Puzzle;
import app.crossword.yourealwaysbe.puz.Zone;
import app.crossword.yourealwaysbe.versions.AndroidVersionUtils;
import app.crossword.yourealwaysbe.view.ScrollingImageView.Point;

import java.util.Set;
import java.util.logging.Logger;


public class PlayboardRenderer {
    private static final float BASE_BOX_SIZE_INCHES = 0.25F;
    private static final Logger LOG = Logger.getLogger(PlayboardRenderer.class.getCanonicalName());
    @SuppressLint("NewApi")
    private static final Typeface TYPEFACE_SEMI_BOLD_SANS =
        AndroidVersionUtils.Factory.getInstance().getSemiBoldTypeface();

    private final Paint blackBox = new Paint();
    private final Paint blackCircle = new Paint();
    private final Paint blackLine = new Paint();
    private final Paint cheated = new Paint();
    private final Paint currentLetterBox = new Paint();
    private final Paint currentLetterHighlight = new Paint();
    private final Paint currentWordHighlight = new Paint();
    private final TextPaint letterText = new TextPaint();
    private final TextPaint numberText = new TextPaint();
    private final TextPaint noteText = new TextPaint();
    private final Paint red = new Paint();
    private final TextPaint redHighlight = new TextPaint();
    private final TextPaint white = new TextPaint();
    private final Paint flag = new Paint();
    private Bitmap bitmap;
    private Playboard board;
    private float dpi;
    private float scale = 1.0F;
    private boolean hintHighlight;
    private int widthPixels;

    private final static AndroidVersionUtils versionUtils
        = AndroidVersionUtils.Factory.getInstance();

    // colors are gotten from context
    public PlayboardRenderer(Playboard board,
                             float dpi, int widthPixels, boolean hintHighlight,
                             Context context) {
        this.dpi = dpi;
        this.widthPixels = widthPixels;
        this.board = board;
        this.hintHighlight = hintHighlight;

        int blankColor = ContextCompat.getColor(context, R.color.blankColor);
        int boxColor = ContextCompat.getColor(context, R.color.boxColor);
        int currentWordHighlightColor
            = ContextCompat.getColor(context, R.color.currentWordHighlightColor);
        int currentLetterHighlightColor
            = ContextCompat.getColor(context, R.color.currentLetterHighlightColor);
        int errorColor
            = ContextCompat.getColor(context, R.color.errorColor);
        int errorHighlightColor
            = ContextCompat.getColor(context, R.color.errorHighlightColor);
        int cheatedColor
            = ContextCompat.getColor(context, R.color.cheatedColor);
        int boardLetterColor
            = ContextCompat.getColor(context, R.color.boardLetterColor);
        int boardNoteColor
            = ContextCompat.getColor(context, R.color.boardNoteColor);
        int flagColor = ContextCompat.getColor(context, R.color.flagColor);

        blackLine.setColor(blankColor);
        blackLine.setStrokeWidth(2.0F);

        numberText.setTextAlign(Align.LEFT);
        numberText.setColor(boardLetterColor);
        numberText.setAntiAlias(true);
        numberText.setTypeface(Typeface.MONOSPACE);

        noteText.setTextAlign(Align.CENTER);
        noteText.setColor(boardNoteColor);
        noteText.setAntiAlias(true);
        noteText.setTypeface(TYPEFACE_SEMI_BOLD_SANS);

        letterText.setTextAlign(Align.CENTER);
        letterText.setColor(boardLetterColor);
        letterText.setAntiAlias(true);
        letterText.setTypeface(Typeface.SANS_SERIF);

        blackBox.setColor(blankColor);

        blackCircle.setColor(boardLetterColor);
        blackCircle.setAntiAlias(true);
        blackCircle.setStyle(Style.STROKE);

        currentWordHighlight.setColor(currentWordHighlightColor);
        currentLetterHighlight.setColor(currentLetterHighlightColor);
        currentLetterBox.setColor(boxColor);
        currentLetterBox.setStrokeWidth(2.0F);

        white.setTextAlign(Align.CENTER);
        white.setColor(boxColor);
        white.setAntiAlias(true);
        white.setTypeface(Typeface.SANS_SERIF);

        red.setTextAlign(Align.CENTER);
        red.setColor(errorColor);
        red.setAntiAlias(true);
        red.setTypeface(Typeface.SANS_SERIF);

        redHighlight.setTextAlign(Align.CENTER);
        redHighlight.setColor(errorHighlightColor);
        redHighlight.setAntiAlias(true);
        redHighlight.setTypeface(Typeface.SANS_SERIF);

        cheated.setColor(cheatedColor);

        flag.setColor(flagColor);
    }

    public float getDeviceMaxScale(){
        float retValue;
        LOG.info("Board "+board.getPuzzle().getWidth() +" widthPixels "+widthPixels);
        // inches * pixels per inch * units
        retValue = 2.2F;
        float puzzleBaseSizeInInches
            = board.getPuzzle().getWidth() * BASE_BOX_SIZE_INCHES;
        //leave a 1/16th in gutter on the puzzle.
        float fitToScreen =  (dpi * (puzzleBaseSizeInInches + 0.0625F)) / dpi;

        if(retValue < fitToScreen){
            retValue = fitToScreen;
        }

        LOG.warning("getDeviceMaxScale "+retValue);
        return retValue;
    }

    public float getDeviceMinScale(){
        //inches * (pixels / pixels per inch);
        float retValue = 0.9F * ((dpi * BASE_BOX_SIZE_INCHES) / dpi);
        LOG.warning("getDeviceMinScale "+retValue);
        return retValue;
    }

    public void setScale(float scale) {
        if (scale > getDeviceMaxScale()) {
            scale = getDeviceMaxScale();
        } else if (scale < getDeviceMinScale()) {
            scale = getDeviceMinScale();
        } else if (String.valueOf(scale).equals("NaN")) {
            scale = 1.0f;
        }
        this.bitmap = null;
        this.scale = scale;
    }

    public float getScale()
    {
        return this.scale;
    }

    /**
     * Draw a word on the board
     *
     * @param suppressNotesLists as in drawBox
     */
    public Bitmap draw(Word reset, Set<String> suppressNotesLists) {
        try {
            Puzzle puz = this.board.getPuzzle();
            Box[][] boxes = this.board.getBoxes();
            int width = puz.getWidth();
            int height = puz.getHeight();
            boolean renderAll = reset == null;

            if (scale > getDeviceMaxScale()) {
                scale = getDeviceMaxScale();
            } else if (scale < getDeviceMinScale()) {
                scale = getDeviceMinScale();
            } else if (Float.isNaN(scale)) {
                scale = 1.0F;
            }

            int boxSize = (int) (BASE_BOX_SIZE_INCHES * dpi * scale);

            if (bitmap == null) {
                LOG.warning("New bitmap box size "+boxSize);
                bitmap = Bitmap.createBitmap(
                    width * boxSize, height * boxSize, Bitmap.Config.RGB_565
                );
                bitmap.eraseColor(Color.BLACK);
                renderAll = true;
            }

            Canvas canvas = new Canvas(bitmap);

            // board data

            Word currentWord = this.board.getCurrentWord();

            for (int row = 0; row < height; row++) {
                for (int col = 0; col < width; col++) {
                    if (!renderAll) {
                        if (!currentWord.checkInWord(row, col) && !reset.checkInWord(row, col)) {
                            continue;
                        }
                    }

                    int x = col * boxSize;
                    int y = row * boxSize;
                    this.drawBox(
                        canvas,
                        x, y, row, col,
                        boxSize,
                        boxes[row][col],
                        currentWord, this.board.getHighlightLetter(),
                        suppressNotesLists,
                        true
                    );
                }
            }

            return bitmap;
        } catch (OutOfMemoryError e) {
            return bitmap;
        }
    }

    /**
     * Draw current word
     *
     * @param suppressNotesLists as in drawBox
     */
    public Bitmap drawWord(Set<String> suppressNotesLists) {
        Zone zone = this.board.getCurrentWord().getZone();
        int length = (zone == null) ? 0 : zone.size();

        Box[] boxes = this.board.getCurrentWordBoxes();
        int boxSize = (int) (BASE_BOX_SIZE_INCHES * this.dpi * scale) ;
        Bitmap bitmap = Bitmap.createBitmap(
            length * boxSize, boxSize, Bitmap.Config.RGB_565
        );
        bitmap.eraseColor(Color.BLACK);

        Canvas canvas = new Canvas(bitmap);

        for (int i = 0; i < length; i++) {
            int x = i * boxSize;
            int y = 0;
            Position pos = zone.getPosition(i);
            this.drawBox(
                canvas,
                x, y,
                pos.getRow(), pos.getCol(),
                boxSize,
                boxes[i],
                null,
                this.board.getHighlightLetter(),
                suppressNotesLists,
                false
            );
        }

        return bitmap;
    }

    /**
     * Draw the boxes
     *
     * @param suppressNotesLists as in drawBox
     */
    public Bitmap drawBoxes(
        Box[] boxes,
        Position highlight,
        Set<String> suppressNotesLists
    ) {
        if (boxes == null || boxes.length == 0) {
            return null;
        }

        int boxSize = (int) (BASE_BOX_SIZE_INCHES * this.dpi * scale);
        Bitmap bitmap = Bitmap.createBitmap(boxes.length * boxSize,
                                            boxSize,
                                            Bitmap.Config.RGB_565);
        bitmap.eraseColor(Color.BLACK);

        Canvas canvas = new Canvas(bitmap);

        for (int i = 0; i < boxes.length; i++) {
            int x = i * boxSize;
            int y = 0;
            this.drawBox(canvas,
                         x, y,
                         0, i,
                         boxSize,
                         boxes[i],
                         null,
                         highlight,
                         suppressNotesLists,
                         false);
        }

        return bitmap;
    }


    public Position findBox(Point p) {
        int boxSize = (int) (BASE_BOX_SIZE_INCHES * dpi * scale);

        if (boxSize == 0) {
            boxSize = (int) (BASE_BOX_SIZE_INCHES * dpi * 0.25F);
        }

        int col = p.x / boxSize;
        int row = p.y / boxSize;

        return new Position(row, col);
    }

    public int findBoxNoScale(Point p) {
        int boxSize =  (int) (BASE_BOX_SIZE_INCHES * dpi);
        LOG.info("DPI "+dpi+" scale "+ scale +" box size "+boxSize);
        return p.x / boxSize;
    }

    public Point findPointBottomRight(Position p) {
        int boxSize = (int) (BASE_BOX_SIZE_INCHES * dpi * scale);
        int x = (p.getCol() * boxSize) + boxSize;
        int y = (p.getRow() * boxSize) + boxSize;

        return new Point(x, y);
    }

    public Point findPointBottomRight(Word word) {
        Zone zone = word.getZone();

        if (zone == null || zone.isEmpty())
            return null;

        // for now assume that last box is bottom right
        Position p = zone.getPosition(zone.size() - 1);

        int boxSize = (int) (BASE_BOX_SIZE_INCHES * dpi * scale);
        int x = (p.getCol() * boxSize) + boxSize;
        int y = (p.getRow() * boxSize) + boxSize;

        return new Point(x, y);
    }

    public Point findPointTopLeft(Position p) {
        int boxSize = (int) (BASE_BOX_SIZE_INCHES  * dpi * scale);
        int x = p.getCol() * boxSize;
        int y = p.getRow() * boxSize;

        return new Point(x, y);
    }

    public Point findPointTopLeft(Word word) {
        // for now, assume first zone position is top left
        Zone zone = word.getZone();
        if (zone == null || zone.isEmpty())
            return null;
        return findPointTopLeft(zone.getPosition(0));
    }

    public float fitTo(int width, int height) {
        this.bitmap = null;
        // (pixels / boxes) / (pixels per inch / inches)
        Puzzle puz = this.board.getPuzzle();
        int numBoxes = Math.min(puz.getWidth(), puz.getHeight());
        return fitTo(width, height, puz.getWidth(), puz.getHeight());
    }

    public float fitTo(
        int width, int height, int numBoxesWidth, int numBoxesHeight
    ) {
        this.bitmap = null;
        float newScaleWidth = calculateScale(width, numBoxesWidth);
        float newScaleHeight = calculateScale(height, numBoxesHeight);
        setScale(Math.min(newScaleWidth, newScaleHeight));
        return getScale();
    }

    public float fitWidthTo(int width, int numBoxes) {
        this.bitmap = null;
        setScale(calculateScale(width, numBoxes));
        return getScale();
    }

    public float zoomIn() {
        this.bitmap = null;
        this.scale = scale * 1.25F;
        if(scale > this.getDeviceMaxScale()){
            this.scale = this.getDeviceMaxScale();
        }
        return scale;
    }

    public float zoomOut() {
        this.bitmap = null;
        this.scale = scale / 1.25F;
        if(scale < this.getDeviceMinScale()){
            scale = this.getDeviceMinScale();
        }
        return scale;
    }

    public float zoomReset() {
        this.bitmap = null;
        this.scale = 1.0F;
        return scale;
    }

    public float zoomInMax() {
        this.bitmap = null;
        this.scale = getDeviceMaxScale();

        return scale;
    }

    /**
     * Dynamic content description describing currently selected box on board
     *
     * @param baseDescription short description of what the board is
     * displaying
     */
    public String getContentDescription(CharSequence baseDescription) {
        Box curBox = board.getCurrentBox();
        return getContentDescription(baseDescription, curBox, true);
    }

    /**
     * Dynamic content description describing currently selected box
     *
     * @param baseDescription short description of what the board is
     * displaying
     * @param boxes
     * @param index which of the boxes to get description for, will return a
     * "no selection" string if index is out of range
     * @param hasCursor true if the box has the cursor
     */
    public String getContentDescription(
        CharSequence baseDescription, Box[] boxes, int index, boolean hasCursor
    ) {
        if (index < 0 || index >= boxes.length) {
            Context context = ForkyzApplication.getInstance();
            return context.getString(
                R.string.cur_box_none_selected, baseDescription
            );
        } else {
            Box curBox = boxes[index];
            return getContentDescription(baseDescription, curBox, hasCursor);
        }
    }
    /**
     * Dynamic content description for the given box
     *
     * @param baseDescription short description of the box
     * @param box the box to describe
     * @param hasCursor if the current box has the cursor
     */
    public String getContentDescription(
        CharSequence baseDescription, Box box, boolean hasCursor
    ) {
        Context context = ForkyzApplication.getInstance();

        String response = box.isBlank()
            ? context.getString(R.string.cur_box_blank)
            : String.valueOf(box.getResponse());

        String clueNumber = box.getClueNumber();
        String number = drawClueNumber(box)
            ? context.getString(R.string.cur_box_number, clueNumber)
            : context.getString(R.string.cur_box_no_number);

        String clueInfo = "";
        for (ClueID cid : box.getIsPartOfClues()) {
            clueInfo += context.getString(
                R.string.cur_box_clue_info,
                cid.getListName(),
                cid.getClueNumber()
            );
        }

        String circled = context.getString(
            box.isCircled()
                ? R.string.cur_box_circled
                : R.string.cur_box_not_circled
        );

        String barTop = context.getString(
            box.isBarredTop()
                ? R.string.cur_box_bar_top
                : R.string.cur_box_no_bar_top
        );

        String barRight = context.getString(
            box.isBarredRight()
                ? R.string.cur_box_bar_right
                : R.string.cur_box_no_bar_right
        );

        String barBottom = context.getString(
            box.isBarredBottom()
                ? R.string.cur_box_bar_bottom
                : R.string.cur_box_no_bar_bottom
        );

        String barLeft = context.getString(
            box.isBarredLeft()
                ? R.string.cur_box_bar_left
                : R.string.cur_box_no_bar_left
        );

        String error = context.getString(
            highlightError(box, hasCursor)
                ? R.string.cur_box_error
                : R.string.cur_box_no_error
        );

       String contentDesc = context.getString(
            R.string.cur_box_desc,
            baseDescription,
            response, clueInfo, number,
            circled, barTop, barRight, barBottom, barLeft,
            error
        );

        return contentDesc;
    }

    /**
     * Draw an individual box
     *
     * @param fullBoard whether to draw details that only make sense when the
     * full board can be seen.
     * @param suppressNotesLists set of lists to not draw notes from.
     * Empty set means draw notes from all lists, null means don't draw
     * any notes.
     */
    private void drawBox(Canvas canvas,
                         int x, int y,
                         int row, int col,
                         int boxSize,
                         Box box,
                         Word currentWord,
                         Position highlight,
                         Set<String> suppressNotesLists,
                         boolean fullBoard) {
        int numberTextSize = boxSize / 4;
        int miniNoteTextSize = boxSize / 2;
        int noteTextSize = Math.round(boxSize * 0.6F);
        int letterTextSize = Math.round(boxSize * 0.7F);
        int barSize = boxSize / 12;
        int numberOffset = barSize;
        int textOffset = boxSize / 30;

        // scale paints
        numberText.setTextSize(numberTextSize);
        letterText.setTextSize(letterTextSize);
        red.setTextSize(letterTextSize);
        redHighlight.setTextSize(letterTextSize);
        white.setTextSize(letterTextSize);

        boolean inCurrentWord = (currentWord != null) && currentWord.checkInWord(row, col);
        boolean isHighlighted
            = (highlight.getCol() == col) && (highlight.getRow() == row);

        TextPaint thisLetter;

        Paint boxColor = (((highlight.getCol() == col) && (highlight.getRow() == row)) && (currentWord != null))
                ? this.currentLetterBox : this.blackLine;

        // Draw left
        if ((col != (highlight.getCol() + 1)) || (row != highlight.getRow())) {
            canvas.drawLine(x, y, x, y + boxSize, boxColor);
        }
        // Draw top
        if ((row != (highlight.getRow() + 1)) || (col != highlight.getCol())) {
            canvas.drawLine(x, y, x + boxSize, y, boxColor);
        }
        // Draw right
        if ((col != (highlight.getCol() - 1)) || (row != highlight.getRow())) {
            canvas.drawLine(x + boxSize, y, x + boxSize, y + boxSize, boxColor);
        }
        // Draw bottom
        if ((row != (highlight.getRow() - 1)) || (col != highlight.getCol())) {
            canvas.drawLine(x, y + boxSize, x + boxSize, y + boxSize, boxColor);
        }

        Rect r = new Rect(x + 1, y + 1, (x + boxSize) - 1, (y + boxSize) - 1);

        if (box == null) {
            canvas.drawRect(r, this.blackBox);
        } else {
            boolean highlightError = highlightError(box, isHighlighted);

            if (highlightError)
                box.setCheated(true);

            // Background colors
            if (isHighlighted && !highlightError) {
                canvas.drawRect(r, this.currentLetterHighlight);
            } else if (isHighlighted && highlightError) {
                canvas.drawRect(r, this.redHighlight);
            } else if ((currentWord != null) && currentWord.checkInWord(row, col)) {
                canvas.drawRect(r, this.currentWordHighlight);
            } else if (highlightError) {
                canvas.drawRect(r, this.red);
            } else if (this.hintHighlight && box.isCheated()) {
                canvas.drawRect(r, this.cheated);
            } else {
                if (!box.hasColor()) {
                    canvas.drawRect(r, this.white);
                } else {
                    Paint paint = getRelativePaint(this.white, box.getColor());
                    canvas.drawRect(r, paint);
                }
            }

            // Bars before clue numbers to avoid obfuscating
            if (fullBoard) {
                if (box.isBarredLeft()) {
                    Rect bar = new Rect(x, y, x + barSize, y + boxSize);
                    canvas.drawRect(bar, this.blackBox);
                }

                if (box.isBarredTop()) {
                    Rect bar = new Rect(x, y, x + boxSize, y + barSize);
                    canvas.drawRect(bar, this.blackBox);
                }

                if (box.isBarredRight()) {
                    Rect bar = new Rect(
                        x + boxSize - barSize, y,
                        x + boxSize, y + boxSize
                    );
                    canvas.drawRect(bar, this.blackBox);
                }

                if (box.isBarredBottom()) {
                    Rect bar = new Rect(
                        x, y + boxSize - barSize,
                        x + boxSize, y + boxSize
                    );
                    canvas.drawRect(bar, this.blackBox);
                }
            }

            if (drawClueNumber(box)) {
                String clueNumber = box.getClueNumber();
                drawHtmlText(
                    canvas,
                    clueNumber,
                    x + numberOffset,
                    y + numberOffset / 2,
                    boxSize,
                    numberText
                );

                Puzzle puz = board.getPuzzle();

                if (fullBoard) {
                    boolean flagAcross = false;
                    boolean flagDown = false;

                    for (ClueID cid : box.getIsPartOfClues()) {
                        if (box.isStartOf(cid) && puz.isFlagged(cid)) {
                            if (isClueProbablyAcross(cid))
                                flagAcross = true;
                            else
                                flagDown = true;

                        }
                    }

                    if (flagDown) {
                        int numDigits = clueNumber.length();
                        int numWidth = numDigits * numberTextSize / 2;
                        Rect bar = new Rect(
                            x + numberOffset + numWidth + barSize,
                            y + 1 * barSize,
                            x + boxSize - barSize,
                            y + 2 * barSize
                        );
                        canvas.drawRect(bar, this.flag);
                    }

                    if (flagAcross) {
                        Rect bar = new Rect(
                            x + 1 * barSize,
                            y + barSize + numberOffset + numberTextSize,
                            x + 2 * barSize,
                            y + boxSize - barSize
                        );
                        canvas.drawRect(bar, this.flag);
                    }
                }
            }

            // Draw circle
            if (box.isCircled()) {
                canvas.drawCircle(x + (boxSize / 2) + 0.5F, y + (boxSize / 2) + 0.5F, (boxSize / 2) - 1.5F, blackCircle);
            }

            thisLetter = this.letterText;
            String letterString = box.isBlank() ? null : Character.toString(box.getResponse());
            String noteStringAcross = null;
            String noteStringDown = null;

            if (highlightError) {
                if (isHighlighted) {
                    thisLetter = this.white;
                } else if (inCurrentWord) {
                    thisLetter = this.redHighlight;
                }
            }

            // check for notes if needed
            if (box.isBlank() && !(suppressNotesLists == null)) {
                for (ClueID cid : box.getIsPartOfClues()) {
                    if (suppressNotesLists.contains(cid.getListName()))
                        continue;

                    Note note = board.getPuzzle().getNote(cid);
                    if (note == null)
                        continue;

                    String scratch = note.getScratch();
                    if (scratch == null)
                        continue;

                    int pos = box.getCluePosition(cid);
                    if (pos < 0 || pos >= scratch.length())
                        continue;

                    char noteChar = scratch.charAt(pos);
                    if (noteChar == ' ')
                        continue;

                    if (isClueProbablyAcross(cid)) {
                        noteStringAcross =
                            Character.toString(noteChar);
                    } else {
                        noteStringDown =
                            Character.toString(noteChar);
                    }
                }
            }

            if (letterString != null) {
                // Full size letter in normal font
                int yoffset = (int) (
                    boxSize - textOffset
                    + letterText.ascent() - letterText.descent()
                );
                drawText(
                    canvas,
                    letterString,
                    x + (boxSize / 2),
                    y + yoffset,
                    boxSize,
                    thisLetter
                );
            } else {
                float[] mWidth = new float[1];
                letterText.getTextWidths("M", mWidth);
                float letterTextHalfWidth = mWidth[0] / 2;

                if (noteStringAcross != null && noteStringDown != null) {
                    if (noteStringAcross.equals(noteStringDown)) {
                        // Same scratch letter in both directions
                        // Align letter with across and down answers
                        noteText.setTextSize(noteTextSize);
                        int noteTextHeight
                            = (int) (noteText.descent() - noteText.ascent());
                        drawText(
                            canvas,
                            noteStringAcross,
                            x + (int)(boxSize - letterTextHalfWidth),
                            y + boxSize - noteTextHeight - textOffset,
                            boxSize,
                            noteText
                        );
                    } else {
                        // Conflicting scratch letters
                        // Display both letters side by side
                        noteText.setTextSize(miniNoteTextSize);
                        int noteTextHeight
                            = (int) (noteText.descent() - noteText.ascent());
                        drawText(
                            canvas,
                            noteStringAcross,
                            x + (int)(boxSize * 0.05 + letterTextHalfWidth),
                            y + boxSize - noteTextHeight - textOffset,
                            boxSize,
                            noteText
                        );
                        int yoffset =
                            boxSize
                            - noteTextHeight
                            + (int) noteText.ascent();
                        drawText(
                            canvas,
                            noteStringDown,
                            x + (int)(boxSize - letterTextHalfWidth),
                            y + yoffset,
                            boxSize,
                            noteText
                        );
                    }
                } else if (noteStringAcross != null) {
                    // Across scratch letter only - display in bottom left
                    noteText.setTextSize(noteTextSize);
                    int noteTextHeight
                        = (int) (noteText.descent() - noteText.ascent());
                    drawText(
                        canvas,
                        noteStringAcross,
                        x + (boxSize / 2),
                        y + boxSize - noteTextHeight - textOffset,
                        boxSize,
                        noteText
                    );
                } else if (noteStringDown != null) {
                    // Down scratch letter only - display in bottom left
                    noteText.setTextSize(noteTextSize);
                    int noteTextHeight
                        = (int) (noteText.descent() - noteText.ascent());
                    drawText(
                        canvas,
                        noteStringDown,
                        x + (int)(boxSize - letterTextHalfWidth),
                        y + boxSize - noteTextHeight - textOffset,
                        boxSize,
                        noteText
                    );
                }
            }
        }
    }

    /**
     * Estimate general direction of clue
     *
     * Bias towards across if unsure
     */
    private boolean isClueProbablyAcross(ClueID cid) {
        Puzzle puz = board.getPuzzle();
        if (puz == null)
            return true;

        Clue clue = puz.getClue(cid);
        Zone zone = (clue == null) ? null : clue.getZone();
        if (zone == null || zone.size() <= 1)
            return true;

        Position pos0 = zone.getPosition(0);
        Position pos1 = zone.getPosition(1);

        return pos1.getCol() > pos0.getCol();
    }

    private boolean highlightError(Box box, boolean hasCursor) {
        boolean showErrors = this.board.isShowErrorsGrid()
            || (this.board.isShowErrorsCursor() && hasCursor);

        return showErrors
            && !box.isBlank()
            && box.hasSolution()
            && box.getSolution() != box.getResponse();
    }

    private boolean drawClueNumber(Box box) {
        return box.hasClueNumber();
    }

    /**
     * Return a new paint based on color
     *
     * For use when "inverting" a color to appear on the board. Relative
     * vs. a pure white background is the pure color. Vs. a pure black
     * background in the inverted color. Somewhere in between is
     * somewhere in between.
     *
     * @param base the standard background color
     * @param color 24-bit 0x00rrggbb "pure" color
     */
    private Paint getRelativePaint(Paint base, int pureColor) {
        int baseCol = base.getColor();

        // the android color library is compatible with 0x00rrggbb
        int mixedR = mixColors(Color.red(baseCol), Color.red(pureColor));
        int mixedG = mixColors(Color.green(baseCol), Color.green(pureColor));
        int mixedB = mixColors(Color.blue(baseCol), Color.blue(pureColor));

        Paint mixedPaint = new Paint(base);
        mixedPaint.setColor(Color.rgb(mixedR, mixedG, mixedB));

        return mixedPaint;
    }

    /**
     * Tint a 0-255 pure color against a base
     *
     * See getRelativePaint
     */
    private int mixColors(int base, int pure) {
        double baseBias = base / 255.0;
        return (int)(
            (baseBias * pure) + ((1- baseBias) * (255 - pure))
        );
    }

    private static void drawText(
        Canvas canvas,
        CharSequence text,
        int x, int  y, int width,
        TextPaint style
    ) {
        // with some help from:
        // https://stackoverflow.com/a/41870464
        StaticLayout staticLayout
            = versionUtils.getStaticLayout(text, style, width);
        canvas.save();
        canvas.translate(x, y);
        staticLayout.draw(canvas);
        canvas.restore();
    }

    private static void drawHtmlText(
        Canvas canvas, String text, int x, int y, int width, TextPaint style
    ) {
        drawText(canvas, HtmlCompat.fromHtml(text, 0), x, y, width, style);
    }

    private float calculateScale(int numPixels, int numBoxes) {
        double density = (double) dpi * (double) BASE_BOX_SIZE_INCHES;
        return (float) ((double) numPixels / (double) numBoxes / density);
    }
}

