---
title: Diagnostic messages
description: Details for diagnostics produced by the Dart analyzer.
body_class: highlight-diagnostics
---
{%- comment %}
WARNING: Do NOT EDIT this file directly. It is autogenerated by the script in
`pkg/analyzer/tool/diagnostics/generate.dart` in the sdk repository.
Update instructions: https://github.com/dart-lang/site-www/issues/1949
{% endcomment -%}

This page lists diagnostic messages produced by the Dart analyzer,
with details about what those messages mean and how you can fix your code.
For more information about the analyzer, see
[Customizing static analysis](/tools/analysis).

[constant context]: /resources/glossary#constant-context
[definite assignment]: /resources/glossary#definite-assignment
[mixin application]: /resources/glossary#mixin-application
[override inference]: /resources/glossary#override-inference
[part file]: /resources/glossary#part-file
[potentially non-nullable]: /resources/glossary#potentially-non-nullable
[public library]: /resources/glossary#public-library

## Diagnostics

The analyzer produces the following diagnostics for code that
doesn't conform to the language specification or
that might work in unexpected ways.

[bottom type]: https://dart.dev/null-safety/understanding-null-safety#top-and-bottom
[debugPrint]: https://api.flutter.dev/flutter/foundation/debugPrint.html
[ffi]: https://dart.dev/interop/c-interop
[IEEE 754]: https://en.wikipedia.org/wiki/IEEE_754
[irrefutable pattern]: https://dart.dev/resources/glossary#irrefutable-pattern
[kDebugMode]: https://api.flutter.dev/flutter/foundation/kDebugMode-constant.html
[meta-doNotStore]: https://pub.dev/documentation/meta/latest/meta/doNotStore-constant.html
[meta-doNotSubmit]: https://pub.dev/documentation/meta/latest/meta/doNotSubmit-constant.html
[meta-factory]: https://pub.dev/documentation/meta/latest/meta/factory-constant.html
[meta-immutable]: https://pub.dev/documentation/meta/latest/meta/immutable-constant.html
[meta-internal]: https://pub.dev/documentation/meta/latest/meta/internal-constant.html
[meta-literal]: https://pub.dev/documentation/meta/latest/meta/literal-constant.html
[meta-mustBeConst]: https://pub.dev/documentation/meta/latest/meta/mustBeConst-constant.html
[meta-mustCallSuper]: https://pub.dev/documentation/meta/latest/meta/mustCallSuper-constant.html
[meta-optionalTypeArgs]: https://pub.dev/documentation/meta/latest/meta/optionalTypeArgs-constant.html
[meta-sealed]: https://pub.dev/documentation/meta/latest/meta/sealed-constant.html
[meta-useResult]: https://pub.dev/documentation/meta/latest/meta/useResult-constant.html
[meta-UseResult]: https://pub.dev/documentation/meta/latest/meta/UseResult-class.html
[meta-visibleForOverriding]: https://pub.dev/documentation/meta/latest/meta/visibleForOverriding-constant.html
[meta-visibleForTesting]: https://pub.dev/documentation/meta/latest/meta/visibleForTesting-constant.html
[package-logging]: https://pub.dev/packages/logging
[refutable pattern]: https://dart.dev/resources/glossary#refutable-pattern

### abi_specific_integer_invalid

_Classes extending 'AbiSpecificInteger' must have exactly one const constructor,
no other members, and no type parameters._

#### Description

The analyzer produces this diagnostic when a class that extends
`AbiSpecificInteger` doesn't meet all of the following requirements:
- there must be exactly one constructor
- the constructor must be marked `const`
- there must not be any members of other than the one constructor
- there must not be any type parameters

#### Examples

The following code produces this diagnostic because the class `C` doesn't
define a const constructor:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
final class [!C!] extends AbiSpecificInteger {
}
```

The following code produces this diagnostic because the constructor isn't
a `const` constructor:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
final class [!C!] extends AbiSpecificInteger {
  C();
}
```

The following code produces this diagnostic because the class `C` defines
multiple constructors:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
final class [!C!] extends AbiSpecificInteger {
  const C.zero();
  const C.one();
}
```

The following code produces this diagnostic because the class `C` defines
a field:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
final class [!C!] extends AbiSpecificInteger {
  final int i;

  const C(this.i);
}
```

The following code produces this diagnostic because the class `C` has a
type parameter:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
final class [!C!]<T> extends AbiSpecificInteger { // type parameters
  const C();
}
```

#### Common fixes

Change the class so that it meets the requirements of having no type
parameters and a single member that is a `const` constructor:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
final class C extends AbiSpecificInteger {
  const C();
}
```

### abi_specific_integer_mapping_extra

_Classes extending 'AbiSpecificInteger' must have exactly one
'AbiSpecificIntegerMapping' annotation specifying the mapping from ABI to a 'NativeType' integer with a fixed size._

#### Description

The analyzer produces this diagnostic when a class that extends
`AbiSpecificInteger` has more than one `AbiSpecificIntegerMapping`
annotation.

#### Example

The following code produces this diagnostic because there are two
`AbiSpecificIntegerMapping` annotations on the class `C`:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
@[!AbiSpecificIntegerMapping!]({Abi.linuxX64 : Uint16()})
final class C extends AbiSpecificInteger {
  const C();
}
```

#### Common fixes

Remove all but one of the annotations, merging the arguments as
appropriate:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8(), Abi.linuxX64 : Uint16()})
final class C extends AbiSpecificInteger {
  const C();
}
```

### abi_specific_integer_mapping_missing

_Classes extending 'AbiSpecificInteger' must have exactly one
'AbiSpecificIntegerMapping' annotation specifying the mapping from ABI to a 'NativeType' integer with a fixed size._

#### Description

The analyzer produces this diagnostic when a class that extends
`AbiSpecificInteger` doesn't have an `AbiSpecificIntegerMapping`
annotation.

#### Example

The following code produces this diagnostic because there's no
`AbiSpecificIntegerMapping` annotation on the class `C`:

```dart
import 'dart:ffi';

final class [!C!] extends AbiSpecificInteger {
  const C();
}
```

#### Common fixes

Add an `AbiSpecificIntegerMapping` annotation to the class:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
final class C extends AbiSpecificInteger {
  const C();
}
```

### abi_specific_integer_mapping_unsupported

_Invalid mapping to '{0}'; only mappings to 'Int8', 'Int16', 'Int32', 'Int64',
'Uint8', 'Uint16', 'UInt32', and 'Uint64' are supported._

#### Description

The analyzer produces this diagnostic when a value in the map argument of
an `AbiSpecificIntegerMapping` annotation is anything other than one of
the following integer types:
- `Int8`
- `Int16`
- `Int32`
- `Int64`
- `Uint8`
- `Uint16`
- `UInt32`
- `Uint64`

#### Example

The following code produces this diagnostic because the value of the map
entry is `Array<Uint8>`, which isn't a valid integer type:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : [!Array<Uint8>(4)!]})
final class C extends AbiSpecificInteger {
  const C();
}
```

#### Common fixes

Use one of the valid types as a value in the map:

```dart
import 'dart:ffi';

@AbiSpecificIntegerMapping({Abi.macosX64 : Int8()})
final class C extends AbiSpecificInteger {
  const C();
}
```

### abstract_field_initializer

_Abstract fields can't have initializers._

#### Description

The analyzer produces this diagnostic when a field that has the `abstract`
modifier also has an initializer.

#### Examples

The following code produces this diagnostic because `f` is marked as
`abstract` and has an initializer:

```dart
abstract class C {
  abstract int [!f!] = 0;
}
```

The following code produces this diagnostic because `f` is marked as
`abstract` and there's an initializer in the constructor:

```dart
abstract class C {
  abstract int f;

  C() : [!f!] = 0;
}
```

#### Common fixes

If the field must be abstract, then remove the initializer:

```dart
abstract class C {
  abstract int f;
}
```

If the field isn't required to be abstract, then remove the keyword:

```dart
abstract class C {
  int f = 0;
}
```

### abstract_sealed_class

_A 'sealed' class can't be marked 'abstract' because it's already implicitly
abstract._

#### Description

The analyzer produces this diagnostic when a class is declared using both
the modifier `abstract` and the modifier `sealed`. Sealed classes are
implicitly abstract, so explicitly using both modifiers is not allowed.

#### Example

The following code produces this diagnostic because the class `C` is
declared using both `abstract` and `sealed`:

```dart
abstract [!sealed!] class C {}
```

#### Common fixes

If the class should be abstract but not sealed, then remove the `sealed`
modifier:

```dart
abstract class C {}
```

If the class should be both abstract and sealed, then remove the
`abstract` modifier:

```dart
sealed class C {}
```

### abstract_super_member_reference

_The {0} '{1}' is always abstract in the supertype._

#### Description

The analyzer produces this diagnostic when an inherited member is
referenced using `super`, but there is no concrete implementation of the
member in the superclass chain. Abstract members can't be invoked.

#### Example

The following code produces this diagnostic because `B` doesn't inherit a
concrete implementation of `a`:

```dart
abstract class A {
  int get a;
}
class B extends A {
  int get a => super.[!a!];
}
```

#### Common fixes

Remove the invocation of the abstract member, possibly replacing it with an
invocation of a concrete member.

### ambiguous_export

_The name '{0}' is defined in the libraries '{1}' and '{2}'._

#### Description

The analyzer produces this diagnostic when two or more export directives
cause the same name to be exported from multiple libraries.

#### Example

Given a file `a.dart` containing

```dart
class C {}
```

And a file `b.dart` containing

```dart
class C {}
```

The following code produces this diagnostic because the name `C` is being
exported from both `a.dart` and `b.dart`:

```dart
export 'a.dart';
export [!'b.dart'!];
```

#### Common fixes

If none of the names in one of the libraries needs to be exported, then
remove the unnecessary export directives:

```dart
export 'a.dart';
```

If all of the export directives are needed, then hide the name in all
except one of the directives:

```dart
export 'a.dart';
export 'b.dart' hide C;
```

### ambiguous_extension_member_access

_A member named '{0}' is defined in {1}, and none are more specific._

#### Description

When code refers to a member of an object (for example, `o.m()` or `o.m` or
`o[i]`) where the static type of `o` doesn't declare the member (`m` or
`[]`, for example), then the analyzer tries to find the member in an
extension. For example, if the member is `m`, then the analyzer looks for
extensions that declare a member named `m` and have an extended type that
the static type of `o` can be assigned to. When there's more than one such
extension in scope, the extension whose extended type is most specific is
selected.

The analyzer produces this diagnostic when none of the extensions has an
extended type that's more specific than the extended types of all of the
other extensions, making the reference to the member ambiguous.

#### Example

The following code produces this diagnostic because there's no way to
choose between the member in `E1` and the member in `E2`:

```dart
extension E1 on String {
  int get charCount => 1;
}

extension E2 on String {
  int get charCount => 2;
}

void f(String s) {
  print(s.[!charCount!]);
}
```

#### Common fixes

If you don't need both extensions, then you can delete or hide one of them.

If you need both, then explicitly select the one you want to use by using
an extension override:

```dart
extension E1 on String {
  int get charCount => length;
}

extension E2 on String {
  int get charCount => length;
}

void f(String s) {
  print(E2(s).charCount);
}
```

### ambiguous_import

_The name '{0}' is defined in the libraries {1}._

#### Description

The analyzer produces this diagnostic when a name is referenced that is
declared in two or more imported libraries.

#### Example

Given a library (`a.dart`) that defines a class (`C` in this example):

```dart
class A {}
class C {}
```

And a library (`b.dart`) that defines a different class with the same name:

```dart
class B {}
class C {}
```

The following code produces this diagnostic:

```dart
import 'a.dart';
import 'b.dart';

void f([!C!] c1, [!C!] c2) {}
```

#### Common fixes

If any of the libraries aren't needed, then remove the import directives
for them:

```dart
import 'a.dart';

void f(C c1, C c2) {}
```

If the name is still defined by more than one library, then add a `hide`
clause to the import directives for all except one library:

```dart
import 'a.dart' hide C;
import 'b.dart';

void f(C c1, C c2) {}
```

If you must be able to reference more than one of these types, then add a
prefix to each of the import directives, and qualify the references with
the appropriate prefix:

```dart
import 'a.dart' as a;
import 'b.dart' as b;

void f(a.C c1, b.C c2) {}
```

### ambiguous_set_or_map_literal_both

_The literal can't be either a map or a set because it contains at least one
literal map entry or a spread operator spreading a 'Map', and at least one element which is neither of these._

#### Description

Because map and set literals use the same delimiters (`{` and `}`), the
analyzer looks at the type arguments and the elements to determine which
kind of literal you meant. When there are no type arguments, then the
analyzer uses the types of the elements. If all of the elements are literal
map entries and all of the spread operators are spreading a `Map` then it's
a `Map`. If none of the elements are literal map entries and all of the
spread operators are spreading an `Iterable`, then it's a `Set`. If neither
of those is true then it's ambiguous.

The analyzer produces this diagnostic when at least one element is a
literal map entry or a spread operator spreading a `Map`, and at least one
element is neither of these, making it impossible for the analyzer to
determine whether you are writing a map literal or a set literal.

#### Example

The following code produces this diagnostic:

```dart
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    [!{...a, ...b, ...c}!];
```

The list `b` can only be spread into a set, and the maps `a` and `c` can
only be spread into a map, and the literal can't be both.

#### Common fixes

There are two common ways to fix this problem. The first is to remove all
of the spread elements of one kind or another, so that the elements are
consistent. In this case, that likely means removing the list and deciding
what to do about the now unused parameter:

```dart
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    {...a, ...c};
```

The second fix is to change the elements of one kind into elements that are
consistent with the other elements. For example, you can add the elements
of the list as keys that map to themselves:

```dart
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    {...a, for (String s in b) s: s, ...c};
```

### ambiguous_set_or_map_literal_either

_This literal must be either a map or a set, but the elements don't have enough
information for type inference to work._

#### Description

Because map and set literals use the same delimiters (`{` and `}`), the
analyzer looks at the type arguments and the elements to determine which
kind of literal you meant. When there are no type arguments and all of the
elements are spread elements (which are allowed in both kinds of literals)
then the analyzer uses the types of the expressions that are being spread.
If all of the expressions have the type `Iterable`, then it's a set
literal; if they all have the type `Map`, then it's a map literal.

This diagnostic is produced when none of the expressions being spread have
a type that allows the analyzer to decide whether you were writing a map
literal or a set literal.

#### Example

The following code produces this diagnostic:

```dart
union(a, b) => [!{...a, ...b}!];
```

The problem occurs because there are no type arguments, and there is no
information about the type of either `a` or `b`.

#### Common fixes

There are three common ways to fix this problem. The first is to add type
arguments to the literal. For example, if the literal is intended to be a
map literal, you might write something like this:

```dart
union(a, b) => <String, String>{...a, ...b};
```

The second fix is to add type information so that the expressions have
either the type `Iterable` or the type `Map`. You can add an explicit cast
or, in this case, add types to the declarations of the two parameters:

```dart
union(List<int> a, List<int> b) => {...a, ...b};
```

The third fix is to add context information. In this case, that means
adding a return type to the function:

```dart
Set<String> union(a, b) => {...a, ...b};
```

In other cases, you might add a type somewhere else. For example, say the
original code looks like this:

```dart
union(a, b) {
  var x = [!{...a, ...b}!];
  return x;
}
```

You might add a type annotation on `x`, like this:

```dart
union(a, b) {
  Map<String, String> x = {...a, ...b};
  return x;
}
```

### annotation_on_pointer_field

_Fields in a struct class whose type is 'Pointer' shouldn't have any
annotations._

#### Description

The analyzer produces this diagnostic when a field that's declared in a
subclass of `Struct` and has the type `Pointer` also has an annotation
associated with it.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `p`, which
has the type `Pointer` and is declared in a subclass of `Struct`, has the
annotation `@Double()`:

```dart
import 'dart:ffi';

final class C extends Struct {
  [!@Double()!]
  external Pointer<Int8> p;
}
```

#### Common fixes

Remove the annotations from the field:

```dart
import 'dart:ffi';

final class C extends Struct {
  external Pointer<Int8> p;
}
```

### argument_must_be_a_constant

_Argument '{0}' must be a constant._

#### Description

The analyzer produces this diagnostic when an invocation of either
`Pointer.asFunction` or `DynamicLibrary.lookupFunction` has an `isLeaf`
argument whose value isn't a constant expression.

The analyzer also produces this diagnostic when an invocation of either
`Pointer.fromFunction` or `NativeCallable.isolateLocal` has an
`exceptionalReturn` argument whose value isn't a constant expression.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the value of the
`isLeaf` argument is a parameter, and hence isn't a constant:

```dart
import 'dart:ffi';

int Function(int) fromPointer(
    Pointer<NativeFunction<Int8 Function(Int8)>> p, bool isLeaf) {
  return p.asFunction(isLeaf: [!isLeaf!]);
}
```

#### Common fixes

If there's a suitable constant that can be used, then replace the argument
with a constant:

```dart
import 'dart:ffi';

const isLeaf = false;

int Function(int) fromPointer(Pointer<NativeFunction<Int8 Function(Int8)>> p) {
  return p.asFunction(isLeaf: isLeaf);
}
```

If there isn't a suitable constant, then replace the argument with a
boolean literal:

```dart
import 'dart:ffi';

int Function(int) fromPointer(Pointer<NativeFunction<Int8 Function(Int8)>> p) {
  return p.asFunction(isLeaf: true);
}
```

### argument_must_be_native

_Argument to 'Native.addressOf' must be annotated with @Native_

#### Description

The analyzer produces this diagnostic when the argument passed to
`Native.addressOf` isn't annotated with the `Native` annotation.

#### Examples

The following code produces this diagnostic because the argument to
`addressOf` is a string, not a field, and strings can't be annotated:

```dart
import 'dart:ffi';

@Native<Void Function()>()
external void f();

void g() {
  print(Native.addressOf([!'f'!]));
}
```

The following code produces this diagnostic because the function `f` is
being passed to `addressOf` but isn't annotated as being `Native`:

```dart
import 'dart:ffi';

external void f();

void g() {
  print(Native.addressOf<NativeFunction<Void Function()>>([!f!]));
}
```

#### Common fixes

If the argument isn't either a field or a function, then replace the
argument with a field or function that's annotated with `Native`:

```dart
import 'dart:ffi';

@Native<Void Function()>()
external void f();

void g() {
  print(Native.addressOf<NativeFunction<Void Function()>>(f));
}
```

If the argument is either a field or a function, then annotate the field
of function with `Native`:

```dart
import 'dart:ffi';

@Native<Void Function()>()
external void f();

void g() {
  print(Native.addressOf<NativeFunction<Void Function()>>(f));
}
```

### argument_type_not_assignable

_The argument type '{0}' can't be assigned to the parameter type '{1}'. {2}_

#### Description

The analyzer produces this diagnostic when the static type of an argument
can't be assigned to the static type of the corresponding parameter.

#### Example

The following code produces this diagnostic because a `num` can't be
assigned to a `String`:

```dart
String f(String x) => x;
String g(num y) => f([!y!]);
```

#### Common fixes

If possible, rewrite the code so that the static type is assignable. In the
example above you might be able to change the type of the parameter `y`:

```dart
String f(String x) => x;
String g(String y) => f(y);
```

If that fix isn't possible, then add code to handle the case where the
argument value isn't the required type. One approach is to coerce other
types to the required type:

```dart
String f(String x) => x;
String g(num y) => f(y.toString());
```

Another approach is to add explicit type tests and fallback code:

```dart
String f(String x) => x;
String g(Object y) => f(y is String ? y : '');
```

If you believe that the runtime type of the argument will always be the
same as the static type of the parameter, and you're willing to risk having
an exception thrown at runtime if you're wrong, then add an explicit cast:

```dart
String f(String x) => x;
String g(num y) => f(y as String);
```

### argument_type_not_assignable_to_error_handler

_The argument type '{0}' can't be assigned to the parameter type '{1}
Function(Object)' or '{1} Function(Object, StackTrace)'._

#### Description

The analyzer produces this diagnostic when an invocation of
`Future.catchError` has an argument that is a function whose parameters
aren't compatible with the arguments that will be passed to the function
when it's invoked. The static type of the first argument to `catchError`
is just `Function`, even though the function that is passed in is expected
to have either a single parameter of type `Object` or two parameters of
type `Object` and `StackTrace`.

#### Examples

The following code produces this diagnostic because the closure being
passed to `catchError` doesn't take any parameters, but the function is
required to take at least one parameter:

```dart
void f(Future<int> f) {
  f.catchError([!() => 0!]);
}
```

The following code produces this diagnostic because the closure being
passed to `catchError` takes three parameters, but it can't have more than
two required parameters:

```dart
void f(Future<int> f) {
  f.catchError([!(one, two, three) => 0!]);
}
```

The following code produces this diagnostic because even though the closure
being passed to `catchError` takes one parameter, the closure doesn't have
a type that is compatible with `Object`:

```dart
void f(Future<int> f) {
  f.catchError([!(String error) => 0!]);
}
```

#### Common fixes

Change the function being passed to `catchError` so that it has either one
or two required parameters, and the parameters have the required types:

```dart
void f(Future<int> f) {
  f.catchError((Object error) => 0);
}
```

### assert_in_redirecting_constructor

_A redirecting constructor can't have an 'assert' initializer._

#### Description

The analyzer produces this diagnostic when a redirecting constructor (a
constructor that redirects to another constructor in the same class) has an
assert in the initializer list.

#### Example

The following code produces this diagnostic because the unnamed constructor
is a redirecting constructor and also has an assert in the initializer
list:

```dart
class C {
  C(int x) : [!assert(x > 0)!], this.name();
  C.name() {}
}
```

#### Common fixes

If the assert isn't needed, then remove it:

```dart
class C {
  C(int x) : this.name();
  C.name() {}
}
```

If the assert is needed, then convert the constructor into a factory
constructor:

```dart
class C {
  factory C(int x) {
    assert(x > 0);
    return C.name();
  }
  C.name() {}
}
```

### asset_directory_does_not_exist

_The asset directory '{0}' doesn't exist._

#### Description

The analyzer produces this diagnostic when an asset list contains a value
referencing a directory that doesn't exist.

#### Example

Assuming that the directory `assets` doesn't exist, the following code
produces this diagnostic because it's listed as a directory containing
assets:

```yaml
name: example
flutter:
  assets:
    - [!assets/!]
```

#### Common fixes

If the path is correct, then create a directory at that path.

If the path isn't correct, then change the path to match the path of the
directory containing the assets.

### asset_does_not_exist

_The asset file '{0}' doesn't exist._

#### Description

The analyzer produces this diagnostic when an asset list contains a value
referencing a file that doesn't exist.

#### Example

Assuming that the file `doesNotExist.gif` doesn't exist, the following code
produces this diagnostic because it's listed as an asset:

```yaml
name: example
flutter:
  assets:
    - [!doesNotExist.gif!]
```

#### Common fixes

If the path is correct, then create a file at that path.

If the path isn't correct, then change the path to match the path of the
file containing the asset.

### asset_field_not_list

_The value of the 'assets' field is expected to be a list of relative file
paths._

#### Description

The analyzer produces this diagnostic when the value of the `assets` key
isn't a list.

#### Example

The following code produces this diagnostic because the value of the
`assets` key is a string when a list is expected:

```yaml
name: example
flutter:
  assets: [!assets/!]
```

#### Common fixes

Change the value of the asset list so that it's a list:

```yaml
name: example
flutter:
  assets:
    - assets/
```

### asset_missing_path

_Asset map entry must contain a 'path' field._

#### Description

The analyzer produces this diagnostic when an asset map is missing a
`path` value.

#### Example

The following code produces this diagnostic because the asset map
is missing a `path` value:

```yaml
name: example
flutter:
  assets:
    - flavors:
      - premium
```

#### Common fixes

Change the asset map so that it contains a `path` field with a string
value (a valid POSIX-style file path):

```yaml
name: example
flutter:
  assets:
    - path: assets/image.gif
      flavors:
      - premium
```

### asset_not_string

_Assets are required to be file paths (strings)._

#### Description

The analyzer produces this diagnostic when an `assets` list contains a
value that isn't a string.

#### Example

The following code produces this diagnostic because the `assets` list
contains a map:

```yaml
name: example
flutter:
  assets:
    - [!image.gif: true!]
```

#### Common fixes

Change the `assets` list so that it only contains valid POSIX-style file
paths:

```yaml
name: example
flutter:
  assets:
    - assets/image.gif
```

### asset_not_string_or_map

_An asset value is required to be a file path (string) or map._

#### Description

The analyzer produces this diagnostic when an asset value isn't a string
or a map.

#### Example

The following code produces this diagnostic because the asset value
is a list:

```yaml
name: example
flutter:
  assets:
    - [![one, two, three]!]
```

#### Common fixes

If you need to specify more than just the path to the asset, then replace
the value with a map with a `path` key (a valid POSIX-style file path):

```yaml
name: example
flutter:
  assets:
    - path: assets/image.gif
      flavors:
      - premium
```

If you only need to specify the path, then replace the value with the path
to the asset (a valid POSIX-style file path):

```yaml
name: example
flutter:
  assets:
    - assets/image.gif
```

### asset_path_not_string

_Asset paths are required to be file paths (strings)._

#### Description

The analyzer produces this diagnostic when an asset map contains a
`path` value that isn't a string.

#### Example

The following code produces this diagnostic because the asset map
contains a `path` value which is a list:

```yaml
name: example
flutter:
  assets:
    - path: [![one, two, three]!]
      flavors:
      - premium
```

#### Common fixes

Change the `asset` map so that it contains a `path` value which is a
string (a valid POSIX-style file path):

```yaml
name: example
flutter:
  assets:
    - path: image.gif
      flavors:
      - premium
```

### assignment_of_do_not_store

_'{0}' is marked 'doNotStore' and shouldn't be assigned to a field or top-level
variable._

#### Description

The analyzer produces this diagnostic when the value of a function
(including methods and getters) that is explicitly or implicitly marked by
the [`doNotStore`][meta-doNotStore] annotation is stored in either a field
or top-level variable.

#### Example

The following code produces this diagnostic because the value of the
function `f` is being stored in the top-level variable `x`:

```dart
import 'package:meta/meta.dart';

@doNotStore
int f() => 1;

var x = [!f()!];
```

#### Common fixes

Replace references to the field or variable with invocations of the
function producing the value.

### assignment_to_const

_Constant variables can't be assigned a value._

#### Description

The analyzer produces this diagnostic when it finds an assignment to a
top-level variable, a static field, or a local variable that has the
`const` modifier. The value of a compile-time constant can't be changed at
runtime.

#### Example

The following code produces this diagnostic because `c` is being assigned a
value even though it has the `const` modifier:

```dart
const c = 0;

void f() {
  [!c!] = 1;
  print(c);
}
```

#### Common fixes

If the variable must be assignable, then remove the `const` modifier:

```dart
var c = 0;

void f() {
  c = 1;
  print(c);
}
```

If the constant shouldn't be changed, then either remove the assignment or
use a local variable in place of references to the constant:

```dart
const c = 0;

void f() {
  var v = 1;
  print(v);
}
```

### assignment_to_final

_'{0}' can't be used as a setter because it's final._

#### Description

The analyzer produces this diagnostic when it finds an invocation of a
setter, but there's no setter because the field with the same name was
declared to be `final` or `const`.

#### Example

The following code produces this diagnostic because `v` is final:

```dart
class C {
  final v = 0;
}

f(C c) {
  c.[!v!] = 1;
}
```

#### Common fixes

If you need to be able to set the value of the field, then remove the
modifier `final` from the field:

```dart
class C {
  int v = 0;
}

f(C c) {
  c.v = 1;
}
```

### assignment_to_final_local

_The final variable '{0}' can only be set once._

#### Description

The analyzer produces this diagnostic when a local variable that was
declared to be final is assigned after it was initialized.

#### Example

The following code produces this diagnostic because `x` is final, so it
can't have a value assigned to it after it was initialized:

```dart
void f() {
  final x = 0;
  [!x!] = 3;
  print(x);
}
```

#### Common fixes

Remove the keyword `final`, and replace it with `var` if there's no type
annotation:

```dart
void f() {
  var x = 0;
  x = 3;
  print(x);
}
```

### assignment_to_final_no_setter

_There isn't a setter named '{0}' in class '{1}'._

#### Description

The analyzer produces this diagnostic when a reference to a setter is
found; there is no setter defined for the type; but there is a getter
defined with the same name.

#### Example

The following code produces this diagnostic because there is no setter
named `x` in `C`, but there is a getter named `x`:

```dart
class C {
  int get x => 0;
  set y(int p) {}
}

void f(C c) {
  c.[!x!] = 1;
}
```

#### Common fixes

If you want to invoke an existing setter, then correct the name:

```dart
class C {
  int get x => 0;
  set y(int p) {}
}

void f(C c) {
  c.y = 1;
}
```

If you want to invoke the setter but it just doesn't exist yet, then
declare it:

```dart
class C {
  int get x => 0;
  set x(int p) {}
  set y(int p) {}
}

void f(C c) {
  c.x = 1;
}
```

### assignment_to_function

_Functions can't be assigned a value._

#### Description

The analyzer produces this diagnostic when the name of a function appears
on the left-hand side of an assignment expression.

#### Example

The following code produces this diagnostic because the assignment to the
function `f` is invalid:

```dart
void f() {}

void g() {
  [!f!] = () {};
}
```

#### Common fixes

If the right-hand side should be assigned to something else, such as a
local variable, then change the left-hand side:

```dart
void f() {}

void g() {
  var x = () {};
  print(x);
}
```

If the intent is to change the implementation of the function, then define
a function-valued variable instead of a function:

```dart
void Function() f = () {};

void g() {
  f = () {};
}
```

### assignment_to_method

_Methods can't be assigned a value._

#### Description

The analyzer produces this diagnostic when the target of an assignment is a
method.

#### Example

The following code produces this diagnostic because `f` can't be assigned a
value because it's a method:

```dart
class C {
  void f() {}

  void g() {
    [!f!] = null;
  }
}
```

#### Common fixes

Rewrite the code so that there isn't an assignment to a method.

### assignment_to_type

_Types can't be assigned a value._

#### Description

The analyzer produces this diagnostic when the name of a type name appears
on the left-hand side of an assignment expression.

#### Example

The following code produces this diagnostic because the assignment to the
class `C` is invalid:

```dart
class C {}

void f() {
  [!C!] = null;
}
```

#### Common fixes

If the right-hand side should be assigned to something else, such as a
local variable, then change the left-hand side:

```dart
void f() {}

void g() {
  var c = null;
  print(c);
}
```

### async_for_in_wrong_context

_The async for-in loop can only be used in an async function._

#### Description

The analyzer produces this diagnostic when an async for-in loop is found in
a function or method whose body isn't marked as being either `async` or
`async*`.

#### Example

The following code produces this diagnostic because the body of `f` isn't
marked as being either `async` or `async*`, but `f` contains an async
for-in loop:

```dart
void f(list) {
  [!await!] for (var e in list) {
    print(e);
  }
}
```

#### Common fixes

If the function should return a `Future`, then mark the body with `async`:

```dart
Future<void> f(list) async {
  await for (var e in list) {
    print(e);
  }
}
```

If the function should return a `Stream` of values, then mark the body with
`async*`:

```dart
Stream<void> f(list) async* {
  await for (var e in list) {
    print(e);
  }
}
```

If the function should be synchronous, then remove the `await` before the
loop:

```dart
void f(list) {
  for (var e in list) {
    print(e);
  }
}
```

### await_in_late_local_variable_initializer

_The 'await' expression can't be used in a 'late' local variable's initializer._

#### Description

The analyzer produces this diagnostic when a local variable that has the
`late` modifier uses an `await` expression in the initializer.

#### Example

The following code produces this diagnostic because an `await` expression
is used in the initializer for `v`, a local variable that is marked `late`:

```dart
Future<int> f() async {
  late var v = [!await!] 42;
  return v;
}
```

#### Common fixes

If the initializer can be rewritten to not use `await`, then rewrite it:

```dart
Future<int> f() async {
  late var v = 42;
  return v;
}
```

If the initializer can't be rewritten, then remove the `late` modifier:

```dart
Future<int> f() async {
  var v = await 42;
  return v;
}
```

### await_of_incompatible_type

_The 'await' expression can't be used for an expression with an extension type
that is not a subtype of 'Future'._

#### Description

The analyzer produces this diagnostic when the type of the expression in
an `await` expression is an extension type, and the extension type isn't a
subclass of `Future`.

#### Example

The following code produces this diagnostic because the extension type `E`
isn't a subclass of `Future`:

```dart
extension type E(int i) {}

void f(E e) async {
  [!await!] e;
}
```

#### Common fixes

If the extension type is correctly defined, then remove the `await`:

```dart
extension type E(int i) {}

void f(E e) {
  e;
}
```

If the extension type is intended to be awaitable, then add `Future` (or a
subtype of `Future`) to the `implements` clause (adding an `implements`
clause if there isn't one already), and make the representation type
match:

```dart
extension type E(Future<int> i) implements Future<int> {}

void f(E e) async {
  await e;
}
```

### body_might_complete_normally

_The body might complete normally, causing 'null' to be returned, but the return
type, '{0}', is a potentially non-nullable type._

#### Description

The analyzer produces this diagnostic when a method or function has a
return type that's [potentially non-nullable][] but would implicitly return
`null` if control reached the end of the function.

#### Examples

The following code produces this diagnostic because the method `m` has an
implicit return of `null` inserted at the end of the method, but the method
is declared to not return `null`:

```dart
class C {
  int [!m!](int t) {
    print(t);
  }
}
```

The following code produces this diagnostic because the method `m` has an
implicit return of `null` inserted at the end of the method, but because
the class `C` can be instantiated with a non-nullable type argument, the
method is effectively declared to not return `null`:

```dart
class C<T> {
  T [!m!](T t) {
    print(t);
  }
}
```

#### Common fixes

If there's a reasonable value that can be returned, then add a `return`
statement at the end of the method:

```dart
class C<T> {
  T m(T t) {
    print(t);
    return t;
  }
}
```

If the method won't reach the implicit return, then add a `throw` at the
end of the method:

```dart
class C<T> {
  T m(T t) {
    print(t);
    throw '';
  }
}
```

If the method intentionally returns `null` at the end, then add an
explicit return of `null` at the end of the method and change the
return type so that it's valid to return `null`:

```dart
class C<T> {
  T? m(T t) {
    print(t);
    return null;
  }
}
```

### body_might_complete_normally_catch_error

_This 'onError' handler must return a value assignable to '{0}', but ends
without returning a value._

#### Description

The analyzer produces this diagnostic when the closure passed to the
`onError` parameter of the `Future.catchError` method is required to
return a non-`null` value (because of the `Future`s type argument) but can
implicitly return `null`.

#### Example

The following code produces this diagnostic because the closure passed to
the `catchError` method is required to return an `int` but doesn't end
with an explicit `return`, causing it to implicitly return `null`:

```dart
void g(Future<int> f) {
  f.catchError((e, st) [!{!]});
}
```

#### Common fixes

If the closure should sometimes return a non-`null` value, then add an
explicit return to the closure:

```dart
void g(Future<int> f) {
  f.catchError((e, st) {
    return -1;
  });
}
```

If the closure should always return `null`, then change the type argument
of the `Future` to be either `void` or `Null`:

```dart
void g(Future<void> f) {
  f.catchError((e, st) {});
}
```

### body_might_complete_normally_nullable

_This function has a nullable return type of '{0}', but ends without returning a
value._

#### Description

The analyzer produces this diagnostic when a method or function can
implicitly return `null` by falling off the end. While this is valid Dart
code, it's better for the return of `null` to be explicit.

#### Example

The following code produces this diagnostic because the function `f`
implicitly returns `null`:

```dart
String? [!f!]() {}
```

#### Common fixes

If the return of `null` is intentional, then make it explicit:

```dart
String? f() {
  return null;
}
```

If the function should return a non-null value along that path, then add
the missing return statement:

```dart
String? f() {
  return '';
}
```

### break_label_on_switch_member

_A break label resolves to the 'case' or 'default' statement._

#### Description

The analyzer produces this diagnostic when a break in a case clause inside
a switch statement has a label that is associated with another case clause.

#### Example

The following code produces this diagnostic because the label `l` is
associated with the case clause for `0`:

```dart
void f(int i) {
  switch (i) {
    l: case 0:
      break;
    case 1:
      break [!l!];
  }
}
```

#### Common fixes

If the intent is to transfer control to the statement after the switch,
then remove the label from the break statement:

```dart
void f(int i) {
  switch (i) {
    case 0:
      break;
    case 1:
      break;
  }
}
```

If the intent is to transfer control to a different case block, then use
`continue` rather than `break`:

```dart
void f(int i) {
  switch (i) {
    l: case 0:
      break;
    case 1:
      continue l;
  }
}
```

### built_in_identifier_as_type

_The built-in identifier '{0}' can't be used as a type._

#### Description

The analyzer produces this diagnostic when a built-in identifier is used
where a type name is expected.

#### Example

The following code produces this diagnostic because `import` can't be used
as a type because it's a built-in identifier:

```dart
[!import!]<int> x;
```

#### Common fixes

Replace the built-in identifier with the name of a valid type:

```dart
List<int> x;
```

### built_in_identifier_in_declaration

_The built-in identifier '{0}' can't be used as a prefix name._

_The built-in identifier '{0}' can't be used as a type name._

_The built-in identifier '{0}' can't be used as a type parameter name._

_The built-in identifier '{0}' can't be used as a typedef name._

_The built-in identifier '{0}' can't be used as an extension name._

_The built-in identifier '{0}' can't be used as an extension type name._

#### Description

The analyzer produces this diagnostic when the name used in the declaration
of a class, extension, mixin, typedef, type parameter, or import prefix is
a built-in identifier. Built-in identifiers can't be used to name any of
these kinds of declarations.

#### Example

The following code produces this diagnostic because `mixin` is a built-in
identifier:

```dart
extension [!mixin!] on int {}
```

#### Common fixes

Choose a different name for the declaration.

### case_block_not_terminated

_The last statement of the 'case' should be 'break', 'continue', 'rethrow',
'return', or 'throw'._

#### Description

The analyzer produces this diagnostic when the last statement in a `case`
block isn't one of the required terminators: `break`, `continue`,
`rethrow`, `return`, or `throw`.

#### Example

The following code produces this diagnostic because the `case` block ends
with an assignment:

```dart
void f(int x) {
  switch (x) {
    [!case!] 0:
      x += 2;
    default:
      x += 1;
  }
}
```

#### Common fixes

Add one of the required terminators:

```dart
void f(int x) {
  switch (x) {
    case 0:
      x += 2;
      break;
    default:
      x += 1;
  }
}
```

### case_expression_type_implements_equals

_The switch case expression type '{0}' can't override the '==' operator._

#### Description

The analyzer produces this diagnostic when the type of the expression
following the keyword `case` has an implementation of the `==` operator
other than the one in `Object`.

#### Example

The following code produces this diagnostic because the expression
following the keyword `case` (`C(0)`) has the type `C`, and the class `C`
overrides the `==` operator:

```dart
class C {
  final int value;

  const C(this.value);

  bool operator ==(Object other) {
    return false;
  }
}

void f(C c) {
  switch (c) {
    case [!C(0)!]:
      break;
  }
}
```

#### Common fixes

If there isn't a strong reason not to do so, then rewrite the code to use
an if-else structure:

```dart
class C {
  final int value;

  const C(this.value);

  bool operator ==(Object other) {
    return false;
  }
}

void f(C c) {
  if (c == C(0)) {
    // ...
  }
}
```

If you can't rewrite the switch statement and the implementation of `==`
isn't necessary, then remove it:

```dart
class C {
  final int value;

  const C(this.value);
}

void f(C c) {
  switch (c) {
    case C(0):
      break;
  }
}
```

If you can't rewrite the switch statement and you can't remove the
definition of `==`, then find some other value that can be used to control
the switch:

```dart
class C {
  final int value;

  const C(this.value);

  bool operator ==(Object other) {
    return false;
  }
}

void f(C c) {
  switch (c.value) {
    case 0:
      break;
  }
}
```

### case_expression_type_is_not_switch_expression_subtype

_The switch case expression type '{0}' must be a subtype of the switch
expression type '{1}'._

#### Description

The analyzer produces this diagnostic when the expression following `case`
in a `switch` statement has a static type that isn't a subtype of the
static type of the expression following `switch`.

#### Example

The following code produces this diagnostic because `1` is an `int`, which
isn't a subtype of `String` (the type of `s`):

```dart
void f(String s) {
  switch (s) {
    case [!1!]:
      break;
  }
}
```

#### Common fixes

If the value of the `case` expression is wrong, then change the `case`
expression so that it has the required type:

```dart
void f(String s) {
  switch (s) {
    case '1':
      break;
  }
}
```

If the value of the `case` expression is correct, then change the `switch`
expression to have the required type:

```dart
void f(int s) {
  switch (s) {
    case 1:
      break;
  }
}
```

### cast_from_nullable_always_fails

_This cast will always throw an exception because the nullable local variable
'{0}' is not assigned._

#### Description

The analyzer produces this diagnostic when a local variable that has a
nullable type hasn't been assigned and is cast to a non-nullable type.
Because the variable hasn't been assigned it has the default value of
`null`, causing the cast to throw an exception.

#### Example

The following code produces this diagnostic because the variable `x` is
cast to a non-nullable type (`int`) when it's known to have the value
`null`:

```dart
void f() {
  num? x;
  [!x!] as int;
  print(x);
}
```

#### Common fixes

If the variable is expected to have a value before the cast, then add an
initializer or an assignment:

```dart
void f() {
  num? x = 3;
  x as int;
  print(x);
}
```

If the variable isn't expected to be assigned, then remove the cast:

```dart
void f() {
  num? x;
  print(x);
}
```

### cast_from_null_always_fails

_This cast always throws an exception because the expression always evaluates to
'null'._

#### Description

The analyzer produces this diagnostic when an expression whose type is
`Null` is being cast to a non-nullable type.

#### Example

The following code produces this diagnostic because `n` is known to always
be `null`, but it's being cast to a non-nullable type:

```dart
void f(Null n) {
  [!n as int!];
}
```

#### Common fixes

Remove the unnecessary cast:

```dart
void f(Null n) {
  n;
}
```

### cast_to_non_type

_The name '{0}' isn't a type, so it can't be used in an 'as' expression._

#### Description

The analyzer produces this diagnostic when the name following the `as` in a
cast expression is defined to be something other than a type.

#### Example

The following code produces this diagnostic because `x` is a variable, not
a type:

```dart
num x = 0;
int y = x as [!x!];
```

#### Common fixes

Replace the name with the name of a type:

```dart
num x = 0;
int y = x as int;
```

### class_used_as_mixin

_The class '{0}' can't be used as a mixin because it's neither a mixin class nor
a mixin._

#### Description

The analyzer produces this diagnostic when a class that is neither a
`mixin class` nor a `mixin` is used in a `with` clause.

#### Example

The following code produces this diagnostic because the class `M` is being
used as a mixin, but it isn't defined as a `mixin class`:

```dart
class M {}
class C with [!M!] {}
```

#### Common fixes

If the class can be a pure mixin, then change `class` to `mixin`:

```dart
mixin M {}
class C with M {}
```

If the class needs to be both a class and a mixin, then add `mixin`:

```dart
mixin class M {}
class C with M {}
```

### collection_element_from_deferred_library

_Constant values from a deferred library can't be used as keys in a 'const' map
literal._

_Constant values from a deferred library can't be used as values in a 'const'
constructor._

_Constant values from a deferred library can't be used as values in a 'const'
list literal._

_Constant values from a deferred library can't be used as values in a 'const'
map literal._

_Constant values from a deferred library can't be used as values in a 'const'
set literal._

#### Description

The analyzer produces this diagnostic when a collection literal that is
either explicitly (because it's prefixed by the `const` keyword) or
implicitly (because it appears in a [constant context][]) a constant
contains a value that is declared in a library that is imported using a
deferred import. Constants are evaluated at compile time, and values from
deferred libraries aren't available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

Given a file `a.dart` that defines the constant `zero`:

```dart
const zero = 0;
```

The following code produces this diagnostic because the constant list
literal contains `a.zero`, which is imported using a `deferred` import:

```dart
import 'a.dart' deferred as a;

var l = const [a.[!zero!]];
```

#### Common fixes

If the collection literal isn't required to be constant, then remove the
`const` keyword:

```dart
import 'a.dart' deferred as a;

var l = [a.zero];
```

If the collection is required to be constant and the imported constant must
be referenced, then remove the keyword `deferred` from the import:

```dart
import 'a.dart' as a;

var l = const [a.zero];
```

If you don't need to reference the constant, then replace it with a
suitable value:

```dart
var l = const [0];
```

### compound_implements_finalizable

_The class '{0}' can't implement Finalizable._

#### Description

The analyzer produces this diagnostic when a subclass of either `Struct`
or `Union` implements `Finalizable`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the class `S`
implements `Finalizable`:

```dart
import 'dart:ffi';

final class [!S!] extends Struct implements Finalizable {
  external Pointer notEmpty;
}
```

#### Common fixes

Try removing the implements clause from the class:

```dart
import 'dart:ffi';

final class S extends Struct {
  external Pointer notEmpty;
}
```

### concrete_class_has_enum_superinterface

_Concrete classes can't have 'Enum' as a superinterface._

#### Description

The analyzer produces this diagnostic when a concrete class indirectly has
the class `Enum` as a superinterface.

#### Example

The following code produces this diagnostic because the concrete class `B`
has `Enum` as a superinterface as a result of implementing `A`:

```dart
abstract class A implements Enum {}

class [!B!] implements A {}
```

#### Common fixes

If the implemented class isn't the class you intend to implement, then
change it:

```dart
abstract class A implements Enum {}

class B implements C {}

class C {}
```

If the implemented class can be changed to not implement `Enum`, then do
so:

```dart
abstract class A {}

class B implements A {}
```

If the implemented class can't be changed to not implement `Enum`, then
remove it from the `implements` clause:

```dart
abstract class A implements Enum {}

class B {}
```

### concrete_class_with_abstract_member

_'{0}' must have a method body because '{1}' isn't abstract._

#### Description

The analyzer produces this diagnostic when a member of a concrete class is
found that doesn't have a concrete implementation. Concrete classes aren't
allowed to contain abstract members.

#### Example

The following code produces this diagnostic because `m` is an abstract
method but `C` isn't an abstract class:

```dart
class C {
  [!void m();!]
}
```

#### Common fixes

If it's valid to create instances of the class, provide an implementation
for the member:

```dart
class C {
  void m() {}
}
```

If it isn't valid to create instances of the class, mark the class as being
abstract:

```dart
abstract class C {
  void m();
}
```

### conflicting_constructor_and_static_member

_'{0}' can't be used to name both a constructor and a static field in this
class._

_'{0}' can't be used to name both a constructor and a static getter in this
class._

_'{0}' can't be used to name both a constructor and a static method in this
class._

_'{0}' can't be used to name both a constructor and a static setter in this
class._

#### Description

The analyzer produces this diagnostic when a named constructor and either a
static method or static field have the same name. Both are accessed using
the name of the class, so having the same name makes the reference
ambiguous.

#### Examples

The following code produces this diagnostic because the static field `foo`
and the named constructor `foo` have the same name:

```dart
class C {
  C.[!foo!]();
  static int foo = 0;
}
```

The following code produces this diagnostic because the static method `foo`
and the named constructor `foo` have the same name:

```dart
class C {
  C.[!foo!]();
  static void foo() {}
}
```

#### Common fixes

Rename either the member or the constructor.

### conflicting_generic_interfaces

_The {0} '{1}' can't implement both '{2}' and '{3}' because the type arguments
are different._

#### Description

The analyzer produces this diagnostic when a class attempts to implement a
generic interface multiple times, and the values of the type arguments
aren't the same.

#### Example

The following code produces this diagnostic because `C` is defined to
implement both `I<int>` (because it extends `A`) and `I<String>` (because
it implements`B`), but `int` and `String` aren't the same type:

```dart
class I<T> {}
class A implements I<int> {}
class B implements I<String> {}
class [!C!] extends A implements B {}
```

#### Common fixes

Rework the type hierarchy to avoid this situation. For example, you might
make one or both of the inherited types generic so that `C` can specify the
same type for both type arguments:

```dart
class I<T> {}
class A<S> implements I<S> {}
class B implements I<String> {}
class C extends A<String> implements B {}
```

### conflicting_type_variable_and_container

_'{0}' can't be used to name both a type variable and the class in which the
type variable is defined._

_'{0}' can't be used to name both a type variable and the enum in which the type
variable is defined._

_'{0}' can't be used to name both a type variable and the extension in which the
type variable is defined._

_'{0}' can't be used to name both a type variable and the extension type in
which the type variable is defined._

_'{0}' can't be used to name both a type variable and the mixin in which the
type variable is defined._

#### Description

The analyzer produces this diagnostic when a class, mixin, or extension
declaration declares a type parameter with the same name as the class,
mixin, or extension that declares it.

#### Example

The following code produces this diagnostic because the type parameter `C`
has the same name as the class `C` of which it's a part:

```dart
class C<[!C!]> {}
```

#### Common fixes

Rename either the type parameter, or the class, mixin, or extension:

```dart
class C<T> {}
```

### conflicting_type_variable_and_member

_'{0}' can't be used to name both a type variable and a member in this class._

_'{0}' can't be used to name both a type variable and a member in this enum._

_'{0}' can't be used to name both a type variable and a member in this extension
type._

_'{0}' can't be used to name both a type variable and a member in this
extension._

_'{0}' can't be used to name both a type variable and a member in this mixin._

#### Description

The analyzer produces this diagnostic when a class, mixin, or extension
declaration declares a type parameter with the same name as one of the
members of the class, mixin, or extension that declares it.

#### Example

The following code produces this diagnostic because the type parameter `T`
has the same name as the field `T`:

```dart
class C<[!T!]> {
  int T = 0;
}
```

#### Common fixes

Rename either the type parameter or the member with which it conflicts:

```dart
class C<T> {
  int total = 0;
}
```

### constant_pattern_never_matches_value_type

_The matched value type '{0}' can never be equal to this constant of type
'{1}'._

#### Description

The analyzer produces this diagnostic when a constant pattern can never
match the value it's being tested against because the type of the constant
is known to never match the type of the value.

#### Example

The following code produces this diagnostic because the type of the
constant pattern `(true)` is `bool`, and the type of the value being
matched (`x`) is `int`, and a Boolean can never match an integer:

```dart
void f(int x) {
  if (x case [!true!]) {}
}
```

#### Common fixes

If the type of the value is correct, then rewrite the pattern to be
compatible:

```dart
void f(int x) {
  if (x case 3) {}
}
```

If the type of the constant is correct, then rewrite the value to be
compatible:

```dart
void f(bool x) {
  if (x case true) {}
}
```

### constant_pattern_with_non_constant_expression

_The expression of a constant pattern must be a valid constant._

#### Description

The analyzer produces this diagnostic when a constant pattern has an
expression that isn't a valid constant.

#### Example

The following code produces this diagnostic because the constant pattern
`i` isn't a constant:

```dart
void f(int e, int i) {
  switch (e) {
    case [!i!]:
      break;
  }
}
```

#### Common fixes

If the value that should be matched is known, then replace the expression
with a constant:

```dart
void f(int e, int i) {
  switch (e) {
    case 0:
      break;
  }
}
```

If the value that should be matched isn't known, then rewrite the code to
not use a pattern:

```dart
void f(int e, int i) {
  if (e == i) {}
}
```

### const_constructor_param_type_mismatch

_A value of type '{0}' can't be assigned to a parameter of type '{1}' in a const
constructor._

#### Description

The analyzer produces this diagnostic when the runtime type of a constant
value can't be assigned to the static type of a constant constructor's
parameter.

#### Example

The following code produces this diagnostic because the runtime type of `i`
is `int`, which can't be assigned to the static type of `s`:

```dart
class C {
  final String s;

  const C(this.s);
}

const dynamic i = 0;

void f() {
  const C([!i!]);
}
```

#### Common fixes

Pass a value of the correct type to the constructor:

```dart
class C {
  final String s;

  const C(this.s);
}

const dynamic i = 0;

void f() {
  const C('$i');
}
```

### const_constructor_with_field_initialized_by_non_const

_Can't define the 'const' constructor because the field '{0}' is initialized
with a non-constant value._

#### Description

The analyzer produces this diagnostic when a constructor has the keyword
`const`, but a field in the class is initialized to a non-constant value.

#### Example

The following code produces this diagnostic because the field `s` is
initialized to a non-constant value:

```dart
String x = '3';
class C {
  final String s = x;
  [!const!] C();
}
```

#### Common fixes

If the field can be initialized to a constant value, then change the
initializer to a constant expression:

```dart
class C {
  final String s = '3';
  const C();
}
```

If the field can't be initialized to a constant value, then remove the
keyword `const` from the constructor:

```dart
String x = '3';
class C {
  final String s = x;
  C();
}
```

### const_constructor_with_non_const_super

_A constant constructor can't call a non-constant super constructor of '{0}'._

#### Description

The analyzer produces this diagnostic when a constructor that is marked as
`const` invokes a constructor from its superclass that isn't marked as
`const`.

#### Example

The following code produces this diagnostic because the `const` constructor
in `B` invokes the constructor `nonConst` from the class `A`, and the
superclass constructor isn't a `const` constructor:

```dart
class A {
  const A();
  A.nonConst();
}

class B extends A {
  const B() : [!super.nonConst()!];
}
```

#### Common fixes

If it isn't essential to invoke the superclass constructor that is
currently being invoked, then invoke a constant constructor from the
superclass:

```dart
class A {
  const A();
  A.nonConst();
}

class B extends A {
  const B() : super();
}
```

If it's essential that the current constructor be invoked and if you can
modify it, then add `const` to the constructor in the superclass:

```dart
class A {
  const A();
  const A.nonConst();
}

class B extends A {
  const B() : super.nonConst();
}
```

If it's essential that the current constructor be invoked and you can't
modify it, then remove `const` from the constructor in the subclass:

```dart
class A {
  const A();
  A.nonConst();
}

class B extends A {
  B() : super.nonConst();
}
```

### const_constructor_with_non_final_field

_Can't define a const constructor for a class with non-final fields._

#### Description

The analyzer produces this diagnostic when a constructor is marked as a
const constructor, but the constructor is defined in a class that has at
least one non-final instance field (either directly or by inheritance).

#### Example

The following code produces this diagnostic because the field `x` isn't
final:

```dart
class C {
  int x;

  const [!C!](this.x);
}
```

#### Common fixes

If it's possible to mark all of the fields as final, then do so:

```dart
class C {
  final int x;

  const C(this.x);
}
```

If it isn't possible to mark all of the fields as final, then remove the
keyword `const` from the constructor:

```dart
class C {
  int x;

  C(this.x);
}
```

### const_deferred_class

_Deferred classes can't be created with 'const'._

#### Description

The analyzer produces this diagnostic when a class from a library that is
imported using a deferred import is used to create a `const` object.
Constants are evaluated at compile time, and classes from deferred
libraries aren't available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

The following code produces this diagnostic because it attempts to create a
`const` instance of a class from a deferred library:

```dart
import 'dart:convert' deferred as convert;

const json2 = [!convert.JsonCodec()!];
```

#### Common fixes

If the object isn't required to be a constant, then change the code so that
a non-constant instance is created:

```dart
import 'dart:convert' deferred as convert;

final json2 = convert.JsonCodec();
```

If the object must be a constant, then remove `deferred` from the import
directive:

```dart
import 'dart:convert' as convert;

const json2 = convert.JsonCodec();
```

### const_initialized_with_non_constant_value

_Const variables must be initialized with a constant value._

#### Description

The analyzer produces this diagnostic when a value that isn't statically
known to be a constant is assigned to a variable that's declared to be a
`const` variable.

#### Example

The following code produces this diagnostic because `x` isn't declared to
be `const`:

```dart
var x = 0;
const y = [!x!];
```

#### Common fixes

If the value being assigned can be declared to be `const`, then change the
declaration:

```dart
const x = 0;
const y = x;
```

If the value can't be declared to be `const`, then remove the `const`
modifier from the variable, possibly using `final` in its place:

```dart
var x = 0;
final y = x;
```

### const_initialized_with_non_constant_value_from_deferred_library

_Constant values from a deferred library can't be used to initialize a 'const'
variable._

#### Description

The analyzer produces this diagnostic when a `const` variable is
initialized using a `const` variable from a library that is imported using
a deferred import. Constants are evaluated at compile time, and values from
deferred libraries aren't available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

The following code produces this diagnostic because the variable `pi` is
being initialized using the constant `math.pi` from the library
`dart:math`, and `dart:math` is imported as a deferred library:

```dart
import 'dart:math' deferred as math;

const pi = math.[!pi!];
```

#### Common fixes

If you need to reference the value of the constant from the imported
library, then remove the keyword `deferred`:

```dart
import 'dart:math' as math;

const pi = math.pi;
```

If you don't need to reference the imported constant, then remove the
reference:

```dart
const pi = 3.14;
```

### const_instance_field

_Only static fields can be declared as const._

#### Description

The analyzer produces this diagnostic when an instance field is marked as
being const.

#### Example

The following code produces this diagnostic because `f` is an instance
field:

```dart
class C {
  [!const!] int f = 3;
}
```

#### Common fixes

If the field needs to be an instance field, then remove the keyword
`const`, or replace it with `final`:

```dart
class C {
  final int f = 3;
}
```

If the field really should be a const field, then make it a static field:

```dart
class C {
  static const int f = 3;
}
```

### const_map_key_not_primitive_equality

_The type of a key in a constant map can't override the '==' operator, or
'hashCode', but the class '{0}' does._

#### Description

The analyzer produces this diagnostic when the class of object used as a
key in a constant map literal implements either the `==` operator, the
getter `hashCode`, or both. The implementation of constant maps uses both
the `==` operator and the `hashCode` getter, so any implementation other
than the ones inherited from `Object` requires executing arbitrary code at
compile time, which isn't supported.

#### Examples

The following code produces this diagnostic because the constant map
contains a key whose type is `C`, and the class `C` overrides the
implementation of `==`:

```dart
class C {
  const C();

  bool operator ==(Object other) => true;
}

const map = {[!C()!] : 0};
```

The following code produces this diagnostic because the constant map
contains a key whose type is `C`, and the class `C` overrides the
implementation of `hashCode`:

```dart
class C {
  const C();

  int get hashCode => 3;
}

const map = {[!C()!] : 0};
```

#### Common fixes

If you can remove the implementation of `==` and `hashCode` from the
class, then do so:

```dart
class C {
  const C();
}

const map = {C() : 0};
```

If you can't remove the implementation of `==` and `hashCode` from the
class, then make the map non-constant:

```dart
class C {
  const C();

  bool operator ==(Object other) => true;
}

final map = {C() : 0};
```

### const_not_initialized

_The constant '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a variable that is declared to
be a constant doesn't have an initializer.

#### Example

The following code produces this diagnostic because `c` isn't initialized:

```dart
const [!c!];
```

#### Common fixes

Add an initializer:

```dart
const c = 'c';
```

### const_set_element_not_primitive_equality

<a id="const_set_element_type_implements_equals" aria-hidden="true"></a>_(Previously known as `const_set_element_type_implements_equals`)_

_An element in a constant set can't override the '==' operator, or 'hashCode',
but the type '{0}' does._

#### Description

The analyzer produces this diagnostic when the class of object used as an
element in a constant set literal implements either the `==` operator, the
getter `hashCode`, or both. The implementation of constant sets uses both
the `==` operator and the `hashCode` getter, so any implementation other
than the ones inherited from `Object` requires executing arbitrary code at
compile time, which isn't supported.

#### Example

The following code produces this diagnostic because the constant set
contains an element whose type is `C`, and the class `C` overrides the
implementation of `==`:

```dart
class C {
  const C();

  bool operator ==(Object other) => true;
}

const set = {[!C()!]};
```

The following code produces this diagnostic because the constant set
contains an element whose type is `C`, and the class `C` overrides the
implementation of `hashCode`:

```dart
class C {
  const C();

  int get hashCode => 3;
}

const map = {[!C()!]};
```

#### Common fixes

If you can remove the implementation of `==` and `hashCode` from the
class, then do so:

```dart
class C {
  const C();
}

const set = {C()};
```

If you can't remove the implementation of `==` and `hashCode` from the
class, then make the set non-constant:

```dart
class C {
  const C();

  bool operator ==(Object other) => true;
}

final set = {C()};
```

### const_spread_expected_list_or_set

_A list or a set is expected in this spread._

#### Description

The analyzer produces this diagnostic when the expression of a spread
operator in a constant list or set evaluates to something other than a list
or a set.

#### Example

The following code produces this diagnostic because the value of `list1` is
`null`, which is neither a list nor a set:

```dart
const dynamic list1 = 42;
const List<int> list2 = [...[!list1!]];
```

#### Common fixes

Change the expression to something that evaluates to either a constant list
or a constant set:

```dart
const dynamic list1 = [42];
const List<int> list2 = [...list1];
```

### const_spread_expected_map

_A map is expected in this spread._

#### Description

The analyzer produces this diagnostic when the expression of a spread
operator in a constant map evaluates to something other than a map.

#### Example

The following code produces this diagnostic because the value of `map1` is
`null`, which isn't a map:

```dart
const dynamic map1 = 42;
const Map<String, int> map2 = {...[!map1!]};
```

#### Common fixes

Change the expression to something that evaluates to a constant map:

```dart
const dynamic map1 = {'answer': 42};
const Map<String, int> map2 = {...map1};
```

### const_with_non_const

_The constructor being called isn't a const constructor._

#### Description

The analyzer produces this diagnostic when the keyword `const` is used to
invoke a constructor that isn't marked with `const`.

#### Example

The following code produces this diagnostic because the constructor in `A`
isn't a const constructor:

```dart
class A {
  A();
}

A f() => [!const!] A();
```

#### Common fixes

If it's desirable and possible to make the class a constant class (by
making all of the fields of the class, including inherited fields, final),
then add the keyword `const` to the constructor:

```dart
class A {
  const A();
}

A f() => const A();
```

Otherwise, remove the keyword `const`:

```dart
class A {
  A();
}

A f() => A();
```

### const_with_non_constant_argument

_Arguments of a constant creation must be constant expressions._

#### Description

The analyzer produces this diagnostic when a const constructor is invoked
with an argument that isn't a constant expression.

#### Example

The following code produces this diagnostic because `i` isn't a constant:

```dart
class C {
  final int i;
  const C(this.i);
}
C f(int i) => const C([!i!]);
```

#### Common fixes

Either make all of the arguments constant expressions, or remove the
`const` keyword to use the non-constant form of the constructor:

```dart
class C {
  final int i;
  const C(this.i);
}
C f(int i) => C(i);
```

### const_with_type_parameters

_A constant constructor tearoff can't use a type parameter as a type argument._

_A constant creation can't use a type parameter as a type argument._

_A constant function tearoff can't use a type parameter as a type argument._

#### Description

The analyzer produces this diagnostic when a type parameter is used as a
type argument in a `const` invocation of a constructor. This isn't allowed
because the value of the type parameter (the actual type that will be used
at runtime) can't be known at compile time.

#### Example

The following code produces this diagnostic because the type parameter `T`
is being used as a type argument when creating a constant:

```dart
class C<T> {
  const C();
}

C<T> newC<T>() => const C<[!T!]>();
```

#### Common fixes

If the type that will be used for the type parameter can be known at
compile time, then remove the use of the type parameter:

```dart
class C<T> {
  const C();
}

C<int> newC() => const C<int>();
```

If the type that will be used for the type parameter can't be known until
runtime, then remove the keyword `const`:

```dart
class C<T> {
  const C();
}

C<T> newC<T>() => C<T>();
```

### continue_label_invalid

<a id="continue_label_on_switch" aria-hidden="true"></a>_(Previously known as `continue_label_on_switch`)_

_The label used in a 'continue' statement must be defined on either a loop or a
switch member._

#### Description

The analyzer produces this diagnostic when the label in a `continue`
statement resolves to a label on a `switch` statement.

#### Example

The following code produces this diagnostic because the label `l`, used to
label a `switch` statement, is used in the `continue` statement:

```dart
void f(int i) {
  l: switch (i) {
    case 0:
      [!continue l;!]
  }
}
```

#### Common fixes

Find a different way to achieve the control flow you need; for example, by
introducing a loop that re-executes the `switch` statement.

### creation_of_struct_or_union

_Subclasses of 'Struct' and 'Union' are backed by native memory, and can't be
instantiated by a generative constructor._

#### Description

The analyzer produces this diagnostic when a subclass of either `Struct`
or `Union` is instantiated using a generative constructor.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the class `C` is being
instantiated using a generative constructor:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int32()
  external int a;
}

void f() {
  [!C!]();
}
```

#### Common fixes

If you need to allocate the structure described by the class, then use the
`ffi` package to do so:

```dart
import 'dart:ffi';
import 'package:ffi/ffi.dart';

final class C extends Struct {
  @Int32()
  external int a;
}

void f() {
  final pointer = calloc.allocate<C>(4);
  final c = pointer.ref;
  print(c);
  calloc.free(pointer);
}
```

### creation_with_non_type

_The name '{0}' isn't a class._

#### Description

The analyzer produces this diagnostic when an instance creation using
either `new` or `const` specifies a name that isn't defined as a class.

#### Example

The following code produces this diagnostic because `f` is a function
rather than a class:

```dart
int f() => 0;

void g() {
  new [!f!]();
}
```

#### Common fixes

If a class should be created, then replace the invalid name with the name
of a valid class:

```dart
int f() => 0;

void g() {
  new Object();
}
```

If the name is the name of a function and you want that function to be
invoked, then remove the `new` or `const` keyword:

```dart
int f() => 0;

void g() {
  f();
}
```

### dead_code

_Dead code._

#### Description

The analyzer produces this diagnostic when code is found that won't be
executed because execution will never reach the code.

#### Example

The following code produces this diagnostic because the invocation of
`print` occurs after the function has returned:

```dart
void f() {
  return;
  [!print('here');!]
}
```

#### Common fixes

If the code isn't needed, then remove it:

```dart
void f() {
  return;
}
```

If the code needs to be executed, then either move the code to a place
where it will be executed:

```dart
void f() {
  print('here');
  return;
}
```

Or, rewrite the code before it, so that it can be reached:

```dart
void f({bool skipPrinting = true}) {
  if (skipPrinting) {
    return;
  }
  print('here');
}
```

### dead_code_catch_following_catch

_Dead code: Catch clauses after a 'catch (e)' or an 'on Object catch (e)' are
never reached._

#### Description

The analyzer produces this diagnostic when a `catch` clause is found that
can't be executed because it's after a `catch` clause of the form
`catch (e)` or `on Object catch (e)`. The first `catch` clause that matches
the thrown object is selected, and both of those forms will match any
object, so no `catch` clauses that follow them will be selected.

#### Example

The following code produces this diagnostic:

```dart
void f() {
  try {
  } catch (e) {
  } [!on String {
  }!]
}
```

#### Common fixes

If the clause should be selectable, then move the clause before the general
clause:

```dart
void f() {
  try {
  } on String {
  } catch (e) {
  }
}
```

If the clause doesn't need to be selectable, then remove it:

```dart
void f() {
  try {
  } catch (e) {
  }
}
```

### dead_code_on_catch_subtype

_Dead code: This on-catch block won't be executed because '{0}' is a subtype of
'{1}' and hence will have been caught already._

#### Description

The analyzer produces this diagnostic when a `catch` clause is found that
can't be executed because it is after a `catch` clause that catches either
the same type or a supertype of the clause's type. The first `catch` clause
that matches the thrown object is selected, and the earlier clause always
matches anything matchable by the highlighted clause, so the highlighted
clause will never be selected.

#### Example

The following code produces this diagnostic:

```dart
void f() {
  try {
  } on num {
  } [!on int {
  }!]
}
```

#### Common fixes

If the clause should be selectable, then move the clause before the general
clause:

```dart
void f() {
  try {
  } on int {
  } on num {
  }
}
```

If the clause doesn't need to be selectable, then remove it:

```dart
void f() {
  try {
  } on num {
  }
}
```

### dead_null_aware_expression

_The left operand can't be null, so the right operand is never executed._

#### Description

The analyzer produces this diagnostic in two cases.

The first is when the left operand of an `??` operator can't be `null`.
The right operand is only evaluated if the left operand has the value
`null`, and because the left operand can't be `null`, the right operand is
never evaluated.

The second is when the left-hand side of an assignment using the `??=`
operator can't be `null`. The right-hand side is only evaluated if the
left-hand side has the value `null`, and because the left-hand side can't
be `null`, the right-hand side is never evaluated.

#### Examples

The following code produces this diagnostic because `x` can't be `null`:

```dart
int f(int x) {
  return x ?? [!0!];
}
```

The following code produces this diagnostic because `f` can't be `null`:

```dart
class C {
  int f = -1;

  void m(int x) {
    f ??= [!x!];
  }
}
```

#### Common fixes

If the diagnostic is reported for an `??` operator, then remove the `??`
operator and the right operand:

```dart
int f(int x) {
  return x;
}
```

If the diagnostic is reported for an assignment, and the assignment isn't
needed, then remove the assignment:

```dart
class C {
  int f = -1;

  void m(int x) {
  }
}
```

If the assignment is needed, but should be based on a different condition,
then rewrite the code to use `=` and the different condition:

```dart
class C {
  int f = -1;

  void m(int x) {
    if (f < 0) {
      f = x;
    }
  }
}
```

### default_list_constructor

_The default 'List' constructor isn't available when null safety is enabled._

#### Description

The analyzer produces this diagnostic when it finds a use of the default
constructor for the class `List` in code that has opted in to null safety.

#### Example

Assuming the following code is opted in to null safety, it produces this
diagnostic because it uses the default `List` constructor:

```dart
var l = [!List<int>!]();
```

#### Common fixes

If no initial size is provided, then convert the code to use a list
literal:

```dart
var l = <int>[];
```

If an initial size needs to be provided and there is a single reasonable
initial value for the elements, then use `List.filled`:

```dart
var l = List.filled(3, 0);
```

If an initial size needs to be provided but each element needs to be
computed, then use `List.generate`:

```dart
var l = List.generate(3, (i) => i);
```

### default_value_in_function_type

_Parameters in a function type can't have default values._

#### Description

The analyzer produces this diagnostic when a function type associated with
a parameter includes optional parameters that have a default value. This
isn't allowed because the default values of parameters aren't part of the
function's type, and therefore including them doesn't provide any value.

#### Example

The following code produces this diagnostic because the parameter `p` has a
default value even though it's part of the type of the parameter `g`:

```dart
void f(void Function([int p [!=!] 0]) g) {
}
```

#### Common fixes

Remove the default value from the function-type's parameter:

```dart
void f(void Function([int p]) g) {
}
```

### default_value_in_redirecting_factory_constructor

_Default values aren't allowed in factory constructors that redirect to another
constructor._

#### Description

The analyzer produces this diagnostic when a factory constructor that
redirects to another constructor specifies a default value for an optional
parameter.

#### Example

The following code produces this diagnostic because the factory constructor
in `A` has a default value for the optional parameter `x`:

```dart
class A {
  factory A([int [!x!] = 0]) = B;
}

class B implements A {
  B([int x = 1]) {}
}
```

#### Common fixes

Remove the default value from the factory constructor:

```dart
class A {
  factory A([int x]) = B;
}

class B implements A {
  B([int x = 1]) {}
}
```

Note that this fix might change the value used when the optional parameter
is omitted. If that happens, and if that change is a problem, then consider
making the optional parameter a required parameter in the factory method:

```dart
class A {
 factory A(int x) = B;
}

class B implements A {
  B([int x = 1]) {}
}
```

### default_value_on_required_parameter

_Required named parameters can't have a default value._

#### Description

The analyzer produces this diagnostic when a named parameter has both the
`required` modifier and a default value. If the parameter is required, then
a value for the parameter is always provided at the call sites, so the
default value can never be used.

#### Example

The following code generates this diagnostic:

```dart
void log({required String [!message!] = 'no message'}) {}
```

#### Common fixes

If the parameter is really required, then remove the default value:

```dart
void log({required String message}) {}
```

If the parameter isn't always required, then remove the `required`
modifier:

```dart
void log({String message = 'no message'}) {}
```

### deferred_import_of_extension

_Imports of deferred libraries must hide all extensions._

#### Description

The analyzer produces this diagnostic when a library that is imported using
a deferred import declares an extension that is visible in the importing
library. Extension methods are resolved at compile time, and extensions
from deferred libraries aren't available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

Given a file `a.dart` that defines a named extension:

```dart
class C {}

extension E on String {
  int get size => length;
}
```

The following code produces this diagnostic because the named extension is
visible to the library:

```dart
import [!'a.dart'!] deferred as a;

void f() {
  a.C();
}
```

#### Common fixes

If the library must be imported as `deferred`, then either add a `show`
clause listing the names being referenced or add a `hide` clause listing
all of the named extensions. Adding a `show` clause would look like this:

```dart
import 'a.dart' deferred as a show C;

void f() {
  a.C();
}
```

Adding a `hide` clause would look like this:

```dart
import 'a.dart' deferred as a hide E;

void f() {
  a.C();
}
```

With the first fix, the benefit is that if new extensions are added to the
imported library, then the extensions won't cause a diagnostic to be
generated.

If the library doesn't need to be imported as `deferred`, or if you need to
make use of the extension method declared in it, then remove the keyword
`deferred`:

```dart
import 'a.dart' as a;

void f() {
  a.C();
}
```

### definitely_unassigned_late_local_variable

_The late local variable '{0}' is definitely unassigned at this point._

#### Description

The analyzer produces this diagnostic when [definite assignment][] analysis
shows that a local variable that's marked as `late` is read before being
assigned.

#### Example

The following code produces this diagnostic because `x` wasn't assigned a
value before being read:

```dart
void f(bool b) {
  late int x;
  print([!x!]);
}
```

#### Common fixes

Assign a value to the variable before reading from it:

```dart
void f(bool b) {
  late int x;
  x = b ? 1 : 0;
  print(x);
}
```

### dependencies_field_not_map

_The value of the '{0}' field is expected to be a map._

#### Description

The analyzer produces this diagnostic when the value of either the
`dependencies` or `dev_dependencies` key isn't a map.

#### Example

The following code produces this diagnostic because the value of the
top-level `dependencies` key is a list:

```yaml
name: example
dependencies:
  [!- meta!]
```

#### Common fixes

Use a map as the value of the `dependencies` key:

```yaml
name: example
dependencies:
  meta: ^1.0.2
```

### deprecated_colon_for_default_value

_Using a colon as the separator before a default value is deprecated and will
not be supported in language version 3.0 and later._

#### Description

The analyzer produces this diagnostic when a colon (`:`) is used as the
separator before the default value of an optional named parameter.
While this syntax is allowed, it is deprecated in favor of
using an equal sign (`=`).

#### Example

The following code produces this diagnostic because a colon is being used
before the default value of the optional parameter `i`:

```dart
void f({int i [!:!] 0}) {}
```

#### Common fixes

Replace the colon with an equal sign.

```dart
void f({int i = 0}) {}
```

### deprecated_export_use

_The ability to import '{0}' indirectly is deprecated._

#### Description

The analyzer produces this diagnostic when one library imports a name from
a second library, and the second library exports the name from a third
library but has indicated that it won't export the third library in the
future.

#### Example

Given a library `a.dart` defining the class `A`:

```dart
class A {}
```

And a second library `b.dart` that exports `a.dart` but has marked the
export as being deprecated:

```dart
import 'a.dart';

@deprecated
export 'a.dart';
```

The following code produces this diagnostic because the class `A` won't be
exported from `b.dart` in some future version:

```dart
import 'b.dart';

[!A!]? a;
```

#### Common fixes

If the name is available from a different library that you can import,
then replace the existing import with an import for that library (or add
an import for the defining library if you still need the old import):

```dart
import 'a.dart';

A? a;
```

If the name isn't available, then look for instructions from the library
author or contact them directly to find out how to update your code.

### deprecated_field

_The '{0}' field is no longer used and can be removed._

#### Description

The analyzer produces this diagnostic when a key is used in a
`pubspec.yaml` file that was deprecated. Unused keys take up space and
might imply semantics that are no longer valid.

#### Example

The following code produces this diagnostic because the `author` key is no
longer being used:

```dart
name: example
author: 'Dash'
```

#### Common fixes

Remove the deprecated key:

```dart
name: example
```

### deprecated_member_use

_'{0}' is deprecated and shouldn't be used._

_'{0}' is deprecated and shouldn't be used. {1}_

#### Description

The analyzer produces this diagnostic when a deprecated library or class
member is used in a different package.

#### Example

If the method `m` in the class `C` is annotated with `@deprecated`, then
the following code produces this diagnostic:

```dart
void f(C c) {
  c.[!m!]();
}
```

#### Common fixes

The documentation for declarations that are annotated with `@deprecated`
should indicate what code to use in place of the deprecated code.

### deprecated_member_use_from_same_package

_'{0}' is deprecated and shouldn't be used._

_'{0}' is deprecated and shouldn't be used. {1}_

#### Description

The analyzer produces this diagnostic when a deprecated library member or
class member is used in the same package in which it's declared.

#### Example

The following code produces this diagnostic because `x` is deprecated:

```dart
@deprecated
var x = 0;
var y = [!x!];
```

#### Common fixes

The fix depends on what's been deprecated and what the replacement is. The
documentation for deprecated declarations should indicate what code to use
in place of the deprecated code.

### deprecated_new_in_comment_reference

_Using the 'new' keyword in a comment reference is deprecated._

#### Description

The analyzer produces this diagnostic when a comment reference (the name
of a declaration enclosed in square brackets in a documentation comment)
uses the keyword `new` to refer to a constructor. This form is deprecated.

#### Examples

The following code produces this diagnostic because the unnamed
constructor is being referenced using `new C`:

```dart
/// See [[!new!] C].
class C {
  C();
}
```

The following code produces this diagnostic because the constructor named
`c` is being referenced using `new C.c`:

```dart
/// See [[!new!] C.c].
class C {
  C.c();
}
```

#### Common fixes

If you're referencing a named constructor, then remove the keyword `new`:

```dart
/// See [C.c].
class C {
  C.c();
}
```

If you're referencing the unnamed constructor, then remove the keyword
`new` and append `.new` after the class name:

```dart
/// See [C.new].
class C {
  C.c();
}
```

### deprecated_subtype_of_function

_Extending 'Function' is deprecated._

_Implementing 'Function' has no effect._

_Mixing in 'Function' is deprecated._

#### Description

The analyzer produces this diagnostic when the class `Function` is used in
either the `extends`, `implements`, or `with` clause of a class or mixin.
Using the class `Function` in this way has no semantic value, so it's
effectively dead code.

#### Example

The following code produces this diagnostic because `Function` is used as
the superclass of `F`:

```dart
class F extends [!Function!] {}
```

#### Common fixes

Remove the class `Function` from whichever clause it's in, and remove the
whole clause if `Function` is the only type in the clause:

```dart
class F {}
```

### disallowed_type_instantiation_expression

_Only a generic type, generic function, generic instance method, or generic
constructor can have type arguments._

#### Description

The analyzer produces this diagnostic when an expression with a value that
is anything other than one of the allowed kinds of values is followed by
type arguments. The allowed kinds of values are:
- generic types,
- generic constructors, and
- generic functions, including top-level functions, static and instance
  members, and local functions.

#### Example

The following code produces this diagnostic because `i` is a top-level
variable, which isn't one of the allowed cases:

```dart
int i = 1;

void f() {
  print([!i!]<int>);
}
```

#### Common fixes

If the referenced value is correct, then remove the type arguments:

```dart
int i = 1;

void f() {
  print(i);
}
```

### division_optimization

_The operator x ~/ y is more efficient than (x / y).toInt()._

#### Description

The analyzer produces this diagnostic when the result of dividing two
numbers is converted to an integer using `toInt`. Dart has a built-in
integer division operator that is both more efficient and more concise.

#### Example

The following code produces this diagnostic because the result of dividing
`x` and `y` is converted to an integer using `toInt`:

```dart
int divide(int x, int y) => [!(x / y).toInt()!];
```

#### Common fixes

Use the integer division operator (`~/`):

```dart
int divide(int x, int y) => x ~/ y;
```

### duplicate_constructor

_The constructor with name '{0}' is already defined._

_The unnamed constructor is already defined._

#### Description

The analyzer produces this diagnostic when a class declares more than one
unnamed constructor or when it declares more than one constructor with the
same name.

#### Examples

The following code produces this diagnostic because there are two
declarations for the unnamed constructor:

```dart
class C {
  C();

  [!C!]();
}
```

The following code produces this diagnostic because there are two
declarations for the constructor named `m`:

```dart
class C {
  C.m();

  [!C.m!]();
}
```

#### Common fixes

If there are multiple unnamed constructors and all of the constructors are
needed, then give all of them, or all except one of them, a name:

```dart
class C {
  C();

  C.n();
}
```

If there are multiple unnamed constructors and all except one of them are
unneeded, then remove the constructors that aren't needed:

```dart
class C {
  C();
}
```

If there are multiple named constructors and all of the constructors are
needed, then rename all except one of them:

```dart
class C {
  C.m();

  C.n();
}
```

If there are multiple named constructors and all except one of them are
unneeded, then remove the constructors that aren't needed:

```dart
class C {
  C.m();
}
```

### duplicate_definition

_The name '{0}' is already defined._

#### Description

The analyzer produces this diagnostic when a name is declared, and there is
a previous declaration with the same name in the same scope.

#### Example

The following code produces this diagnostic because the name `x` is
declared twice:

```dart
int x = 0;
int [!x!] = 1;
```

#### Common fixes

Choose a different name for one of the declarations.

```dart
int x = 0;
int y = 1;
```

### duplicate_export

_Duplicate export._

#### Description

The analyzer produces this diagnostic when an export directive is found
that is the same as an export before it in the file. The second export
doesn't add value and should be removed.

#### Example

The following code produces this diagnostic because the same library is
being exported twice:

```dart
export 'package:meta/meta.dart';
export [!'package:meta/meta.dart'!];
```

#### Common fixes

Remove the unnecessary export:

```dart
export 'package:meta/meta.dart';
```

### duplicate_field_formal_parameter

_The field '{0}' can't be initialized by multiple parameters in the same
constructor._

#### Description

The analyzer produces this diagnostic when there's more than one
initializing formal parameter for the same field in a constructor's
parameter list. It isn't useful to assign a value that will immediately be
overwritten.

#### Example

The following code produces this diagnostic because `this.f` appears twice
in the parameter list:

```dart
class C {
  int f;

  C(this.f, this.[!f!]) {}
}
```

#### Common fixes

Remove one of the initializing formal parameters:

```dart
class C {
  int f;

  C(this.f) {}
}
```

### duplicate_field_name

_The field name '{0}' is already used in this record._

#### Description

The analyzer produces this diagnostic when either a record literal or a
record type annotation contains a field whose name is the same as a
previously declared field in the same literal or type.

#### Examples

The following code produces this diagnostic because the record literal has
two fields named `a`:

```dart
var r = (a: 1, [!a!]: 2);
```

The following code produces this diagnostic because the record type
annotation has two fields named `a`, one a positional field and the other
a named field:

```dart
void f((int a, {int [!a!]}) r) {}
```

#### Common fixes

Rename one or both of the fields:

```dart
var r = (a: 1, b: 2);
```

### duplicate_hidden_name

_Duplicate hidden name._

#### Description

The analyzer produces this diagnostic when a name occurs multiple times in
a `hide` clause. Repeating the name is unnecessary.

#### Example

The following code produces this diagnostic because the name `min` is
hidden more than once:

```dart
import 'dart:math' hide min, [!min!];

var x = pi;
```

#### Common fixes

If the name was mistyped in one or more places, then correct the mistyped
names:

```dart
import 'dart:math' hide max, min;

var x = pi;
```

If the name wasn't mistyped, then remove the unnecessary name from the
list:

```dart
import 'dart:math' hide min;

var x = pi;
```

### duplicate_ignore

_The diagnostic '{0}' doesn't need to be ignored here because it's already being
ignored._

#### Description

The analyzer produces this diagnostic when a diagnostic name appears in an
`ignore` comment, but the diagnostic is already being ignored, either
because it's already included in the same `ignore` comment or because it
appears in an `ignore-in-file` comment.

#### Examples

The following code produces this diagnostic because the diagnostic named
`unused_local_variable` is already being ignored for the whole file so it
doesn't need to be ignored on a specific line:

```dart
// ignore_for_file: unused_local_variable
void f() {
  // ignore: [!unused_local_variable!]
  var x = 0;
}
```

The following code produces this diagnostic because the diagnostic named
`unused_local_variable` is being ignored twice on the same line:

```dart
void f() {
  // ignore: unused_local_variable, [!unused_local_variable!]
  var x = 0;
}
```

#### Common fixes

Remove the ignore comment, or remove the unnecessary diagnostic name if the
ignore comment is ignoring more than one diagnostic:

```dart
// ignore_for_file: unused_local_variable
void f() {
  var x = 0;
}
```

### duplicate_import

_Duplicate import._

#### Description

The analyzer produces this diagnostic when an import directive is found
that is the same as an import before it in the file. The second import
doesn't add value and should be removed.

#### Example

The following code produces this diagnostic:

```dart
import 'package:meta/meta.dart';
import [!'package:meta/meta.dart'!];

@sealed class C {}
```

#### Common fixes

Remove the unnecessary import:

```dart
import 'package:meta/meta.dart';

@sealed class C {}
```

### duplicate_named_argument

_The argument for the named parameter '{0}' was already specified._

#### Description

The analyzer produces this diagnostic when an invocation has two or more
named arguments that have the same name.

#### Example

The following code produces this diagnostic because there are two arguments
with the name `a`:

```dart
void f(C c) {
  c.m(a: 0, [!a!]: 1);
}

class C {
  void m({int? a, int? b}) {}
}
```

#### Common fixes

If one of the arguments should have a different name, then change the name:

```dart
void f(C c) {
  c.m(a: 0, b: 1);
}

class C {
  void m({int? a, int? b}) {}
}
```

If one of the arguments is wrong, then remove it:

```dart
void f(C c) {
  c.m(a: 1);
}

class C {
  void m({int? a, int? b}) {}
}
```

### duplicate_part

_The library already contains a part with the URI '{0}'._

#### Description

The analyzer produces this diagnostic when a single file is referenced in
multiple part directives.

#### Example

Given a file `part.dart` containing

```dart
part of lib;
```

The following code produces this diagnostic because the file `part.dart` is
included multiple times:

```dart
library lib;

part 'part.dart';
part [!'part.dart'!];
```

#### Common fixes

Remove all except the first of the duplicated part directives:

```dart
library lib;

part 'part.dart';
```

### duplicate_pattern_assignment_variable

_The variable '{0}' is already assigned in this pattern._

#### Description

The analyzer produces this diagnostic when a single pattern variable is
assigned a value more than once in the same pattern assignment.

#### Example

The following code produces this diagnostic because the variable `a` is
assigned twice in the pattern `(a, a)`:

```dart
int f((int, int) r) {
  int a;
  (a, [!a!]) = r;
  return a;
}
```

#### Common fixes

If you need to capture all of the values, then use a unique variable for
each of the subpatterns being matched:

```dart
int f((int, int) r) {
  int a, b;
  (a, b) = r;
  return a + b;
}
```

If some of the values don't need to be captured, then use a wildcard
pattern `_` to avoid having to bind the value to a variable:

```dart
int f((int, int) r) {
  int a;
  (_, a) = r;
  return a;
}
```

### duplicate_pattern_field

_The field '{0}' is already matched in this pattern._

#### Description

The analyzer produces this diagnostic when a record pattern matches the
same field more than once, or when an object pattern matches the same
getter more than once.

#### Examples

The following code produces this diagnostic because the record field `a`
is matched twice in the same record pattern:

```dart
void f(({int a, int b}) r) {
  switch (r) {
    case (a: 1, [!a!]: 2):
      return;
  }
}
```

The following code produces this diagnostic because the getter `f` is
matched twice in the same object pattern:

```dart
void f(Object o) {
  switch (o) {
    case C(f: 1, [!f!]: 2):
      return;
  }
}
class C {
  int? f;
}
```

#### Common fixes

If the pattern should match for more than one value of the duplicated
field, then use a logical-or pattern:

```dart
void f(({int a, int b}) r) {
  switch (r) {
    case (a: 1, b: _) || (a: 2, b: _):
      break;
  }
}
```

If the pattern should match against multiple fields, then change the name
of one of the fields:

```dart
void f(({int a, int b}) r) {
  switch (r) {
    case (a: 1, b: 2):
      return;
  }
}
```

### duplicate_rest_element_in_pattern

_At most one rest element is allowed in a list or map pattern._

#### Description

The analyzer produces this diagnostic when there's more than one rest
pattern in either a list or map pattern. A rest pattern will capture any
values unmatched by other subpatterns, making subsequent rest patterns
unnecessary because there's nothing left to capture.

#### Example

The following code produces this diagnostic because there are two rest
patterns in the list pattern:

```dart
void f(List<int> x) {
  if (x case [0, ..., [!...!]]) {}
}
```

#### Common fixes

Remove all but one of the rest patterns:

```dart
void f(List<int> x) {
  if (x case [0, ...]) {}
}
```

### duplicate_shown_name

_Duplicate shown name._

#### Description

The analyzer produces this diagnostic when a name occurs multiple times in
a `show` clause. Repeating the name is unnecessary.

#### Example

The following code produces this diagnostic because the name `min` is shown
more than once:

```dart
import 'dart:math' show min, [!min!];

var x = min(2, min(0, 1));
```

#### Common fixes

If the name was mistyped in one or more places, then correct the mistyped
names:

```dart
import 'dart:math' show max, min;

var x = max(2, min(0, 1));
```

If the name wasn't mistyped, then remove the unnecessary name from the
list:

```dart
import 'dart:math' show min;

var x = min(2, min(0, 1));
```

### duplicate_variable_pattern

_The variable '{0}' is already defined in this pattern._

#### Description

The analyzer produces this diagnostic when a branch of a logical-and
pattern declares a variable that is already declared in an earlier branch
of the same pattern.

#### Example

The following code produces this diagnostic because the variable `a` is
declared in both branches of the logical-and pattern:

```dart
void f((int, int) r) {
  if (r case (var a, 0) && (0, var [!a!])) {
    print(a);
  }
}
```

#### Common fixes

If you need to capture the matched value in multiple branches, then change
the names of the variables so that they are unique:

```dart
void f((int, int) r) {
  if (r case (var a, 0) && (0, var b)) {
    print(a + b);
  }
}
```

If you only need to capture the matched value on one branch, then remove
the variable pattern from all but one branch:

```dart
void f((int, int) r) {
  if (r case (var a, 0) && (0, _)) {
    print(a);
  }
}
```

### empty_map_pattern

_A map pattern must have at least one entry._

#### Description

The analyzer produces this diagnostic when a map pattern is empty.

#### Example

The following code produces this diagnostic because the map pattern
is empty:

```dart
void f(Map<int, String> x) {
  if (x case [!{}!]) {}
}
```

#### Common fixes

If the pattern should match any map, then replace it with an object
pattern:

```dart
void f(Map<int, String> x) {
  if (x case Map()) {}
}
```

If the pattern should only match an empty map, then check the length
in the pattern:

```dart
void f(Map<int, String> x) {
  if (x case Map(isEmpty: true)) {}
}
```

### empty_record_literal_with_comma

_A record literal without fields can't have a trailing comma._

#### Description

The analyzer produces this diagnostic when a record literal that has no
fields has a trailing comma. Empty record literals can't contain a comma.

#### Example

The following code produces this diagnostic because the empty record
literal has a trailing comma:

```dart
var r = ([!,!]);
```

#### Common fixes

If the record is intended to be empty, then remove the comma:

```dart
var r = ();
```

If the record is intended to have one or more fields, then add the
expressions used to compute the values of those fields:

```dart
var r = (3, 4);
```

### empty_record_type_named_fields_list

_The list of named fields in a record type can't be empty._

#### Description

The analyzer produces this diagnostic when a record type has an empty list
of named fields.

#### Example

The following code produces this diagnostic because the record type has an
empty list of named fields:

```dart
void f((int, int, {[!}!]) r) {}
```

#### Common fixes

If the record is intended to have named fields, then add the types and
names of the fields:

```dart
void f((int, int, {int z}) r) {}
```

If the record isn't intended to have named fields, then remove the curly
braces:

```dart
void f((int, int) r) {}
```

### empty_record_type_with_comma

_A record type without fields can't have a trailing comma._

#### Description

The analyzer produces this diagnostic when a record type that has no
fields has a trailing comma. Empty record types can't contain a comma.

#### Example

The following code produces this diagnostic because the empty record type
has a trailing comma:

```dart
void f(([!,!]) r) {}
```

#### Common fixes

If the record type is intended to be empty, then remove the comma:

```dart
void f(() r) {}
```

If the record type is intended to have one or more fields, then add the
types of those fields:

```dart
void f((int, int) r) {}
```

### empty_struct

_The class '{0}' can't be empty because it's a subclass of '{1}'._

#### Description

The analyzer produces this diagnostic when a subclass of `Struct` or
`Union` doesn't have any fields. Having an empty `Struct` or `Union`
isn't supported.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the class `C`, which
extends `Struct`, doesn't declare any fields:

```dart
import 'dart:ffi';

final class [!C!] extends Struct {}
```

#### Common fixes

If the class is intended to be a struct, then declare one or more fields:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int32()
  external int x;
}
```

If the class is intended to be used as a type argument to `Pointer`, then
make it a subclass of `Opaque`:

```dart
import 'dart:ffi';

final class C extends Opaque {}
```

If the class isn't intended to be a struct, then remove or change the
extends clause:

```dart
class C {}
```

### enum_constant_same_name_as_enclosing

_The name of the enum value can't be the same as the enum's name._

#### Description

The analyzer produces this diagnostic when an enum value has the same name
as the enum in which it's declared.

#### Example

The following code produces this diagnostic because the enum value `E` has
the same name as the enclosing enum `E`:

```dart
enum E {
  [!E!]
}
```

#### Common fixes

If the name of the enum is correct, then rename the constant:

```dart
enum E {
  e
}
```

If the name of the constant is correct, then rename the enum:

```dart
enum F {
  E
}
```

### enum_constant_with_non_const_constructor

_The invoked constructor isn't a 'const' constructor._

#### Description

The analyzer produces this diagnostic when an enum value is being created
using either a factory constructor or a generative constructor that isn't
marked as being `const`.

#### Example

The following code produces this diagnostic because the enum value `e` is
being initialized by a factory constructor:

```dart
enum E {
  [!e!]();

  factory E() => e;
}
```

#### Common fixes

Use a generative constructor marked as `const`:

```dart
enum E {
  e._();

  factory E() => e;

  const E._();
}
```

### enum_mixin_with_instance_variable

_Mixins applied to enums can't have instance variables._

#### Description

The analyzer produces this diagnostic when a mixin that's applied to an
enum declares one or more instance variables. This isn't allowed because
the enum values are constant, and there isn't any way for the constructor
in the enum to initialize any of the mixin's fields.

#### Example

The following code produces this diagnostic because the mixin `M` defines
the instance field `x`:

```dart
mixin M {
  int x = 0;
}

enum E with [!M!] {
  a
}
```

#### Common fixes

If you need to apply the mixin, then change all instance fields into
getter and setter pairs and implement them in the enum if necessary:

```dart
mixin M {
  int get x => 0;
}

enum E with M {
  a
}
```

If you don't need to apply the mixin, then remove it:

```dart
enum E {
  a
}
```

### enum_with_abstract_member

_'{0}' must have a method body because '{1}' is an enum._

#### Description

The analyzer produces this diagnostic when a member of an enum is found
that doesn't have a concrete implementation. Enums aren't allowed to
contain abstract members.

#### Example

The following code produces this diagnostic because `m` is an abstract
method and `E` is an enum:

```dart
enum E {
  e;

  [!void m();!]
}
```

#### Common fixes

Provide an implementation for the member:

```dart
enum E {
  e;

  void m() {}
}
```

### enum_with_name_values

_The name 'values' is not a valid name for an enum._

#### Description

The analyzer produces this diagnostic when an enum is declared to have the
name `values`. This isn't allowed because the enum has an implicit static
field named `values`, and the two would collide.

#### Example

The following code produces this diagnostic because there's an enum
declaration that has the name `values`:

```dart
enum [!values!] {
  c
}
```

#### Common fixes

Rename the enum to something other than `values`.

### equal_elements_in_const_set

_Two elements in a constant set literal can't be equal._

#### Description

The analyzer produces this diagnostic when two elements in a constant set
literal have the same value. The set can only contain each value once,
which means that one of the values is unnecessary.

#### Example

The following code produces this diagnostic because the string `'a'` is
specified twice:

```dart
const Set<String> set = {'a', [!'a'!]};
```

#### Common fixes

Remove one of the duplicate values:

```dart
const Set<String> set = {'a'};
```

Note that literal sets preserve the order of their elements, so the choice
of which element to remove might affect the order in which elements are
returned by an iterator.

### equal_elements_in_set

_Two elements in a set literal shouldn't be equal._

#### Description

The analyzer produces this diagnostic when an element in a non-constant set
is the same as a previous element in the same set. If two elements are the
same, then the second value is ignored, which makes having both elements
pointless and likely signals a bug.

#### Example

The following code produces this diagnostic because the element `1` appears
twice:

```dart
const a = 1;
const b = 1;
var s = <int>{a, [!b!]};
```

#### Common fixes

If both elements should be included in the set, then change one of the
elements:

```dart
const a = 1;
const b = 2;
var s = <int>{a, b};
```

If only one of the elements is needed, then remove the one that isn't
needed:

```dart
const a = 1;
var s = <int>{a};
```

Note that literal sets preserve the order of their elements, so the choice
of which element to remove might affect the order in which elements are
returned by an iterator.

### equal_keys_in_const_map

_Two keys in a constant map literal can't be equal._

#### Description

The analyzer produces this diagnostic when a key in a constant map is the
same as a previous key in the same map. If two keys are the same, then the
second value would overwrite the first value, which makes having both pairs
pointless.

#### Example

The following code produces this diagnostic because the key `1` is used
twice:

```dart
const map = <int, String>{1: 'a', 2: 'b', [!1!]: 'c', 4: 'd'};
```

#### Common fixes

If both entries should be included in the map, then change one of the keys
to be different:

```dart
const map = <int, String>{1: 'a', 2: 'b', 3: 'c', 4: 'd'};
```

If only one of the entries is needed, then remove the one that isn't
needed:

```dart
const map = <int, String>{1: 'a', 2: 'b', 4: 'd'};
```

Note that literal maps preserve the order of their entries, so the choice
of which entry to remove might affect the order in which keys and values
are returned by an iterator.

### equal_keys_in_map

_Two keys in a map literal shouldn't be equal._

#### Description

The analyzer produces this diagnostic when a key in a non-constant map is
the same as a previous key in the same map. If two keys are the same, then
the second value overwrites the first value, which makes having both pairs
pointless and likely signals a bug.

#### Example

The following code produces this diagnostic because the keys `a` and `b`
have the same value:

```dart
const a = 1;
const b = 1;
var m = <int, String>{a: 'a', [!b!]: 'b'};
```

#### Common fixes

If both entries should be included in the map, then change one of the keys:

```dart
const a = 1;
const b = 2;
var m = <int, String>{a: 'a', b: 'b'};
```

If only one of the entries is needed, then remove the one that isn't
needed:

```dart
const a = 1;
var m = <int, String>{a: 'a'};
```

Note that literal maps preserve the order of their entries, so the choice
of which entry to remove might affect the order in which the keys and
values are returned by an iterator.

### equal_keys_in_map_pattern

_Two keys in a map pattern can't be equal._

#### Description

The analyzer produces this diagnostic when a map pattern contains more
than one key with the same name. The same key can't be matched twice.

#### Example

The following code produces this diagnostic because the key `'a'` appears
twice:

```dart
void f(Map<String, int> x) {
  if (x case {'a': 1, [!'a'!]: 2}) {}
}
```

#### Common fixes

If you are trying to match two different keys, then change one of the keys
in the pattern:

```dart
void f(Map<String, int> x) {
  if (x case {'a': 1, 'b': 2}) {}
}
```

If you are trying to match the same key, but allow any one of multiple
patterns to match, the use a logical-or pattern:

```dart
void f(Map<String, int> x) {
  if (x case {'a': 1 || 2}) {}
}
```

### expected_one_list_pattern_type_arguments

_List patterns require one type argument or none, but {0} found._

#### Description

The analyzer produces this diagnostic when a list pattern has more than
one type argument. List patterns can have either zero type arguments or
one type argument, but can't have more than one.

#### Example

The following code produces this diagnostic because the list pattern
(`[0]`) has two type arguments:

```dart
void f(Object x) {
  if (x case [!<int, int>!][0]) {}
}
```

#### Common fixes

Remove all but one of the type arguments:

```dart
void f(Object x) {
  if (x case <int>[0]) {}
}
```

### expected_one_list_type_arguments

_List literals require one type argument or none, but {0} found._

#### Description

The analyzer produces this diagnostic when a list literal has more than one
type argument.

#### Example

The following code produces this diagnostic because the list literal has
two type arguments when it can have at most one:

```dart
var l = [!<int, int>!][];
```

#### Common fixes

Remove all except one of the type arguments:

```dart
var l = <int>[];
```

### expected_one_set_type_arguments

_Set literals require one type argument or none, but {0} were found._

#### Description

The analyzer produces this diagnostic when a set literal has more than one
type argument.

#### Example

The following code produces this diagnostic because the set literal has
three type arguments when it can have at most one:

```dart
var s = [!<int, String, int>!]{0, 'a', 1};
```

#### Common fixes

Remove all except one of the type arguments:

```dart
var s = <int>{0, 1};
```

### expected_two_map_pattern_type_arguments

_Map patterns require two type arguments or none, but {0} found._

#### Description

The analyzer produces this diagnostic when a map pattern has either one
type argument or more than two type arguments. Map patterns can have
either two type arguments or zero type arguments, but can't have any other
number.

#### Example

The following code produces this diagnostic because the map pattern
(`<int>{}`) has one type argument:

```dart
void f(Object x) {
  if (x case [!<int>!]{0: _}) {}
}
```

#### Common fixes

Add or remove type arguments until there are two, or none:

```dart
void f(Object x) {
  if (x case <int, int>{0: _}) {}
}
```

### expected_two_map_type_arguments

_Map literals require two type arguments or none, but {0} found._

#### Description

The analyzer produces this diagnostic when a map literal has either one or
more than two type arguments.

#### Example

The following code produces this diagnostic because the map literal has
three type arguments when it can have either two or zero:

```dart
var m = [!<int, String, int>!]{};
```

#### Common fixes

Remove all except two of the type arguments:

```dart
var m = <int, String>{};
```

### export_internal_library

_The library '{0}' is internal and can't be exported._

#### Description

The analyzer produces this diagnostic when it finds an export whose `dart:`
URI references an internal library.

#### Example

The following code produces this diagnostic because `_interceptors` is an
internal library:

```dart
export [!'dart:_interceptors'!];
```

#### Common fixes

Remove the export directive.

### export_legacy_symbol

_The symbol '{0}' is defined in a legacy library, and can't be re-exported from
a library with null safety enabled._

#### Description

The analyzer produces this diagnostic when a library that was opted in to
null safety exports another library, and the exported library is opted out
of null safety.

#### Example

Given a library that is opted out of null safety:

```dart
// @dart = 2.8
String s;
```

The following code produces this diagnostic because it's exporting symbols
from an opted-out library:

```dart
export [!'optedOut.dart'!];

class C {}
```

#### Common fixes

If you're able to do so, migrate the exported library so that it doesn't
need to opt out:

```dart
String? s;
```

If you can't migrate the library, then remove the export:

```dart
class C {}
```

If the exported library (the one that is opted out) itself exports an
opted-in library, then it's valid for your library to indirectly export the
symbols from the opted-in library. You can do so by adding a hide
combinator to the export directive in your library that hides all of the
names declared in the opted-out library.

### export_of_non_library

_The exported library '{0}' can't have a part-of directive._

#### Description

The analyzer produces this diagnostic when an export directive references a
part rather than a library.

#### Example

Given a file `part.dart` containing

```dart
part of lib;
```

The following code produces this diagnostic because the file `part.dart` is
a part, and only libraries can be exported:

```dart
library lib;

export [!'part.dart'!];
```

#### Common fixes

Either remove the export directive, or change the URI to be the URI of the
library containing the part.

### expression_in_map

_Expressions can't be used in a map literal._

#### Description

The analyzer produces this diagnostic when the analyzer finds an
expression, rather than a map entry, in what appears to be a map literal.

#### Example

The following code produces this diagnostic:

```dart
var map = <String, int>{'a': 0, 'b': 1, [!'c'!]};
```

#### Common fixes

If the expression is intended to compute either a key or a value in an
entry, fix the issue by replacing the expression with the key or the value.
For example:

```dart
var map = <String, int>{'a': 0, 'b': 1, 'c': 2};
```

### extends_non_class

_Classes can only extend other classes._

#### Description

The analyzer produces this diagnostic when an `extends` clause contains a
name that is declared to be something other than a class.

#### Example

The following code produces this diagnostic because `f` is declared to be a
function:

```dart
void f() {}

class C extends [!f!] {}
```

#### Common fixes

If you want the class to extend a class other than `Object`, then replace
the name in the `extends` clause with the name of that class:

```dart
void f() {}

class C extends B {}

class B {}
```

If you want the class to extend `Object`, then remove the `extends` clause:

```dart
void f() {}

class C {}
```

### extension_as_expression

_Extension '{0}' can't be used as an expression._

#### Description

The analyzer produces this diagnostic when the name of an extension is used
in an expression other than in an extension override or to qualify an
access to a static member of the extension. Because classes define a type,
the name of a class can be used to refer to the instance of `Type`
representing the type of the class. Extensions, on the other hand, don't
define a type and can't be used as a type literal.

#### Example

The following code produces this diagnostic because `E` is an extension:

```dart
extension E on int {
  static String m() => '';
}

var x = [!E!];
```

#### Common fixes

Replace the name of the extension with a name that can be referenced, such
as a static member defined on the extension:

```dart
extension E on int {
  static String m() => '';
}

var x = E.m();
```

### extension_conflicting_static_and_instance

_An extension can't define static member '{0}' and an instance member with the
same name._

#### Description

The analyzer produces this diagnostic when an extension declaration
contains both an instance member and a static member that have the same
name. The instance member and the static member can't have the same name
because it's unclear which member is being referenced by an unqualified use
of the name within the body of the extension.

#### Example

The following code produces this diagnostic because the name `a` is being
used for two different members:

```dart
extension E on Object {
  int get a => 0;
  static int [!a!]() => 0;
}
```

#### Common fixes

Rename or remove one of the members:

```dart
extension E on Object {
  int get a => 0;
  static int b() => 0;
}
```

### extension_declares_abstract_member

_Extensions can't declare abstract members._

#### Description

The analyzer produces this diagnostic when an abstract declaration is
declared in an extension. Extensions can declare only concrete members.

#### Example

The following code produces this diagnostic because the method `a` doesn't
have a body:

```dart
extension E on String {
  int [!a!]();
}
```

#### Common fixes

Either provide an implementation for the member or remove it.

### extension_declares_constructor

_Extensions can't declare constructors._

#### Description

The analyzer produces this diagnostic when a constructor declaration is
found in an extension. It isn't valid to define a constructor because
extensions aren't classes, and it isn't possible to create an instance of
an extension.

#### Example

The following code produces this diagnostic because there is a constructor
declaration in `E`:

```dart
extension E on String {
  [!E!]() : super();
}
```

#### Common fixes

Remove the constructor or replace it with a static method.

### extension_declares_instance_field

_Extensions can't declare instance fields_

#### Description

The analyzer produces this diagnostic when an instance field declaration is
found in an extension. It isn't valid to define an instance field because
extensions can only add behavior, not state.

#### Example

The following code produces this diagnostic because `s` is an instance
field:

```dart
extension E on String {
  String [!s!];
}
```

#### Common fixes

Remove the field, make it a static field, or convert it to be a getter,
setter, or method.

### extension_declares_member_of_object

_Extensions can't declare members with the same name as a member declared by
'Object'._

#### Description

The analyzer produces this diagnostic when an extension declaration
declares a member with the same name as a member declared in the class
`Object`. Such a member can never be used because the member in `Object` is
always found first.

#### Example

The following code produces this diagnostic because `toString` is defined
by `Object`:

```dart
extension E on String {
  String [!toString!]() => this;
}
```

#### Common fixes

Remove the member or rename it so that the name doesn't conflict with the
member in `Object`:

```dart
extension E on String {
  String displayString() => this;
}
```

### extension_override_access_to_static_member

_An extension override can't be used to access a static member from an
extension._

#### Description

The analyzer produces this diagnostic when an extension override is the
receiver of the invocation of a static member. Similar to static members in
classes, the static members of an extension should be accessed using the
name of the extension, not an extension override.

#### Example

The following code produces this diagnostic because `m` is static:

```dart
extension E on String {
  static void m() {}
}

void f() {
  E('').[!m!]();
}
```

#### Common fixes

Replace the extension override with the name of the extension:

```dart
extension E on String {
  static void m() {}
}

void f() {
  E.m();
}
```

### extension_override_argument_not_assignable

_The type of the argument to the extension override '{0}' isn't assignable to
the extended type '{1}'._

#### Description

The analyzer produces this diagnostic when the argument to an extension
override isn't assignable to the type being extended by the extension.

#### Example

The following code produces this diagnostic because `3` isn't a `String`:

```dart
extension E on String {
  void method() {}
}

void f() {
  E([!3!]).method();
}
```

#### Common fixes

If you're using the correct extension, then update the argument to have the
correct type:

```dart
extension E on String {
  void method() {}
}

void f() {
  E(3.toString()).method();
}
```

If there's a different extension that's valid for the type of the argument,
then either replace the name of the extension or unwrap the argument so
that the correct extension is found.

### extension_override_without_access

_An extension override can only be used to access instance members._

#### Description

The analyzer produces this diagnostic when an extension override is found
that isn't being used to access one of the members of the extension. The
extension override syntax doesn't have any runtime semantics; it only
controls which member is selected at compile time.

#### Example

The following code produces this diagnostic because `E(i)` isn't an
expression:

```dart
extension E on int {
  int get a => 0;
}

void f(int i) {
  print([!E(i)!]);
}
```

#### Common fixes

If you want to invoke one of the members of the extension, then add the
invocation:

```dart
extension E on int {
  int get a => 0;
}

void f(int i) {
  print(E(i).a);
}
```

If you don't want to invoke a member, then unwrap the argument:

```dart
extension E on int {
  int get a => 0;
}

void f(int i) {
  print(i);
}
```

### extension_override_with_cascade

_Extension overrides have no value so they can't be used as the receiver of a
cascade expression._

#### Description

The analyzer produces this diagnostic when an extension override is used as
the receiver of a cascade expression. The value of a cascade expression
`e..m` is the value of the receiver `e`, but extension overrides aren't
expressions and don't have a value.

#### Example

The following code produces this diagnostic because `E(3)` isn't an
expression:

```dart
extension E on int {
  void m() {}
}
f() {
  [!E!](3)..m();
}
```

#### Common fixes

Use `.` rather than `..`:

```dart
extension E on int {
  void m() {}
}
f() {
  E(3).m();
}
```

If there are multiple cascaded accesses, you'll need to duplicate the
extension override for each one.

### extension_type_constructor_with_super_formal_parameter

_Extension type constructors can't declare super formal parameters._

#### Description

The analyzer produces this diagnostic when a constructor in an extension
type has a super parameter. Super parameters aren't valid because
extension types don't have a superclass.

#### Example

The following code produces this diagnostic because the named constructor
`n` contains a super parameter:

```dart
extension type E(int i) {
  E.n(this.i, [!super!].foo);
}
```

#### Common fixes

If you need the parameter, replace the super parameter with a normal
parameter:

```dart
extension type E(int i) {
  E.n(this.i, String foo);
}
```

If you don't need the parameter, remove the super parameter:

```dart
extension type E(int i) {
  E.n(this.i);
}
```

### extension_type_constructor_with_super_invocation

_Extension type constructors can't include super initializers._

#### Description

The analyzer produces this diagnostic when a constructor in an extension
type includes an invocation of a super constructor in the initializer
list. Because extension types don't have a superclass, there's no
constructor to invoke.

#### Example

The following code produces this diagnostic because the constructor `E.n`
invokes a super constructor in its initializer list:

```dart
extension type E(int i) {
  E.n() : i = 0, [!super!].n();
}
```

#### Common fixes

Remove the invocation of the super constructor:

```dart
extension type E(int i) {
  E.n() : i = 0;
}
```

### extension_type_declares_instance_field

_Extension types can't declare instance fields._

#### Description

The analyzer produces this diagnostic when there's a field declaration in
the body of an extension type declaration.

#### Example

The following code produces this diagnostic because the extension type `E`
declares a field named `f`:

```dart
extension type E(int i) {
  final int [!f!] = 0;
}
```

#### Common fixes

If you don't need the field, then remove it or replace it with a getter
and/or setter:

```dart
extension type E(int i) {
  int get f => 0;
}
```

If you need the field, then convert the extension type into a class:

```dart
class E {
  final int i;

  final int f = 0;

  E(this.i);
}
```

### extension_type_declares_member_of_object

_Extension types can't declare members with the same name as a member declared
by 'Object'._

#### Description

The analyzer produces this diagnostic when the body of an extension type
declaration contains a member with the same name as one of the members
declared by `Object`.

#### Example

The following code produces this diagnostic because the class `Object`
already defines a member named `hashCode`:

```dart
extension type E(int i) {
  int get [!hashCode!] => 0;
}
```

#### Common fixes

If you need a member with the implemented semantics, then rename the
member:

```dart
extension type E(int i) {
  int get myHashCode => 0;
}
```

If you don't need a member with the implemented semantics, then remove the
member:

```dart
extension type E(int i) {}
```

### extension_type_implements_disallowed_type

_Extension types can't implement '{0}'._

#### Description

The analyzer produces this diagnostic when an extension type implements a
type that it isn't allowed to implement.

#### Example

The following code produces this diagnostic because extension types can't
implement the type `dynamic`:

```dart
extension type A(int i) implements [!dynamic!] {}
```

#### Common fixes

Remove the disallowed type from the implements clause:

```dart
extension type A(int i) {}
```

### extension_type_implements_itself

_The extension type can't implement itself._

#### Description

The analyzer produces this diagnostic when an extension type implements
itself, either directly or indirectly.

#### Example

The following code produces this diagnostic because the extension type `A`
directly implements itself:

```dart
extension type [!A!](int i) implements A {}
```

The following code produces this diagnostic because the extension type `A`
indirectly implements itself (through `B`):

```dart
extension type [!A!](int i) implements B {}

extension type [!B!](int i) implements A {}
```

#### Common fixes

Break the cycle by removing a type from the implements clause of at least
one of the types involved in the cycle:

```dart
extension type A(int i) implements B {}

extension type B(int i) {}
```

### extension_type_implements_not_supertype

_'{0}' is not a supertype of '{1}', the representation type._

#### Description

The analyzer produces this diagnostic when an extension type implements a
type that isn't a supertype of the representation type.

#### Example

The following code produces this diagnostic because the extension type `A`
implements `String`, but `String` isn't a supertype of the representation
type `int`:

```dart
extension type A(int i) implements [!String!] {}
```

#### Common fixes

If the representation type is correct, then remove or replace the type in
the implements clause:

```dart
extension type A(int i) {}
```

If the representation type isn't correct, then replace it with the correct
type:

```dart
extension type A(String s) implements String {}
```

### extension_type_implements_representation_not_supertype

_'{0}', the representation type of '{1}', is not a supertype of '{2}', the
representation type of '{3}'._

#### Description

The analyzer produces this diagnostic when an extension type implements
another extension type, and the representation type of the implemented
extension type isn't a subtype of the representation type of the implementing
extension type.

#### Example

The following code produces this diagnostic because the extension type `B`
implements `A`, but the representation type of `A` (`num`) isn't a
subtype of the representation type of `B` (`String`):

```dart
extension type A(num i) {}

extension type B(String s) implements [!A!] {}
```

#### Common fixes

Either change the representation types of the two extension types so that
the representation type of the implemented type is a supertype of the
representation type of the implementing type:

```dart
extension type A(num i) {}

extension type B(int n) implements A {}
```

Or remove the implemented type from the implements clause:

```dart
extension type A(num i) {}

extension type B(String s) {}
```

### extension_type_inherited_member_conflict

_The extension type '{0}' has more than one distinct member named '{1}' from
implemented types._

#### Description

The analyzer produces this diagnostic when an extension type implements
two or more other types, and at least two of those types declare a member
with the same name.

#### Example

The following code produces this diagnostic because the extension type `C`
implements both `A` and `B`, and both declare a member named `m`:

```dart
class A {
  void m() {}
}

extension type B(A a) {
  void m() {}
}

extension type [!C!](A a) implements A, B {}
```

#### Common fixes

If the extension type doesn't need to implement all of the listed types,
then remove all but one of the types introducing the conflicting members:

```dart
class A {
  void m() {}
}

extension type B(A a) {
  void m() {}
}

extension type C(A a) implements A {}
```

If the extension type needs to implement all of the listed types but you
can rename the members in those types, then give the conflicting members
unique names:

```dart
class A {
  void m() {}
}

extension type B(A a) {
  void n() {}
}

extension type C(A a) implements A, B {}
```

### extension_type_representation_depends_on_itself

_The extension type representation can't depend on itself._

#### Description

The analyzer produces this diagnostic when an extension type has a
representation type that depends on the extension type itself, either
directly or indirectly.

#### Example

The following code produces this diagnostic because the representation
type of the extension type `A` depends on `A` directly:

```dart
extension type [!A!](A a) {}
```

The following two code examples produce this diagnostic because the
representation type of the extension type `A` depends on `A`
indirectly through the extension type `B`:

```dart
extension type [!A!](B b) {}

extension type [!B!](A a) {}
```

```dart
extension type [!A!](List<B> b) {}

extension type [!B!](List<A> a) {}
```

#### Common fixes

Remove the dependency by choosing a different representation type for at
least one of the types in the cycle:

```dart
extension type A(String s) {}
```

### extension_type_representation_type_bottom

_The representation type can't be a bottom type._

#### Description

The analyzer produces this diagnostic when the representation type of an
extension type is the [bottom type][] `Never`. The type `Never` can't be
the representation type of an extension type because there are no values
that can be extended.

#### Example

The following code produces this diagnostic because the representation
type of the extension type `E` is `Never`:

```dart
extension type E([!Never!] n) {}
```

#### Common fixes

Replace the extension type with a different type:

```dart
extension type E(String s) {}
```

### extension_type_with_abstract_member

_'{0}' must have a method body because '{1}' is an extension type._

#### Description

The analyzer produces this diagnostic when an extension type declares an
abstract member. Because extension type member references are resolved
statically, an abstract member in an extension type could never be
executed.

#### Example

The following code produces this diagnostic because the method `m` in the
extension type `E` is abstract:

```dart
extension type E(String s) {
  [!void m();!]
}
```

#### Common fixes

If the member is intended to be executable, then provide an implementation
of the member:

```dart
extension type E(String s) {
  void m() {}
}
```

If the member isn't intended to be executable, then remove it:

```dart
extension type E(String s) {}
```

### external_with_initializer

_External fields can't have initializers._

_External variables can't have initializers._

#### Description

The analyzer produces this diagnostic when a field or variable marked with
the keyword `external` has an initializer, or when an external field is
initialized in a constructor.

#### Examples

The following code produces this diagnostic because the external field `x`
is assigned a value in an initializer:

```dart
class C {
  external int x;
  C() : [!x!] = 0;
}
```

The following code produces this diagnostic because the external field `x`
has an initializer:

```dart
class C {
  external final int [!x!] = 0;
}
```

The following code produces this diagnostic because the external top level
variable `x` has an initializer:

```dart
external final int [!x!] = 0;
```

#### Common fixes

Remove the initializer:

```dart
class C {
  external final int x;
}
```

### extra_annotation_on_struct_field

_Fields in a struct class must have exactly one annotation indicating the native
type._

#### Description

The analyzer produces this diagnostic when a field in a subclass of
`Struct` has more than one annotation describing the native type of the
field.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `x` has two
annotations describing the native type of the field:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int32()
  [!@Int16()!]
  external int x;
}
```

#### Common fixes

Remove all but one of the annotations:

```dart
import 'dart:ffi';
final class C extends Struct {
  @Int32()
  external int x;
}
```

### extra_positional_arguments

_Too many positional arguments: {0} expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has more positional arguments than the method or function allows.

#### Example

The following code produces this diagnostic because `f` defines 2
parameters but is invoked with 3 arguments:

```dart
void f(int a, int b) {}
void g() {
  f(1, 2, [!3!]);
}
```

#### Common fixes

Remove the arguments that don't correspond to parameters:

```dart
void f(int a, int b) {}
void g() {
  f(1, 2);
}
```

### extra_positional_arguments_could_be_named

_Too many positional arguments: {0} expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has more positional arguments than the method or function allows, but the
method or function defines named parameters.

#### Example

The following code produces this diagnostic because `f` defines 2
positional parameters but has a named parameter that could be used for the
third argument:

```dart
void f(int a, int b, {int? c}) {}
void g() {
  f(1, 2, [!3!]);
}
```

#### Common fixes

If some of the arguments should be values for named parameters, then add
the names before the arguments:

```dart
void f(int a, int b, {int? c}) {}
void g() {
  f(1, 2, c: 3);
}
```

Otherwise, remove the arguments that don't correspond to positional
parameters:

```dart
void f(int a, int b, {int? c}) {}
void g() {
  f(1, 2);
}
```

### extra_size_annotation_carray

_'Array's must have exactly one 'Array' annotation._

#### Description

The analyzer produces this diagnostic when a field in a subclass of
`Struct` has more than one annotation describing the size of the native
array.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `a0` has two
annotations that specify the size of the native array:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Array(4)
  [!@Array(8)!]
  external Array<Uint8> a0;
}
```

#### Common fixes

Remove all but one of the annotations:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Array(8)
  external Array<Uint8> a0;
}
```

### ffi_native_invalid_duplicate_default_asset

_There may be at most one @DefaultAsset annotation on a library._

#### Description

The analyzer produces this diagnostic when a library directive has more
than one `DefaultAsset` annotation associated with it.

#### Example

The following code produces this diagnostic because the library directive
has two `DefaultAsset` annotations associated with it:

```dart
@DefaultAsset('a')
@[!DefaultAsset!]('b')
library;

import 'dart:ffi';
```

#### Common fixes

Remove all but one of the `DefaultAsset` annotations:

```dart
@DefaultAsset('a')
library;

import 'dart:ffi';
```

### ffi_native_invalid_multiple_annotations

_Native functions and fields must have exactly one `@Native` annotation._

#### Description

The analyzer produces this diagnostic when there is more than one `Native`
annotation on a single declaration.

#### Example

The following code produces this diagnostic because the function `f` has
two `Native` annotations associated with it:

```dart
import 'dart:ffi';

@Native<Int32 Function(Int32)>()
@[!Native!]<Int32 Function(Int32)>(isLeaf: true)
external int f(int v);
```

#### Common fixes

Remove all but one of the annotations:

```dart
import 'dart:ffi';

@Native<Int32 Function(Int32)>(isLeaf: true)
external int f(int v);
```

### ffi_native_must_be_external

_Native functions must be declared external._

#### Description

The analyzer produces this diagnostic when a function annotated as being
`@Native` isn't marked as `external`.

#### Example

The following code produces this diagnostic because the function `free` is
annotated as being `@Native`, but the function isn't marked as `external`:

```dart
import 'dart:ffi';

[!@Native<Void Function(Pointer<Void>)>()
void free(Pointer<Void> ptr) {}!]
```

#### Common fixes

If the function is a native function, then add the modifier `external`
before the return type:

```dart
import 'dart:ffi';

@Native<Void Function(Pointer<Void>)>()
external void free(Pointer<Void> ptr);
```

### ffi_native_unexpected_number_of_parameters

_Unexpected number of Native annotation parameters. Expected {0} but has {1}._

#### Description

The analyzer produces this diagnostic when the number of parameters in the
function type used as a type argument for the `@Native` annotation doesn't
match the number of parameters in the function being annotated.

#### Example

The following code produces this diagnostic because the function type used
as a type argument for the `@Native` annotation (`Void Function(Double)`)
has one argument and the type of the annotated function
(`void f(double, double)`) has two arguments:

```dart
import 'dart:ffi';

[!@Native<Void Function(Double)>(symbol: 'f')
external void f(double x, double y);!]
```

#### Common fixes

If the annotated function is correct, then update the function type in the
`@Native` annotation to match:

```dart
import 'dart:ffi';

@Native<Void Function(Double, Double)>(symbol: 'f')
external void f(double x, double y);
```

If the function type in the `@Native` annotation is correct, then update
the annotated function to match:

```dart
import 'dart:ffi';

@Native<Void Function(Double)>(symbol: 'f')
external void f(double x);
```

### ffi_native_unexpected_number_of_parameters_with_receiver

_Unexpected number of Native annotation parameters. Expected {0} but has {1}.
Native instance method annotation must have receiver as first argument._

#### Description

The analyzer produces this diagnostic when the type argument used on the
`@Native` annotation of a native method doesn't include a type for the
receiver of the method.

#### Example

The following code produces this diagnostic because the type argument on
the `@Native` annotation (`Void Function(Double)`) doesn't include a type
for the receiver of the method:

```dart
import 'dart:ffi';

class C {
  [!@Native<Void Function(Double)>()
  external void f(double x);!]
}
```

#### Common fixes

Add an initial parameter whose type is the same as the class in which the
native method is being declared:

```dart
import 'dart:ffi';

class C {
  @Native<Void Function(C, Double)>()
  external void f(double x);
}
```

### field_initialized_by_multiple_initializers

_The field '{0}' can't be initialized twice in the same constructor._

#### Description

The analyzer produces this diagnostic when the initializer list of a
constructor initializes a field more than once. There is no value to allow
both initializers because only the last value is preserved.

#### Example

The following code produces this diagnostic because the field `f` is being
initialized twice:

```dart
class C {
  int f;

  C() : f = 0, [!f!] = 1;
}
```

#### Common fixes

Remove one of the initializers:

```dart
class C {
  int f;

  C() : f = 0;
}
```

### field_initialized_in_initializer_and_declaration

_Fields can't be initialized in the constructor if they are final and were
already initialized at their declaration._

#### Description

The analyzer produces this diagnostic when a final field is initialized in
both the declaration of the field and in an initializer in a constructor.
Final fields can only be assigned once, so it can't be initialized in both
places.

#### Example

The following code produces this diagnostic because `f` is :

```dart
class C {
  final int f = 0;
  C() : [!f!] = 1;
}
```

#### Common fixes

If the initialization doesn't depend on any values passed to the
constructor, and if all of the constructors need to initialize the field to
the same value, then remove the initializer from the constructor:

```dart
class C {
  final int f = 0;
  C();
}
```

If the initialization depends on a value passed to the constructor, or if
different constructors need to initialize the field differently, then
remove the initializer in the field's declaration:

```dart
class C {
  final int f;
  C() : f = 1;
}
```

### field_initialized_in_parameter_and_initializer

_Fields can't be initialized in both the parameter list and the initializers._

#### Description

The analyzer produces this diagnostic when a field is initialized in both
the parameter list and in the initializer list of a constructor.

#### Example

The following code produces this diagnostic because the field `f` is
initialized both by an initializing formal parameter and in the
initializer list:

```dart
class C {
  int f;

  C(this.f) : [!f!] = 0;
}
```

#### Common fixes

If the field should be initialized by the parameter, then remove the
initialization in the initializer list:

```dart
class C {
  int f;

  C(this.f);
}
```

If the field should be initialized in the initializer list and the
parameter isn't needed, then remove the parameter:

```dart
class C {
  int f;

  C() : f = 0;
}
```

If the field should be initialized in the initializer list and the
parameter is needed, then make it a normal parameter:

```dart
class C {
  int f;

  C(int g) : f = g * 2;
}
```

### field_initializer_factory_constructor

_Initializing formal parameters can't be used in factory constructors._

#### Description

The analyzer produces this diagnostic when a factory constructor has an
initializing formal parameter. Factory constructors can't assign values to
fields because no instance is created; hence, there is no field to assign.

#### Example

The following code produces this diagnostic because the factory constructor
uses an initializing formal parameter:

```dart
class C {
  int? f;

  factory C([!this.f!]) => throw 0;
}
```

#### Common fixes

Replace the initializing formal parameter with a normal parameter:

```dart
class C {
  int? f;

  factory C(int f) => throw 0;
}
```

### field_initializer_in_struct

_Constructors in subclasses of 'Struct' and 'Union' can't have field
initializers._

#### Description

The analyzer produces this diagnostic when a constructor in a subclass of
either `Struct` or `Union` has one or more field initializers.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the class `C` has a
constructor with an initializer for the field `f`:

```dart
// @dart = 2.9
import 'dart:ffi';

final class C extends Struct {
  @Int32()
  int f;

  C() : [!f = 0!];
}
```

#### Common fixes

Remove the field initializer:

```dart
// @dart = 2.9
import 'dart:ffi';

final class C extends Struct {
  @Int32()
  int f;

  C();
}
```

### field_initializer_not_assignable

_The initializer type '{0}' can't be assigned to the field type '{1}' in a const
constructor._

_The initializer type '{0}' can't be assigned to the field type '{1}'._

#### Description

The analyzer produces this diagnostic when the initializer list of a
constructor initializes a field to a value that isn't assignable to the
field.

#### Example

The following code produces this diagnostic because `0` has the type `int`,
and an `int` can't be assigned to a field of type `String`:

```dart
class C {
  String s;

  C() : s = [!0!];
}
```

#### Common fixes

If the type of the field is correct, then change the value assigned to it
so that the value has a valid type:

```dart
class C {
  String s;

  C() : s = '0';
}
```

If the type of the value is correct, then change the type of the field to
allow the assignment:

```dart
class C {
  int s;

  C() : s = 0;
}
```

### field_initializer_outside_constructor

_Field formal parameters can only be used in a constructor._

_Initializing formal parameters can only be used in constructors._

#### Description

The analyzer produces this diagnostic when an initializing formal
parameter is used in the parameter list for anything other than a
constructor.

#### Example

The following code produces this diagnostic because the initializing
formal parameter `this.x` is being used in the method `m`:

```dart
class A {
  int x = 0;

  m([[!this.x!] = 0]) {}
}
```

#### Common fixes

Replace the initializing formal parameter with a normal parameter and
assign the field within the body of the method:

```dart
class A {
  int x = 0;

  m([int x = 0]) {
    this.x = x;
  }
}
```

### field_initializer_redirecting_constructor

_The redirecting constructor can't have a field initializer._

#### Description

The analyzer produces this diagnostic when a redirecting constructor
initializes a field in the object. This isn't allowed because the instance
that has the field hasn't been created at the point at which it should be
initialized.

#### Examples

The following code produces this diagnostic because the constructor
`C.zero`, which redirects to the constructor `C`, has an initializing
formal parameter that initializes the field `f`:

```dart
class C {
  int f;

  C(this.f);

  C.zero([!this.f!]) : this(f);
}
```

The following code produces this diagnostic because the constructor
`C.zero`, which redirects to the constructor `C`, has an initializer that
initializes the field `f`:

```dart
class C {
  int f;

  C(this.f);

  C.zero() : [!f = 0!], this(1);
}
```

#### Common fixes

If the initialization is done by an initializing formal parameter, then
use a normal parameter:

```dart
class C {
  int f;

  C(this.f);

  C.zero(int f) : this(f);
}
```

If the initialization is done in an initializer, then remove the
initializer:

```dart
class C {
  int f;

  C(this.f);

  C.zero() : this(0);
}
```

### field_initializing_formal_not_assignable

_The parameter type '{0}' is incompatible with the field type '{1}'._

#### Description

The analyzer produces this diagnostic when the type of an initializing
formal parameter isn't assignable to the type of the field being
initialized.

#### Example

The following code produces this diagnostic because the initializing
formal parameter has the type `String`, but the type of the field is
`int`. The parameter must have a type that is a subtype of the field's
type.

```dart
class C {
  int f;

  C([!String this.f!]);
}
```

#### Common fixes

If the type of the field is incorrect, then change the type of the field to
match the type of the parameter, and consider removing the type from the
parameter:

```dart
class C {
  String f;

  C(this.f);
}
```

If the type of the parameter is incorrect, then remove the type of the
parameter:

```dart
class C {
  int f;

  C(this.f);
}
```

If the types of both the field and the parameter are correct, then use an
initializer rather than an initializing formal parameter to convert the
parameter value into a value of the correct type:

```dart
class C {
  int f;

  C(String s) : f = int.parse(s);
}
```

### field_in_struct_with_initializer

_Fields in subclasses of 'Struct' and 'Union' can't have initializers._

#### Description

The analyzer produces this diagnostic when a field in a subclass of
`Struct` has an initializer.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `p` has an
initializer:

```dart
// @dart = 2.9
import 'dart:ffi';

final class C extends Struct {
  Pointer [!p!] = nullptr;
}
```

#### Common fixes

Remove the initializer:

```dart
// @dart = 2.9
import 'dart:ffi';

final class C extends Struct {
  Pointer p;
}
```

### field_must_be_external_in_struct

_Fields of 'Struct' and 'Union' subclasses must be marked external._

#### Description

The analyzer produces this diagnostic when a field in a subclass of either
`Struct` or `Union` isn't marked as being `external`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `a` isn't
marked as being `external`:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int16()
  int [!a!];
}
```

#### Common fixes

Add the required `external` modifier:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int16()
  external int a;
}
```

### final_initialized_in_declaration_and_constructor

_'{0}' is final and was given a value when it was declared, so it can't be set
to a new value._

#### Description

The analyzer produces this diagnostic when a final field is initialized
twice: once where it's declared and once by a constructor's parameter.

#### Example

The following code produces this diagnostic because the field `f` is
initialized twice:

```dart
class C {
  final int f = 0;

  C(this.[!f!]);
}
```

#### Common fixes

If the field should have the same value for all instances, then remove the
initialization in the parameter list:

```dart
class C {
  final int f = 0;

  C();
}
```

If the field can have different values in different instances, then remove
the initialization in the declaration:

```dart
class C {
  final int f;

  C(this.f);
}
```

### final_not_initialized

_The final variable '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a final field or variable isn't
initialized.

#### Example

The following code produces this diagnostic because `x` doesn't have an
initializer:

```dart
final [!x!];
```

#### Common fixes

For variables and static fields, you can add an initializer:

```dart
final x = 0;
```

For instance fields, you can add an initializer as shown in the previous
example, or you can initialize the field in every constructor. You can
initialize the field by using an initializing formal parameter:

```dart
class C {
  final int x;
  C(this.x);
}
```

You can also initialize the field by using an initializer in the
constructor:

```dart
class C {
  final int x;
  C(int y) : x = y * 2;
}
```

### final_not_initialized_constructor

_All final variables must be initialized, but '{0}' and '{1}' aren't._

_All final variables must be initialized, but '{0}' isn't._

_All final variables must be initialized, but '{0}', '{1}', and {2} others
aren't._

#### Description

The analyzer produces this diagnostic when a class defines one or more
final instance fields without initializers and has at least one constructor
that doesn't initialize those fields. All final instance fields must be
initialized when the instance is created, either by the field's initializer
or by the constructor.

#### Example

The following code produces this diagnostic:

```dart
class C {
  final String value;

  [!C!]();
}
```

#### Common fixes

If the value should be passed in to the constructor directly, then use an
initializing formal parameter to initialize the field `value`:

```dart
class C {
  final String value;

  C(this.value);
}
```

If the value should be computed indirectly from a value provided by the
caller, then add a parameter and include an initializer:

```dart
class C {
  final String value;

  C(Object o) : value = o.toString();
}
```

If the value of the field doesn't depend on values that can be passed to
the constructor, then add an initializer for the field as part of the field
declaration:

```dart
class C {
  final String value = '';

  C();
}
```

If the value of the field doesn't depend on values that can be passed to
the constructor but different constructors need to initialize it to
different values, then add an initializer for the field in the initializer
list:

```dart
class C {
  final String value;

  C() : value = '';

  C.named() : value = 'c';
}
```

However, if the value is the same for all instances, then consider using a
static field instead of an instance field:

```dart
class C {
  static const String value = '';

  C();
}
```

### flutter_field_not_map

_The value of the 'flutter' field is expected to be a map._

#### Description

The analyzer produces this diagnostic when the value of the `flutter` key
isn't a map.

#### Example

The following code produces this diagnostic because the value of the
top-level `flutter` key is a string:

```yaml
name: example
flutter: [!true!]
```

#### Common fixes

If you need to specify Flutter-specific options, then change the value to
be a map:

```yaml
name: example
flutter:
  uses-material-design: true
```

If you don't need to specify Flutter-specific options, then remove the
`flutter` key:

```yaml
name: example
```

### for_in_of_invalid_element_type

_The type '{0}' used in the 'for' loop must implement '{1}' with a type argument
that can be assigned to '{2}'._

#### Description

The analyzer produces this diagnostic when the `Iterable` or `Stream` in a
for-in loop has an element type that can't be assigned to the loop
variable.

#### Example

The following code produces this diagnostic because `<String>[]` has an
element type of `String`, and `String` can't be assigned to the type of `e`
(`int`):

```dart
void f() {
  for (int e in [!<String>[]!]) {
    print(e);
  }
}
```

#### Common fixes

If the type of the loop variable is correct, then update the type of the
iterable:

```dart
void f() {
  for (int e in <int>[]) {
    print(e);
  }
}
```

If the type of the iterable is correct, then update the type of the loop
variable:

```dart
void f() {
  for (String e in <String>[]) {
    print(e);
  }
}
```

### for_in_of_invalid_type

_The type '{0}' used in the 'for' loop must implement '{1}'._

#### Description

The analyzer produces this diagnostic when the expression following `in` in
a for-in loop has a type that isn't a subclass of `Iterable`.

#### Example

The following code produces this diagnostic because `m` is a `Map`, and
`Map` isn't a subclass of `Iterable`:

```dart
void f(Map<String, String> m) {
  for (String s in [!m!]) {
    print(s);
  }
}
```

#### Common fixes

Replace the expression with one that produces an iterable value:

```dart
void f(Map<String, String> m) {
  for (String s in m.values) {
    print(s);
  }
}
```

### for_in_with_const_variable

_A for-in loop variable can't be a 'const'._

#### Description

The analyzer produces this diagnostic when the loop variable declared in a
for-in loop is declared to be a `const`. The variable can't be a `const`
because the value can't be computed at compile time.

#### Example

The following code produces this diagnostic because the loop variable `x`
is declared to be a `const`:

```dart
void f() {
  for ([!const!] x in [0, 1, 2]) {
    print(x);
  }
}
```

#### Common fixes

If there's a type annotation, then remove the `const` modifier from the
declaration.

If there's no type, then replace the `const` modifier with `final`, `var`,
or a type annotation:

```dart
void f() {
  for (final x in [0, 1, 2]) {
    print(x);
  }
}
```

### generic_method_type_instantiation_on_dynamic

_A method tear-off on a receiver whose type is 'dynamic' can't have type
arguments._

#### Description

The analyzer produces this diagnostic when an instance method is being torn
off from a receiver whose type is `dynamic`, and the tear-off includes type
arguments. Because the analyzer can't know how many type parameters the
method has, or whether it has any type parameters, there's no way it can
validate that the type arguments are correct. As a result, the type
arguments aren't allowed.

#### Example

The following code produces this diagnostic because the type of `p` is
`dynamic` and the tear-off of `m` has type arguments:

```dart
void f(dynamic list) {
  [!list.fold!]<int>;
}
```

#### Common fixes

If you can use a more specific type than `dynamic`, then change the type of
the receiver:

```dart
void f(List<Object> list) {
  list.fold<int>;
}
```

If you can't use a more specific type, then remove the type arguments:

```dart
void f(dynamic list) {
  list.cast;
}
```

### generic_struct_subclass

_The class '{0}' can't extend 'Struct' or 'Union' because '{0}' is generic._

#### Description

The analyzer produces this diagnostic when a subclass of either `Struct`
or `Union` has a type parameter.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the class `S` defines
the type parameter `T`:

```dart
import 'dart:ffi';

final class [!S!]<T> extends Struct {
  external Pointer notEmpty;
}
```

#### Common fixes

Remove the type parameters from the class:

```dart
import 'dart:ffi';

final class S extends Struct {
  external Pointer notEmpty;
}
```

### getter_not_subtype_setter_types

_The return type of getter '{0}' is '{1}' which isn't a subtype of the type
'{2}' of its setter '{3}'._

#### Description

The analyzer produces this diagnostic when the return type of a getter
isn't a subtype of the type of the parameter of a setter with the same
name.

The subtype relationship is a requirement whether the getter and setter are
in the same class or whether one of them is in a superclass of the other.

#### Example

The following code produces this diagnostic because the return type of the
getter `x` is `num`, the parameter type of the setter `x` is `int`, and
`num` isn't a subtype of `int`:

```dart
class C {
  num get [!x!] => 0;

  set x(int y) {}
}
```

#### Common fixes

If the type of the getter is correct, then change the type of the setter:

```dart
class C {
  num get x => 0;

  set x(num y) {}
}
```

If the type of the setter is correct, then change the type of the getter:

```dart
class C {
  int get x => 0;

  set x(int y) {}
}
```

### illegal_async_generator_return_type

_Functions marked 'async*' must have a return type that is a supertype of
'Stream<T>' for some type 'T'._

#### Description

The analyzer produces this diagnostic when the body of a function has the
`async*` modifier even though the return type of the function isn't either
`Stream` or a supertype of `Stream`.

#### Example

The following code produces this diagnostic because the body of the
function `f` has the 'async*' modifier even though the return type `int`
isn't a supertype of `Stream`:

```dart
[!int!] f() async* {}
```

#### Common fixes

If the function should be asynchronous, then change the return type to be
either `Stream` or a supertype of `Stream`:

```dart
Stream<int> f() async* {}
```

If the function should be synchronous, then remove the `async*` modifier:

```dart
int f() => 0;
```

### illegal_async_return_type

_Functions marked 'async' must have a return type which is a supertype of
'Future'._

#### Description

The analyzer produces this diagnostic when the body of a function has the
`async` modifier even though the return type of the function isn't
assignable to `Future`.

#### Example

The following code produces this diagnostic because the body of the
function `f` has the `async` modifier even though the return type isn't
assignable to `Future`:

```dart
[!int!] f() async {
  return 0;
}
```

#### Common fixes

If the function should be asynchronous, then change the return type to be
assignable to `Future`:

```dart
Future<int> f() async {
  return 0;
}
```

If the function should be synchronous, then remove the `async` modifier:

```dart
int f() => 0;
```

### illegal_concrete_enum_member

_A concrete instance member named '{0}' can't be declared in a class that
implements 'Enum'._

_A concrete instance member named '{0}' can't be inherited from '{1}' in a class
that implements 'Enum'._

#### Description

The analyzer produces this diagnostic when either an enum declaration, a
class that implements `Enum`, or a mixin with a superclass constraint of
`Enum`, declares or inherits a concrete instance member named either
`index`, `hashCode`, or `==`.

#### Examples

The following code produces this diagnostic because the enum `E` declares
an instance getter named `index`:

```dart
enum E {
  v;

  int get [!index!] => 0;
}
```

The following code produces this diagnostic because the class `C`, which
implements `Enum`, declares an instance field named `hashCode`:

```dart
abstract class C implements Enum {
  int [!hashCode!] = 0;
}
```

The following code produces this diagnostic because the class `C`, which
indirectly implements `Enum` through the class `A`, declares an instance
getter named `hashCode`:

```dart
abstract class A implements Enum {}

abstract class C implements A {
  int get [!hashCode!] => 0;
}
```

The following code produces this diagnostic because the mixin `M`, which
has `Enum` in the `on` clause, declares an explicit operator named `==`:

```dart
mixin M on Enum {
  bool operator [!==!](Object other) => false;
}
```

#### Common fixes

Rename the conflicting member:

```dart
enum E {
  v;

  int get getIndex => 0;
}
```

### illegal_enum_values

_An instance member named 'values' can't be declared in a class that implements
'Enum'._

_An instance member named 'values' can't be inherited from '{0}' in a class that
implements 'Enum'._

#### Description

The analyzer produces this diagnostic when either a class that implements
`Enum` or a mixin with a superclass constraint of `Enum` has an instance
member named `values`.

#### Examples

The following code produces this diagnostic because the class `C`, which
implements `Enum`, declares an instance field named `values`:

```dart
abstract class C implements Enum {
  int get [!values!] => 0;
}
```

The following code produces this diagnostic because the class `B`, which
implements `Enum`, inherits an instance method named `values` from `A`:

```dart
abstract class A {
  int values() => 0;
}

abstract class [!B!] extends A implements Enum {}
```

#### Common fixes

Change the name of the conflicting member:

```dart
abstract class C implements Enum {
  int get value => 0;
}
```

### illegal_sync_generator_return_type

_Functions marked 'sync*' must have a return type that is a supertype of
'Iterable<T>' for some type 'T'._

#### Description

The analyzer produces this diagnostic when the body of a function has the
`sync*` modifier even though the return type of the function isn't either
`Iterable` or a supertype of `Iterable`.

#### Example

The following code produces this diagnostic because the body of the
function `f` has the 'sync*' modifier even though the return type `int`
isn't a supertype of `Iterable`:

```dart
[!int!] f() sync* {}
```

#### Common fixes

If the function should return an iterable, then change the return type to
be either `Iterable` or a supertype of `Iterable`:

```dart
Iterable<int> f() sync* {}
```

If the function should return a single value, then remove the `sync*`
modifier:

```dart
int f() => 0;
```

### implements_non_class

_Classes and mixins can only implement other classes and mixins._

#### Description

The analyzer produces this diagnostic when a name used in the `implements`
clause of a class or mixin declaration is defined to be something other
than a class or mixin.

#### Example

The following code produces this diagnostic because `x` is a variable
rather than a class or mixin:

```dart
var x;
class C implements [!x!] {}
```

#### Common fixes

If the name is the name of an existing class or mixin that's already being
imported, then add a prefix to the import so that the local definition of
the name doesn't shadow the imported name.

If the name is the name of an existing class or mixin that isn't being
imported, then add an import, with a prefix, for the library in which it's
declared.

Otherwise, either replace the name in the `implements` clause with the name
of an existing class or mixin, or remove the name from the `implements`
clause.

### implements_repeated

_'{0}' can only be implemented once._

#### Description

The analyzer produces this diagnostic when a single class is specified more
than once in an `implements` clause.

#### Example

The following code produces this diagnostic because `A` is in the list
twice:

```dart
class A {}
class B implements A, [!A!] {}
```

#### Common fixes

Remove all except one occurrence of the class name:

```dart
class A {}
class B implements A {}
```

### implements_super_class

_'{0}' can't be used in both the 'extends' and 'implements' clauses._

_'{0}' can't be used in both the 'extends' and 'with' clauses._

#### Description

The analyzer produces this diagnostic when a class is listed in the
`extends` clause of a class declaration and also in either the
`implements` or `with` clause of the same declaration.

#### Example

The following code produces this diagnostic because the class `A` is used
in both the `extends` and `implements` clauses for the class `B`:

```dart
class A {}

class B extends A implements [!A!] {}
```

The following code produces this diagnostic because the class `A` is used
in both the `extends` and `with` clauses for the class `B`:

```dart
mixin class A {}

class B extends A with [!A!] {}
```

#### Common fixes

If you want to inherit the implementation from the class, then remove the
class from the `implements` clause:

```dart
class A {}

class B extends A {}
```

If you don't want to inherit the implementation from the class, then remove
the `extends` clause:

```dart
class A {}

class B implements A {}
```

### implicit_super_initializer_missing_arguments

_The implicitly invoked unnamed constructor from '{0}' has required parameters._

#### Description

The analyzer produces this diagnostic when a constructor implicitly
invokes the unnamed constructor from the superclass, the unnamed
constructor of the superclass has a required parameter, and there's no
super parameter corresponding to the required parameter.

#### Examples

The following code produces this diagnostic because the unnamed
constructor in the class `B` implicitly invokes the unnamed constructor in
the class `A`, but the constructor in `A` has a required positional
parameter named `x`:

```dart
class A {
  A(int x);
}

class B extends A {
  [!B!]();
}
```

The following code produces this diagnostic because the unnamed
constructor in the class `B` implicitly invokes the unnamed constructor in
the class `A`, but the constructor in `A` has a required named parameter
named `x`:

```dart
class A {
  A({required int x});
}

class B extends A {
  [!B!]();
}
```

#### Common fixes

If you can add a parameter to the constructor in the subclass, then add a
super parameter corresponding to the required parameter in the superclass'
constructor. The new parameter can either be required:

```dart
class A {
  A({required int x});
}

class B extends A {
  B({required super.x});
}
```

or it can be optional:

```dart
class A {
  A({required int x});
}

class B extends A {
  B({super.x = 0});
}
```

If you can't add a parameter to the constructor in the subclass, then add
an explicit super constructor invocation with the required argument:

```dart
class A {
  A(int x);
}

class B extends A {
  B() : super(0);
}
```

### implicit_this_reference_in_initializer

_The instance member '{0}' can't be accessed in an initializer._

#### Description

The analyzer produces this diagnostic when it finds a reference to an
instance member in a constructor's initializer list.

#### Example

The following code produces this diagnostic because `defaultX` is an
instance member:

```dart
class C {
  int x;

  C() : x = [!defaultX!];

  int get defaultX => 0;
}
```

#### Common fixes

If the member can be made static, then do so:

```dart
class C {
  int x;

  C() : x = defaultX;

  static int get defaultX => 0;
}
```

If not, then replace the reference in the initializer with a different
expression that doesn't use an instance member:

```dart
class C {
  int x;

  C() : x = 0;

  int get defaultX => 0;
}
```

### import_deferred_library_with_load_function

_The imported library defines a top-level function named 'loadLibrary' that is
hidden by deferring this library._

#### Description

The analyzer produces this diagnostic when a library that declares a
function named `loadLibrary` is imported using a deferred import. A
deferred import introduces an implicit function named `loadLibrary`. This
function is used to load the contents of the deferred library, and the
implicit function hides the explicit declaration in the deferred library.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

Given a file `a.dart` that defines a function named `loadLibrary`:

```dart
void loadLibrary(Library library) {}

class Library {}
```

The following code produces this diagnostic because the implicit
declaration of `a.loadLibrary` is hiding the explicit declaration of
`loadLibrary` in `a.dart`:

```dart
[!import 'a.dart' deferred as a;!]

void f() {
  a.Library();
}
```

#### Common fixes

If the imported library isn't required to be deferred, then remove the
keyword `deferred`:

```dart
import 'a.dart' as a;

void f() {
  a.Library();
}
```

If the imported library is required to be deferred and you need to
reference the imported function, then rename the function in the imported
library:

```dart
void populateLibrary(Library library) {}

class Library {}
```

If the imported library is required to be deferred and you don't need to
reference the imported function, then add a `hide` clause:

```dart
import 'a.dart' deferred as a hide loadLibrary;

void f() {
  a.Library();
}
```

### import_internal_library

_The library '{0}' is internal and can't be imported._

#### Description

The analyzer produces this diagnostic when it finds an import whose `dart:`
URI references an internal library.

#### Example

The following code produces this diagnostic because `_interceptors` is an
internal library:

```dart
import [!'dart:_interceptors'!];
```

#### Common fixes

Remove the import directive.

### import_of_legacy_library_into_null_safe

_The library '{0}' is legacy, and shouldn't be imported into a null safe
library._

#### Description

The analyzer produces this diagnostic when a library that is null safe
imports a library that isn't null safe.

#### Example

Given a file `a.dart` that contains the following:

```dart
// @dart = 2.9

class A {}
```

The following code produces this diagnostic because a library that null
safe is importing a library that isn't null safe:

```dart
import [!'a.dart'!];

A? f() => null;
```

#### Common fixes

If you can migrate the imported library to be null safe, then migrate it
and update or remove the migrated library's language version.

If you can't migrate the imported library, then the importing library
needs to have a language version that is before 2.12, when null safety was
enabled by default.

### import_of_non_library

_The imported library '{0}' can't have a part-of directive._

#### Description

The analyzer produces this diagnostic when a [part file][] is imported
into a library.

#### Example

Given a [part file][] named `part.dart` containing the following:

```dart
part of lib;
```

The following code produces this diagnostic because imported files can't
have a part-of directive:

```dart
library lib;

import [!'part.dart'!];
```

#### Common fixes

Import the library that contains the [part file][] rather than the
[part file][] itself.

### inconsistent_inheritance

_Superinterfaces don't have a valid override for '{0}': {1}._

#### Description

The analyzer produces this diagnostic when a class inherits two or more
conflicting signatures for a member and doesn't provide an implementation
that satisfies all the inherited signatures.

#### Example

The following code produces this diagnostic because `C` is inheriting the
declaration of `m` from `A`, and that implementation isn't consistent with
the signature of `m` that's inherited from `B`:

```dart
class A {
  void m({int? a}) {}
}

class B {
  void m({int? b}) {}
}

class [!C!] extends A implements B {
}
```

#### Common fixes

Add an implementation of the method that satisfies all the inherited
signatures:

```dart
class A {
  void m({int? a}) {}
}

class B {
  void m({int? b}) {}
}

class C extends A implements B {
  void m({int? a, int? b}) {}
}
```

### inconsistent_language_version_override

_Parts must have exactly the same language version override as the library._

#### Description

The analyzer produces this diagnostic when a [part file][] has a language
version override comment that specifies a different language version than
the one being used for the library to which the part belongs.

#### Example

Given a [part file][] named `part.dart` that contains the following:

```dart
// @dart = 2.14
part of 'test.dart';
```

The following code produces this diagnostic because the parts of a library
must have the same language version as the defining compilation unit:

```dart
// @dart = 2.15
part [!'part.dart'!];
```

#### Common fixes

Remove the language version override from the [part file][], so that it
implicitly uses the same version as the defining compilation unit:

```dart
part of 'test.dart';
```

If necessary, either adjust the language version override in the defining
compilation unit to be appropriate for the code in the part, or migrate
the code in the [part file][] to be consistent with the new language
version.

### inconsistent_pattern_variable_logical_or

_The variable '{0}' has a different type and/or finality in this branch of the
logical-or pattern._

#### Description

The analyzer produces this diagnostic when a pattern variable that is
declared on all branches of a logical-or pattern doesn't have the same
type on every branch. It is also produced when the variable has a
different finality on different branches. A pattern variable declared on
multiple branches of a logical-or pattern is required to have the same
type and finality in each branch, so that the type and finality of the
variable can be known in code that's guarded by the logical-or pattern.

#### Examples

The following code produces this diagnostic because the variable `a` is
defined to be an `int` on one branch and a `double` on the other:

```dart
void f(Object? x) {
  if (x case (int a) || (double [!a!])) {
    print(a);
  }
}
```

The following code produces this diagnostic because the variable `a` is
`final` in the first branch and isn't `final` in the second branch:

```dart
void f(Object? x) {
  if (x case (final int a) || (int [!a!])) {
    print(a);
  }
}
```

#### Common fixes

If the finality of the variable is different, decide whether it should be
`final` or not `final` and make the cases consistent:

```dart
void f(Object? x) {
  if (x case (int a) || (int a)) {
    print(a);
  }
}
```

If the type of the variable is different and the type isn't critical to
the condition being matched, then ensure that the variable has the same
type on both branches:

```dart
void f(Object? x) {
  if (x case (num a) || (num a)) {
    print(a);
  }
}
```

If the type of the variable is different and the type is critical to the
condition being matched, then consider breaking the condition into
multiple `if` statements or `case` clauses:

```dart
void f(Object? x) {
  if (x case int a) {
    print(a);
  } else if (x case double a) {
    print(a);
  }
}
```

### initializer_for_non_existent_field

_'{0}' isn't a field in the enclosing class._

#### Description

The analyzer produces this diagnostic when a constructor initializes a
field that isn't declared in the class containing the constructor.
Constructors can't initialize fields that aren't declared and fields that
are inherited from superclasses.

#### Example

The following code produces this diagnostic because the initializer is
initializing `x`, but `x` isn't a field in the class:

```dart
class C {
  int? y;

  C() : [!x = 0!];
}
```

#### Common fixes

If a different field should be initialized, then change the name to the
name of the field:

```dart
class C {
  int? y;

  C() : y = 0;
}
```

If the field must be declared, then add a declaration:

```dart
class C {
  int? x;
  int? y;

  C() : x = 0;
}
```

### initializer_for_static_field

_'{0}' is a static field in the enclosing class. Fields initialized in a
constructor can't be static._

#### Description

The analyzer produces this diagnostic when a static field is initialized
in a constructor using either an initializing formal parameter or an
assignment in the initializer list.

#### Example

The following code produces this diagnostic because the static field `a`
is being initialized by the initializing formal parameter `this.a`:

```dart
class C {
  static int? a;
  C([!this.a!]);
}
```

#### Common fixes

If the field should be an instance field, then remove the keyword `static`:

```dart
class C {
  int? a;
  C(this.a);
}
```

If you intended to initialize an instance field and typed the wrong name,
then correct the name of the field being initialized:

```dart
class C {
  static int? a;
  int? b;
  C(this.b);
}
```

If you really want to initialize the static field, then move the
initialization into the constructor body:

```dart
class C {
  static int? a;
  C(int? c) {
    a = c;
  }
}
```

### initializing_formal_for_non_existent_field

_'{0}' isn't a field in the enclosing class._

#### Description

The analyzer produces this diagnostic when an initializing formal
parameter is found in a constructor in a class that doesn't declare the
field being initialized. Constructors can't initialize fields that aren't
declared and fields that are inherited from superclasses.

#### Example

The following code produces this diagnostic because the field `x` isn't
defined:

```dart
class C {
  int? y;

  C([!this.x!]);
}
```

#### Common fixes

If the field name was wrong, then change it to the name of an existing
field:

```dart
class C {
  int? y;

  C(this.y);
}
```

If the field name is correct but hasn't yet been defined, then declare the
field:

```dart
class C {
  int? x;
  int? y;

  C(this.x);
}
```

If the parameter is needed but shouldn't initialize a field, then convert
it to a normal parameter and use it:

```dart
class C {
  int y;

  C(int x) : y = x * 2;
}
```

If the parameter isn't needed, then remove it:

```dart
class C {
  int? y;

  C();
}
```

### instance_access_to_static_member

_The static {1} '{0}' can't be accessed through an instance._

#### Description

The analyzer produces this diagnostic when an access operator is used to
access a static member through an instance of the class.

#### Example

The following code produces this diagnostic because `zero` is a static
field, but it's being accessed as if it were an instance field:

```dart
void f(C c) {
  c.[!zero!];
}

class C {
  static int zero = 0;
}
```

#### Common fixes

Use the class to access the static member:

```dart
void f(C c) {
  C.zero;
}

class C {
  static int zero = 0;
}
```

### instance_member_access_from_factory

_Instance members can't be accessed from a factory constructor._

#### Description

The analyzer produces this diagnostic when a factory constructor contains
an unqualified reference to an instance member. In a generative
constructor, the instance of the class is created and initialized before
the body of the constructor is executed, so the instance can be bound to
`this` and accessed just like it would be in an instance method. But, in a
factory constructor, the instance isn't created before executing the body,
so `this` can't be used to reference it.

#### Example

The following code produces this diagnostic because `x` isn't in scope in
the factory constructor:

```dart
class C {
  int x;
  factory C() {
    return C._([!x!]);
  }
  C._(this.x);
}
```

#### Common fixes

Rewrite the code so that it doesn't reference the instance member:

```dart
class C {
  int x;
  factory C() {
    return C._(0);
  }
  C._(this.x);
}
```

### instance_member_access_from_static

_Instance members can't be accessed from a static method._

#### Description

The analyzer produces this diagnostic when a static method contains an
unqualified reference to an instance member.

#### Example

The following code produces this diagnostic because the instance field `x`
is being referenced in a static method:

```dart
class C {
  int x = 0;

  static int m() {
    return [!x!];
  }
}
```

#### Common fixes

If the method must reference the instance member, then it can't be static,
so remove the keyword:

```dart
class C {
  int x = 0;

  int m() {
    return x;
  }
}
```

If the method can't be made an instance method, then add a parameter so
that an instance of the class can be passed in:

```dart
class C {
  int x = 0;

  static int m(C c) {
    return c.x;
  }
}
```

### instantiate_abstract_class

_Abstract classes can't be instantiated._

#### Description

The analyzer produces this diagnostic when it finds a constructor
invocation and the constructor is declared in an abstract class. Even
though you can't create an instance of an abstract class, abstract classes
can declare constructors that can be invoked by subclasses.

#### Example

The following code produces this diagnostic because `C` is an abstract
class:

```dart
abstract class C {}

var c = new [!C!]();
```

#### Common fixes

If there's a concrete subclass of the abstract class that can be used, then
create an instance of the concrete subclass.

### instantiate_enum

_Enums can't be instantiated._

#### Description

The analyzer produces this diagnostic when an enum is instantiated. It's
invalid to create an instance of an enum by invoking a constructor; only
the instances named in the declaration of the enum can exist.

#### Example

The following code produces this diagnostic because the enum `E` is being
instantiated:

```dart
// @dart = 2.16
enum E {a}

var e = [!E!]();
```

#### Common fixes

If you intend to use an instance of the enum, then reference one of the
constants defined in the enum:

```dart
// @dart = 2.16
enum E {a}

var e = E.a;
```

If you intend to use an instance of a class, then use the name of that class in place of the name of the enum.

### instantiate_type_alias_expands_to_type_parameter

_Type aliases that expand to a type parameter can't be instantiated._

#### Description

The analyzer produces this diagnostic when a constructor invocation is
found where the type being instantiated is a type alias for one of the type
parameters of the type alias. This isn't allowed because the value of the
type parameter is a type rather than a class.

#### Example

The following code produces this diagnostic because it creates an instance
of `A`, even though `A` is a type alias that is defined to be equivalent to
a type parameter:

```dart
typedef A<T> = T;

void f() {
  const [!A!]<int>();
}
```

#### Common fixes

Use either a class name or a type alias defined to be a class, rather than
a type alias defined to be a type parameter:

```dart
typedef A<T> = C<T>;

void f() {
  const A<int>();
}

class C<T> {
  const C();
}
```

### integer_literal_imprecise_as_double

_The integer literal is being used as a double, but can't be represented as a
64-bit double without overflow or loss of precision: '{0}'._

#### Description

The analyzer produces this diagnostic when an integer literal is being
implicitly converted to a double, but can't be represented as a 64-bit
double without overflow or loss of precision. Integer literals are
implicitly converted to a double if the context requires the type `double`.

#### Example

The following code produces this diagnostic because the integer value
`9223372036854775807` can't be represented exactly as a double:

```dart
double x = [!9223372036854775807!];
```

#### Common fixes

If you need to use the exact value, then use the class `BigInt` to
represent the value:

```dart
var x = BigInt.parse('9223372036854775807');
```

If you need to use a double, then change the value to one that can be
represented exactly:

```dart
double x = 9223372036854775808;
```

### integer_literal_out_of_range

_The integer literal {0} can't be represented in 64 bits._

#### Description

The analyzer produces this diagnostic when an integer literal has a value
that is too large (positive) or too small (negative) to be represented in a
64-bit word.

#### Example

The following code produces this diagnostic because the value can't be
represented in 64 bits:

```dart
var x = [!9223372036854775810!];
```

#### Common fixes

If you need to represent the current value, then wrap it in an instance of
the class `BigInt`:

```dart
var x = BigInt.parse('9223372036854775810');
```

### invalid_annotation

_Annotation must be either a const variable reference or const constructor
invocation._

#### Description

The analyzer produces this diagnostic when an annotation is found that is
using something that is neither a variable marked as `const` or the
invocation of a `const` constructor.

Getters can't be used as annotations.

#### Examples

The following code produces this diagnostic because the variable `v` isn't
a `const` variable:

```dart
var v = 0;

[!@v!]
void f() {
}
```

The following code produces this diagnostic because `f` isn't a variable:

```dart
[!@f!]
void f() {
}
```

The following code produces this diagnostic because `f` isn't a
constructor:

```dart
[!@f()!]
void f() {
}
```

The following code produces this diagnostic because `g` is a getter:

```dart
[!@g!]
int get g => 0;
```

#### Common fixes

If the annotation is referencing a variable that isn't a `const`
constructor, add the keyword `const` to the variable's declaration:

```dart
const v = 0;

@v
void f() {
}
```

If the annotation isn't referencing a variable, then remove it:

```dart
int v = 0;

void f() {
}
```

### invalid_annotation_constant_value_from_deferred_library

_Constant values from a deferred library can't be used in annotations._

#### Description

The analyzer produces this diagnostic when a constant defined in a library
that is imported as a deferred library is referenced in the argument list
of an annotation. Annotations are evaluated at compile time, and values
from deferred libraries aren't available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

The following code produces this diagnostic because the constant `pi` is
being referenced in the argument list of an annotation, even though the
library that defines it is being imported as a deferred library:

```dart
import 'dart:math' deferred as math;

class C {
  const C(double d);
}

@C(math.[!pi!])
void f () {}
```

#### Common fixes

If you need to reference the imported constant, then remove the `deferred`
keyword:

```dart
import 'dart:math' as math;

class C {
  const C(double d);
}

@C(math.pi)
void f () {}
```

If the import is required to be deferred and there's another constant that
is appropriate, then use that constant in place of the constant from the
deferred library.

### invalid_annotation_from_deferred_library

_Constant values from a deferred library can't be used as annotations._

#### Description

The analyzer produces this diagnostic when a constant from a library that
is imported using a deferred import is used as an annotation. Annotations
are evaluated at compile time, and constants from deferred libraries aren't
available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

The following code produces this diagnostic because the constant `pi` is
being used as an annotation when the library `dart:math` is imported as
`deferred`:

```dart
import 'dart:math' deferred as math;

@[!math.pi!]
void f() {}
```

#### Common fixes

If you need to reference the constant as an annotation, then remove the
keyword `deferred` from the import:

```dart
import 'dart:math' as math;

@math.pi
void f() {}
```

If you can use a different constant as an annotation, then replace the
annotation with a different constant:

```dart
@deprecated
void f() {}
```

### invalid_annotation_target

_The annotation '{0}' can only be used on {1}._

#### Description

The analyzer produces this diagnostic when an annotation is applied to a
kind of declaration that it doesn't support.

#### Example

The following code produces this diagnostic because the `optionalTypeArgs`
annotation isn't defined to be valid for top-level variables:

```dart
import 'package:meta/meta.dart';

@[!optionalTypeArgs!]
int x = 0;
```

#### Common fixes

Remove the annotation from the declaration.

### invalid_assignment

_A value of type '{0}' can't be assigned to a variable of type '{1}'._

#### Description

The analyzer produces this diagnostic when the static type of an expression
that is assigned to a variable isn't assignable to the type of the
variable.

#### Example

The following code produces this diagnostic because the type of the
initializer (`int`) isn't assignable to the type of the variable
(`String`):

```dart
int i = 0;
String s = [!i!];
```

#### Common fixes

If the value being assigned is always assignable at runtime, even though
the static types don't reflect that, then add an explicit cast.

Otherwise, change the value being assigned so that it has the expected
type. In the previous example, this might look like:

```dart
int i = 0;
String s = i.toString();
```

If you can't change the value, then change the type of the variable to be
compatible with the type of the value being assigned:

```dart
int i = 0;
int s = i;
```

### invalid_dependency

_Publishable packages can't have '{0}' dependencies._

#### Description

The analyzer produces this diagnostic when a publishable package
includes a package in the `dependencies` list of its `pubspec.yaml` file
that isn't a pub-hosted dependency.

To learn more about the different types of dependency sources,
check out [Package dependencies](https://dart.dev/tools/pub/dependencies).

#### Example

The following code produces this diagnostic because the dependency on
the package `transmogrify` isn't a pub-hosted dependency.

```yaml
name: example
dependencies:
  transmogrify:
    [!path!]: ../transmogrify
```

#### Common fixes

If you want to publish the package to `pub.dev`, then change
the dependency to a hosted package that is published on `pub.dev`.

If the package isn't intended to be published on `pub.dev`, then
add a `publish_to: none` entry to its `pubspec.yaml` file to
mark it as not intended to be published:

```yaml
name: example
publish_to: none
dependencies:
  transmogrify:
    path: ../transmogrify
```

### invalid_exception_value

_The method {0} can't have an exceptional return value (the second argument)
when the return type of the function is either 'void', 'Handle' or 'Pointer'._

#### Description

The analyzer produces this diagnostic when an invocation of the method
`Pointer.fromFunction` or `NativeCallable.isolateLocal`
has a second argument (the exceptional return
value) and the type to be returned from the invocation is either `void`,
`Handle` or `Pointer`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because a second argument is
provided when the return type of `f` is `void`:

```dart
import 'dart:ffi';

typedef T = Void Function(Int8);

void f(int i) {}

void g() {
  Pointer.fromFunction<T>(f, [!42!]);
}
```

#### Common fixes

Remove the exception value:

```dart
import 'dart:ffi';

typedef T = Void Function(Int8);

void f(int i) {}

void g() {
  Pointer.fromFunction<T>(f);
}
```

### invalid_export_of_internal_element

_The member '{0}' can't be exported as a part of a package's public API._

#### Description

The analyzer produces this diagnostic when a [public library][] exports a
declaration that is marked with the [`internal`][meta-internal]
annotation.

#### Example

Given a file `a.dart` in the `src` directory that contains:

```dart
import 'package:meta/meta.dart';

@internal class One {}
```

The following code, when found in a [public library][] produces this
diagnostic because the `export` directive is exporting a name that is only
intended to be used internally:

```dart
[!export 'src/a.dart';!]
```

#### Common fixes

If the export is needed, then add a `hide` clause to hide the internal
names:

```dart
export 'src/a.dart' hide One;
```

If the export isn't needed, then remove it.

### invalid_export_of_internal_element_indirectly

_The member '{0}' can't be exported as a part of a package's public API, but is
indirectly exported as part of the signature of '{1}'._

#### Description

The analyzer produces this diagnostic when a [public library][] exports a
top-level function  with a return type or at least one parameter type that
is marked with the [`internal`][meta-internal] annotation.

#### Example

Given a file `a.dart` in the `src` directory that contains the
following:

```dart
import 'package:meta/meta.dart';

@internal
typedef IntFunction = int Function();

int f(IntFunction g) => g();
```

The following code produces this diagnostic because the function `f` has a
parameter of type `IntFunction`, and `IntFunction` is only intended to be
used internally:

```dart
[!export 'src/a.dart' show f;!]
```

#### Common fixes

If the function must be public, then make all the types in the function's
signature public types.

If the function doesn't need to be exported, then stop exporting it,
either by removing it from the `show` clause, adding it to the `hide`
clause, or by removing the export.

### invalid_extension_argument_count

_Extension overrides must have exactly one argument: the value of 'this' in the
extension method._

#### Description

The analyzer produces this diagnostic when an extension override doesn't
have exactly one argument. The argument is the expression used to compute
the value of `this` within the extension method, so there must be one
argument.

#### Examples

The following code produces this diagnostic because there are no arguments:

```dart
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E[!()!].join('b');
}
```

And, the following code produces this diagnostic because there's more than
one argument:

```dart
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E[!('a', 'b')!].join('c');
}
```

#### Common fixes

Provide one argument for the extension override:

```dart
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E('a').join('b');
}
```

### invalid_factory_method_decl

_Factory method '{0}' must have a return type._

#### Description

The analyzer produces this diagnostic when a method that is annotated with
the [`factory`][meta-factory] annotation has a return type of `void`.

#### Example

The following code produces this diagnostic because the method `createC`
is annotated with the [`factory`][meta-factory] annotation but doesn't
return any value:

```dart
import 'package:meta/meta.dart';

class Factory {
  @factory
  void [!createC!]() {}
}

class C {}
```

#### Common fixes

Change the return type to something other than `void`:

```dart
import 'package:meta/meta.dart';

class Factory {
  @factory
  C createC() => C();
}

class C {}
```

### invalid_factory_method_impl

_Factory method '{0}' doesn't return a newly allocated object._

#### Description

The analyzer produces this diagnostic when a method that is annotated with
the [`factory`][meta-factory] annotation doesn't return a newly allocated
object.

#### Example

The following code produces this diagnostic because the method `createC`
returns the value of a field rather than a newly created instance of `C`:

```dart
import 'package:meta/meta.dart';

class Factory {
  C c = C();

  @factory
  C [!createC!]() => c;
}

class C {}
```

#### Common fixes

Change the method to return a newly created instance of the return type:

```dart
import 'package:meta/meta.dart';

class Factory {
  @factory
  C createC() => C();
}

class C {}
```

### invalid_factory_name_not_a_class

_The name of a factory constructor must be the same as the name of the
immediately enclosing class._

#### Description

The analyzer produces this diagnostic when the name of a factory
constructor isn't the same as the name of the surrounding class.

#### Example

The following code produces this diagnostic because the name of the factory
constructor (`A`) isn't the same as the surrounding class (`C`):

```dart
class A {}

class C {
  factory [!A!]() => throw 0;
}
```

#### Common fixes

If the factory returns an instance of the surrounding class, and you
intend it to be an unnamed factory constructor, then rename the factory:

```dart
class A {}

class C {
  factory C() => throw 0;
}
```

If the factory returns an instance of the surrounding class, and you
intend it to be a named factory constructor, then prefix the name of the
factory constructor with the name of the surrounding class:

```dart
class A {}

class C {
  factory C.a() => throw 0;
}
```

If the factory returns an instance of a different class, then move the
factory to that class:

```dart
class A {
  factory A() => throw 0;
}

class C {}
```

If the factory returns an instance of a different class, but you can't
modify that class or don't want to move the factory, then convert it to be
a static method:

```dart
class A {}

class C {
  static A a() => throw 0;
}
```

### invalid_field_name

_Record field names can't be a dollar sign followed by an integer when the
integer is the index of a positional field._

_Record field names can't be private._

_Record field names can't be the same as a member from 'Object'._

#### Description

The analyzer produces this diagnostic when either a record literal or a
record type annotation has a field whose name is invalid. The name is
invalid if it is:
- private (starts with `_`)
- the same as one of the members defined on `Object`
- the same as the name of a positional field (an exception is made if the
  field is a positional field with the specified name)

#### Examples

The following code produces this diagnostic because the record literal has
a field named `toString`, which is a method defined on `Object`:

```dart
var r = (a: 1, [!toString!]: 4);
```

The following code produces this diagnostic because the record type
annotation has a field named `hashCode`, which is a getter defined on
`Object`:

```dart
void f(({int a, int [!hashCode!]}) r) {}
```

The following code produces this diagnostic because the record literal has
a private field named `_a`:

```dart
var r = ([!_a!]: 1, b: 2);
```

The following code produces this diagnostic because the record type
annotation has a private field named `_a`:

```dart
void f(({int [!_a!], int b}) r) {}
```

The following code produces this diagnostic because the record literal has
a field named `$1`, which is also the name of a different positional
parameter:

```dart
var r = (2, [!$1!]: 1);
```

The following code produces this diagnostic because the record type
annotation has a field named `$1`, which is also the name of a different
positional parameter:

```dart
void f((int, String, {int [!$1!]}) r) {}
```

#### Common fixes

Rename the field:

```dart
var r = (a: 1, d: 4);
```

### invalid_field_type_in_struct

_Fields in struct classes can't have the type '{0}'. They can only be declared
as 'int', 'double', 'Array', 'Pointer', or subtype of 'Struct' or 'Union'._

#### Description

The analyzer produces this diagnostic when a field in a subclass of
`Struct` has a type other than `int`, `double`, `Array`, `Pointer`, or
subtype of `Struct` or `Union`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `str` has
the type `String`, which isn't one of the allowed types for fields in a
subclass of `Struct`:

```dart
import 'dart:ffi';

final class C extends Struct {
  external [!String!] s;

  @Int32()
  external int i;
}
```

#### Common fixes

Use one of the allowed types for the field:

```dart
import 'dart:ffi';
import 'package:ffi/ffi.dart';

final class C extends Struct {
  external Pointer<Utf8> s;

  @Int32()
  external int i;
}
```

### invalid_implementation_override

_'{1}.{0}' ('{2}') isn't a valid concrete implementation of '{3}.{0}' ('{4}')._

_The setter '{1}.{0}' ('{2}') isn't a valid concrete implementation of '{3}.{0}'
('{4}')._

#### Description

The analyzer produces this diagnostic when all of the following are true:

- A class defines an abstract member.
- There is a concrete implementation of that member in a superclass.
- The concrete implementation isn't a valid implementation of the abstract
  method.

The concrete implementation can be invalid because of incompatibilities in
either the return type, the types of parameters, or the type variables.

#### Example

The following code produces this diagnostic because the method `A.add` has
a parameter of type `int`, and the overriding method `B.add` has a
corresponding parameter of type `num`:

```dart
class A {
  int add(int a) => a;
}
class [!B!] extends A {
  int add(num a);
}
```

This is a problem because in an invocation of `B.add` like the following:

```dart
void f(B b) {
  b.add(3.4);
}
```

`B.add` is expecting to be able to take, for example, a `double`, but when
the method `A.add` is executed (because it's the only concrete
implementation of `add`), a runtime exception will be thrown because a
`double` can't be assigned to a parameter of type `int`.

#### Common fixes

If the method in the subclass can conform to the implementation in the
superclass, then change the declaration in the subclass (or remove it if
it's the same):

```dart
class A {
  int add(int a) => a;
}
class B	extends A {
  int add(int a);
}
```

If the method in the superclass can be generalized to be a valid
implementation of the method in the subclass, then change the superclass
method:

```dart
class A {
  int add(num a) => a.floor();
}
class B	extends A {
  int add(num a);
}
```

If neither the method in the superclass nor the method in the subclass can
be changed, then provide a concrete implementation of the method in the
subclass:

```dart
class A {
  int add(int a) => a;
}
class B	extends A {
  int add(num a) => a.floor();
}
```

### invalid_inline_function_type

_Inline function types can't be used for parameters in a generic function type._

#### Description

The analyzer produces this diagnostic when a generic function type has a
function-valued parameter that is written using the older inline function
type syntax.

#### Example

The following code produces this diagnostic because the parameter `f`, in
the generic function type used to define `F`, uses the inline function
type syntax:

```dart
typedef F = int Function(int f[!(!]String s));
```

#### Common fixes

Use the generic function syntax for the parameter's type:

```dart
typedef F = int Function(int Function(String));
```

### invalid_internal_annotation

_Only public elements in a package's private API can be annotated as being
internal._

#### Description

The analyzer produces this diagnostic when a declaration is annotated with
the [`internal`][meta-internal] annotation and that declaration is either
in a [public library][] or has a private name.

#### Example

The following code, when in a [public library][], produces this diagnostic
because the [`internal`][meta-internal] annotation can't be applied to
declarations in a [public library][]:

```dart
import 'package:meta/meta.dart';

@[!internal!]
class C {}
```

The following code, whether in a public or internal library, produces this
diagnostic because the [`internal`][meta-internal] annotation can't be
applied to declarations with private names:

```dart
import 'package:meta/meta.dart';

@[!internal!]
class _C {}

void f(_C c) {}
```

#### Common fixes

If the declaration has a private name, then remove the annotation:

```dart
class _C {}

void f(_C c) {}
```

If the declaration has a public name and is intended to be internal to the
package, then move the annotated declaration into an internal library (in
other words, a library inside the `src` directory).

Otherwise, remove the use of the annotation:

```dart
class C {}
```

### invalid_language_version_override

_The Dart language version override comment can't be followed by any
non-whitespace characters._

_The Dart language version override comment must be specified with a version
number, like '2.0', after the '=' character._

_The Dart language version override comment must be specified with an '='
character._

_The Dart language version override comment must be specified with exactly two
slashes._

_The Dart language version override comment must be specified with the word
'dart' in all lower case._

_The Dart language version override number can't be prefixed with a letter._

_The Dart language version override number must begin with '@dart'._

_The language version override can't specify a version greater than the latest
known language version: {0}.{1}._

_The language version override must be specified before any declaration or
directive._

#### Description

The analyzer produces this diagnostic when a comment that appears to be an
attempt to specify a language version override doesn't conform to the
requirements for such a comment. For more information, see
[Per-library language version selection](https://dart.dev/guides/language/evolution#per-library-language-version-selection).

#### Example

The following code produces this diagnostic because the word `dart` must
be lowercase in such a comment and because there's no equal sign between
the word `dart` and the version number:

```dart
[!// @Dart 2.13!]
```

#### Common fixes

If the comment is intended to be a language version override, then change
the comment to follow the correct format:

```dart
// @dart = 2.13
```

### invalid_literal_annotation

_Only const constructors can have the `@literal` annotation._

#### Description

The analyzer produces this diagnostic when the [`literal`][meta-literal]
annotation is applied to anything other than a const constructor.

#### Examples

The following code produces this diagnostic because the constructor isn't
a `const` constructor:

```dart
import 'package:meta/meta.dart';

class C {
  @[!literal!]
  C();
}
```

The following code produces this diagnostic because `x` isn't a
constructor:

```dart
import 'package:meta/meta.dart';

@[!literal!]
var x;
```

#### Common fixes

If the annotation is on a constructor and the constructor should always be
invoked with `const`, when possible, then mark the constructor with the
`const` keyword:

```dart
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}
```

If the constructor can't be marked as `const`, then remove the annotation.

If the annotation is on anything other than a constructor, then remove the
annotation:

```dart
var x;
```

### invalid_modifier_on_constructor

_The modifier '{0}' can't be applied to the body of a constructor._

#### Description

The analyzer produces this diagnostic when the body of a constructor is
prefixed by one of the following modifiers: `async`, `async*`, or `sync*`.
Constructor bodies must be synchronous.

#### Example

The following code produces this diagnostic because the body of the
constructor for `C` is marked as being `async`:

```dart
class C {
  C() [!async!] {}
}
```

#### Common fixes

If the constructor can be synchronous, then remove the modifier:

```dart
class C {
  C();
}
```

If the constructor can't be synchronous, then use a static method to create
the instance instead:

```dart
class C {
  C();
  static Future<C> c() async {
    return C();
  }
}
```

### invalid_modifier_on_setter

_Setters can't use 'async', 'async*', or 'sync*'._

#### Description

The analyzer produces this diagnostic when the body of a setter is prefixed
by one of the following modifiers: `async`, `async*`, or `sync*`. Setter
bodies must be synchronous.

#### Example

The following code produces this diagnostic because the body of the setter
`x` is marked as being `async`:

```dart
class C {
  set x(int i) [!async!] {}
}
```

#### Common fixes

If the setter can be synchronous, then remove the modifier:

```dart
class C {
  set x(int i) {}
}
```

If the setter can't be synchronous, then use a method to set the value
instead:

```dart
class C {
  void x(int i) async {}
}
```

### invalid_non_virtual_annotation

_The annotation '@nonVirtual' can only be applied to a concrete instance
member._

#### Description

The analyzer produces this diagnostic when the `nonVirtual` annotation is
found on a declaration other than a member of a class, mixin, or enum, or
if the member isn't a concrete instance member.

#### Examples

The following code produces this diagnostic because the annotation is on a
class declaration rather than a member inside the class:

```dart
import 'package:meta/meta.dart';

@[!nonVirtual!]
class C {}
```

The following code produces this diagnostic because the method `m` is an
abstract method:

```dart
import 'package:meta/meta.dart';

abstract class C {
  @[!nonVirtual!]
  void m();
}
```

The following code produces this diagnostic because the method `m` is a
static method:

```dart
import 'package:meta/meta.dart';

abstract class C {
  @[!nonVirtual!]
  static void m() {}
}
```

#### Common fixes

If the declaration isn't a member of a class, mixin, or enum, then remove
the annotation:

```dart
class C {}
```

If the member is intended to be a concrete instance member, then make it
so:

```dart
import 'package:meta/meta.dart';

abstract class C {
  @nonVirtual
  void m() {}
}
```

If the member is not intended to be a concrete instance member, then
remove the annotation:

```dart
abstract class C {
  static void m() {}
}
```

### invalid_null_aware_operator

_The receiver can't be null because of short-circuiting, so the null-aware
operator '{0}' can't be used._

_The receiver can't be null, so the null-aware operator '{0}' is unnecessary._

#### Description

The analyzer produces this diagnostic when a null-aware operator (`?.`,
`?..`, `?[`, `?..[`, or `...?`) is used on a receiver that's known to be
non-nullable.

#### Examples

The following code produces this diagnostic because `s` can't be `null`:

```dart
int? getLength(String s) {
  return s[!?.!]length;
}
```

The following code produces this diagnostic because `a` can't be `null`:

```dart
var a = [];
var b = [[!...?!]a];
```

The following code produces this diagnostic because `s?.length` can't
return `null`:

```dart
void f(String? s) {
  s?.length[!?.!]isEven;
}
```

The reason `s?.length` can't return `null` is because the null-aware
operator following `s` short-circuits the evaluation of both `length` and
`isEven` if `s` is `null`. In other words, if `s` is `null`, then neither
`length` nor `isEven` will be invoked, and if `s` is non-`null`, then
`length` can't return a `null` value. Either way, `isEven` can't be invoked
on a `null` value, so the null-aware operator isn't necessary. See
[Understanding null safety](/null-safety/understanding-null-safety#smarter-null-aware-methods)
for more details.

The following code produces this diagnostic because `s` can't be `null`.

```dart
void f(Object? o) {
  var s = o as String;
  s[!?.!]length;
}
```

The reason `s` can't be null, despite the fact that `o` can be `null`, is
because of the cast to `String`, which is a non-nullable type. If `o` ever
has the value `null`, the cast will fail and the invocation of `length`
will not happen.

#### Common fixes

Replace the null-aware operator with a non-null-aware equivalent; for
example, change `?.` to  `.`:

```dart
int getLength(String s) {
  return s.length;
}
```

(Note that the return type was also changed to be non-nullable, which might
not be appropriate in some cases.)

### invalid_override

_'{1}.{0}' ('{2}') isn't a valid override of '{3}.{0}' ('{4}')._

_The setter '{1}.{0}' ('{2}') isn't a valid override of '{3}.{0}' ('{4}')._

#### Description

The analyzer produces this diagnostic when a member of a class is found
that overrides a member from a supertype and the override isn't valid. An
override is valid if all of these are true:
- It allows all of the arguments allowed by the overridden member.
- It doesn't require any arguments that aren't required by the overridden
  member.
- The type of every parameter of the overridden member is assignable to the
  corresponding parameter of the override.
- The return type of the override is assignable to the return type of the
  overridden member.

#### Example

The following code produces this diagnostic because the type of the
parameter `s` (`String`) isn't assignable to the type of the parameter `i`
(`int`):

```dart
class A {
  void m(int i) {}
}

class B extends A {
  void [!m!](String s) {}
}
```

#### Common fixes

If the invalid method is intended to override the method from the
superclass, then change it to conform:

```dart
class A {
  void m(int i) {}
}

class B extends A {
  void m(int i) {}
}
```

If it isn't intended to override the method from the superclass, then
rename it:

```dart
class A {
  void m(int i) {}
}

class B extends A {
  void m2(String s) {}
}
```

### invalid_override_of_non_virtual_member

_The member '{0}' is declared non-virtual in '{1}' and can't be overridden in
subclasses._

#### Description

The analyzer produces this diagnostic when a member of a class, mixin, or
enum overrides a member that has the `@nonVirtual` annotation on it.

#### Example

The following code produces this diagnostic because the method `m` in `B`
overrides the method `m` in `A`, and the method `m` in `A` is annotated
with the `@nonVirtual` annotation:

```dart
import 'package:meta/meta.dart';

class A {
  @nonVirtual
  void m() {}
}

class B extends A {
  @override
  void [!m!]() {}
}
```

#### Common fixes

If the annotation on the method in the superclass is correct (the method
in the superclass is not intended to be overridden), then remove or rename
the overriding method:

```dart
import 'package:meta/meta.dart';

class A {
  @nonVirtual
  void m() {}
}

class B extends A {}
```

If the method in the superclass is intended to be overridden, then remove
the `@nonVirtual` annotation:

```dart
class A {
  void m() {}
}

class B extends A {
  @override
  void m() {}
}
```

### invalid_pattern_variable_in_shared_case_scope

_The variable '{0}' doesn't have the same type and/or finality in all cases that
share this body._

_The variable '{0}' is available in some, but not all cases that share this
body._

_The variable '{0}' is not available because there is a label or 'default'
case._

#### Description

The analyzer produces this diagnostic when multiple case clauses in a
switch statement share a body, and at least one of them declares a
variable that is referenced in the shared statements, but the variable is
either not declared in all of the case clauses or it is declared in
inconsistent ways.

If the variable isn't declared in all of the case clauses, then it won't
have a value if one of the clauses that doesn't declare the variable is
the one that matches and executes the body. This includes the situation
where one of the case clauses is the `default` clause.

If the variable is declared in inconsistent ways, either being `final` in
some cases and not `final` in others or having a different type in
different cases, then the semantics of what the type or finality of the
variable should be are not defined.

#### Examples

The following code produces this diagnostic because the variable `a` is
only declared in one of the case clauses, and won't have a value if the
second clause is the one that matched `x`:

```dart
void f(Object? x) {
  switch (x) {
    case int a when a > 0:
    case 0:
      [!a!];
  }
}
```

The following code produces this diagnostic because the variable `a` isn't
declared in the `default` clause, and won't have a value if the body is
executed because none of the other clauses matched `x`:

```dart
void f(Object? x) {
  switch (x) {
    case int a when a > 0:
    default:
      [!a!];
  }
}
```

The following code produces this diagnostic because the variable `a` won't
have a value if the body is executed because a different group of cases
caused control to continue at the label:

```dart
void f(Object? x) {
  switch (x) {
    someLabel:
    case int a when a > 0:
      [!a!];
    case int b when b < 0:
      continue someLabel;
  }
}
```

The following code produces this diagnostic because the variable `a`,
while being assigned in all of the case clauses, doesn't have then same
type associated with it in every clause:

```dart
void f(Object? x) {
  switch (x) {
    case int a when a < 0:
    case num a when a > 0:
      [!a!];
  }
}
```

The following code produces this diagnostic because the variable `a` is
`final` in the first case clause and isn't `final` in the second case
clause:

```dart
void f(Object? x) {
  switch (x) {
    case final int a when a < 0:
    case int a when a > 0:
      [!a!];
  }
}
```

#### Common fixes

If the variable isn't declared in all of the cases, and you need to
reference it in the statements, then declare it in the other cases:

```dart
void f(Object? x) {
  switch (x) {
    case int a when a > 0:
    case int a when a == 0:
      a;
  }
}
```

If the variable isn't declared in all of the cases, and you don't need to
reference it in the statements, then remove the references to it and
remove the declarations from the other cases:

```dart
void f(int x) {
  switch (x) {
    case > 0:
    case 0:
  }
}
```

If the type of the variable is different, decide the type the variable
should have and make the cases consistent:

```dart
void f(Object? x) {
  switch (x) {
    case num a when a < 0:
    case num a when a > 0:
      a;
  }
}
```

If the finality of the variable is different, decide whether it should be
`final` or not `final` and make the cases consistent:

```dart
void f(Object? x) {
  switch (x) {
    case final int a when a < 0:
    case final int a when a > 0:
      a;
  }
}
```

### invalid_platforms_field

_The 'platforms' field must be a map with platforms as keys._

#### Description

The analyzer produces this diagnostic when a top-level `platforms`
field is specified, but its value is not a map with keys.
To learn more about specifying your package's supported platforms,
check out the [documentation on platform declarations](https://dart.dev/tools/pub/pubspec#platforms).

#### Example

The following `pubspec.yaml` produces this diagnostic because `platforms`
should be a map.

```yaml
name: example
platforms:
  [!- android
  - web
  - ios!]
```

#### Common fixes

If you can rely on automatic platform detection, then omit the
top-level `platforms` field.

```yaml
name: example
```

If you need to manually specify the list of supported platforms, then
write the `platforms` field as a map with platform names as keys.

```yaml
name: example
platforms:
  android:
  web:
  ios:
```

### invalid_reference_to_generative_enum_constructor

_Generative enum constructors can only be used as targets of redirection._

#### Description

The analyzer produces this diagnostic when a generative constructor
defined on an enum is used anywhere other than to create one of the enum
constants or as the target of a redirection from another constructor in
the same enum.

#### Example

The following code produces this diagnostic because the constructor for
`E` is being used to create an instance in the function `f`:

```dart
enum E {
  a(0);

  const E(int x);
}

E f() => const [!E!](2);
```

#### Common fixes

If there's an enum value with the same value, or if you add such a
constant, then reference the constant directly:

```dart
enum E {
  a(0), b(2);

  const E(int x);
}

E f() => E.b;
```

If you need to use a constructor invocation, then use a factory
constructor:

```dart
enum E {
  a(0);

  const E(int x);

  factory E.c(int x) => a;
}

E f() => E.c(2);
```

### invalid_reference_to_this

_Invalid reference to 'this' expression._

#### Description

The analyzer produces this diagnostic when `this` is used outside of an
instance method or a generative constructor. The reserved word `this` is
only defined in the context of an instance method, a generative
constructor, or the initializer of a late instance field declaration.

#### Example

The following code produces this diagnostic because `v` is a top-level
variable:

```dart
C f() => [!this!];

class C {}
```

#### Common fixes

Use a variable of the appropriate type in place of `this`, declaring it if
necessary:

```dart
C f(C c) => c;

class C {}
```

### invalid_return_type_for_catch_error

_A value of type '{0}' can't be returned by the 'onError' handler because it
must be assignable to '{1}'._

_The return type '{0}' isn't assignable to '{1}', as required by
'Future.catchError'._

#### Description

The analyzer produces this diagnostic when an invocation of
`Future.catchError` has an argument whose return type isn't compatible with
the type returned by the instance of `Future`. At runtime, the method
`catchError` attempts to return the value from the callback as the result
of the future, which results in another exception being thrown.

#### Examples

The following code produces this diagnostic because `future` is declared to
return an `int` while `callback` is declared to return a `String`, and
`String` isn't a subtype of `int`:

```dart
void f(Future<int> future, String Function(dynamic, StackTrace) callback) {
  future.catchError([!callback!]);
}
```

The following code produces this diagnostic because the closure being
passed to `catchError` returns an `int` while `future` is declared to
return a `String`:

```dart
void f(Future<String> future) {
  future.catchError((error, stackTrace) => [!3!]);
}
```

#### Common fixes

If the instance of `Future` is declared correctly, then change the callback
to match:

```dart
void f(Future<int> future, int Function(dynamic, StackTrace) callback) {
  future.catchError(callback);
}
```

If the declaration of the instance of `Future` is wrong, then change it to
match the callback:

```dart
void f(Future<String> future, String Function(dynamic, StackTrace) callback) {
  future.catchError(callback);
}
```

### invalid_sealed_annotation

_The annotation '@sealed' can only be applied to classes._

#### Description

The analyzer produces this diagnostic when a declaration other than a
class declaration has the `@sealed` annotation on it.

#### Example

The following code produces this diagnostic because the `@sealed`
annotation is on a method declaration:

```dart
import 'package:meta/meta.dart';

class A {
  @[!sealed!]
  void m() {}
}
```

#### Common fixes

Remove the annotation:

```dart
class A {
  void m() {}
}
```

### invalid_super_formal_parameter_location

_Super parameters can only be used in non-redirecting generative constructors._

#### Description

The analyzer produces this diagnostic when a super parameter is used
anywhere other than a non-redirecting generative constructor.

#### Examples

The following code produces this diagnostic because the super parameter
`x` is in a redirecting generative constructor:

```dart
class A {
  A(int x);
}

class B extends A {
  B.b([!super!].x) : this._();
  B._() : super(0);
}
```

The following code produces this diagnostic because the super parameter
`x` isn't in a generative constructor:

```dart
class A {
  A(int x);
}

class C extends A {
  factory C.c([!super!].x) => C._();
  C._() : super(0);
}
```

The following code produces this diagnostic because the super parameter
`x` is in a method:

```dart
class A {
  A(int x);
}

class D extends A {
  D() : super(0);

  void m([!super!].x) {}
}
```

#### Common fixes

If the function containing the super parameter can be changed to be a
non-redirecting generative constructor, then do so:

```dart
class A {
  A(int x);
}

class B extends A {
  B.b(super.x);
}
```

If the function containing the super parameter can't be changed to be a
non-redirecting generative constructor, then remove the `super`:

```dart
class A {
  A(int x);
}

class D extends A {
  D() : super(0);

  void m(int x) {}
}
```

### invalid_type_argument_in_const_literal

_Constant list literals can't include a type parameter as a type argument, such
as '{0}'._

_Constant map literals can't include a type parameter as a type argument, such
as '{0}'._

_Constant set literals can't include a type parameter as a type argument, such
as '{0}'._

#### Description

The analyzer produces this diagnostic when a type parameter is used as a
type argument in a list, map, or set literal that is prefixed by `const`.
This isn't allowed because the value of the type parameter (the actual type
that will be used at runtime) can't be known at compile time.

#### Examples

The following code produces this diagnostic because the type parameter `T`
is being used as a type argument when creating a constant list:

```dart
List<T> newList<T>() => const <[!T!]>[];
```

The following code produces this diagnostic because the type parameter `T`
is being used as a type argument when creating a constant map:

```dart
Map<String, T> newSet<T>() => const <String, [!T!]>{};
```

The following code produces this diagnostic because the type parameter `T`
is being used as a type argument when creating a constant set:

```dart
Set<T> newSet<T>() => const <[!T!]>{};
```

#### Common fixes

If the type that will be used for the type parameter can be known at
compile time, then remove the type parameter:

```dart
List<int> newList() => const <int>[];
```

If the type that will be used for the type parameter can't be known until
runtime, then remove the keyword `const`:

```dart
List<T> newList<T>() => <T>[];
```

### invalid_uri

_Invalid URI syntax: '{0}'._

#### Description

The analyzer produces this diagnostic when a URI in a directive doesn't
conform to the syntax of a valid URI.

#### Example

The following code produces this diagnostic because `'#'` isn't a valid
URI:

```dart
import [!'#'!];
```

#### Common fixes

Replace the invalid URI with a valid URI.

### invalid_use_of_covariant_in_extension

_Can't have modifier '{0}' in an extension._

#### Description

The analyzer produces this diagnostic when a member declared inside an
extension uses the keyword `covariant` in the declaration of a parameter.
Extensions aren't classes and don't have subclasses, so the keyword serves
no purpose.

#### Example

The following code produces this diagnostic because `i` is marked as being
covariant:

```dart
extension E on String {
  void a([!covariant!] int i) {}
}
```

#### Common fixes

Remove the `covariant` keyword:

```dart
extension E on String {
  void a(int i) {}
}
```

### invalid_use_of_internal_member

_The member '{0}' can only be used within its package._

#### Description

The analyzer produces this diagnostic when a reference to a declaration
that is annotated with the [`internal`][meta-internal] annotation is found
outside the package containing the declaration.

#### Example

Given a package `p` that defines a library containing a declaration marked
with the [`internal`][meta-internal] annotation:

```dart
import 'package:meta/meta.dart';

@internal
class C {}
```

The following code produces this diagnostic because it's referencing the
class `C`, which isn't intended to be used outside the package `p`:

```dart
import 'package:p/src/p.dart';

void f([!C!] c) {}
```

#### Common fixes

Remove the reference to the internal declaration.

### invalid_use_of_null_value

_An expression whose value is always 'null' can't be dereferenced._

#### Description

The analyzer produces this diagnostic when an expression whose value will
always be `null` is dereferenced.

#### Example

The following code produces this diagnostic because `x` will always be
`null`:

```dart
int f(Null x) {
  return x.[!length!];
}
```

#### Common fixes

If the value is allowed to be something other than `null`, then change the
type of the expression:

```dart
int f(String? x) {
  return x!.length;
}
```

### invalid_use_of_type_outside_library

_The class '{0}' can't be extended outside of its library because it's a final
class._

_The class '{0}' can't be extended outside of its library because it's an
interface class._

_The class '{0}' can't be extended, implemented, or mixed in outside of its
library because it's a sealed class._

_The class '{0}' can't be implemented outside of its library because it's a base
class._

_The class '{0}' can't be implemented outside of its library because it's a
final class._

_The class '{0}' can't be used as a mixin superclass constraint outside of its
library because it's a final class._

_The mixin '{0}' can't be implemented outside of its library because it's a base
mixin._

#### Description

The analyzer produces this diagnostic when an `extends`, `implements`,
`with`, or `on` clause uses a class or mixin in a way that isn't allowed
given the modifiers on that class or mixin's declaration.

The message specifies how the declaration is being used and why it isn't
allowed.

#### Example

Given a file `a.dart` that defines a base class `A`:

```dart
base class A {}
```

The following code produces this diagnostic because the class `B`
implements the class `A`, but the `base` modifier prevents `A` from being
implemented outside of the library where it's defined:

```dart
import 'a.dart';

final class B implements [!A!] {}
```

#### Common fixes

Use of this type is restricted outside of its declaring library. If a
different, unrestricted type is available that can provide similar
functionality, then replace the type:

```dart
class B implements C {}
class C {}
```

If there isn't a different type that would be appropriate, then remove the
type, and possibly the whole clause:

```dart
class B {}
```

### invalid_use_of_visible_for_overriding_member

_The member '{0}' can only be used for overriding._

#### Description

The analyzer produces this diagnostic when an instance member that is
annotated with [`visibleForOverriding`][meta-visibleForOverriding] is
referenced outside the library in which it's declared for any reason other
than to override it.

#### Example

Given a file `a.dart` containing the following declaration:

```dart
import 'package:meta/meta.dart';

class A {
  @visibleForOverriding
  void a() {}
}
```

The following code produces this diagnostic because the method `m` is being
invoked even though the only reason it's public is to allow it to be
overridden:

```dart
import 'a.dart';

class B extends A {
  void b() {
    [!a!]();
  }
}
```

#### Common fixes

Remove the invalid use of the member.

### invalid_use_of_visible_for_testing_member

_The member '{0}' can only be used within '{1}' or a test._

#### Description

The analyzer produces this diagnostic when a member annotated with
`@visibleForTesting` is referenced anywhere other than the library in
which it is declared or in a library in the `test` directory.

#### Example

Given a file `c.dart` that contains the following:

```dart
import 'package:meta/meta.dart';

class C {
  @visibleForTesting
  void m() {}
}
```

The following code, when not inside the `test` directory, produces this
diagnostic because the method `m` is marked as being visible only for
tests:

```dart
import 'c.dart';

void f(C c) {
  c.[!m!]();
}
```

#### Common fixes

If the annotated member should not be referenced outside of tests, then
remove the reference:

```dart
import 'c.dart';

void f(C c) {}
```

If it's OK to reference the annotated member outside of tests, then remove
the annotation:

```dart
class C {
  void m() {}
}
```

### invalid_visibility_annotation

_The member '{0}' is annotated with '{1}', but this annotation is only
meaningful on declarations of public members._

#### Description

The analyzer produces this diagnostic when either the `visibleForTemplate`
or [`visibleForTesting`][meta-visibleForTesting] annotation is applied to
a non-public declaration.

#### Example

The following code produces this diagnostic:

```dart
import 'package:meta/meta.dart';

@[!visibleForTesting!]
void _someFunction() {}

void f() => _someFunction();
```

#### Common fixes

If the declaration doesn't need to be used by test code, then remove the
annotation:

```dart
void _someFunction() {}

void f() => _someFunction();
```

If it does, then make it public:

```dart
import 'package:meta/meta.dart';

@visibleForTesting
void someFunction() {}

void f() => someFunction();
```

### invalid_visible_for_overriding_annotation

_The annotation 'visibleForOverriding' can only be applied to a public instance
member that can be overridden._

#### Description

The analyzer produces this diagnostic when anything other than a public
instance member of a class is annotated with
[`visibleForOverriding`][meta-visibleForOverriding]. Because only public
instance members can be overridden outside the defining library, there's
no value to annotating any other declarations.

#### Example

The following code produces this diagnostic because the annotation is on a
class, and classes can't be overridden:

```dart
import 'package:meta/meta.dart';

@[!visibleForOverriding!]
class C {}
```

#### Common fixes

Remove the annotation:

```dart
class C {}
```

### invalid_visible_outside_template_annotation

_The annotation 'visibleOutsideTemplate' can only be applied to a member of a
class, enum, or mixin that is annotated with 'visibleForTemplate'._

#### Description

The analyzer produces this diagnostic when the `@visibleOutsideTemplate`
annotation is used incorrectly. This annotation is only meant to annotate
members of a class, enum, or mixin that has the `@visibleForTemplate`
annotation, to opt those members out of the visibility restrictions that
`@visibleForTemplate` imposes.

#### Examples

The following code produces this diagnostic because there is no
`@visibleForTemplate` annotation at the class level:

```dart
import 'package:angular_meta/angular_meta.dart';

class C {
  @[!visibleOutsideTemplate!]
  int m() {
    return 1;
  }
}
```

The following code produces this diagnostic because the annotation is on
a class declaration, not a member of a class, enum, or mixin:

```dart
import 'package:angular_meta/angular_meta.dart';

@[!visibleOutsideTemplate!]
class C {}
```

#### Common fixes

If the class is only visible so that templates can reference it, then add
the `@visibleForTemplate` annotation to the class:

```dart
import 'package:angular_meta/angular_meta.dart';

@visibleForTemplate
class C {
  @visibleOutsideTemplate
  int m() {
    return 1;
  }
}
```

If the `@visibleOutsideTemplate` annotation is on anything other than a
member of a class, enum, or mixin with the `@visibleForTemplate`
annotation, remove the annotation:

```dart
class C {}
```

### invocation_of_extension_without_call

_The extension '{0}' doesn't define a 'call' method so the override can't be
used in an invocation._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a function but the extension doesn't declare a `call` method.

#### Example

The following code produces this diagnostic because the extension `E`
doesn't define a `call` method:

```dart
extension E on String {}

void f() {
  [!E('')!]();
}
```

#### Common fixes

If the extension is intended to define a `call` method, then declare it:

```dart
extension E on String {
  int call() => 0;
}

void f() {
  E('')();
}
```

If the extended type defines a `call` method, then remove the extension
override.

If the `call` method isn't defined, then rewrite the code so that it
doesn't invoke the `call` method.

### invocation_of_non_function

_'{0}' isn't a function._

#### Description

The analyzer produces this diagnostic when it finds a function invocation,
but the name of the function being invoked is defined to be something other
than a function.

#### Example

The following code produces this diagnostic because `Binary` is the name of
a function type, not a function:

```dart
typedef Binary = int Function(int, int);

int f() {
  return [!Binary!](1, 2);
}
```

#### Common fixes

Replace the name with the name of a function.

### invocation_of_non_function_expression

_The expression doesn't evaluate to a function, so it can't be invoked._

#### Description

The analyzer produces this diagnostic when a function invocation is found,
but the name being referenced isn't the name of a function, or when the
expression computing the function doesn't compute a function.

#### Examples

The following code produces this diagnostic because `x` isn't a function:

```dart
int x = 0;

int f() => x;

var y = [!x!]();
```

The following code produces this diagnostic because `f()` doesn't return a
function:

```dart
int x = 0;

int f() => x;

var y = [!f()!]();
```

#### Common fixes

If you need to invoke a function, then replace the code before the argument
list with the name of a function or with an expression that computes a
function:

```dart
int x = 0;

int f() => x;

var y = f();
```

### label_in_outer_scope

_Can't reference label '{0}' declared in an outer method._

#### Description

The analyzer produces this diagnostic when a `break` or `continue`
statement references a label that is declared in a method or function
containing the function in which the `break` or `continue` statement
appears. The `break` and `continue` statements can't be used to transfer
control outside the function that contains them.

#### Example

The following code produces this diagnostic because the label `loop` is
declared outside the local function `g`:

```dart
void f() {
  loop:
  while (true) {
    void g() {
      break [!loop!];
    }

    g();
  }
}
```

#### Common fixes

Try rewriting the code so that it isn't necessary to transfer control
outside the local function, possibly by inlining the local function:

```dart
void f() {
  loop:
  while (true) {
    break loop;
  }
}
```

If that isn't possible, then try rewriting the local function so that a
value returned by the function can be used to determine whether control is
transferred:

```dart
void f() {
  loop:
  while (true) {
    bool g() {
      return true;
    }

    if (g()) {
      break loop;
    }
  }
}
```

### label_undefined

_Can't reference an undefined label '{0}'._

#### Description

The analyzer produces this diagnostic when it finds a reference to a label
that isn't defined in the scope of the `break` or `continue` statement that
is referencing it.

#### Example

The following code produces this diagnostic because the label `loop` isn't
defined anywhere:

```dart
void f() {
  for (int i = 0; i < 10; i++) {
    for (int j = 0; j < 10; j++) {
      if (j != 0) {
        break [!loop!];
      }
    }
  }
}
```

#### Common fixes

If the label should be on the innermost enclosing `do`, `for`, `switch`, or
`while` statement, then remove the label:

```dart
void f() {
  for (int i = 0; i < 10; i++) {
    for (int j = 0; j < 10; j++) {
      if (j != 0) {
        break;
      }
    }
  }
}
```

If the label should be on some other statement, then add the label:

```dart
void f() {
  loop: for (int i = 0; i < 10; i++) {
    for (int j = 0; j < 10; j++) {
      if (j != 0) {
        break loop;
      }
    }
  }
}
```

### late_final_field_with_const_constructor

_Can't have a late final field in a class with a generative const constructor._

#### Description

The analyzer produces this diagnostic when a class that has at least one
`const` constructor also has a field marked both `late` and `final`.

#### Example

The following code produces this diagnostic because the class `A` has a
`const` constructor and the `final` field `f` is marked as `late`:

```dart
class A {
  [!late!] final int f;

  const A();
}
```

#### Common fixes

If the field doesn't need to be marked `late`, then remove the `late`
modifier from the field:

```dart
class A {
  final int f = 0;

  const A();
}
```

If the field must be marked `late`, then remove the `const` modifier from
the constructors:

```dart
class A {
  late final int f;

  A();
}
```

### late_final_local_already_assigned

_The late final local variable is already assigned._

#### Description

The analyzer produces this diagnostic when the analyzer can prove that a
local variable marked as both `late` and `final` was already assigned a
value at the point where another assignment occurs.

Because `final` variables can only be assigned once, subsequent assignments
are guaranteed to fail, so they're flagged.

#### Example

The following code produces this diagnostic because the `final` variable
`v` is assigned a value in two places:

```dart
int f() {
  late final int v;
  v = 0;
  [!v!] += 1;
  return v;
}
```

#### Common fixes

If you need to be able to reassign the variable, then remove the `final`
keyword:

```dart
int f() {
  late int v;
  v = 0;
  v += 1;
  return v;
}
```

If you don't need to reassign the variable, then remove all except the
first of the assignments:

```dart
int f() {
  late final int v;
  v = 0;
  return v;
}
```

### leaf_call_must_not_return_handle

_FFI leaf call can't return a 'Handle'._

#### Description

The analyzer produces this diagnostic when the value of the `isLeaf`
argument in an invocation of either `Pointer.asFunction` or
`DynamicLibrary.lookupFunction` is `true` and the function that would be
returned would have a return type of `Handle`.

The analyzer also produces this diagnostic when the value of the `isLeaf`
argument in an `Native` annotation is `true` and the type argument on
the annotation is a function type whose return type is `Handle`.

In all of these cases, leaf calls are only supported for the types `bool`,
`int`, `float`, `double`, and, as a return type `void`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the function `p`
returns a `Handle`, but the `isLeaf` argument is `true`:

```dart
import 'dart:ffi';

void f(Pointer<NativeFunction<Handle Function()>> p) {
  [!p.asFunction<Object Function()>(isLeaf: true)!];
}
```

#### Common fixes

If the function returns a handle, then remove the `isLeaf` argument:

```dart
import 'dart:ffi';

void f(Pointer<NativeFunction<Handle Function()>> p) {
  p.asFunction<Object Function()>();
}
```

If the function returns one of the supported types, then correct the type
information:

```dart
import 'dart:ffi';

void f(Pointer<NativeFunction<Int32 Function()>> p) {
  p.asFunction<int Function()>(isLeaf: true);
}
```

### leaf_call_must_not_take_handle

_FFI leaf call can't take arguments of type 'Handle'._

#### Description

The analyzer produces this diagnostic when the value of the `isLeaf`
argument in an invocation of either `Pointer.asFunction` or
`DynamicLibrary.lookupFunction` is `true` and the function that would be
returned would have a parameter of type `Handle`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the function `p` has a
parameter of type `Handle`, but the `isLeaf` argument is `true`:

```dart
import 'dart:ffi';

void f(Pointer<NativeFunction<Void Function(Handle)>> p) {
  [!p.asFunction<void Function(Object)>(isLeaf: true)!];
}
```

#### Common fixes

If the function has at least one parameter of type `Handle`, then remove
the `isLeaf` argument:

```dart
import 'dart:ffi';

void f(Pointer<NativeFunction<Void Function(Handle)>> p) {
  p.asFunction<void Function(Object)>();
}
```

If none of the function's parameters are `Handle`s, then correct the type
information:

```dart
import 'dart:ffi';

void f(Pointer<NativeFunction<Void Function(Int8)>> p) {
  p.asFunction<void Function(int)>(isLeaf: true);
}
```

### list_element_type_not_assignable

_The element type '{0}' can't be assigned to the list type '{1}'._

#### Description

The analyzer produces this diagnostic when the type of an element in a list
literal isn't assignable to the element type of the list.

#### Example

The following code produces this diagnostic because `2.5` is a double, and
the list can hold only integers:

```dart
List<int> x = [1, [!2.5!], 3];
```

#### Common fixes

If you intended to add a different object to the list, then replace the
element with an expression that computes the intended object:

```dart
List<int> x = [1, 2, 3];
```

If the object shouldn't be in the list, then remove the element:

```dart
List<int> x = [1, 3];
```

If the object being computed is correct, then widen the element type of the
list to allow all of the different types of objects it needs to contain:

```dart
List<num> x = [1, 2.5, 3];
```

### main_first_positional_parameter_type

_The type of the first positional parameter of the 'main' function must be a
supertype of 'List<String>'._

#### Description

The analyzer produces this diagnostic when the first positional parameter
of a function named `main` isn't a supertype of `List<String>`.

#### Example

The following code produces this diagnostic because `List<int>` isn't a
supertype of `List<String>`:

```dart
void main([!List<int>!] args) {}
```

#### Common fixes

If the function is an entry point, then change the type of the first
positional parameter to be a supertype of `List<String>`:

```dart
void main(List<String> args) {}
```

If the function isn't an entry point, then change the name of the function:

```dart
void f(List<int> args) {}
```

### main_has_required_named_parameters

_The function 'main' can't have any required named parameters._

#### Description

The analyzer produces this diagnostic when a function named `main` has one
or more required named parameters.

#### Example

The following code produces this diagnostic because the function named
`main` has a required named parameter (`x`):

```dart
void [!main!]({required int x}) {}
```

#### Common fixes

If the function is an entry point, then remove the `required` keyword:

```dart
void main({int? x}) {}
```

If the function isn't an entry point, then change the name of the function:

```dart
void f({required int x}) {}
```

### main_has_too_many_required_positional_parameters

_The function 'main' can't have more than two required positional parameters._

#### Description

The analyzer produces this diagnostic when a function named `main` has more
than two required positional parameters.

#### Example

The following code produces this diagnostic because the function `main` has
three required positional parameters:

```dart
void [!main!](List<String> args, int x, int y) {}
```

#### Common fixes

If the function is an entry point and the extra parameters aren't used,
then remove them:

```dart
void main(List<String> args, int x) {}
```

If the function is an entry point, but the extra parameters used are for
when the function isn't being used as an entry point, then make the extra
parameters optional:

```dart
void main(List<String> args, int x, [int y = 0]) {}
```

If the function isn't an entry point, then change the name of the function:

```dart
void f(List<String> args, int x, int y) {}
```

### main_is_not_function

_The declaration named 'main' must be a function._

#### Description

The analyzer produces this diagnostic when a library contains a declaration
of the name `main` that isn't the declaration of a top-level function.

#### Example

The following code produces this diagnostic because the name `main` is
being used to declare a top-level variable:

```dart
var [!main!] = 3;
```

#### Common fixes

Use a different name for the declaration:

```dart
var mainIndex = 3;
```

### map_entry_not_in_map

_Map entries can only be used in a map literal._

#### Description

The analyzer produces this diagnostic when a map entry (a key/value pair)
is found in a set literal.

#### Example

The following code produces this diagnostic because the literal has a map
entry even though it's a set literal:

```dart
var collection = <String>{[!'a' : 'b'!]};
```

#### Common fixes

If you intended for the collection to be a map, then change the code so
that it is a map. In the previous example, you could do this by adding
another type argument:

```dart
var collection = <String, String>{'a' : 'b'};
```

In other cases, you might need to change the explicit type from `Set` to
`Map`.

If you intended for the collection to be a set, then remove the map entry,
possibly by replacing the colon with a comma if both values should be
included in the set:

```dart
var collection = <String>{'a', 'b'};
```

### map_key_type_not_assignable

_The element type '{0}' can't be assigned to the map key type '{1}'._

#### Description

The analyzer produces this diagnostic when a key of a key-value pair in a
map literal has a type that isn't assignable to the key type of the map.

#### Example

The following code produces this diagnostic because `2` is an `int`, but
the keys of the map are required to be `String`s:

```dart
var m = <String, String>{[!2!] : 'a'};
```

#### Common fixes

If the type of the map is correct, then change the key to have the correct
type:

```dart
var m = <String, String>{'2' : 'a'};
```

If the type of the key is correct, then change the key type of the map:

```dart
var m = <int, String>{2 : 'a'};
```

### map_value_type_not_assignable

_The element type '{0}' can't be assigned to the map value type '{1}'._

#### Description

The analyzer produces this diagnostic when a value of a key-value pair in a
map literal has a type that isn't assignable to the value type of the
map.

#### Example

The following code produces this diagnostic because `2` is an `int`, but/
the values of the map are required to be `String`s:

```dart
var m = <String, String>{'a' : [!2!]};
```

#### Common fixes

If the type of the map is correct, then change the value to have the
correct type:

```dart
var m = <String, String>{'a' : '2'};
```

If the type of the value is correct, then change the value type of the map:

```dart
var m = <String, int>{'a' : 2};
```

### mismatched_annotation_on_struct_field

_The annotation doesn't match the declared type of the field._

#### Description

The analyzer produces this diagnostic when the annotation on a field in a
subclass of `Struct` or `Union` doesn't match the Dart type of the field.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the annotation
`Double` doesn't match the Dart type `int`:

```dart
import 'dart:ffi';

final class C extends Struct {
  [!@Double()!]
  external int x;
}
```

#### Common fixes

If the type of the field is correct, then change the annotation to match:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int32()
  external int x;
}
```

If the annotation is correct, then change the type of the field to match:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Double()
  external double x;
}
```

### missing_annotation_on_struct_field

_Fields of type '{0}' in a subclass of '{1}' must have an annotation indicating
the native type._

#### Description

The analyzer produces this diagnostic when a field in a subclass of
`Struct` or `Union` whose type requires an annotation doesn't have one.
The Dart types `int`, `double`, and `Array` are used to represent multiple
C types, and the annotation specifies which of the compatible C types the
field represents.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `x` doesn't
have an annotation indicating the underlying width of the integer value:

```dart
import 'dart:ffi';

final class C extends Struct {
  external [!int!] x;
}
```

#### Common fixes

Add an appropriate annotation to the field:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int64()
  external int x;
}
```

### missing_dart_library

_Required library '{0}' is missing._

#### Description

The analyzer produces this diagnostic when either the Dart or Flutter SDK
isn't installed correctly, and, as a result, one of the `dart:` libraries
can't be found.

#### Common fixes

Reinstall the Dart or Flutter SDK.

### missing_default_value_for_parameter

_The parameter '{0}' can't have a value of 'null' because of its type, but the
implicit default value is 'null'._

_With null safety, use the 'required' keyword, not the '@required' annotation._

#### Description

The analyzer produces this diagnostic when an optional parameter, whether
positional or named, has a [potentially non-nullable][] type and doesn't
specify a default value. Optional parameters that have no explicit default
value have an implicit default value of `null`. If the type of the
parameter doesn't allow the parameter to have a value of `null`, then the
implicit default value isn't valid.

#### Examples

The following code produces this diagnostic because `x` can't be `null`,
and no non-`null` default value is specified:

```dart
void f([int [!x!]]) {}
```

As does this:

```dart
void g({int [!x!]}) {}
```

#### Common fixes

If you want to use `null` to indicate that no value was provided, then you
need to make the type nullable:

```dart
void f([int? x]) {}
void g({int? x}) {}
```

If the parameter can't be null, then either provide a default value:

```dart
void f([int x = 1]) {}
void g({int x = 2}) {}
```

or make the parameter a required parameter:

```dart
void f(int x) {}
void g({required int x}) {}
```

### missing_dependency

_Missing a dependency on imported package '{0}'._

#### Description

The analyzer produces this diagnostic when there's a package that has been
imported in the source but is not listed as a dependency of the
importing package.

#### Example

The following code produces this diagnostic because the package `path` is
not listed as a dependency, while there is an import statement
with package `path` in the source code of package `example`:

```yaml
name: example
dependencies:
  meta: ^1.0.2
```

#### Common fixes

Add the missing package `path` to the `dependencies` field:

```yaml
name: example
dependencies:
  meta: ^1.0.2
  path: any
```

### missing_enum_constant_in_switch

_Missing case clause for '{0}'._

#### Description

The analyzer produces this diagnostic when a `switch` statement for an enum
doesn't include an option for one of the values in the enum.

Note that `null` is always a possible value for an enum and therefore also
must be handled.

#### Example

The following code produces this diagnostic because the enum value `e2`
isn't handled:

```dart
enum E { e1, e2 }

void f(E e) {
  [!switch (e)!] {
    case E.e1:
      break;
  }
}
```

#### Common fixes

If there's special handling for the missing values, then add a `case`
clause for each of the missing values:

```dart
enum E { e1, e2 }

void f(E e) {
  switch (e) {
    case E.e1:
      break;
    case E.e2:
      break;
  }
}
```

If the missing values should be handled the same way, then add a `default`
clause:

```dart
enum E { e1, e2 }

void f(E e) {
  switch (e) {
    case E.e1:
      break;
    default:
      break;
  }
}
```

### missing_exception_value

_The method {0} must have an exceptional return value (the second argument) when
the return type of the function is neither 'void', 'Handle', nor 'Pointer'._

#### Description

The analyzer produces this diagnostic when an invocation of the method
`Pointer.fromFunction` or `NativeCallable.isolateLocal`
doesn't have a second argument (the exceptional
return value) when the type to be returned from the invocation is neither
`void`, `Handle`, nor `Pointer`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the type returned by
`f` is expected to be an 8-bit integer but the call to `fromFunction`
doesn't include an exceptional return argument:

```dart
import 'dart:ffi';

int f(int i) => i * 2;

void g() {
  Pointer.[!fromFunction!]<Int8 Function(Int8)>(f);
}
```

#### Common fixes

Add an exceptional return type:

```dart
import 'dart:ffi';

int f(int i) => i * 2;

void g() {
  Pointer.fromFunction<Int8 Function(Int8)>(f, 0);
}
```

### missing_field_type_in_struct

_Fields in struct classes must have an explicitly declared type of 'int',
'double' or 'Pointer'._

#### Description

The analyzer produces this diagnostic when a field in a subclass of
`Struct` or `Union` doesn't have a type annotation. Every field must have
an explicit type, and the type must either be `int`, `double`, `Pointer`,
or a subclass of either `Struct` or `Union`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `str`
doesn't have a type annotation:

```dart
import 'dart:ffi';

final class C extends Struct {
  external var [!str!];

  @Int32()
  external int i;
}
```

#### Common fixes

Explicitly specify the type of the field:

```dart
import 'dart:ffi';
import 'package:ffi/ffi.dart';

final class C extends Struct {
  external Pointer<Utf8> str;

  @Int32()
  external int i;
}
```

### missing_name

_The 'name' field is required but missing._

#### Description

The analyzer produces this diagnostic when there's no top-level `name` key.
The `name` key provides the name of the package, which is required.

#### Example

The following code produces this diagnostic because the package doesn't
have a name:

```yaml
dependencies:
  meta: ^1.0.2
```

#### Common fixes

Add the top-level key `name` with a value that's the name of the package:

```yaml
name: example
dependencies:
  meta: ^1.0.2
```

### missing_named_pattern_field_name

_The getter name is not specified explicitly, and the pattern is not a
variable._

#### Description

The analyzer produces this diagnostic when, within an object pattern, the
specification of a property and the pattern used to match the property's
value doesn't have either:

- a getter name before the colon
- a variable pattern from which the getter name can be inferred

#### Example

The following code produces this diagnostic because there is no getter
name before the colon and no variable pattern after the colon in the
object pattern (`C(:0)`):

```dart
abstract class C {
  int get f;
}

void f(C c) {
  switch (c) {
    case C([!:0!]):
      break;
  }
}
```

#### Common fixes

If you need to use the actual value of the property within the pattern's
scope, then add a variable pattern where the name of the variable is the
same as the name of the property being matched:

```dart
abstract class C {
  int get f;
}

void f(C c) {
  switch (c) {
    case C(:var f) when f == 0:
      print(f);
  }
}
```

If you don't need to use the actual value of the property within the
pattern's scope, then add the name of the property being matched before
the colon:

```dart
abstract class C {
  int get f;
}

void f(C c) {
  switch (c) {
    case C(f: 0):
      break;
  }
}
```

### missing_override_of_must_be_overridden

_Missing concrete implementation of '{0}'._

_Missing concrete implementations of '{0}' and '{1}'._

_Missing concrete implementations of '{0}', '{1}', and {2} more._

#### Description

The analyzer produces this diagnostic when an instance member that has the
`@mustBeOverridden` annotation isn't overridden in a subclass.

#### Example

The following code produces this diagnostic because the class `B` doesn't
have an override of the inherited method `A.m` when `A.m` is annotated
with `@mustBeOverridden`:

```dart
import 'package:meta/meta.dart';

class A {
  @mustBeOverridden
  void m() {}
}

class [!B!] extends A {}
```

#### Common fixes

If the annotation is appropriate for the member, then override the member
in the subclass:

```dart
import 'package:meta/meta.dart';

class A {
  @mustBeOverridden
  void m() {}
}

class B extends A {
  @override
  void m() {}
}
```

If the annotation isn't appropriate for the member, then remove the
annotation:

```dart
class A {
  void m() {}
}

class B extends A {}
```

### missing_required_argument

_The named parameter '{0}' is required, but there's no corresponding argument._

#### Description

The analyzer produces this diagnostic when an invocation of a function is
missing a required named parameter.

#### Example

The following code produces this diagnostic because the invocation of `f`
doesn't include a value for the required named parameter `end`:

```dart
void f(int start, {required int end}) {}
void g() {
  [!f!](3);
}
```

#### Common fixes

Add a named argument corresponding to the missing required parameter:

```dart
void f(int start, {required int end}) {}
void g() {
  f(3, end: 5);
}
```

### missing_required_param

_The parameter '{0}' is required._

_The parameter '{0}' is required. {1}._

#### Description

The analyzer produces this diagnostic when a method or function with a
named parameter that is annotated as being required is invoked without
providing a value for the parameter.

#### Example

The following code produces this diagnostic because the named parameter `x`
is required:

```dart
import 'package:meta/meta.dart';

void f({@required int? x}) {}

void g() {
  [!f!]();
}
```

#### Common fixes

Provide the required value:

```dart
import 'package:meta/meta.dart';

void f({@required int? x}) {}

void g() {
  f(x: 2);
}
```

### missing_return

_This function has a return type of '{0}', but doesn't end with a return
statement._

#### Description

Any function or method that doesn't end with either an explicit return or a
throw implicitly returns `null`. This is rarely the desired behavior. The
analyzer produces this diagnostic when it finds an implicit return.

#### Example

The following code produces this diagnostic because `f` doesn't end with a
return:

```dart
int [!f!](int x) {
  if (x < 0) {
    return 0;
  }
}
```

#### Common fixes

Add a `return` statement that makes the return value explicit, even if
`null` is the appropriate value.

### missing_size_annotation_carray

_Fields of type 'Array' must have exactly one 'Array' annotation._

#### Description

The analyzer produces this diagnostic when a field in a subclass of either
`Struct` or `Union` has a type of `Array` but doesn't have a single
`Array` annotation indicating the dimensions of the array.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `a0` doesn't
have an `Array` annotation:

```dart
import 'dart:ffi';

final class C extends Struct {
  external [!Array<Uint8>!] a0;
}
```

#### Common fixes

Ensure that there's exactly one `Array` annotation on the field:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Array(8)
  external Array<Uint8> a0;
}
```

### missing_variable_pattern

_Variable pattern '{0}' is missing in this branch of the logical-or pattern._

#### Description

The analyzer produces this diagnostic when one branch of a logical-or
pattern doesn't declare a variable that is declared on the other branch of
the same pattern.

#### Example

The following code produces this diagnostic because the right-hand side of
the logical-or pattern doesn't declare the variable `a`:

```dart
void f((int, int) r) {
  if (r case (var a, 0) || [!(0, _)!]) {
    print(a);
  }
}
```

#### Common fixes

If the variable needs to be referenced in the controlled statements, then
add a declaration of the variable to every branch of the logical-or
pattern:

```dart
void f((int, int) r) {
  if (r case (var a, 0) || (0, var a)) {
    print(a);
  }
}
```

If the variable doesn't need to be referenced in the controlled
statements, then remove the declaration of the variable from every branch
of the logical-or pattern:

```dart
void f((int, int) r) {
  if (r case (_, 0) || (0, _)) {
    print('found a zero');
  }
}
```

If the variable needs to be referenced if one branch of the pattern
matches but not when the other matches, then break the pattern into two
pieces:

```dart
void f((int, int) r) {
  switch (r) {
    case (var a, 0):
      print(a);
    case (0, _):
      print('found a zero');
  }
}
```

### mixin_application_concrete_super_invoked_member_type

_The super-invoked member '{0}' has the type '{1}', and the concrete member in
the class has the type '{2}'._

#### Description

The analyzer produces this diagnostic when a mixin that invokes a method
using `super` is used in a class where the concrete implementation of that
method has a different signature than the signature defined for that method
by the mixin's `on` type. The reason this is an error is because the
invocation in the mixin might invoke the method in a way that's
incompatible with the method that will actually be executed.

#### Example

The following code produces this diagnostic because the class `C` uses the
mixin `M`, the mixin `M` invokes `foo` using `super`, and the abstract
version of `foo` declared in `I` (the mixin's `on` type) doesn't have the
same signature as the concrete version of `foo` declared in `A`:

```dart
class I {
  void foo([int? p]) {}
}

class A {
  void foo(int p) {}
}

abstract class B extends A implements I {
  @override
  void foo([int? p]);
}

mixin M on I {
  void bar() {
    super.foo(42);
  }
}

abstract class C extends B with [!M!] {}
```

#### Common fixes

If the class doesn't need to use the mixin, then remove it from the `with`
clause:

```dart
class I {
  void foo([int? p]) {}
}

class A {
  void foo(int? p) {}
}

abstract class B extends A implements I {
  @override
  void foo([int? p]);
}

mixin M on I {
  void bar() {
    super.foo(42);
  }
}

abstract class C extends B {}
```

If the class needs to use the mixin, then ensure that there's a concrete
implementation of the method that conforms to the signature expected by the
mixin:

```dart
class I {
  void foo([int? p]) {}
}

class A {
  void foo(int? p) {}
}

abstract class B extends A implements I {
  @override
  void foo([int? p]) {
    super.foo(p);
  }
}

mixin M on I {
  void bar() {
    super.foo(42);
  }
}

abstract class C extends B with M {}
```

### mixin_application_not_implemented_interface

_'{0}' can't be mixed onto '{1}' because '{1}' doesn't implement '{2}'._

#### Description

The analyzer produces this diagnostic when a mixin that has a superclass
constraint is used in a [mixin application][] with a superclass that
doesn't implement the required constraint.

#### Example

The following code produces this diagnostic because the mixin `M` requires
that the class to which it's applied be a subclass of `A`, but `Object`
isn't a subclass of `A`:

```dart
class A {}

mixin M on A {}

class X = Object with [!M!];
```

#### Common fixes

If you need to use the mixin, then change the superclass to be either the
same as or a subclass of the superclass constraint:

```dart
class A {}

mixin M on A {}

class X = A with M;
```

### mixin_application_no_concrete_super_invoked_member

_The class doesn't have a concrete implementation of the super-invoked member
'{0}'._

_The class doesn't have a concrete implementation of the super-invoked setter
'{0}'._

#### Description

The analyzer produces this diagnostic when a [mixin application][] contains
an invocation of a member from its superclass, and there's no concrete
member of that name in the mixin application's superclass.

#### Example

The following code produces this diagnostic because the mixin `M` contains
the invocation `super.m()`, and the class `A`, which is the superclass of
the [mixin application][] `A+M`, doesn't define a concrete implementation
of `m`:

```dart
abstract class A {
  void m();
}

mixin M on A {
  void bar() {
    super.m();
  }
}

abstract class B extends A with [!M!] {}
```

#### Common fixes

If you intended to apply the mixin `M` to a different class, one that has a
concrete implementation of `m`, then change the superclass of `B` to that
class:

```dart
abstract class A {
  void m();
}

mixin M on A {
  void bar() {
    super.m();
  }
}

class C implements A {
  void m() {}
}

abstract class B extends C with M {}
```

If you need to make `B` a subclass of `A`, then add a concrete
implementation of `m` in `A`:

```dart
abstract class A {
  void m() {}
}

mixin M on A {
  void bar() {
    super.m();
  }
}

abstract class B extends A with M {}
```

### mixin_class_declaration_extends_not_object

_The class '{0}' can't be declared a mixin because it extends a class other than
'Object'._

#### Description

The analyzer produces this diagnostic when a class that is marked with
the `mixin` modifier extends a class other than `Object`. A mixin class
can't have a superclass other than `Object`.

#### Example

The following code produces this diagnostic because the class `B`, which
has the modifier `mixin`, extends `A`:

```dart
class A {}

mixin class B extends [!A!] {}
```

#### Common fixes

If you want the class to be used as a mixin, then change the superclass to
`Object`, either explicitly or by removing the extends clause:

```dart
class A {}

mixin class B {}
```

If the class needs to have a superclass other than `Object`, then remove
the `mixin` modifier:

```dart
class A {}

class B extends A {}
```

If you need both a mixin and a subclass of a class other than `Object`,
then move the members of the subclass to a new mixin, remove the `mixin`
modifier from the subclass, and apply the new mixin to the subclass:

```dart
class A {}

class B extends A with M {}

mixin M {}
```

Depending on the members of the subclass this might require adding an `on`
clause to the mixin.

### mixin_class_declares_constructor

_The class '{0}' can't be used as a mixin because it declares a constructor._

#### Description

The analyzer produces this diagnostic when a class is used as a mixin and
the mixed-in class defines a constructor.

#### Example

The following code produces this diagnostic because the class `A`, which
defines a constructor, is being used as a mixin:

```dart
//@dart=2.19
class A {
  A();
}

class B with [!A!] {}
```

#### Common fixes

If it's possible to convert the class to a mixin, then do so:

```dart
mixin A {
}

class B with A {}
```

If the class can't be a mixin and it's possible to remove the constructor,
then do so:

```dart
//@dart=2.19
class A {
}

class B with A {}
```

If the class can't be a mixin and you can't remove the constructor, then
try extending or implementing the class rather than mixing it in:

```dart
class A {
  A();
}

class B extends A {}
```

### mixin_inherits_from_not_object

_The class '{0}' can't be used as a mixin because it extends a class other than
'Object'._

#### Description

The analyzer produces this diagnostic when a class that extends a class
other than `Object` is used as a mixin.

#### Example

The following code produces this diagnostic because the class `B`, which
extends `A`, is being used as a mixin by `C`:

```dart
//@dart=2.19
class A {}

class B extends A {}

class C with [!B!] {}
```

#### Common fixes

If the class being used as a mixin can be changed to extend `Object`, then
change it:

```dart
//@dart=2.19
class A {}

class B {}

class C with B {}
```

If the class being used as a mixin can't be changed and the class that's
using it extends `Object`, then extend the class being used as a mixin:

```dart
class A {}

class B extends A {}

class C extends B {}
```

If the class doesn't extend `Object` or if you want to be able to mix in
the behavior from `B` in other places, then create a real mixin:

```dart
class A {}

mixin M on A {}

class B extends A with M {}

class C extends A with M {}
```

### mixin_instantiate

_Mixins can't be instantiated._

#### Description

The analyzer produces this diagnostic when a mixin is instantiated.

#### Example

The following code produces this diagnostic because the mixin `M` is being
instantiated:

```dart
mixin M {}

var m = [!M!]();
```

#### Common fixes

If you intend to use an instance of a class, then use the name of that
class in place of the name of the mixin.

### mixin_of_non_class

_Classes can only mix in mixins and classes._

#### Description

The analyzer produces this diagnostic when a name in a `with` clause is
defined to be something other than a mixin or a class.

#### Example

The following code produces this diagnostic because `F` is defined to be a
function type:

```dart
typedef F = int Function(String);

class C with [!F!] {}
```

#### Common fixes

Remove the invalid name from the list, possibly replacing it with the name
of the intended mixin or class:

```dart
typedef F = int Function(String);

class C {}
```

### mixin_on_sealed_class

_The class '{0}' shouldn't be used as a mixin constraint because it is sealed,
and any class mixing in this mixin must have '{0}' as a superclass._

#### Description

The analyzer produces this diagnostic when the superclass constraint of a
mixin is a class from a different package that was marked as
[`sealed`][meta-sealed]. Classes that are sealed can't be extended,
implemented, mixed in, or used as a superclass constraint.

#### Example

If the package `p` defines a sealed class:

```dart
import 'package:meta/meta.dart';

@sealed
class C {}
```

Then, the following code, when in a package other than `p`, produces this
diagnostic:

```dart
import 'package:p/p.dart';

[!mixin M on C {}!]
```

#### Common fixes

If the classes that use the mixin don't need to be subclasses of the sealed
class, then consider adding a field and delegating to the wrapped instance
of the sealed class.

### mixin_super_class_constraint_deferred_class

_Deferred classes can't be used as superclass constraints._

#### Description

The analyzer produces this diagnostic when a superclass constraint of a
mixin is imported from a deferred library.

#### Example

The following code produces this diagnostic because the superclass
constraint of `math.Random` is imported from a deferred library:

```dart
import 'dart:async' deferred as async;

mixin M<T> on [!async.Stream<T>!] {}
```

#### Common fixes

If the import doesn't need to be deferred, then remove the `deferred`
keyword:

```dart
import 'dart:async' as async;

mixin M<T> on async.Stream<T> {}
```

If the import does need to be deferred, then remove the superclass
constraint:

```dart
mixin M<T> {}
```

### mixin_super_class_constraint_non_interface

_Only classes and mixins can be used as superclass constraints._

#### Description

The analyzer produces this diagnostic when a type following the `on`
keyword in a mixin declaration is neither a class nor a mixin.

#### Example

The following code produces this diagnostic because `F` is neither a class
nor a mixin:

```dart
typedef F = void Function();

mixin M on [!F!] {}
```

#### Common fixes

If the type was intended to be a class but was mistyped, then replace the
name.

Otherwise, remove the type from the `on` clause.

### multiple_redirecting_constructor_invocations

_Constructors can have only one 'this' redirection, at most._

#### Description

The analyzer produces this diagnostic when a constructor redirects to more
than one other constructor in the same class (using `this`).

#### Example

The following code produces this diagnostic because the unnamed
constructor in `C` is redirecting to both `this.a` and `this.b`:

```dart
class C {
  C() : this.a(), [!this.b()!];
  C.a();
  C.b();
}
```

#### Common fixes

Remove all but one of the redirections:

```dart
class C {
  C() : this.a();
  C.a();
  C.b();
}
```

### multiple_super_initializers

_A constructor can have at most one 'super' initializer._

#### Description

The analyzer produces this diagnostic when the initializer list of a
constructor contains more than one invocation of a constructor from the
superclass. The initializer list is required to have exactly one such call,
which can either be explicit or implicit.

#### Example

The following code produces this diagnostic because the initializer list
for `B`'s constructor invokes both the constructor `one` and the
constructor `two` from the superclass `A`:

```dart
class A {
  int? x;
  String? s;
  A.one(this.x);
  A.two(this.s);
}

class B extends A {
  B() : super.one(0), [!super.two('')!];
}
```

#### Common fixes

If one of the super constructors will initialize the instance fully, then
remove the other:

```dart
class A {
  int? x;
  String? s;
  A.one(this.x);
  A.two(this.s);
}

class B extends A {
  B() : super.one(0);
}
```

If the initialization achieved by one of the super constructors can be
performed in the body of the constructor, then remove its super invocation
and perform the initialization in the body:

```dart
class A {
  int? x;
  String? s;
  A.one(this.x);
  A.two(this.s);
}

class B extends A {
  B() : super.one(0) {
    s = '';
  }
}
```

If the initialization can only be performed in a constructor in the
superclass, then either add a new constructor or modify one of the existing
constructors so there's a constructor that allows all the required
initialization to occur in a single call:

```dart
class A {
  int? x;
  String? s;
  A.one(this.x);
  A.two(this.s);
  A.three(this.x, this.s);
}

class B extends A {
  B() : super.three(0, '');
}
```

### must_be_a_native_function_type

_The type '{0}' given to '{1}' must be a valid 'dart:ffi' native function type._

#### Description

The analyzer produces this diagnostic when an invocation of either
`Pointer.fromFunction`, `DynamicLibrary.lookupFunction`,
or a `NativeCallable` constructor, has a type
argument(whether explicit or inferred) that isn't a native function type.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the type `T` can be
any subclass of `Function` but the type argument for `fromFunction` is
required to be a native function type:

```dart
import 'dart:ffi';

int f(int i) => i * 2;

class C<T extends Function> {
  void g() {
    Pointer.fromFunction<[!T!]>(f, 0);
  }
}
```

#### Common fixes

Use a native function type as the type argument to the invocation:

```dart
import 'dart:ffi';

int f(int i) => i * 2;

class C<T extends Function> {
  void g() {
    Pointer.fromFunction<Int32 Function(Int32)>(f, 0);
  }
}
```

### must_be_a_subtype

_The type '{0}' must be a subtype of '{1}' for '{2}'._

#### Description

The analyzer produces this diagnostic in two cases:
- In an invocation of `Pointer.fromFunction`, or a
  `NativeCallable` constructor where the type argument
  (whether explicit or inferred) isn't a supertype of the type of the
  function passed as the first argument to the method.
- In an invocation of `DynamicLibrary.lookupFunction` where the first type
  argument isn't a supertype of the second type argument.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the type of the
function `f` (`String Function(int)`) isn't a subtype of the type
argument `T` (`Int8 Function(Int8)`):

```dart
import 'dart:ffi';

typedef T = Int8 Function(Int8);

double f(double i) => i;

void g() {
  Pointer.fromFunction<T>([!f!], 5.0);
}
```

#### Common fixes

If the function is correct, then change the type argument to match:

```dart
import 'dart:ffi';

typedef T = Float Function(Float);

double f(double i) => i;

void g() {
  Pointer.fromFunction<T>(f, 5.0);
}
```

If the type argument is correct, then change the function to match:

```dart
import 'dart:ffi';

typedef T = Int8 Function(Int8);

int f(int i) => i;

void g() {
  Pointer.fromFunction<T>(f, 5);
}
```

### must_be_immutable

_This class (or a class that this class inherits from) is marked as
'@immutable', but one or more of its instance fields aren't final: {0}_

#### Description

The analyzer produces this diagnostic when an immutable class defines one
or more instance fields that aren't final. A class is immutable if it's
marked as being immutable using the annotation
[`immutable`][meta-immutable] or if it's a subclass of an immutable class.

#### Example

The following code produces this diagnostic because the field `x` isn't
final:

```dart
import 'package:meta/meta.dart';

@immutable
class [!C!] {
  int x;

  C(this.x);
}
```

#### Common fixes

If instances of the class should be immutable, then add the keyword `final`
to all non-final field declarations:

```dart
import 'package:meta/meta.dart';

@immutable
class C {
  final int x;

  C(this.x);
}
```

If the instances of the class should be mutable, then remove the
annotation, or choose a different superclass if the annotation is
inherited:

```dart
class C {
  int x;

  C(this.x);
}
```

### must_call_super

_This method overrides a method annotated as '@mustCallSuper' in '{0}', but
doesn't invoke the overridden method._

#### Description

The analyzer produces this diagnostic when a method that overrides a method
that is annotated as [`mustCallSuper`][meta-mustCallSuper] doesn't invoke
the overridden method as required.

#### Example

The following code produces this diagnostic because the method `m` in `B`
doesn't invoke the overridden method `m` in `A`:

```dart
import 'package:meta/meta.dart';

class A {
  @mustCallSuper
  m() {}
}

class B extends A {
  @override
  [!m!]() {}
}
```

#### Common fixes

Add an invocation of the overridden method in the overriding method:

```dart
import 'package:meta/meta.dart';

class A {
  @mustCallSuper
  m() {}
}

class B extends A {
  @override
  m() {
    super.m();
  }
}
```

### must_return_void

_The return type of the function passed to 'NativeCallable.listener' must be
'void' rather than '{0}'._

#### Description

The analyzer produces this diagnostic when you pass a function
that doesn't return `void` to the `NativeCallable.listener` constructor.

`NativeCallable.listener` creates a native callable that can be invoked
from any thread. The native code that invokes the callable sends a message
back to the isolate that created the callable, and doesn't wait for a
response. So it isn't possible to return a result from the callable.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the function
`f` returns `int` rather than `void`.

```dart
import 'dart:ffi';

int f(int i) => i * 2;

void g() {
  NativeCallable<Int32 Function(Int32)>.listener([!f!]);
}
```

#### Common fixes

Change the return type of the function to `void`.

```dart
import 'dart:ffi';

void f(int i) => print(i * 2);

void g() {
  NativeCallable<Void Function(Int32)>.listener(f);
}
```

### name_not_string

_The value of the 'name' field is required to be a string._

#### Description

The analyzer produces this diagnostic when the top-level `name` key has a
value that isn't a string.

#### Example

The following code produces this diagnostic because the value following the
`name` key is a list:

```yaml
name:
  [!- example!]
```

#### Common fixes

Replace the value with a string:

```yaml
name: example
```

### native_field_invalid_type

_'{0}' is an unsupported type for native fields. Native fields only support
pointers, arrays or numeric and compound types._

#### Description

The analyzer produces this diagnostic when an `@Native`-annotated field
has a type not supported for native fields.

Native fields support pointers, arrays, numeric types and subtypes of
`Compound` (i.e., structs or unions). Other subtypes of `NativeType`,
such as `Handle` or `NativeFunction` are not allowed as native fields.

Native functions should be used with external functions instead of
external fields.

Handles are unsupported because there is no way to transparently load and
store Dart objects into pointers.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `free` uses
an unsupported native type, `NativeFunction`:

```dart
import 'dart:ffi';

@Native<NativeFunction<Void Function()>>()
external void Function() [!free!];
```

#### Common fixes

If you meant to bind to an existing native function with a
`NativeFunction` field, use `@Native` methods instead:

```dart
import 'dart:ffi';

@Native<Void Function(Pointer<Void>)>()
external void free(Pointer<Void> ptr);
```

To bind to a field storing a function pointer in C, use a pointer type
for the Dart field:

```dart
import 'dart:ffi';

@Native()
external Pointer<NativeFunction<Void Function(Pointer<Void>)>> free;
```

### native_field_missing_type

_The native type of this field could not be inferred and must be specified in
the annotation._

#### Description

The analyzer produces this diagnostic when an `@Native`-annotated field
requires a type hint on the annotation to infer the native type.

Dart types like `int` and `double` have multiple possible native
representations. Since the native type needs to be known at compile time
to generate the correct load and stores when accessing the field, an
explicit type must be given.

#### Example

The following code produces this diagnostic because the field `f` has
the type `int` (for which multiple native representations exist), but no
explicit type parameter on the `Native` annotation:

```dart
import 'dart:ffi';

@Native()
external int [!f!];
```

#### Common fixes

To fix this diagnostic, find out the correct native representation from
the native declaration of the field. Then, add the corresponding type to
the annotation. For instance, if `f` was declared as an `uint8_t` in C,
the Dart field should be declared as:

```dart
import 'dart:ffi';

@Native<Uint8>()
external int f;
```

For more information about FFI, see [C interop using dart:ffi][ffi].

### native_field_not_static

_Native fields must be static._

#### Description

The analyzer produces this diagnostic when an instance field in a class
has been annotated with `@Native`.
Native fields refer to global variables in C, C++ or other native
languages, whereas instance fields in Dart are specific to an instance of
that class. Hence, native fields must be static.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `f` in the
class `C` is `@Native`, but not `static`:

```dart
import 'dart:ffi';

class C {
  @Native<Int>()
  external int [!f!];
}
```

#### Common fixes

Either make the field static:

```dart
import 'dart:ffi';

class C {
  @Native<Int>()
  external static int f;
}
```

Or move it out of a class, in which case no explicit `static` modifier is
required:

```dart
import 'dart:ffi';

class C {
}

@Native<Int>()
external int f;
```

If you meant to annotate an instance field that should be part of a
struct, omit the `@Native` annotation:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int()
  external int f;
}
```

### new_with_undefined_constructor_default

_The class '{0}' doesn't have an unnamed constructor._

#### Description

The analyzer produces this diagnostic when an unnamed constructor is
invoked on a class that defines named constructors but the class doesn't
have an unnamed constructor.

#### Example

The following code produces this diagnostic because `A` doesn't define an
unnamed constructor:

```dart
class A {
  A.a();
}

A f() => [!A!]();
```

#### Common fixes

If one of the named constructors does what you need, then use it:

```dart
class A {
  A.a();
}

A f() => A.a();
```

If none of the named constructors does what you need, and you're able to
add an unnamed constructor, then add the constructor:

```dart
class A {
  A();
  A.a();
}

A f() => A();
```

### non_abstract_class_inherits_abstract_member

_Missing concrete implementation of '{0}'._

_Missing concrete implementations of '{0}' and '{1}'._

_Missing concrete implementations of '{0}', '{1}', '{2}', '{3}', and {4} more._

_Missing concrete implementations of '{0}', '{1}', '{2}', and '{3}'._

_Missing concrete implementations of '{0}', '{1}', and '{2}'._

#### Description

The analyzer produces this diagnostic when a concrete class inherits one or
more abstract members, and doesn't provide or inherit an implementation for
at least one of those abstract members.

#### Example

The following code produces this diagnostic because the class `B` doesn't
have a concrete implementation of `m`:

```dart
abstract class A {
  void m();
}

class [!B!] extends A {}
```

#### Common fixes

If the subclass can provide a concrete implementation for some or all of
the abstract inherited members, then add the concrete implementations:

```dart
abstract class A {
  void m();
}

class B extends A {
  void m() {}
}
```

If there is a mixin that provides an implementation of the inherited
methods, then apply the mixin to the subclass:

```dart
abstract class A {
  void m();
}

class B extends A with M {}

mixin M {
  void m() {}
}
```

If the subclass can't provide a concrete implementation for all of the
abstract inherited members, then mark the subclass as being abstract:

```dart
abstract class A {
  void m();
}

abstract class B extends A {}
```

### non_bool_condition

_Conditions must have a static type of 'bool'._

#### Description

The analyzer produces this diagnostic when a condition, such as an `if` or
`while` loop, doesn't have the static type `bool`.

#### Example

The following code produces this diagnostic because `x` has the static type
`int`:

```dart
void f(int x) {
  if ([!x!]) {
    // ...
  }
}
```

#### Common fixes

Change the condition so that it produces a Boolean value:

```dart
void f(int x) {
  if (x == 0) {
    // ...
  }
}
```

### non_bool_expression

_The expression in an assert must be of type 'bool'._

#### Description

The analyzer produces this diagnostic when the first expression in an
assert has a type other than `bool`.

#### Example

The following code produces this diagnostic because the type of `p` is
`int`, but a `bool` is required:

```dart
void f(int p) {
  assert([!p!]);
}
```

#### Common fixes

Change the expression so that it has the type `bool`:

```dart
void f(int p) {
  assert(p > 0);
}
```

### non_bool_negation_expression

_A negation operand must have a static type of 'bool'._

#### Description

The analyzer produces this diagnostic when the operand of the unary
negation operator (`!`) doesn't have the type `bool`.

#### Example

The following code produces this diagnostic because `x` is an `int` when it
must be a `bool`:

```dart
int x = 0;
bool y = ![!x!];
```

#### Common fixes

Replace the operand with an expression that has the type `bool`:

```dart
int x = 0;
bool y = !(x > 0);
```

### non_bool_operand

_The operands of the operator '{0}' must be assignable to 'bool'._

#### Description

The analyzer produces this diagnostic when one of the operands of either
the `&&` or `||` operator doesn't have the type `bool`.

#### Example

The following code produces this diagnostic because `a` isn't a Boolean
value:

```dart
int a = 3;
bool b = [!a!] || a > 1;
```

#### Common fixes

Change the operand to a Boolean value:

```dart
int a = 3;
bool b = a == 0 || a > 1;
```

### non_constant_annotation_constructor

_Annotation creation can only call a const constructor._

#### Description

The analyzer produces this diagnostic when an annotation is the invocation
of an existing constructor even though the invoked constructor isn't a
const constructor.

#### Example

The following code produces this diagnostic because the constructor for `C`
isn't a const constructor:

```dart
[!@C()!]
void f() {
}

class C {
  C();
}
```

#### Common fixes

If it's valid for the class to have a const constructor, then create a
const constructor that can be used for the annotation:

```dart
@C()
void f() {
}

class C {
  const C();
}
```

If it isn't valid for the class to have a const constructor, then either
remove the annotation or use a different class for the annotation.

### non_constant_case_expression

_Case expressions must be constant._

#### Description

The analyzer produces this diagnostic when the expression in a `case`
clause isn't a constant expression.

#### Example

The following code produces this diagnostic because `j` isn't a constant:

```dart
void f(int i, int j) {
  switch (i) {
    case [!j!]:
      // ...
      break;
  }
}
```

#### Common fixes

Either make the expression a constant expression, or rewrite the `switch`
statement as a sequence of `if` statements:

```dart
void f(int i, int j) {
  if (i == j) {
    // ...
  }
}
```

### non_constant_case_expression_from_deferred_library

_Constant values from a deferred library can't be used as a case expression._

#### Description

The analyzer produces this diagnostic when the expression in a case clause
references a constant from a library that is imported using a deferred
import. In order for switch statements to be compiled efficiently, the
constants referenced in case clauses need to be available at compile time,
and constants from deferred libraries aren't available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

Given a file `a.dart` that defines the constant `zero`:

```dart
const zero = 0;
```

The following code produces this diagnostic because the library `a.dart` is
imported using a `deferred` import, and the constant `a.zero`, declared in
the imported library, is used in a case clause:

```dart
import 'a.dart' deferred as a;

void f(int x) {
  switch (x) {
    case a.[!zero!]:
      // ...
      break;
  }
}
```

#### Common fixes

If you need to reference the constant from the imported library, then
remove the `deferred` keyword:

```dart
import 'a.dart' as a;

void f(int x) {
  switch (x) {
    case a.zero:
      // ...
      break;
  }
}
```

If you need to reference the constant from the imported library and also
need the imported library to be deferred, then rewrite the switch statement
as a sequence of `if` statements:

```dart
import 'a.dart' deferred as a;

void f(int x) {
  if (x == a.zero) {
    // ...
  }
}
```

If you don't need to reference the constant, then replace the case
expression:

```dart
void f(int x) {
  switch (x) {
    case 0:
      // ...
      break;
  }
}
```

### non_constant_default_value

_The default value of an optional parameter must be constant._

#### Description

The analyzer produces this diagnostic when an optional parameter, either
named or positional, has a default value that isn't a compile-time
constant.

#### Example

The following code produces this diagnostic:

```dart
var defaultValue = 3;

void f([int value = [!defaultValue!]]) {}
```

#### Common fixes

If the default value can be converted to be a constant, then convert it:

```dart
const defaultValue = 3;

void f([int value = defaultValue]) {}
```

If the default value needs to change over time, then apply the default
value inside the function:

```dart
var defaultValue = 3;

void f([int? value]) {
  value ??= defaultValue;
}
```

### non_constant_default_value_from_deferred_library

_Constant values from a deferred library can't be used as a default parameter
value._

#### Description

The analyzer produces this diagnostic when the default value of an optional
parameter uses a constant from a library imported using a deferred import.
Default values need to be available at compile time, and constants from
deferred libraries aren't available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

Given a file `a.dart` that defines the constant `zero`:

```dart
const zero = 0;
```

The following code produces this diagnostic because `zero` is declared in a
library imported using a deferred import:

```dart
import 'a.dart' deferred as a;

void f({int x = a.[!zero!]}) {}
```

#### Common fixes

If you need to reference the constant from the imported library, then
remove the `deferred` keyword:

```dart
import 'a.dart' as a;

void f({int x = a.zero}) {}
```

If you don't need to reference the constant, then replace the default
value:

```dart
void f({int x = 0}) {}
```

### non_constant_list_element

_The values in a const list literal must be constants._

#### Description

The analyzer produces this diagnostic when an element in a constant list
literal isn't a constant value. The list literal can be constant either
explicitly (because it's prefixed by the `const` keyword) or implicitly
(because it appears in a [constant context][]).

#### Example

The following code produces this diagnostic because `x` isn't a constant,
even though it appears in an implicitly constant list literal:

```dart
var x = 2;
var y = const <int>[0, 1, [!x!]];
```

#### Common fixes

If the list needs to be a constant list, then convert the element to be a
constant. In the example above, you might add the `const` keyword to the
declaration of `x`:

```dart
const x = 2;
var y = const <int>[0, 1, x];
```

If the expression can't be made a constant, then the list can't be a
constant either, so you must change the code so that the list isn't a
constant. In the example above this means removing the `const` keyword
before the list literal:

```dart
var x = 2;
var y = <int>[0, 1, x];
```

### non_constant_map_element

_The elements in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when an `if` element or a spread
element in a constant map isn't a constant element.

#### Examples

The following code produces this diagnostic because it's attempting to
spread a non-constant map:

```dart
var notConst = <int, int>{};
var map = const <int, int>{...[!notConst!]};
```

Similarly, the following code produces this diagnostic because the
condition in the `if` element isn't a constant expression:

```dart
bool notConst = true;
var map = const <int, int>{if ([!notConst!]) 1 : 2};
```

#### Common fixes

If the map needs to be a constant map, then make the elements constants.
In the spread example, you might do that by making the collection being
spread a constant:

```dart
const notConst = <int, int>{};
var map = const <int, int>{...notConst};
```

If the map doesn't need to be a constant map, then remove the `const`
keyword:

```dart
bool notConst = true;
var map = <int, int>{if (notConst) 1 : 2};
```

### non_constant_map_key

_The keys in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when a key in a constant map literal
isn't a constant value.

#### Example

The following code produces this diagnostic because `a` isn't a constant:

```dart
var a = 'a';
var m = const {[!a!]: 0};
```

#### Common fixes

If the map needs to be a constant map, then make the key a constant:

```dart
const a = 'a';
var m = const {a: 0};
```

If the map doesn't need to be a constant map, then remove the `const`
keyword:

```dart
var a = 'a';
var m = {a: 0};
```

### non_constant_map_pattern_key

_Key expressions in map patterns must be constants._

#### Description

The analyzer produces this diagnostic when a key in a map pattern isn't a
constant expression.

#### Example

The following code produces this diagnostic because the key `A()` isn't a
constant:

```dart
void f(Object x) {
  if (x case {[!A()!]: 0}) {}
}

class A {
  const A();
}
```

#### Common fixes

Use a constant for the key:

```dart
void f(Object x) {
  if (x case {const A(): 0}) {}
}

class A {
  const A();
}
```

### non_constant_map_value

_The values in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when a value in a constant map
literal isn't a constant value.

#### Example

The following code produces this diagnostic because `a` isn't a constant:

```dart
var a = 'a';
var m = const {0: [!a!]};
```

#### Common fixes

If the map needs to be a constant map, then make the key a constant:

```dart
const a = 'a';
var m = const {0: a};
```

If the map doesn't need to be a constant map, then remove the `const`
keyword:

```dart
var a = 'a';
var m = {0: a};
```

### non_constant_relational_pattern_expression

_The relational pattern expression must be a constant._

#### Description

The analyzer produces this diagnostic when the value in a relational
pattern expression isn't a constant expression.

#### Example

The following code produces this diagnostic because the operand of the `>`
operator, `a`, isn't a constant:

```dart
final a = 0;

void f(int x) {
  if (x case > [!a!]) {}
}
```

#### Common fixes

Replace the value with a constant expression:

```dart
const a = 0;

void f(int x) {
  if (x case > a) {}
}
```

### non_constant_set_element

_The values in a const set literal must be constants._

#### Description

The analyzer produces this diagnostic when a constant set literal contains
an element that isn't a compile-time constant.

#### Example

The following code produces this diagnostic because `i` isn't a constant:

```dart
var i = 0;

var s = const {[!i!]};
```

#### Common fixes

If the element can be changed to be a constant, then change it:

```dart
const i = 0;

var s = const {i};
```

If the element can't be a constant, then remove the keyword `const`:

```dart
var i = 0;

var s = {i};
```

### non_constant_type_argument

_The type arguments to '{0}' must be known at compile time, so they can't be
type parameters._

#### Description

The analyzer produces this diagnostic when the type arguments to a method
are required to be known at compile time, but a type parameter, whose
value can't be known at compile time, is used as a type argument.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the type argument to
`Pointer.asFunction` must be known at compile time, but the type parameter
`R`, which isn't known at compile time, is being used as the type
argument:

```dart
import 'dart:ffi';

typedef T = int Function(int);

class C<R extends T> {
  void m(Pointer<NativeFunction<T>> p) {
    p.asFunction<[!R!]>();
  }
}
```

#### Common fixes

Remove any uses of type parameters:

```dart
import 'dart:ffi';

class C {
  void m(Pointer<NativeFunction<Int64 Function(Int64)>> p) {
    p.asFunction<int Function(int)>();
  }
}
```

### non_const_argument_for_const_parameter

_Argument '{0}' must be a constant._

#### Description

The analyzer produces this diagnostic when a parameter is
annotated with the [`mustBeConst`][meta-mustBeConst] annotation and
the corresponding argument is not a constant expression.

#### Example

The following code produces this diagnostic on the invocation of
the function `f` because the value of the argument passed to the
function `g` isn't a constant:

```dart
import 'package:meta/meta.dart' show mustBeConst;

int f(int value) => g([!value!]);

int g(@mustBeConst int value) => value + 1;
```

#### Common fixes

If a suitable constant is available to use, then replace the argument
with a constant:

```dart
import 'package:meta/meta.dart' show mustBeConst;

const v = 3;

int f() => g(v);

int g(@mustBeConst int value) => value + 1;
```

### non_const_call_to_literal_constructor

_This instance creation must be 'const', because the {0} constructor is marked
as '@literal'._

#### Description

The analyzer produces this diagnostic when a constructor that has the
[`literal`][meta-literal] annotation is invoked without using the `const`
keyword, but all of the arguments to the constructor are constants. The
annotation indicates that the constructor should be used to create a
constant value whenever possible.

#### Example

The following code produces this diagnostic:

```dart
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}

C f() => [!C()!];
```

#### Common fixes

Add the keyword `const` before the constructor invocation:

```dart
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}

void f() => const C();
```

### non_const_generative_enum_constructor

_Generative enum constructors must be 'const'._

#### Description

The analyzer produces this diagnostic when an enum declaration contains a
generative constructor that isn't marked as `const`.

#### Example

The following code produces this diagnostic because the constructor in `E`
isn't marked as being `const`:

```dart
enum E {
  e;

  [!E!]();
}
```

#### Common fixes

Add the `const` keyword before the constructor:

```dart
enum E {
  e;

  const E();
}
```

### non_covariant_type_parameter_position_in_representation_type

_An extension type parameter can't be used in a non-covariant position of its
representation type._

#### Description

The analyzer produces this diagnostic when a type parameter of an
extension type is used in a non-covariant position in the representation
type of that extension type.

#### Example

The following code produces this diagnostic because the type parameter `T`
is used as a parameter type in the function type `void Function(T)`, and
parameters are not covariant:

```dart
extension type A<[!T!]>(void Function(T) f) {}
```

#### Common fixes

Remove the use of the type parameter:

```dart
extension type A(void Function(String) f) {}
```

### non_exhaustive_switch_expression

_The type '{0}' is not exhaustively matched by the switch cases since it doesn't
match '{1}'._

#### Description

The analyzer produces this diagnostic when a `switch` expression is
missing a case for one or more of the possible values that could flow
through it.

#### Example

The following code produces this diagnostic because the switch expression
doesn't have a case for the value `E.three`:

```dart
enum E { one, two, three }

String f(E e) => [!switch!] (e) {
    E.one => 'one',
    E.two => 'two',
  };
```

#### Common fixes

If the missing values are distinctly meaningful to the switch expression,
then add a case for each of the values missing a match:

```dart
enum E { one, two, three }

String f(E e) => switch (e) {
    E.one => 'one',
    E.two => 'two',
    E.three => 'three',
  };
```

If the missing values don't need to be matched, then add a wildcard
pattern that returns a simple default:

```dart
enum E { one, two, three }

String f(E e) => switch (e) {
    E.one => 'one',
    E.two => 'two',
    _ => 'unknown',
  };
```

Be aware that a wildcard pattern will handle any values added to the type
in the future. You will lose the ability to have the compiler warn you if
the `switch` needs to be updated to account for newly added types.

### non_exhaustive_switch_statement

_The type '{0}' is not exhaustively matched by the switch cases since it doesn't
match '{1}'._

#### Description

The analyzer produces this diagnostic when a `switch` statement switching
over an exhaustive type is missing a case for one or more of the possible
values that could flow through it.

#### Example

The following code produces this diagnostic because the switch statement
doesn't have a case for the value `E.three`, and `E` is an exhaustive
type:

```dart
enum E { one, two, three }

void f(E e) {
  [!switch!] (e) {
    case E.one:
    case E.two:
  }
}
```

#### Common fixes

Add a case for each of the constants that aren't currently being matched:

```dart
enum E { one, two, three }

void f(E e) {
  switch (e) {
    case E.one:
    case E.two:
      break;
    case E.three:
  }
}
```

If the missing values don't need to be matched, then add a `default`
clause or a wildcard pattern:

```dart
enum E { one, two, three }

void f(E e) {
  switch (e) {
    case E.one:
    case E.two:
      break;
    default:
  }
}
```

But be aware that adding a `default` clause or wildcard pattern will cause
any future values of the exhaustive type to also be handled, so you will
have lost the ability for the compiler to warn you if the `switch` needs
to be updated.

### non_final_field_in_enum

_Enums can only declare final fields._

#### Description

The analyzer produces this diagnostic when an instance field in an enum
isn't marked as `final`.

#### Example

The following code produces this diagnostic because the field `f` isn't a
final field:

```dart
enum E {
  c;

  int [!f!] = 0;
}
```

#### Common fixes

If the field must be defined for the enum, then mark the field as being
`final`:

```dart
enum E {
  c;

  final int f = 0;
}
```

If the field can be removed, then remove it:

```dart
enum E {
  c
}
```

### non_generative_constructor

_The generative constructor '{0}' is expected, but a factory was found._

#### Description

The analyzer produces this diagnostic when the initializer list of a
constructor invokes a constructor from the superclass, and the invoked
constructor is a factory constructor. Only a generative constructor can be
invoked in the initializer list.

#### Example

The following code produces this diagnostic because the invocation of the
constructor `super.one()` is invoking a factory constructor:

```dart
class A {
  factory A.one() = B;
  A.two();
}

class B extends A {
  B() : [!super.one()!];
}
```

#### Common fixes

Change the super invocation to invoke a generative constructor:

```dart
class A {
  factory A.one() = B;
  A.two();
}

class B extends A {
  B() : super.two();
}
```

If the generative constructor is the unnamed constructor, and if there are
no arguments being passed to it, then you can remove the super invocation.

### non_generative_implicit_constructor

_The unnamed constructor of superclass '{0}' (called by the default constructor
of '{1}') must be a generative constructor, but factory found._

#### Description

The analyzer produces this diagnostic when a class has an implicit
generative constructor and the superclass has an explicit unnamed factory
constructor. The implicit constructor in the subclass implicitly invokes
the unnamed constructor in the superclass, but generative constructors can
only invoke another generative constructor, not a factory constructor.

#### Example

The following code produces this diagnostic because the implicit
constructor in `B` invokes the unnamed constructor in `A`, but the
constructor in `A` is a factory constructor, when a generative constructor
is required:

```dart
class A {
  factory A() => throw 0;
  A.named();
}

class [!B!] extends A {}
```

#### Common fixes

If the unnamed constructor in the superclass can be a generative
constructor, then change it to be a generative constructor:

```dart
class A {
  A();
  A.named();
}

class B extends A { }
```

If the unnamed constructor can't be a generative constructor and there are
other generative constructors in the superclass, then explicitly invoke
one of them:

```dart
class A {
  factory A() => throw 0;
  A.named();
}

class B extends A {
  B() : super.named();
}
```

If there are no generative constructors that can be used and none can be
added, then implement the superclass rather than extending it:

```dart
class A {
  factory A() => throw 0;
  A.named();
}

class B implements A {}
```

### non_native_function_type_argument_to_pointer

_Can't invoke 'asFunction' because the function signature '{0}' for the pointer
isn't a valid C function signature._

#### Description

The analyzer produces this diagnostic when the method `asFunction` is
invoked on a pointer to a native function, but the signature of the native
function isn't a valid C function signature.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because function signature
associated with the pointer `p` (`FNative`) isn't a valid C function
signature:

```dart
import 'dart:ffi';

typedef FNative = int Function(int);
typedef F = int Function(int);

class C {
  void f(Pointer<NativeFunction<FNative>> p) {
    p.asFunction<[!F!]>();
  }
}
```

#### Common fixes

Make the `NativeFunction` signature a valid C signature:

```dart
import 'dart:ffi';

typedef FNative = Int8 Function(Int8);
typedef F = int Function(int);

class C {
  void f(Pointer<NativeFunction<FNative>> p) {
    p.asFunction<F>();
  }
}
```

### non_positive_array_dimension

_Array dimensions must be positive numbers._

#### Description

The analyzer produces this diagnostic when a dimension given in an `Array`
annotation is less than or equal to zero (`0`).

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because an array dimension of
`-1` was provided:

```dart
import 'dart:ffi';

final class MyStruct extends Struct {
  @Array([!-8!])
  external Array<Uint8> a0;
}
```

#### Common fixes

Change the dimension to be a positive integer:

```dart
import 'dart:ffi';

final class MyStruct extends Struct {
  @Array(8)
  external Array<Uint8> a0;
}
```

### non_sized_type_argument

_The type '{1}' isn't a valid type argument for '{0}'. The type argument must be
a native integer, 'Float', 'Double', 'Pointer', or subtype of 'Struct', 'Union', or 'AbiSpecificInteger'._

#### Description

The analyzer produces this diagnostic when the type argument for the class
`Array` isn't one of the valid types: either a native integer, `Float`,
`Double`, `Pointer`, or subtype of `Struct`, `Union`, or
`AbiSpecificInteger`.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the type argument to
`Array` is `Void`, and `Void` isn't one of the valid types:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Array(8)
  external Array<[!Void!]> a0;
}
```

#### Common fixes

Change the type argument to one of the valid types:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Array(8)
  external Array<Uint8> a0;
}
```

### non_sync_factory

_Factory bodies can't use 'async', 'async*', or 'sync*'._

#### Description

The analyzer produces this diagnostic when the body of a factory
constructor is marked with `async`, `async*`, or `sync*`. All constructors,
including factory constructors, are required to return an instance of the
class in which they're declared, not a `Future`, `Stream`, or `Iterator`.

#### Example

The following code produces this diagnostic because the body of the factory
constructor is marked with `async`:

```dart
class C {
  factory C() [!async!] {
    return C._();
  }
  C._();
}
```

#### Common fixes

If the member must be declared as a factory constructor, then remove the
keyword appearing before the body:

```dart
class C {
  factory C() {
    return C._();
  }
  C._();
}
```

If the member must return something other than an instance of the enclosing
class, then make the member a static method:

```dart
class C {
  static Future<C> m() async {
    return C._();
  }
  C._();
}
```

### non_type_as_type_argument

_The name '{0}' isn't a type, so it can't be used as a type argument._

#### Description

The analyzer produces this diagnostic when an identifier that isn't a type
is used as a type argument.

#### Example

The following code produces this diagnostic because `x` is a variable, not
a type:

```dart
var x = 0;
List<[!x!]> xList = [];
```

#### Common fixes

Change the type argument to be a type:

```dart
var x = 0;
List<int> xList = [];
```

### non_type_in_catch_clause

_The name '{0}' isn't a type and can't be used in an on-catch clause._

#### Description

The analyzer produces this diagnostic when the identifier following the
`on` in a `catch` clause is defined to be something other than a type.

#### Example

The following code produces this diagnostic because `f` is a function, not
a type:

```dart
void f() {
  try {
    // ...
  } on [!f!] {
    // ...
  }
}
```

#### Common fixes

Change the name to the type of object that should be caught:

```dart
void f() {
  try {
    // ...
  } on FormatException {
    // ...
  }
}
```

### non_void_return_for_operator

_The return type of the operator []= must be 'void'._

#### Description

The analyzer produces this diagnostic when a declaration of the operator
`[]=` has a return type other than `void`.

#### Example

The following code produces this diagnostic because the declaration of the
operator `[]=` has a return type of `int`:

```dart
class C {
  [!int!] operator []=(int index, int value) => 0;
}
```

#### Common fixes

Change the return type to `void`:

```dart
class C {
  void operator []=(int index, int value) => 0;
}
```

### non_void_return_for_setter

_The return type of the setter must be 'void' or absent._

#### Description

The analyzer produces this diagnostic when a setter is defined with a
return type other than `void`.

#### Example

The following code produces this diagnostic because the setter `p` has a
return type of `int`:

```dart
class C {
  [!int!] set p(int i) => 0;
}
```

#### Common fixes

Change the return type to `void` or omit the return type:

```dart
class C {
  set p(int i) => 0;
}
```

### not_assigned_potentially_non_nullable_local_variable

_The non-nullable local variable '{0}' must be assigned before it can be used._

#### Description

The analyzer produces this diagnostic when a local variable is referenced
and has all these characteristics:
- Has a type that's [potentially non-nullable][].
- Doesn't have an initializer.
- Isn't marked as `late`.
- The analyzer can't prove that the local variable will be assigned before
  the reference based on the specification of [definite assignment][].

#### Examples

The following code produces this diagnostic because `x` can't have a value
of `null`, but is referenced before a value was assigned to it:

```dart
String f() {
  int x;
  return [!x!].toString();
}
```

The following code produces this diagnostic because the assignment to `x`
might not be executed, so it might have a value of `null`:

```dart
int g(bool b) {
  int x;
  if (b) {
    x = 1;
  }
  return [!x!] * 2;
}
```

The following code produces this diagnostic because the analyzer can't
prove, based on definite assignment analysis, that `x` won't be referenced
without having a value assigned to it:

```dart
int h(bool b) {
  int x;
  if (b) {
    x = 1;
  }
  if (b) {
    return [!x!] * 2;
  }
  return 0;
}
```

#### Common fixes

If `null` is a valid value, then make the variable nullable:

```dart
String f() {
  int? x;
  return x!.toString();
}
```

If `null` isn't a valid value, and there's a reasonable default value, then
add an initializer:

```dart
int g(bool b) {
  int x = 2;
  if (b) {
    x = 1;
  }
  return x * 2;
}
```

Otherwise, ensure that a value was assigned on every possible code path
before the value is accessed:

```dart
int g(bool b) {
  int x;
  if (b) {
    x = 1;
  } else {
    x = 2;
  }
  return x * 2;
}
```

You can also mark the variable as `late`, which removes the diagnostic, but
if the variable isn't assigned a value before it's accessed, then it
results in an exception being thrown at runtime. This approach should only
be used if you're sure that the variable will always be assigned, even
though the analyzer can't prove it based on definite assignment analysis.

```dart
int h(bool b) {
  late int x;
  if (b) {
    x = 1;
  }
  if (b) {
    return x * 2;
  }
  return 0;
}
```

### not_a_type

_{0} isn't a type._

#### Description

The analyzer produces this diagnostic when a name is used as a type but
declared to be something other than a type.

#### Example

The following code produces this diagnostic because `f` is a function:

```dart
f() {}
g([!f!] v) {}
```

#### Common fixes

Replace the name with the name of a type.

### not_binary_operator

_'{0}' isn't a binary operator._

#### Description

The analyzer produces this diagnostic when an operator that can only be
used as a unary operator is used as a binary operator.

#### Example

The following code produces this diagnostic because the operator `~` can
only be used as a unary operator:

```dart
var a = 5 [!~!] 3;
```

#### Common fixes

Replace the operator with the correct binary operator:

```dart
var a = 5 - 3;
```

### not_enough_positional_arguments

_1 positional argument expected by '{0}', but 0 found._

_1 positional argument expected, but 0 found._

_{0} positional arguments expected by '{2}', but {1} found._

_{0} positional arguments expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has fewer positional arguments than the number of required positional
parameters.

#### Example

The following code produces this diagnostic because `f` declares two
required parameters, but only one argument is provided:

```dart
void f(int a, int b) {}
void g() {
  f(0[!)!];
}
```

#### Common fixes

Add arguments corresponding to the remaining parameters:

```dart
void f(int a, int b) {}
void g() {
  f(0, 1);
}
```

### not_initialized_non_nullable_instance_field

_Non-nullable instance field '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a field is declared and has all
these characteristics:
- Has a type that's [potentially non-nullable][]
- Doesn't have an initializer
- Isn't marked as `late`

#### Examples

The following code produces this diagnostic because `x` is implicitly
initialized to `null` when it isn't allowed to be `null`:

```dart
class C {
  int [!x!];
}
```

Similarly, the following code produces this diagnostic because `x` is
implicitly initialized to `null`, when it isn't allowed to be `null`, by
one of the constructors, even though it's initialized by other
constructors:

```dart
class C {
  int x;

  C(this.x);

  [!C!].n();
}
```

#### Common fixes

If there's a reasonable default value for the field that's the same for all
instances, then add an initializer expression:

```dart
class C {
  int x = 0;
}
```

If the value of the field should be provided when an instance is created,
then add a constructor that sets the value of the field or update an
existing constructor:

```dart
class C {
  int x;

  C(this.x);
}
```

You can also mark the field as `late`, which removes the diagnostic, but if
the field isn't assigned a value before it's accessed, then it results in
an exception being thrown at runtime. This approach should only be used if
you're sure that the field will always be assigned before it's referenced.

```dart
class C {
  late int x;
}
```

### not_initialized_non_nullable_variable

_The non-nullable variable '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a static field or top-level
variable has a type that's non-nullable and doesn't have an initializer.
Fields and variables that don't have an initializer are normally
initialized to `null`, but the type of the field or variable doesn't allow
it to be set to `null`, so an explicit initializer must be provided.

#### Examples

The following code produces this diagnostic because the field `f` can't be
initialized to `null`:

```dart
class C {
  static int [!f!];
}
```

Similarly, the following code produces this diagnostic because the
top-level variable `v` can't be initialized to `null`:

```dart
int [!v!];
```

#### Common fixes

If the field or variable can't be initialized to `null`, then add an
initializer that sets it to a non-null value:

```dart
class C {
  static int f = 0;
}
```

If the field or variable should be initialized to `null`, then change the
type to be nullable:

```dart
int? v;
```

If the field or variable can't be initialized in the declaration but will
always be initialized before it's referenced, then mark it as being `late`:

```dart
class C {
  static late int f;
}
```

### not_iterable_spread

_Spread elements in list or set literals must implement 'Iterable'._

#### Description

The analyzer produces this diagnostic when the static type of the
expression of a spread element that appears in either a list literal or a
set literal doesn't implement the type `Iterable`.

#### Example

The following code produces this diagnostic:

```dart
var m = <String, int>{'a': 0, 'b': 1};
var s = <String>{...[!m!]};
```

#### Common fixes

The most common fix is to replace the expression with one that produces an
iterable object:

```dart
var m = <String, int>{'a': 0, 'b': 1};
var s = <String>{...m.keys};
```

### not_map_spread

_Spread elements in map literals must implement 'Map'._

#### Description

The analyzer produces this diagnostic when the static type of the
expression of a spread element that appears in a map literal doesn't
implement the type `Map`.

#### Example

The following code produces this diagnostic because `l` isn't a `Map`:

```dart
var l =  <String>['a', 'b'];
var m = <int, String>{...[!l!]};
```

#### Common fixes

The most common fix is to replace the expression with one that produces a
map:

```dart
var l =  <String>['a', 'b'];
var m = <int, String>{...l.asMap()};
```

### no_annotation_constructor_arguments

_Annotation creation must have arguments._

#### Description

The analyzer produces this diagnostic when an annotation consists of a
single identifier, but that identifier is the name of a class rather than a
variable. To create an instance of the class, the identifier must be
followed by an argument list.

#### Example

The following code produces this diagnostic because `C` is a class, and a
class can't be used as an annotation without invoking a `const` constructor
from the class:

```dart
class C {
  const C();
}

[!@C!]
var x;
```

#### Common fixes

Add the missing argument list:

```dart
class C {
  const C();
}

@C()
var x;
```

### no_combined_super_signature

_Can't infer missing types in '{0}' from overridden methods: {1}._

#### Description

The analyzer produces this diagnostic when there is a method declaration
for which one or more types needs to be inferred, and those types can't be
inferred because none of the overridden methods has a function type that is
a supertype of all the other overridden methods, as specified by
[override inference][].

#### Example

The following code produces this diagnostic because the method `m` declared
in the class `C` is missing both the return type and the type of the
parameter `a`, and neither of the missing types can be inferred for it:

```dart
abstract class A {
  A m(String a);
}

abstract class B {
  B m(int a);
}

abstract class C implements A, B {
  [!m!](a);
}
```

In this example, override inference can't be performed because the
overridden methods are incompatible in these ways:
- Neither parameter type (`String` and `int`) is a supertype of the other.
- Neither return type is a subtype of the other.

#### Common fixes

If possible, add types to the method in the subclass that are consistent
with the types from all the overridden methods:

```dart
abstract class A {
  A m(String a);
}

abstract class B {
  B m(int a);
}

abstract class C implements A, B {
  C m(Object a);
}
```

### no_generative_constructors_in_superclass

_The class '{0}' can't extend '{1}' because '{1}' only has factory constructors
(no generative constructors), and '{0}' has at least one generative constructor._

#### Description

The analyzer produces this diagnostic when a class that has at least one
generative constructor (whether explicit or implicit) has a superclass
that doesn't have any generative constructors. Every generative
constructor, except the one defined in `Object`, invokes, either
explicitly or implicitly, one of the generative constructors from its
superclass.

#### Example

The following code produces this diagnostic because the class `B` has an
implicit generative constructor that can't invoke a generative constructor
from `A` because `A` doesn't have any generative constructors:

```dart
class A {
  factory A.none() => throw '';
}

class B extends [!A!] {}
```

#### Common fixes

If the superclass should have a generative constructor, then add one:

```dart
class A {
  A();
  factory A.none() => throw '';
}

class B extends A {}
```

If the subclass shouldn't have a generative constructor, then remove it by
adding a factory constructor:

```dart
class A {
  factory A.none() => throw '';
}

class B extends A {
  factory B.none() => throw '';
}
```

If the subclass must have a generative constructor but the superclass
can't have one, then implement the superclass instead:

```dart
class A {
  factory A.none() => throw '';
}

class B implements A {}
```

### nullable_type_in_catch_clause

_A potentially nullable type can't be used in an 'on' clause because it isn't
valid to throw a nullable expression._

#### Description

The analyzer produces this diagnostic when the type following `on` in a
`catch` clause is a nullable type. It isn't valid to specify a nullable
type because it isn't possible to catch `null` (because it's a runtime
error to throw `null`).

#### Example

The following code produces this diagnostic because the exception type is
specified to allow `null` when `null` can't be thrown:

```dart
void f() {
  try {
    // ...
  } on [!FormatException?!] {
  }
}
```

#### Common fixes

Remove the question mark from the type:

```dart
void f() {
  try {
    // ...
  } on FormatException {
  }
}
```

### nullable_type_in_extends_clause

_A class can't extend a nullable type._

#### Description

The analyzer produces this diagnostic when a class declaration uses an
`extends` clause to specify a superclass, and the superclass is followed by
a `?`.

It isn't valid to specify a nullable superclass because doing so would have
no meaning; it wouldn't change either the interface or implementation being
inherited by the class containing the `extends` clause.

Note, however, that it _is_ valid to use a nullable type as a type argument
to the superclass, such as `class A extends B<C?> {}`.

#### Example

The following code produces this diagnostic because `A?` is a nullable
type, and nullable types can't be used in an `extends` clause:

```dart
class A {}
class B extends [!A?!] {}
```

#### Common fixes

Remove the question mark from the type:

```dart
class A {}
class B extends A {}
```

### nullable_type_in_implements_clause

_A class, mixin, or extension type can't implement a nullable type._

#### Description

The analyzer produces this diagnostic when a class, mixin, or
extension type declaration has an `implements` clause, and an
interface is followed by a `?`.

It isn't valid to specify a nullable interface because doing so would have
no meaning; it wouldn't change the interface being inherited by the class
containing the `implements` clause.

Note, however, that it _is_ valid to use a nullable type as a type argument
to the interface, such as `class A implements B<C?> {}`.


#### Example

The following code produces this diagnostic because `A?` is a nullable
type, and nullable types can't be used in an `implements` clause:

```dart
class A {}
class B implements [!A?!] {}
```

#### Common fixes

Remove the question mark from the type:

```dart
class A {}
class B implements A {}
```

### nullable_type_in_on_clause

_A mixin can't have a nullable type as a superclass constraint._

#### Description

The analyzer produces this diagnostic when a mixin declaration uses an `on`
clause to specify a superclass constraint, and the class that's specified
is followed by a `?`.

It isn't valid to specify a nullable superclass constraint because doing so
would have no meaning; it wouldn't change the interface being depended on
by the mixin containing the `on` clause.

Note, however, that it _is_ valid to use a nullable type as a type argument
to the superclass constraint, such as `mixin A on B<C?> {}`.


#### Example

The following code produces this diagnostic because `A?` is a nullable type
and nullable types can't be used in an `on` clause:

```dart
class C {}
mixin M on [!C?!] {}
```

#### Common fixes

Remove the question mark from the type:

```dart
class C {}
mixin M on C {}
```

### nullable_type_in_with_clause

_A class or mixin can't mix in a nullable type._

#### Description

The analyzer produces this diagnostic when a class or mixin declaration has
a `with` clause, and a mixin is followed by a `?`.

It isn't valid to specify a nullable mixin because doing so would have no
meaning; it wouldn't change either the interface or implementation being
inherited by the class containing the `with` clause.

Note, however, that it _is_ valid to use a nullable type as a type argument
to the mixin, such as `class A with B<C?> {}`.

#### Example

The following code produces this diagnostic because `A?` is a nullable
type, and nullable types can't be used in a `with` clause:

```dart
mixin M {}
class C with [!M?!] {}
```

#### Common fixes

Remove the question mark from the type:

```dart
mixin M {}
class C with M {}
```

### null_argument_to_non_null_type

_'{0}' shouldn't be called with a null argument for the non-nullable type
argument '{1}'._

#### Description

The analyzer produces this diagnostic when `null` is passed to either the
constructor `Future.value` or the method `Completer.complete` when the type
argument used to create the instance was non-nullable. Even though the type
system can't express this restriction, passing in a `null` results in a
runtime exception.

#### Example

The following code produces this diagnostic because `null` is being passed
to the constructor `Future.value` even though the type argument is the
non-nullable type `String`:

```dart
Future<String> f() {
  return Future.value([!null!]);
}
```

#### Common fixes

Pass in a non-null value:

```dart
Future<String> f() {
  return Future.value('');
}
```

### null_check_always_fails

_This null-check will always throw an exception because the expression will
always evaluate to 'null'._

#### Description

The analyzer produces this diagnostic when the null check operator (`!`)
is used on an expression whose value can only be `null`. In such a case
the operator always throws an exception, which likely isn't the intended
behavior.

#### Example

The following code produces this diagnostic because the function `g` will
always return `null`, which means that the null check in `f` will always
throw:

```dart
void f() {
  [!g()!!];
}

Null g() => null;
```

#### Common fixes

If you intend to always throw an exception, then replace the null check
with an explicit `throw` expression to make the intent more clear:

```dart
void f() {
  g();
  throw TypeError();
}

Null g() => null;
```

### obsolete_colon_for_default_value

_Using a colon as the separator before a default value is no longer supported._

#### Description

The analyzer produces this diagnostic when a colon (`:`) is used as the
separator before the default value of an optional named parameter.
While this syntax used to be allowed, it was removed in favor of
using an equal sign (`=`).

#### Example

The following code produces this diagnostic because a colon is being used
before the default value of the optional parameter `i`:

```dart
void f({int i [!:!] 0}) {}
```

#### Common fixes

Replace the colon with an equal sign:

```dart
void f({int i = 0}) {}
```

### on_repeated

_The type '{0}' can be included in the superclass constraints only once._

#### Description

The analyzer produces this diagnostic when the same type is listed in the
superclass constraints of a mixin multiple times.

#### Example

The following code produces this diagnostic because `A` is included twice
in the superclass constraints for `M`:

```dart
mixin M on A, [!A!] {
}

class A {}
class B {}
```

#### Common fixes

If a different type should be included in the superclass constraints, then
replace one of the occurrences with the other type:

```dart
mixin M on A, B {
}

class A {}
class B {}
```

If no other type was intended, then remove the repeated type name:

```dart
mixin M on A {
}

class A {}
class B {}
```

### optional_parameter_in_operator

_Optional parameters aren't allowed when defining an operator._

#### Description

The analyzer produces this diagnostic when one or more of the parameters in
an operator declaration are optional.

#### Example

The following code produces this diagnostic because the parameter `other`
is an optional parameter:

```dart
class C {
  C operator +([[!C? other!]]) => this;
}
```

#### Common fixes

Make all of the parameters be required parameters:

```dart
class C {
  C operator +(C other) => this;
}
```

### override_on_non_overriding_member

_The field doesn't override an inherited getter or setter._

_The getter doesn't override an inherited getter._

_The method doesn't override an inherited method._

_The setter doesn't override an inherited setter._

#### Description

The analyzer produces this diagnostic when a class member is annotated with
the `@override` annotation, but the member isn't declared in any of the
supertypes of the class.

#### Example

The following code produces this diagnostic because `m` isn't declared in
any of the supertypes of `C`:

```dart
class C {
  @override
  String [!m!]() => '';
}
```

#### Common fixes

If the member is intended to override a member with a different name, then
update the member to have the same name:

```dart
class C {
  @override
  String toString() => '';
}
```

If the member is intended to override a member that was removed from the
superclass, then consider removing the member from the subclass.

If the member can't be removed, then remove the annotation.

### packed_annotation

_Structs must have at most one 'Packed' annotation._

#### Description

The analyzer produces this diagnostic when a subclass of `Struct` has more
than one `Packed` annotation.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the class `C`, which
is a subclass of `Struct`, has two `Packed` annotations:

```dart
import 'dart:ffi';

@Packed(1)
[!@Packed(1)!]
final class C extends Struct {
  external Pointer<Uint8> notEmpty;
}
```

#### Common fixes

Remove all but one of the annotations:

```dart
import 'dart:ffi';

@Packed(1)
final class C extends Struct {
  external Pointer<Uint8> notEmpty;
}
```

### packed_annotation_alignment

_Only packing to 1, 2, 4, 8, and 16 bytes is supported._

#### Description

The analyzer produces this diagnostic when the argument to the `Packed`
annotation isn't one of the allowed values: 1, 2, 4, 8, or 16.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the argument to the
`Packed` annotation (`3`) isn't one of the allowed values:

```dart
import 'dart:ffi';

@Packed([!3!])
final class C extends Struct {
  external Pointer<Uint8> notEmpty;
}
```

#### Common fixes

Change the alignment to be one of the allowed values:

```dart
import 'dart:ffi';

@Packed(4)
final class C extends Struct {
  external Pointer<Uint8> notEmpty;
}
```

### part_of_different_library

_Expected this library to be part of '{0}', not '{1}'._

#### Description

The analyzer produces this diagnostic when a library attempts to include a
file as a part of itself when the other file is a part of a different
library.

#### Example

Given a file `part.dart` containing

```dart
part of 'library.dart';
```

The following code, in any file other than `library.dart`, produces this
diagnostic because it attempts to include `part.dart` as a part of itself
when `part.dart` is a part of a different library:

```dart
part [!'package:a/part.dart'!];
```

#### Common fixes

If the library should be using a different file as a part, then change the
URI in the part directive to be the URI of the other file.

If the [part file][] should be a part of this library, then update the URI
(or library name) in the part-of directive to be the URI (or name) of the
correct library.

### part_of_non_part

_The included part '{0}' must have a part-of directive._

#### Description

The analyzer produces this diagnostic when a part directive is found and
the referenced file doesn't have a part-of directive.

#### Example

Given a file `a.dart` containing:

```dart
class A {}
```

The following code produces this diagnostic because `a.dart` doesn't
contain a part-of directive:

```dart
part [!'a.dart'!];
```

#### Common fixes

If the referenced file is intended to be a part of another library, then
add a part-of directive to the file:

```dart
part of 'test.dart';

class A {}
```

If the referenced file is intended to be a library, then replace the part
directive with an import directive:

```dart
import 'a.dart';
```

### part_of_unnamed_library

_The library is unnamed. A URI is expected, not a library name '{0}', in the
part-of directive._

#### Description

The analyzer produces this diagnostic when a library that doesn't have a
`library` directive (and hence has no name) contains a `part` directive
and the `part of` directive in the [part file][] uses a name to specify
the library that it's a part of.

#### Example

Given a [part file][] named `part_file.dart` containing the following
code:

```dart
part of lib;
```

The following code produces this diagnostic because the library including
the [part file][] doesn't have a name even though the [part file][] uses a
name to specify which library it's a part of:

```dart
part [!'part_file.dart'!];
```

#### Common fixes

Change the `part of` directive in the [part file][] to specify its library
by URI:

```dart
part of 'test.dart';
```

### path_does_not_exist

_The path '{0}' doesn't exist._

#### Description

The analyzer produces this diagnostic when a dependency has a `path` key
referencing a directory that doesn't exist.

#### Example

Assuming that the directory `doesNotExist` doesn't exist, the following
code produces this diagnostic because it's listed as the path of a package:

```yaml
name: example
dependencies:
  local_package:
    path: [!doesNotExist!]
```

#### Common fixes

If the path is correct, then create a directory at that path.

If the path isn't correct, then change the path to match the path to the
root of the package.

### path_not_posix

_The path '{0}' isn't a POSIX-style path._

#### Description

The analyzer produces this diagnostic when a dependency has a `path` key
whose value is a string, but isn't a POSIX-style path.

#### Example

The following code produces this diagnostic because the path following the
`path` key is a Windows path:

```yaml
name: example
dependencies:
  local_package:
    path: [!E:\local_package!]
```

#### Common fixes

Convert the path to a POSIX path.

### path_pubspec_does_not_exist

_The directory '{0}' doesn't contain a pubspec._

#### Description

The analyzer produces this diagnostic when a dependency has a `path` key
that references a directory that doesn't contain a `pubspec.yaml` file.

#### Example

Assuming that the directory `local_package` doesn't contain a file
`pubspec.yaml`, the following code produces this diagnostic because it's
listed as the path of a package:

```yaml
name: example
dependencies:
  local_package:
    path: [!local_package!]
```

#### Common fixes

If the path is intended to be the root of a package, then add a
`pubspec.yaml` file in the directory:

```yaml
name: local_package
```

If the path is wrong, then replace it with the correct path.

### pattern_assignment_not_local_variable

_Only local variables can be assigned in pattern assignments._

#### Description

The analyzer produces this diagnostic when a pattern assignment assigns a
value to anything other than a local variable. Patterns can't assign to
fields or top-level variables.

#### Example

If the code is cleaner when destructuring with a pattern, then rewrite the
code to assign the value to a local variable in a pattern declaration,
assigning the non-local variable separately:

```dart
class C {
  var x = 0;

  void f((int, int) r) {
    ([!x!], _) = r;
  }
}
```

#### Common fixes

If the code is cleaner when using a pattern assignment, then rewrite the
code to assign the value to a local variable, assigning the non-local
variable separately:

```dart
class C {
  var x = 0;

  void f((int, int) r) {
    var (a, _) = r;
    x = a;
  }
}
```

If the code is cleaner without using a pattern assignment, then rewrite
the code to not use a pattern assignment:

```dart
class C {
  var x = 0;

  void f((int, int) r) {
    x = r.$1;
  }
}
```

### pattern_constant_from_deferred_library

_Constant values from a deferred library can't be used in patterns._

#### Description

The analyzer produces this diagnostic when a pattern contains a value
declared in a different library, and that library is imported using a
deferred import. Constants are evaluated at compile time, but values from
deferred libraries aren't available at compile time.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

Given a file `a.dart` that defines the constant `zero`:

```dart
const zero = 0;
```

The following code produces this diagnostic because the constant pattern
`a.zero` is imported using a deferred import:

```dart
import 'a.dart' deferred as a;

void f(int x) {
  switch (x) {
    case a.[!zero!]:
      // ...
      break;
  }
}
```

#### Common fixes

If you need to reference the constant from the imported library, then
remove the `deferred` keyword:

```dart
import 'a.dart' as a;

void f(int x) {
  switch (x) {
    case a.zero:
      // ...
      break;
  }
}
```

If you need to reference the constant from the imported library and also
need the imported library to be deferred, then rewrite the switch
statement as a sequence of `if` statements:

```dart
import 'a.dart' deferred as a;

void f(int x) {
  if (x == a.zero) {
    // ...
  }
}
```

If you don't need to reference the constant, then replace the case
expression:

```dart
void f(int x) {
  switch (x) {
    case 0:
      // ...
      break;
  }
}
```

### pattern_type_mismatch_in_irrefutable_context

_The matched value of type '{0}' isn't assignable to the required type '{1}'._

#### Description

The analyzer produces this diagnostic when the type of the value on the
right-hand side of a pattern assignment or pattern declaration doesn't
match the type required by the pattern being used to match it.

#### Example

The following code produces this diagnostic because `x` might not be a
`String` and hence might not match the object pattern:

```dart
void f(Object x) {
  var [!String(length: a)!] = x;
  print(a);
}
```

#### Common fixes

Change the code so that the type of the expression on the right-hand side
matches the type required by the pattern:

```dart
void f(String x) {
  var String(length: a) = x;
  print(a);
}
```

### pattern_variable_assignment_inside_guard

_Pattern variables can't be assigned inside the guard of the enclosing guarded
pattern._

#### Description

The analyzer produces this diagnostic when a pattern variable is assigned
a value inside a guard (`when`) clause.

#### Example

The following code produces this diagnostic because the variable `a` is
assigned a value inside the guard clause:

```dart
void f(int x) {
  if (x case var a when ([!a!] = 1) > 0) {
    print(a);
  }
}
```

#### Common fixes

If there's a value you need to capture, then assign it to a different
variable:

```dart
void f(int x) {
  var b;
  if (x case var a when (b = 1) > 0) {
    print(a + b);
  }
}
```

If there isn't a value you need to capture, then remove the assignment:

```dart
void f(int x) {
  if (x case var a when 1 > 0) {
    print(a);
  }
}
```

### platform_value_disallowed

_Keys in the `platforms` field can't have values._

#### Description

The analyzer produces this diagnostic when a key in the `platforms` map
has a value.
To learn more about specifying your package's supported platforms,
check out the [documentation on platform declarations](https://dart.dev/tools/pub/pubspec#platforms).

#### Example

The following `pubspec.yaml` produces this diagnostic because the key
`web` has a value.

```yaml
name: example
platforms:
  web: [!"chrome"!]
```

#### Common fixes

Omit the value and leave the key without a value:

```yaml
name: example
platforms:
  web:
```

Values for keys in the `platforms` field are currently reserved for
potential future behavior.

### positional_field_in_object_pattern

_Object patterns can only use named fields._

#### Description

The analyzer produces this diagnostic when an object pattern contains a
field without specifying the getter name. Object pattern fields match
against values that the object's getters return. Without a getter name
specified, the pattern field can't access a value to attempt to match against.

#### Example

The following code produces this diagnostic because the object pattern
`String(1)` doesn't specify which getter of `String` to access and compare
with the value `1`:

```dart
void f(Object o) {
  if (o case String([!1!])) {}
}
```

#### Common fixes

Add the getter name to access the value, followed
by a colon before the pattern to match against:

```dart
void f(Object o) {
  if (o case String(length: 1)) {}
}
```

### positional_super_formal_parameter_with_positional_argument

_Positional super parameters can't be used when the super constructor invocation
has a positional argument._

#### Description

The analyzer produces this diagnostic when some, but not all, of the
positional parameters provided to the constructor of the superclass are
using a super parameter.

Positional super parameters are associated with positional parameters in
the super constructor by their index. That is, the first super parameter
is associated with the first positional parameter in the super
constructor, the second with the second, and so on. The same is true for
positional arguments. Having both positional super parameters and
positional arguments means that there are two values associated with the
same parameter in the superclass's constructor, and hence isn't allowed.

#### Example

The following code produces this diagnostic because the constructor
`B.new` is using a super parameter to pass one of the required positional
parameters to the super constructor in `A`, but is explicitly passing the
other in the super constructor invocation:

```dart
class A {
  A(int x, int y);
}

class B extends A {
  B(int x, super.[!y!]) : super(x);
}
```

#### Common fixes

If all the positional parameters can be super parameters, then convert the
normal positional parameters to be super parameters:

```dart
class A {
  A(int x, int y);
}

class B extends A {
  B(super.x, super.y);
}
```

If some positional parameters can't be super parameters, then convert the
super parameters to be normal parameters:

```dart
class A {
  A(int x, int y);
}

class B extends A {
  B(int x, int y) : super(x, y);
}
```

### prefix_collides_with_top_level_member

_The name '{0}' is already used as an import prefix and can't be used to name a
top-level element._

#### Description

The analyzer produces this diagnostic when a name is used as both an import
prefix and the name of a top-level declaration in the same library.

#### Example

The following code produces this diagnostic because `f` is used as both an
import prefix and the name of a function:

```dart
import 'dart:math' as f;

int [!f!]() => f.min(0, 1);
```

#### Common fixes

If you want to use the name for the import prefix, then rename the
top-level declaration:

```dart
import 'dart:math' as f;

int g() => f.min(0, 1);
```

If you want to use the name for the top-level declaration, then rename the
import prefix:

```dart
import 'dart:math' as math;

int f() => math.min(0, 1);
```

### prefix_identifier_not_followed_by_dot

_The name '{0}' refers to an import prefix, so it must be followed by '.'._

#### Description

The analyzer produces this diagnostic when an import prefix is used by
itself, without accessing any of the names declared in the libraries
associated with the prefix. Prefixes aren't variables, and therefore can't
be used as a value.

#### Example

The following code produces this diagnostic because the prefix `math` is
being used as if it were a variable:

```dart
import 'dart:math' as math;

void f() {
  print([!math!]);
}
```

#### Common fixes

If the code is incomplete, then reference something in one of the libraries
associated with the prefix:

```dart
import 'dart:math' as math;

void f() {
  print(math.pi);
}
```

If the name is wrong, then correct the name.

### prefix_shadowed_by_local_declaration

_The prefix '{0}' can't be used here because it's shadowed by a local
declaration._

#### Description

The analyzer produces this diagnostic when an import prefix is used in a
context where it isn't visible because it was shadowed by a local
declaration.

#### Example

The following code produces this diagnostic because the prefix `a` is
being used to access the class `Future`, but isn't visible because it's
shadowed by the parameter `a`:

```dart
import 'dart:async' as a;

a.Future? f(int a) {
  [!a!].Future? x;
  return x;
}
```

#### Common fixes

Rename either the prefix:

```dart
import 'dart:async' as p;

p.Future? f(int a) {
  p.Future? x;
  return x;
}
```

Or rename the local variable:

```dart
import 'dart:async' as a;

a.Future? f(int p) {
  a.Future? x;
  return x;
}
```

### private_collision_in_mixin_application

_The private name '{0}', defined by '{1}', conflicts with the same name defined
by '{2}'._

#### Description

The analyzer produces this diagnostic when two mixins that define the same
private member are used together in a single class in a library other than
the one that defines the mixins.

#### Example

Given a file `a.dart` containing the following code:

```dart
mixin A {
  void _foo() {}
}

mixin B {
  void _foo() {}
}
```

The following code produces this diagnostic because the mixins `A` and `B`
both define the method `_foo`:

```dart
import 'a.dart';

class C extends Object with A, [!B!] {}
```

#### Common fixes

If you don't need both of the mixins, then remove one of them from the
`with` clause:

```dart
import 'a.dart';

class C extends Object with A, [!B!] {}
```

If you need both of the mixins, then rename the conflicting member in one
of the two mixins.

### private_optional_parameter

_Named parameters can't start with an underscore._

#### Description

The analyzer produces this diagnostic when the name of a named parameter
starts with an underscore.

#### Example

The following code produces this diagnostic because the named parameter
`_x` starts with an underscore:

```dart
class C {
  void m({int [!_x!] = 0}) {}
}
```

#### Common fixes

Rename the parameter so that it doesn't start with an underscore:

```dart
class C {
  void m({int x = 0}) {}
}
```

### private_setter

_The setter '{0}' is private and can't be accessed outside the library that
declares it._

#### Description

The analyzer produces this diagnostic when a private setter is used in a
library where it isn't visible.

#### Example

Given a file `a.dart` that contains the following:

```dart
class A {
  static int _f = 0;
}
```

The following code produces this diagnostic because it references the
private setter `_f` even though the setter isn't visible:

```dart
import 'a.dart';

void f() {
  A.[!_f!] = 0;
}
```

#### Common fixes

If you're able to make the setter public, then do so:

```dart
class A {
  static int f = 0;
}
```

If you aren't able to make the setter public, then find a different way to
implement the code.

### read_potentially_unassigned_final

_The final variable '{0}' can't be read because it's potentially unassigned at
this point._

#### Description

The analyzer produces this diagnostic when a final local variable that
isn't initialized at the declaration site is read at a point where the
compiler can't prove that the variable is always initialized before it's
referenced.

#### Example

The following code produces this diagnostic because the final local
variable `x` is read (on line 3) when it's possible that it hasn't yet
been initialized:

```dart
int f() {
  final int x;
  return [!x!];
}
```

#### Common fixes

Ensure that the variable has been initialized before it's read:

```dart
int f(bool b) {
  final int x;
  if (b) {
    x = 0;
  } else {
    x = 1;
  }
  return x;
}
```

### record_literal_one_positional_no_trailing_comma

_A record literal with exactly one positional field requires a trailing comma._

#### Description

The analyzer produces this diagnostic when a record literal with a single
positional field doesn't have a trailing comma after the field.

In some locations a record literal with a single positional field could
also be a parenthesized expression. A trailing comma is required to
disambiguate these two valid interpretations.

#### Example

The following code produces this diagnostic because the record literal has
one positional field but doesn't have a trailing comma:

```dart
var r = const (1[!)!];
```

#### Common fixes

Add a trailing comma:

```dart
var r = const (1,);
```

### record_type_one_positional_no_trailing_comma

_A record type with exactly one positional field requires a trailing comma._

#### Description

The analyzer produces this diagnostic when a record type annotation with a
single positional field doesn't have a trailing comma after the field.

In some locations a record type with a single positional field could also
be a parenthesized expression. A trailing comma is required to
disambiguate these two valid interpretations.

#### Example

The following code produces this diagnostic because the record type has
one positional field, but doesn't have a trailing comma:

```dart
void f((int[!)!] r) {}
```

#### Common fixes

Add a trailing comma:

```dart
void f((int,) r) {}
```

### recursive_compile_time_constant

_The compile-time constant expression depends on itself._

#### Description

The analyzer produces this diagnostic when the value of a compile-time
constant is defined in terms of itself, either directly or indirectly,
creating an infinite loop.

#### Example

The following code produces this diagnostic twice because both of the
constants are defined in terms of the other:

```dart
const [!secondsPerHour!] = minutesPerHour * 60;
const [!minutesPerHour!] = secondsPerHour / 60;
```

#### Common fixes

Break the cycle by finding an alternative way of defining at least one of
the constants:

```dart
const secondsPerHour = minutesPerHour * 60;
const minutesPerHour = 60;
```

### recursive_constructor_redirect

_Constructors can't redirect to themselves either directly or indirectly._

#### Description

The analyzer produces this diagnostic when a constructor redirects to
itself, either directly or indirectly, creating an infinite loop.

#### Examples

The following code produces this diagnostic because the generative
constructors `C.a` and `C.b` each redirect to the other:

```dart
class C {
  C.a() : [!this.b()!];
  C.b() : [!this.a()!];
}
```

The following code produces this diagnostic because the factory
constructors `A` and `B` each redirect to the other:

```dart
abstract class A {
  factory A() = [!B!];
}
class B implements A {
  factory B() = [!A!];
  B.named();
}
```

#### Common fixes

In the case of generative constructors, break the cycle by finding defining
at least one of the constructors to not redirect to another constructor:

```dart
class C {
  C.a() : this.b();
  C.b();
}
```

In the case of factory constructors, break the cycle by defining at least
one of the factory constructors to do one of the following:

- Redirect to a generative constructor:

```dart
abstract class A {
  factory A() = B;
}
class B implements A {
  factory B() = B.named;
  B.named();
}
```

- Not redirect to another constructor:

```dart
abstract class A {
  factory A() = B;
}
class B implements A {
  factory B() {
    return B.named();
  }

  B.named();
}
```

- Not be a factory constructor:

```dart
abstract class A {
  factory A() = B;
}
class B implements A {
  B();
  B.named();
}
```

### recursive_interface_inheritance

_'{0}' can't be a superinterface of itself: {1}._

_'{0}' can't extend itself._

_'{0}' can't implement itself._

_'{0}' can't use itself as a mixin._

_'{0}' can't use itself as a superclass constraint._

#### Description

The analyzer produces this diagnostic when there's a circularity in the
type hierarchy. This happens when a type, either directly or indirectly,
is declared to be a subtype of itself.

#### Example

The following code produces this diagnostic because the class `A` is
declared to be a subtype of `B`, and `B` is a subtype of `A`:

```dart
class [!A!] extends B {}
class B implements A {}
```

#### Common fixes

Change the type hierarchy so that there's no circularity.

### redeclare_on_non_redeclaring_member

_The {0} doesn't redeclare a {0} declared in a superinterface._

#### Description

The analyzer produces this diagnostic when a member of an extension type
is annotated with `@redeclare`, but none of the implemented interfaces
has a member with the same name.

#### Example

The following code produces this diagnostic because the member `n`
declared by the extension type `E` is annotated with `@redeclare`, but `C`
doesn't have a member named `n`:

```dart
import 'package:meta/meta.dart';

class C {
  void m() {}
}

extension type E(C c) implements C {
  @redeclare
  void [!n!]() {}
}
```

#### Common fixes

If the annotated member has the right name, then remove the annotation:

```dart
class C {
  void m() {}
}

extension type E(C c) implements C {
  void n() {}
}
```

If the annotated member is suppose to replace a member from the
implemented interfaces, then change the name of the annotated member to
match the member being replaced:

```dart
import 'package:meta/meta.dart';

class C {
  void m() {}
}

extension type E(C c) implements C {
  @redeclare
  void m() {}
}
```

### redirect_generative_to_missing_constructor

_The constructor '{0}' couldn't be found in '{1}'._

#### Description

The analyzer produces this diagnostic when a generative constructor
redirects to a constructor that isn't defined.

#### Example

The following code produces this diagnostic because the constructor `C.a`
redirects to the constructor `C.b`, but `C.b` isn't defined:

```dart
class C {
  C.a() : [!this.b()!];
}
```

#### Common fixes

If the missing constructor must be called, then define it:

```dart
class C {
  C.a() : this.b();
  C.b();
}
```

If the missing constructor doesn't need to be called, then remove the
redirect:

```dart
class C {
  C.a();
}
```

### redirect_generative_to_non_generative_constructor

_Generative constructors can't redirect to a factory constructor._

#### Description

The analyzer produces this diagnostic when a generative constructor
redirects to a factory constructor.

#### Example

The following code produces this diagnostic because the generative
constructor `C.a` redirects to the factory constructor `C.b`:

```dart
class C {
  C.a() : [!this.b()!];
  factory C.b() => C.a();
}
```

#### Common fixes

If the generative constructor doesn't need to redirect to another
constructor, then remove the redirect.

```dart
class C {
  C.a();
  factory C.b() => C.a();
}
```

If the generative constructor must redirect to another constructor, then
make the other constructor be a generative (non-factory) constructor:

```dart
class C {
  C.a() : this.b();
  C.b();
}
```

### redirect_to_abstract_class_constructor

_The redirecting constructor '{0}' can't redirect to a constructor of the
abstract class '{1}'._

#### Description

The analyzer produces this diagnostic when a constructor redirects to a
constructor in an abstract class.

#### Example

The following code produces this diagnostic because the factory
constructor in `A` redirects to a constructor in `B`, but `B` is an
abstract class:

```dart
class A {
  factory A() = [!B!];
}

abstract class B implements A {}
```

#### Common fixes

If the code redirects to the correct constructor, then change the class so
that it isn't abstract:

```dart
class A {
  factory A() = B;
}

class B implements A {}
```

Otherwise, change the factory constructor so that it either redirects to a
constructor in a concrete class, or has a concrete implementation.

### redirect_to_invalid_function_type

_The redirected constructor '{0}' has incompatible parameters with '{1}'._

#### Description

The analyzer produces this diagnostic when a factory constructor attempts
to redirect to another constructor, but the two have incompatible
parameters. The parameters are compatible if all of the parameters of the
redirecting constructor can be passed to the other constructor and if the
other constructor doesn't require any parameters that aren't declared by
the redirecting constructor.

#### Examples

The following code produces this diagnostic because the constructor for `A`
doesn't declare a parameter that the constructor for `B` requires:

```dart
abstract class A {
  factory A() = [!B!];
}

class B implements A {
  B(int x);
  B.zero();
}
```

The following code produces this diagnostic because the constructor for `A`
declares a named parameter (`y`) that the constructor for `B` doesn't
allow:

```dart
abstract class A {
  factory A(int x, {int y}) = [!B!];
}

class B implements A {
  B(int x);
}
```

#### Common fixes

If there's a different constructor that is compatible with the redirecting
constructor, then redirect to that constructor:

```dart
abstract class A {
  factory A() = B.zero;
}

class B implements A {
  B(int x);
  B.zero();
}
```

Otherwise, update the redirecting constructor to be compatible:

```dart
abstract class A {
  factory A(int x) = B;
}

class B implements A {
  B(int x);
}
```

### redirect_to_invalid_return_type

_The return type '{0}' of the redirected constructor isn't a subtype of '{1}'._

#### Description

The analyzer produces this diagnostic when a factory constructor redirects
to a constructor whose return type isn't a subtype of the type that the
factory constructor is declared to produce.

#### Example

The following code produces this diagnostic because `A` isn't a subclass
of `C`, which means that the value returned by the constructor `A()`
couldn't be returned from the constructor `C()`:

```dart
class A {}

class B implements C {}

class C {
  factory C() = [!A!];
}
```

#### Common fixes

If the factory constructor is redirecting to a constructor in the wrong
class, then update the factory constructor to redirect to the correct
constructor:

```dart
class A {}

class B implements C {}

class C {
  factory C() = B;
}
```

If the class defining the constructor being redirected to is the class that
should be returned, then make it a subtype of the factory's return type:

```dart
class A implements C {}

class B implements C {}

class C {
  factory C() = A;
}
```

### redirect_to_missing_constructor

_The constructor '{0}' couldn't be found in '{1}'._

#### Description

The analyzer produces this diagnostic when a constructor redirects to a
constructor that doesn't exist.

#### Example

The following code produces this diagnostic because the factory
constructor in `A` redirects to a constructor in `B` that doesn't exist:

```dart
class A {
  factory A() = [!B.name!];
}

class B implements A {
  B();
}
```

#### Common fixes

If the constructor being redirected to is correct, then define the
constructor:

```dart
class A {
  factory A() = B.name;
}

class B implements A {
  B();
  B.name();
}
```

If a different constructor should be invoked, then update the redirect:

```dart
class A {
  factory A() = B;
}

class B implements A {
  B();
}
```

### redirect_to_non_class

_The name '{0}' isn't a type and can't be used in a redirected constructor._

#### Description

One way to implement a factory constructor is to redirect to another
constructor by referencing the name of the constructor. The analyzer
produces this diagnostic when the redirect is to something other than a
constructor.

#### Example

The following code produces this diagnostic because `f` is a function:

```dart
C f() => throw 0;

class C {
  factory C() = [!f!];
}
```

#### Common fixes

If the constructor isn't defined, then either define it or replace it with
a constructor that is defined.

If the constructor is defined but the class that defines it isn't visible,
then you probably need to add an import.

If you're trying to return the value returned by a function, then rewrite
the constructor to return the value from the constructor's body:

```dart
C f() => throw 0;

class C {
  factory C() => f();
}
```

### redirect_to_non_const_constructor

_A constant redirecting constructor can't redirect to a non-constant
constructor._

#### Description

The analyzer produces this diagnostic when a constructor marked as `const`
redirects to a constructor that isn't marked as `const`.

#### Example

The following code produces this diagnostic because the constructor `C.a`
is marked as `const` but redirects to the constructor `C.b`, which isn't:

```dart
class C {
  const C.a() : this.[!b!]();
  C.b();
}
```

#### Common fixes

If the non-constant constructor can be marked as `const`, then mark it as
`const`:

```dart
class C {
  const C.a() : this.b();
  const C.b();
}
```

If the non-constant constructor can't be marked as `const`, then either
remove the redirect or remove `const` from the redirecting constructor:

```dart
class C {
  C.a() : this.b();
  C.b();
}
```

### redirect_to_type_alias_expands_to_type_parameter

_A redirecting constructor can't redirect to a type alias that expands to a type
parameter._

#### Description

The analyzer produces this diagnostic when a redirecting factory
constructor redirects to a type alias, and the type alias expands to one of
the type parameters of the type alias. This isn't allowed because the value
of the type parameter is a type rather than a class.

#### Example

The following code produces this diagnostic because the redirect to `B<A>`
is to a type alias whose value is `T`, even though it looks like the value
should be `A`:

```dart
class A implements C {}

typedef B<T> = T;

abstract class C {
  factory C() = [!B!]<A>;
}
```

#### Common fixes

Use either a class name or a type alias that is defined to be a class
rather than a type alias defined to be a type parameter:

```dart
class A implements C {}

abstract class C {
  factory C() = A;
}
```

### referenced_before_declaration

_Local variable '{0}' can't be referenced before it is declared._

#### Description

The analyzer produces this diagnostic when a variable is referenced before
it's declared. In Dart, variables are visible everywhere in the block in
which they are declared, but can only be referenced after they are
declared.

The analyzer also produces a context message that indicates where the
declaration is located.

#### Example

The following code produces this diagnostic because `i` is used before it
is declared:

```dart
void f() {
  print([!i!]);
  int i = 5;
}
```

#### Common fixes

If you intended to reference the local variable, move the declaration
before the first reference:

```dart
void f() {
  int i = 5;
  print(i);
}
```

If you intended to reference a name from an outer scope, such as a
parameter, instance field or top-level variable, then rename the local
declaration so that it doesn't hide the outer variable.

```dart
void f(int i) {
  print(i);
  int x = 5;
  print(x);
}
```

### refutable_pattern_in_irrefutable_context

_Refutable patterns can't be used in an irrefutable context._

#### Description

The analyzer produces this diagnostic when a [refutable pattern][] is used
in a context where only an [irrefutable pattern][] is allowed.

The refutable patterns that are disallowed are:
- logical-or
- relational
- null-check
- constant

The contexts that are checked are:
- pattern-based variable declarations
- pattern-based for loops
- assignments with a pattern on the left-hand side

#### Example

The following code produces this diagnostic because the null-check
pattern, which is a refutable pattern, is in a pattern-based variable
declaration, which doesn't allow refutable patterns:

```dart
void f(int? x) {
  var ([!_?!]) = x;
}
```

#### Common fixes

Rewrite the code to not use a refutable pattern in an irrefutable context.

### relational_pattern_operand_type_not_assignable

_The constant expression type '{0}' is not assignable to the parameter type
'{1}' of the '{2}' operator._

#### Description

The analyzer produces this diagnostic when the operand of a relational
pattern has a type that isn't assignable to the parameter of the operator
that will be invoked.

#### Example

The following code produces this diagnostic because the operand in the
relational pattern (`0`) is an `int`, but the `>` operator defined in `C`
expects an object of type `C`:

```dart
class C {
  const C();

  bool operator >(C other) => true;
}

void f(C c) {
  switch (c) {
    case > [!0!]:
      print('positive');
  }
}
```

#### Common fixes

If the switch is using the correct value, then change the case to compare
the value to the right type of object:

```dart
class C {
  const C();

  bool operator >(C other) => true;
}

void f(C c) {
  switch (c) {
    case > const C():
      print('positive');
  }
}
```

If the switch is using the wrong value, then change the expression used to
compute the value being matched:

```dart
class C {
  const C();

  bool operator >(C other) => true;

  int get toInt => 0;
}

void f(C c) {
  switch (c.toInt) {
    case > 0:
      print('positive');
  }
}
```

### relational_pattern_operator_return_type_not_assignable_to_bool

_The return type of operators used in relational patterns must be assignable to
'bool'._

#### Description

The analyzer produces this diagnostic when a relational pattern references
an operator that doesn't produce a value of type `bool`.

#### Example

The following code produces this diagnostic because the operator `>`, used
in the relational pattern `> c2`, returns a value of type `int` rather
than a `bool`:

```dart
class C {
  const C();

  int operator >(C c) => 3;

  bool operator <(C c) => false;
}

const C c2 = C();

void f(C c1) {
  if (c1 case [!>!] c2) {}
}
```

#### Common fixes

If there's a different operator that should be used, then change the
operator:

```dart
class C {
  const C();

  int operator >(C c) => 3;

  bool operator <(C c) => false;
}

const C c2 = C();

void f(C c1) {
  if (c1 case < c2) {}
}
```

If the operator is expected to return `bool`, then update the declaration
of the operator:

```dart
class C {
  const C();

  bool operator >(C c) => true;

  bool operator <(C c) => false;
}

const C c2 = C();

void f(C c1) {
  if (c1 case > c2) {}
}
```

### rest_element_in_map_pattern

_A map pattern can't contain a rest pattern._

#### Description

The analyzer produces this diagnostic when a map pattern contains a rest
pattern. Map patterns match a map with more keys
than those explicitly given in the pattern (as long as the given keys match),
so a rest pattern is unnecessary.

#### Example

The following code produces this diagnostic because the map pattern contains
a rest pattern:

```dart
void f(Map<int, String> x) {
  if (x case {0: _, [!...!]}) {}
}
```

#### Common fixes

Remove the rest pattern:

```dart
void f(Map<int, String> x) {
  if (x case {0: _}) {}
}
```

### rethrow_outside_catch

_A rethrow must be inside of a catch clause._

#### Description

The analyzer produces this diagnostic when a `rethrow` statement is outside
a `catch` clause. The `rethrow` statement is used to throw a caught
exception again, but there's no caught exception outside of a `catch`
clause.

#### Example

The following code produces this diagnostic because the`rethrow` statement
is outside of a `catch` clause:

```dart
void f() {
  [!rethrow!];
}
```

#### Common fixes

If you're trying to rethrow an exception, then wrap the `rethrow` statement
in a `catch` clause:

```dart
void f() {
  try {
    // ...
  } catch (exception) {
    rethrow;
  }
}
```

If you're trying to throw a new exception, then replace the `rethrow`
statement with a `throw` expression:

```dart
void f() {
  throw UnsupportedError('Not yet implemented');
}
```

### return_in_generative_constructor

_Constructors can't return values._

#### Description

The analyzer produces this diagnostic when a generative constructor
contains a `return` statement that specifies a value to be returned.
Generative constructors always return the object that was created, and
therefore can't return a different object.

#### Example

The following code produces this diagnostic because the `return` statement
has an expression:

```dart
class C {
  C() {
    return [!this!];
  }
}
```

#### Common fixes

If the constructor should create a new instance, then remove either the
`return` statement or the expression:

```dart
class C {
  C();
}
```

If the constructor shouldn't create a new instance, then convert it to be a
factory constructor:

```dart
class C {
  factory C() {
    return _instance;
  }

  static C _instance = C._();

  C._();
}
```

### return_in_generator

_Can't return a value from a generator function that uses the 'async*' or
'sync*' modifier._

#### Description

The analyzer produces this diagnostic when a generator function (one whose
body is marked with either `async*` or `sync*`) uses either a `return`
statement to return a value or implicitly returns a value because of using
`=>`. In any of these cases, they should use `yield` instead of `return`.

#### Examples

The following code produces this diagnostic because the method `f` is a
generator and is using `return` to return a value:

```dart
Iterable<int> f() sync* {
  [!return!] 3;
}
```

The following code produces this diagnostic because the function `f` is a
generator and is implicitly returning a value:

```dart
Stream<int> f() async* [!=>!] 3;
```

#### Common fixes

If the function is using `=>` for the body of the function, then convert it
to a block function body, and use `yield` to return a value:

```dart
Stream<int> f() async* {
  yield 3;
}
```

If the method is intended to be a generator, then use `yield` to return a
value:

```dart
Iterable<int> f() sync* {
  yield 3;
}
```

If the method isn't intended to be a generator, then remove the modifier
from the body (or use `async` if you're returning a future):

```dart
int f() {
  return 3;
}
```

### return_of_do_not_store

_'{0}' is annotated with 'doNotStore' and shouldn't be returned unless '{1}' is
also annotated._

#### Description

The analyzer produces this diagnostic when a value that is annotated with
the [`doNotStore`][meta-doNotStore] annotation is returned from a method,
getter, or function that doesn't have the same annotation.

#### Example

The following code produces this diagnostic because the result of invoking
`f` shouldn't be stored, but the function `g` isn't annotated to preserve
that semantic:

```dart
import 'package:meta/meta.dart';

@doNotStore
int f() => 0;

int g() => [!f()!];
```

#### Common fixes

If the value that shouldn't be stored is the correct value to return, then
mark the function with the [`doNotStore`][meta-doNotStore] annotation:

```dart
import 'package:meta/meta.dart';

@doNotStore
int f() => 0;

@doNotStore
int g() => f();
```

Otherwise, return a different value from the function:

```dart
import 'package:meta/meta.dart';

@doNotStore
int f() => 0;

int g() => 0;
```

### return_of_invalid_type

_A value of type '{0}' can't be returned from the constructor '{2}' because it
has a return type of '{1}'._

_A value of type '{0}' can't be returned from the function '{2}' because it has
a return type of '{1}'._

_A value of type '{0}' can't be returned from the method '{2}' because it has a
return type of '{1}'._

#### Description

The analyzer produces this diagnostic when a method or function returns a
value whose type isn't assignable to the declared return type.

#### Example

The following code produces this diagnostic because `f` has a return type
of `String` but is returning an `int`:

```dart
String f() => [!3!];
```

#### Common fixes

If the return type is correct, then replace the value being returned with a
value of the correct type, possibly by converting the existing value:

```dart
String f() => 3.toString();
```

If the value is correct, then change the return type to match:

```dart
int f() => 3;
```

### return_of_invalid_type_from_closure

_The return type '{0}' isn't a '{1}', as required by the closure's context._

#### Description

The analyzer produces this diagnostic when the static type of a returned
expression isn't assignable to the return type that the closure is required
to have.

#### Example

The following code produces this diagnostic because `f` is defined to be a
function that returns a `String`, but the closure assigned to it returns an
`int`:

```dart
String Function(String) f = (s) => [!3!];
```

#### Common fixes

If the return type is correct, then replace the returned value with a value
of the correct type, possibly by converting the existing value:

```dart
String Function(String) f = (s) => 3.toString();
```

### return_without_value

_The return value is missing after 'return'._

#### Description

The analyzer produces this diagnostic when it finds a `return` statement
without an expression in a function that declares a return type.

#### Example

The following code produces this diagnostic because the function `f` is
expected to return an `int`, but no value is being returned:

```dart
int f() {
  [!return!];
}
```

#### Common fixes

Add an expression that computes the value to be returned:

```dart
int f() {
  return 0;
}
```

### sdk_version_async_exported_from_core

_The class '{0}' wasn't exported from 'dart:core' until version 2.1, but this
code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when either the class `Future` or
`Stream` is referenced in a library that doesn't import `dart:async` in
code that has an SDK constraint whose lower bound is less than 2.1.0. In
earlier versions, these classes weren't defined in `dart:core`, so the
import was necessary.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.1.0:

```yaml
environment:
  sdk: '>=2.0.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

```dart
void f([!Future!] f) {}
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the classes to be referenced:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

If you need to support older versions of the SDK, then import the
`dart:async` library.

```dart
import 'dart:async';

void f(Future f) {}
```

### sdk_version_as_expression_in_const_context

_The use of an as expression in a constant expression wasn't supported until
version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an `as` expression inside a
[constant context][] is found in code that has an SDK constraint whose
lower bound is less than 2.3.2. Using an `as` expression in a
[constant context][] wasn't supported in earlier versions, so this code
won't be able to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

```dart
const num n = 3;
const int i = [!n as int!];
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the expression to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use an `as` expression, or change the code so that the `as`
expression isn't in a [constant context][]:

```dart
num x = 3;
int y = x as int;
```

### sdk_version_bool_operator_in_const_context

_The use of the operator '{0}' for 'bool' operands in a constant context wasn't
supported until version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when any use of the `&`, `|`, or `^`
operators on the class `bool` inside a [constant context][] is found in
code that has an SDK constraint whose lower bound is less than 2.3.2. Using
these operators in a [constant context][] wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

```dart
const bool a = true;
const bool b = false;
const bool c = a [!&!] b;
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the operators to be used:

```yaml
environment:
 sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use these operators, or change the code so that the expression
isn't in a [constant context][]:

```dart
const bool a = true;
const bool b = false;
bool c = a & b;
```

### sdk_version_constructor_tearoffs

_Tearing off a constructor requires the 'constructor-tearoffs' language
feature._

#### Description

The analyzer produces this diagnostic when a constructor tear-off is found
in code that has an SDK constraint whose lower bound is less than 2.15.
Constructor tear-offs weren't supported in earlier versions, so this code
won't be able to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.15:

```yaml
environment:
  sdk: '>=2.9.0 <2.15.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

```dart
var setConstructor = [!Set.identity!];
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the operator to be used:

```yaml
environment:
  sdk: '>=2.15.0 <2.16.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not use constructor tear-offs:

```dart
var setConstructor = () => Set.identity();
```

### sdk_version_eq_eq_operator_in_const_context

_Using the operator '==' for non-primitive types wasn't supported until version
2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when the operator `==` is used on a
non-primitive type inside a [constant context][] is found in code that has
an SDK constraint whose lower bound is less than 2.3.2. Using this operator
in a [constant context][] wasn't supported in earlier versions, so this
code won't be able to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

```dart
class C {}
const C a = null;
const C b = null;
const bool same = a [!==!] b;
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the operator to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use the `==` operator, or change the code so that the
expression isn't in a [constant context][]:

```dart
class C {}
const C a = null;
const C b = null;
bool same = a == b;
```

### sdk_version_extension_methods

_Extension methods weren't supported until version 2.6.0, but this code is
required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an extension declaration or an
extension override is found in code that has an SDK constraint whose lower
bound is less than 2.6.0. Using extensions wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.6.0:

```yaml
environment:
 sdk: '>=2.4.0 <2.7.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

```dart
[!extension!] E on String {
  void sayHello() {
    print('Hello $this');
  }
}
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.6.0 <2.7.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of extensions. The most common way to do this is to rewrite
the members of the extension as top-level functions (or methods) that take
the value that would have been bound to `this` as a parameter:

```dart
void sayHello(String s) {
  print('Hello $s');
}
```

### sdk_version_gt_gt_gt_operator

_The operator '>>>' wasn't supported until version 2.14.0, but this code is
required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when the operator `>>>` is used in
code that has an SDK constraint whose lower bound is less than 2.14.0. This
operator wasn't supported in earlier versions, so this code won't be able
to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.14.0:

```yaml
environment:
 sdk: '>=2.0.0 <2.15.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

```dart
int x = 3 [!>>>!] 4;
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the operator to be used:

```yaml
environment:
  sdk: '>=2.14.0 <2.15.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not use the `>>>` operator:

```dart
int x = logicalShiftRight(3, 4);

int logicalShiftRight(int leftOperand, int rightOperand) {
  int divisor = 1 << rightOperand;
  if (divisor == 0) {
    return 0;
  }
  return leftOperand ~/ divisor;
}
```

### sdk_version_is_expression_in_const_context

_The use of an is expression in a constant context wasn't supported until
version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an `is` expression inside a
[constant context][] is found in code that has an SDK constraint whose
lower bound is less than 2.3.2. Using an `is` expression in a
[constant context][] wasn't supported in earlier versions, so this code
won't be able to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

```dart
const Object x = 4;
const y = [!x is int!] ? 0 : 1;
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the expression to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use the `is` operator, or, if that isn't possible, change the
code so that the `is` expression isn't in a
[constant context][]:

```dart
const Object x = 4;
var y = x is int ? 0 : 1;
```

### sdk_version_never

_The type 'Never' wasn't supported until version 2.12.0, but this code is
required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when a reference to the class `Never`
is found in code that has an SDK constraint whose lower bound is less than
2.12.0. This class wasn't defined in earlier versions, so this code won't
be able to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.12.0:

```yaml
environment:
  sdk: '>=2.5.0 <2.6.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

```dart
[!Never!] n;
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the type to be used:

```yaml
environment:
  sdk: '>=2.12.0 <2.13.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not reference this class:

```dart
dynamic x;
```

### sdk_version_set_literal

_Set literals weren't supported until version 2.2, but this code is required to
be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when a set literal is found in code
that has an SDK constraint whose lower bound is less than 2.2.0. Set
literals weren't supported in earlier versions, so this code won't be able
to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.2.0:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

```dart
var s = [!<int>{}!];
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.2.0 <2.4.0'
```

If you do need to support older versions of the SDK, then replace the set
literal with code that creates the set without the use of a literal:

```dart
var s = new Set<int>();
```

### sdk_version_ui_as_code

_The for, if, and spread elements weren't supported until version 2.3.0, but
this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when a for, if, or spread element is
found in code that has an SDK constraint whose lower bound is less than
2.3.0. Using a for, if, or spread element wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.0:

```yaml
environment:
  sdk: '>=2.2.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

```dart
var digits = [[!for (int i = 0; i < 10; i++) i!]];
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.3.0 <2.4.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of those elements:

```dart
var digits = _initializeDigits();

List<int> _initializeDigits() {
  var digits = <int>[];
  for (int i = 0; i < 10; i++) {
    digits.add(i);
  }
  return digits;
}
```

### sdk_version_ui_as_code_in_const_context

_The if and spread elements weren't supported in constant expressions until
version 2.5.0, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an if or spread element inside
a [constant context][] is found in code that has an SDK constraint whose
lower bound is less than 2.5.0. Using an if or spread element inside a
[constant context][] wasn't supported in earlier versions, so this code
won't be able to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.5.0:

```yaml
environment:
  sdk: '>=2.4.0 <2.6.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

```dart
const a = [1, 2];
const b = [[!...a!]];
```

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.5.0 <2.6.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of those elements:

```dart
const a = [1, 2];
const b = [1, 2];
```

If that isn't possible, change the code so that the element isn't in a
[constant context][]:

```dart
const a = [1, 2];
var b = [...a];
```

### set_element_type_not_assignable

_The element type '{0}' can't be assigned to the set type '{1}'._

#### Description

The analyzer produces this diagnostic when an element in a set literal has
a type that isn't assignable to the element type of the set.

#### Example

The following code produces this diagnostic because the type of the string
literal `'0'` is `String`, which isn't assignable to `int`, the element
type of the set:

```dart
var s = <int>{[!'0'!]};
```

#### Common fixes

If the element type of the set literal is wrong, then change the element
type of the set:

```dart
var s = <String>{'0'};
```

If the type of the element is wrong, then change the element:

```dart
var s = <int>{'0'.length};
```

### shared_deferred_prefix

_The prefix of a deferred import can't be used in other import directives._

#### Description

The analyzer produces this diagnostic when a prefix in a deferred import is
also used as a prefix in other imports (whether deferred or not). The
prefix in a deferred import can't be shared with other imports because the
prefix is used to load the imported library.

#### Example

The following code produces this diagnostic because the prefix `x` is used
as the prefix for a deferred import and is also used for one other import:

```dart
import 'dart:math' [!deferred!] as x;
import 'dart:convert' as x;

var y = x.json.encode(x.min(0, 1));
```

#### Common fixes

If you can use a different name for the deferred import, then do so:

```dart
import 'dart:math' deferred as math;
import 'dart:convert' as x;

var y = x.json.encode(math.min(0, 1));
```

If you can use a different name for the other imports, then do so:

```dart
import 'dart:math' deferred as x;
import 'dart:convert' as convert;

var y = convert.json.encode(x.min(0, 1));
```

### size_annotation_dimensions

_'Array's must have an 'Array' annotation that matches the dimensions._

#### Description

The analyzer produces this diagnostic when the number of dimensions
specified in an `Array` annotation doesn't match the number of nested
arrays specified by the type of a field.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the field `a0` has a
type with three nested arrays, but only two dimensions are given in the
`Array` annotation:

```dart
import 'dart:ffi';

final class C extends Struct {
  [!@Array(8, 8)!]
  external Array<Array<Array<Uint8>>> a0;
}
```

#### Common fixes

If the type of the field is correct, then fix the annotation to have the
required number of dimensions:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Array(8, 8, 4)
  external Array<Array<Array<Uint8>>> a0;
}
```

If the type of the field is wrong, then fix the type of the field:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Array(8, 8)
  external Array<Array<Uint8>> a0;
}
```

### static_access_to_instance_member

_Instance member '{0}' can't be accessed using static access._

#### Description

The analyzer produces this diagnostic when a class name is used to access
an instance field. Instance fields don't exist on a class; they exist only
on an instance of the class.

#### Example

The following code produces this diagnostic because `x` is an instance
field:

```dart
class C {
  static int a = 0;

  int b = 0;
}

int f() => C.[!b!];
```

#### Common fixes

If you intend to access a static field, then change the name of the field
to an existing static field:

```dart
class C {
  static int a = 0;

  int b = 0;
}

int f() => C.a;
```

If you intend to access the instance field, then use an instance of the
class to access the field:

```dart
class C {
  static int a = 0;

  int b = 0;
}

int f(C c) => c.b;
```

### subtype_of_base_or_final_is_not_base_final_or_sealed

_The mixin '{0}' must be 'base' because the supertype '{1}' is 'base'._

_The mixin '{0}' must be 'base' because the supertype '{1}' is 'final'._

_The type '{0}' must be 'base', 'final' or 'sealed' because the supertype '{1}'
is 'base'._

_The type '{0}' must be 'base', 'final' or 'sealed' because the supertype '{1}'
is 'final'._

#### Description

The analyzer produces this diagnostic when a class or mixin has a direct
or indirect supertype that is either `base` or `final`, but the class or
mixin itself isn't marked either `base`, `final`, or `sealed`.

#### Example

The following code produces this diagnostic because the class `B` is a
subtype of `A`, and `A` is a `base` class, but `B` is neither `base`,
`final` or `sealed`:

```dart
base class A {}
class [!B!] extends A {}
```

#### Common fixes

Add either `base`, `final` or `sealed` to the class or mixin declaration:

```dart
base class A {}
final class B extends A {}
```

### subtype_of_deferred_class

_Classes and mixins can't implement deferred classes._

_Classes can't extend deferred classes._

_Classes can't mixin deferred classes._

#### Description

The analyzer produces this diagnostic when a type (class or mixin) is a
subtype of a class from a library being imported using a deferred import.
The supertypes of a type must be compiled at the same time as the type, and
classes from deferred libraries aren't compiled until the library is
loaded.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

Given a file `a.dart` that defines the class `A`:

```dart
class A {}
```

The following code produces this diagnostic because the superclass of `B`
is declared in a deferred library:

```dart
import 'a.dart' deferred as a;

class B extends [!a.A!] {}
```

#### Common fixes

If you need to create a subtype of a type from the deferred library, then
remove the `deferred` keyword:

```dart
import 'a.dart' as a;

class B extends a.A {}
```

### subtype_of_disallowed_type

_'{0}' can't be used as a superclass constraint._

_Classes and mixins can't implement '{0}'._

_Classes can't extend '{0}'._

_Classes can't mixin '{0}'._

#### Description

The analyzer produces this diagnostic when one of the restricted classes is
used in either an `extends`, `implements`, `with`, or `on` clause. The
classes `bool`, `double`, `FutureOr`, `int`, `Null`, `num`, and `String`
are all restricted in this way, to allow for more efficient
implementations.

#### Examples

The following code produces this diagnostic because `String` is used in an
`extends` clause:

```dart
class A extends [!String!] {}
```

The following code produces this diagnostic because `String` is used in an
`implements` clause:

```dart
class B implements [!String!] {}
```

The following code produces this diagnostic because `String` is used in a
`with` clause:

```dart
class C with [!String!] {}
```

The following code produces this diagnostic because `String` is used in an
`on` clause:

```dart
mixin M on [!String!] {}
```

#### Common fixes

If a different type should be specified, then replace the type:

```dart
class A extends Object {}
```

If there isn't a different type that would be appropriate, then remove the
type, and possibly the whole clause:

```dart
class B {}
```

### subtype_of_ffi_class

_The class '{0}' can't extend '{1}'._

_The class '{0}' can't implement '{1}'._

_The class '{0}' can't mix in '{1}'._

#### Description

The analyzer produces this diagnostic when a class extends any FFI class
other than `Struct` or `Union`, or implements or mixes in any FFI class.
`Struct` and `Union` are the only FFI classes that can be subtyped, and
then only by extending them.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the class `C` extends
`Double`:

```dart
import 'dart:ffi';

final class C extends [!Double!] {}
```

#### Common fixes

If the class should extend either `Struct` or `Union`, then change the
declaration of the class:

```dart
import 'dart:ffi';

final class C extends Struct {
  @Int32()
  external int i;
}
```

If the class shouldn't extend either `Struct` or `Union`, then remove any
references to FFI classes:

```dart
final class C {}
```

### subtype_of_sealed_class

_The class '{0}' shouldn't be extended, mixed in, or implemented because it's
sealed._

#### Description

The analyzer produces this diagnostic when a sealed class (one that either
has the [`sealed`][meta-sealed] annotation or inherits or mixes in a
sealed class) is referenced in either the `extends`, `implements`, or
`with` clause of a class or mixin declaration if the declaration isn't in
the same package as the sealed class.

#### Example

Given a library in a package other than the package being analyzed that
contains the following:

```dart
import 'package:meta/meta.dart';

class A {}

@sealed
class B {}
```

The following code produces this diagnostic because `C`, which isn't in the
same package as `B`, is extending the sealed class `B`:

```dart
import 'package:a/a.dart';

[!class C extends B {}!]
```

#### Common fixes

If the class doesn't need to be a subtype of the sealed class, then change
the declaration so that it isn't:

```dart
import 'package:a/a.dart';

class B extends A {}
```

If the class needs to be a subtype of the sealed class, then either change
the sealed class so that it's no longer sealed or move the subclass into
the same package as the sealed class.

### subtype_of_struct_class

_The class '{0}' can't extend '{1}' because '{1}' is a subtype of 'Struct',
'Union', or 'AbiSpecificInteger'._

_The class '{0}' can't implement '{1}' because '{1}' is a subtype of 'Struct',
'Union', or 'AbiSpecificInteger'._

_The class '{0}' can't mix in '{1}' because '{1}' is a subtype of 'Struct',
'Union', or 'AbiSpecificInteger'._

#### Description

The analyzer produces this diagnostic when a class extends, implements, or
mixes in a class that extends either `Struct` or `Union`. Classes can only
extend either `Struct` or `Union` directly.

For more information about FFI, see [C interop using dart:ffi][ffi].

#### Example

The following code produces this diagnostic because the class `C` extends
`S`, and `S` extends `Struct`:

```dart
import 'dart:ffi';

final class S extends Struct {
  external Pointer f;
}

final class C extends [!S!] {
  external Pointer g;
}
```

#### Common fixes

If you're trying to define a struct or union that shares some fields
declared by a different struct or union, then extend `Struct` or `Union`
directly and copy the shared fields:

```dart
import 'dart:ffi';

final class S extends Struct {
  external Pointer f;
}

final class C extends Struct {
  external Pointer f;

  external Pointer g;
}
```

### supertype_expands_to_type_parameter

_A type alias that expands to a type parameter can't be implemented._

_A type alias that expands to a type parameter can't be mixed in._

_A type alias that expands to a type parameter can't be used as a superclass
constraint._

_A type alias that expands to a type parameter can't be used as a superclass._

#### Description

The analyzer produces this diagnostic when a type alias that expands to a
type parameter is used in an `extends`, `implements`, `with`, or `on`
clause.

#### Example

The following code produces this diagnostic because the type alias `T`,
which expands to the type parameter `S`, is used in the `extends` clause of
the class `C`:

```dart
typedef T<S> = S;

class C extends [!T!]<Object> {}
```

#### Common fixes

Use the value of the type argument directly:

```dart
typedef T<S> = S;

class C extends Object {}
```

### super_formal_parameter_type_is_not_subtype_of_associated

_The type '{0}' of this parameter isn't a subtype of the type '{1}' of the
associated super constructor parameter._

#### Description

The analyzer produces this diagnostic when the type of a super parameter
isn't a subtype of the corresponding parameter from the super constructor.

#### Example

The following code produces this diagnostic because the type of the super
parameter `x` in the constructor for `B` isn't a subtype of the parameter
`x` in the constructor for `A`:

```dart
class A {
  A(num x);
}

class B extends A {
  B(String super.[!x!]);
}
```

#### Common fixes

If the type of the super parameter can be the same as the parameter from
the super constructor, then remove the type annotation from the super
parameter (if the type is implicit, it is inferred from the type in the
super constructor):

```dart
class A {
  A(num x);
}

class B extends A {
  B(super.x);
}
```

If the type of the super parameter can be a subtype of the corresponding
parameter's type, then change the type of the super parameter:

```dart
class A {
  A(num x);
}

class B extends A {
  B(int super.x);
}
```

If the type of the super parameter can't be changed, then use a normal
parameter instead of a super parameter:

```dart
class A {
  A(num x);
}

class B extends A {
  B(String x) : super(x.length);
}
```

### super_formal_parameter_without_associated_named

_No associated named super constructor parameter._

#### Description

The analyzer produces this diagnostic when there's a named super parameter
in a constructor and the implicitly or explicitly invoked super
constructor doesn't have a named parameter with the same name.

Named super parameters are associated by name with named parameters in the
super constructor.

#### Example

The following code produces this diagnostic because the constructor in `A`
doesn't have a parameter named `y`:

```dart
class A {
  A({int? x});
}

class B extends A {
  B({super.[!y!]});
}
```

#### Common fixes

If the super parameter should be associated with an existing parameter
from the super constructor, then change the name to match the name of the
corresponding parameter:

```dart
class A {
  A({int? x});
}

class B extends A {
  B({super.x});
}
```

If the super parameter should be associated with a parameter that hasn't
yet been added to the super constructor, then add it:

```dart
class A {
  A({int? x, int? y});
}

class B extends A {
  B({super.y});
}
```

If the super parameter doesn't correspond to a named parameter from the
super constructor, then change it to be a normal parameter:

```dart
class A {
  A({int? x});
}

class B extends A {
  B({int? y});
}
```

### super_formal_parameter_without_associated_positional

_No associated positional super constructor parameter._

#### Description

The analyzer produces this diagnostic when there's a positional super
parameter in a constructor and the implicitly or explicitly invoked super
constructor doesn't have a positional parameter at the corresponding
index.

Positional super parameters are associated with positional parameters in
the super constructor by their index. That is, the first super parameter
is associated with the first positional parameter in the super
constructor, the second with the second, and so on.

#### Examples

The following code produces this diagnostic because the constructor in `B`
has a positional super parameter, but there's no positional parameter in
the super constructor in `A`:

```dart
class A {
  A({int? x});
}

class B extends A {
  B(super.[!x!]);
}
```

The following code produces this diagnostic because the constructor in `B`
has two positional super parameters, but there's only one positional
parameter in the super constructor in `A`, which means that there's no
corresponding parameter for `y`:

```dart
class A {
  A(int x);
}

class B extends A {
  B(super.x, super.[!y!]);
}
```

#### Common fixes

If the super constructor should have a positional parameter corresponding
to the super parameter, then update the super constructor appropriately:

```dart
class A {
  A(int x, int y);
}

class B extends A {
  B(super.x, super.y);
}
```

If the super constructor is correct, or can't be changed, then convert the
super parameter into a normal parameter:

```dart
class A {
  A(int x);
}

class B extends A {
  B(super.x, int y);
}
```

### super_invocation_not_last

<a id="invalid_super_invocation" aria-hidden="true"></a>_(Previously known as `invalid_super_invocation`)_

_The superconstructor call must be last in an initializer list: '{0}'._

#### Description

The analyzer produces this diagnostic when the initializer list of a
constructor contains an invocation of a constructor in the superclass, but
the invocation isn't the last item in the initializer list.

#### Example

The following code produces this diagnostic because the invocation of the
superclass' constructor isn't the last item in the initializer list:

```dart
class A {
  A(int x);
}

class B extends A {
  B(int x) : [!super!](x), assert(x >= 0);
}
```

#### Common fixes

Move the invocation of the superclass' constructor to the end of the
initializer list:

```dart
class A {
  A(int x);
}

class B extends A {
  B(int x) : assert(x >= 0), super(x);
}
```

### super_in_enum_constructor

_The enum constructor can't have a 'super' initializer._

#### Description

The analyzer produces this diagnostic when the initializer list in a
constructor in an enum contains an invocation of a super constructor.

#### Example

The following code produces this diagnostic because the constructor in
the enum `E` has a super constructor invocation in the initializer list:

```dart
enum E {
  e;

  const E() : [!super!]();
}
```

#### Common fixes

Remove the super constructor invocation:

```dart
enum E {
  e;

  const E();
}
```

### super_in_extension

_The 'super' keyword can't be used in an extension because an extension doesn't
have a superclass._

#### Description

The analyzer produces this diagnostic when a member declared inside an
extension uses the `super` keyword . Extensions aren't classes and don't
have superclasses, so the `super` keyword serves no purpose.

#### Example

The following code produces this diagnostic because `super` can't be used
in an extension:

```dart
extension E on Object {
  String get displayString => [!super!].toString();
}
```

#### Common fixes

Remove the `super` keyword :

```dart
extension E on Object {
  String get displayString => toString();
}
```

### super_in_extension_type

_The 'super' keyword can't be used in an extension type because an extension
type doesn't have a superclass._

#### Description

The analyzer produces this diagnostic when `super` is used in an instance
member of an extension type. Extension types don't have superclasses, so
there's no inherited member that could be invoked.

#### Example

The following code produces this diagnostic because :

```dart
extension type E(String s) {
  void m() {
    [!super!].m();
  }
}
```

#### Common fixes

Replace or remove the `super` invocation:

```dart
extension type E(String s) {
  void m() {
    s.toLowerCase();
  }
}
```

### super_in_invalid_context

_Invalid context for 'super' invocation._

#### Description

The analyzer produces this diagnostic when the keyword `super` is used
outside of an instance method.

#### Example

The following code produces this diagnostic because `super` is used in a
top-level function:

```dart
void f() {
  [!super!].f();
}
```

#### Common fixes

Rewrite the code to not use `super`.

### super_in_redirecting_constructor

_The redirecting constructor can't have a 'super' initializer._

#### Description

The analyzer produces this diagnostic when a constructor that redirects to
another constructor also attempts to invoke a constructor from the
superclass. The superclass constructor will be invoked when the constructor
that the redirecting constructor is redirected to is invoked.

#### Example

The following code produces this diagnostic because the constructor `C.a`
both redirects to `C.b` and invokes a constructor from the superclass:

```dart
class C {
  C.a() : this.b(), [!super()!];
  C.b();
}
```

#### Common fixes

Remove the invocation of the `super` constructor:

```dart
class C {
  C.a() : this.b();
  C.b();
}
```

### switch_case_completes_normally

_The 'case' shouldn't complete normally._

#### Description

The analyzer produces this diagnostic when the statements following a
`case` label in a `switch` statement could fall through to the next `case`
or `default` label.

#### Example

The following code produces this diagnostic because the `case` label with
 a value of zero (`0`) falls through to the `default` statements:

```dart
void f(int a) {
  switch (a) {
    [!case!] 0:
      print(0);
    default:
      return;
  }
}
```

#### Common fixes

Change the flow of control so that the `case` won't fall through. There
are several ways that this can be done, including adding one of the
following at the end of the current list of statements:
- a `return` statement,
- a `throw` expression,
- a `break` statement,
- a `continue`, or
- an invocation of a function or method whose return type is `Never`.

### switch_expression_not_assignable

_Type '{0}' of the switch expression isn't assignable to the type '{1}' of case
expressions._

#### Description

The analyzer produces this diagnostic when the type of the expression in a
`switch` statement isn't assignable to the type of the expressions in the
`case` clauses.

#### Example

The following code produces this diagnostic because the type of `s`
(`String`) isn't assignable to the type of `0` (`int`):

```dart
void f(String s) {
  switch ([!s!]) {
    case 0:
      break;
  }
}
```

#### Common fixes

If the type of the `case` expressions is correct, then change the
expression in the `switch` statement to have the correct type:

```dart
void f(String s) {
  switch (int.parse(s)) {
    case 0:
      break;
  }
}
```

If the type of the `switch` expression is correct, then change the `case`
expressions to have the correct type:

```dart
void f(String s) {
  switch (s) {
    case '0':
      break;
  }
}
```

### tearoff_of_generative_constructor_of_abstract_class

_A generative constructor of an abstract class can't be torn off._

#### Description

The analyzer produces this diagnostic when a generative constructor from an
abstract class is being torn off. This isn't allowed because it isn't valid
to create an instance of an abstract class, which means that there isn't
any valid use for the torn off constructor.

#### Example

The following code produces this diagnostic because the constructor `C.new`
is being torn off and the class `C` is an abstract class:

```dart
abstract class C {
  C();
}

void f() {
  [!C.new!];
}
```

#### Common fixes

Tear off the constructor of a concrete class.

### text_direction_code_point_in_comment

_The Unicode code point 'U+{0}' changes the appearance of text from how it's
interpreted by the compiler._

#### Description

The analyzer produces this diagnostic when it encounters source that
contains text direction Unicode code points. These code points cause
source code in either a string literal or a comment to be interpreted
and compiled differently than how it appears in editors, leading to
possible security vulnerabilities.

#### Example

The following code produces this diagnostic twice because there are
hidden characters at the start and end of the label string:

```dart
var label = '[!I!]nteractive text[!'!];
```

#### Common fixes

If the code points are intended to be included in the string literal,
then escape them:

```dart
var label = '\u202AInteractive text\u202C';
```

If the code points aren't intended to be included in the string literal,
then remove them:

```dart
var label = 'Interactive text';
```

### text_direction_code_point_in_literal

_The Unicode code point 'U+{0}' changes the appearance of text from how it's
interpreted by the compiler._

#### Description

The analyzer produces this diagnostic when it encounters source that
contains text direction Unicode code points. These code points cause
source code in either a string literal or a comment to be interpreted
and compiled differently than how it appears in editors, leading to
possible security vulnerabilities.

#### Example

The following code produces this diagnostic twice because there are
hidden characters at the start and end of the label string:

```dart
var label = '[!I!]nteractive text[!'!];
```

#### Common fixes

If the code points are intended to be included in the string literal,
then escape them:

```dart
var label = '\u202AInteractive text\u202C';
```

If the code points aren't intended to be included in the string literal,
then remove them:

```dart
var label = 'Interactive text';
```

### throw_of_invalid_type

_The type '{0}' of the thrown expression must be assignable to 'Object'._

#### Description

The analyzer produces this diagnostic when the type of the expression in a
throw expression isn't assignable to `Object`. It isn't valid to throw
`null`, so it isn't valid to use an expression that might evaluate to
`null`.

#### Example

The following code produces this diagnostic because `s` might be `null`:

```dart
void f(String? s) {
  throw [!s!];
}
```

#### Common fixes

Add an explicit null-check to the expression:

```dart
void f(String? s) {
  throw s!;
}
```

### top_level_cycle

_The type of '{0}' can't be inferred because it depends on itself through the
cycle: {1}._

#### Description

The analyzer produces this diagnostic when a top-level variable has no type
annotation and the variable's initializer refers to the variable, either
directly or indirectly.

#### Example

The following code produces this diagnostic because the variables `x` and
`y` are defined in terms of each other, and neither has an explicit type,
so the type of the other can't be inferred:

```dart
var x = y;
var y = [!x!];
```

#### Common fixes

If the two variables don't need to refer to each other, then break the
cycle:

```dart
var x = 0;
var y = x;
```

If the two variables need to refer to each other, then give at least one of
them an explicit type:

```dart
int x = y;
var y = x;
```

Note, however, that while this code doesn't produce any diagnostics, it
will produce a stack overflow at runtime unless at least one of the
variables is assigned a value that doesn't depend on the other variables
before any of the variables in the cycle are referenced.

### type_alias_cannot_reference_itself

_Typedefs can't reference themselves directly or recursively via another
typedef._

#### Description

The analyzer produces this diagnostic when a typedef refers to itself,
either directly or indirectly.

#### Example

The following code produces this diagnostic because `F` depends on itself
indirectly through `G`:

```dart
typedef [!F!] = void Function(G);
typedef G = void Function(F);
```

#### Common fixes

Change one or more of the typedefs in the cycle so that none of them refer
to themselves:

```dart
typedef F = void Function(G);
typedef G = void Function(int);
```

### type_annotation_deferred_class

_The deferred type '{0}' can't be used in a declaration, cast, or type test._

#### Description

The analyzer produces this diagnostic when the type annotation is in a
variable declaration, or the type used in a cast (`as`) or type test (`is`)
is a type declared in a library that is imported using a deferred import.
These types are required to be available at compile time, but aren't.

For more information, check out
[Lazily loading a library](https://dart.dev/language/libraries#lazily-loading-a-library).

#### Example

The following code produces this diagnostic because the type of the
parameter `f` is imported from a deferred library:

```dart
import 'dart:io' deferred as io;

void f([!io.File!] f) {}
```

#### Common fixes

If you need to reference the imported type, then remove the `deferred`
keyword:

```dart
import 'dart:io' as io;

void f(io.File f) {}
```

If the import is required to be deferred and there's another type that is
appropriate, then use that type in place of the type from the deferred
library.

### type_argument_not_matching_bounds

_'{0}' doesn't conform to the bound '{2}' of the type parameter '{1}'._

#### Description

The analyzer produces this diagnostic when a type argument isn't the same
as or a subclass of the bounds of the corresponding type parameter.

#### Example

The following code produces this diagnostic because `String` isn't a
subclass of `num`:

```dart
class A<E extends num> {}

var a = A<[!String!]>();
```

#### Common fixes

Change the type argument to be a subclass of the bounds:

```dart
class A<E extends num> {}

var a = A<int>();
```

### type_check_with_null

_Tests for non-null should be done with '!= null'._

_Tests for null should be done with '== null'._

#### Description

The analyzer produces this diagnostic when there's a type check (using the
`as` operator) where the type is `Null`. There's only one value whose type
is `Null`, so the code is both more readable and more performant when it
tests for `null` explicitly.

#### Examples

The following code produces this diagnostic because the code is testing to
see whether the value of `s` is `null` by using a type check:

```dart
void f(String? s) {
  if ([!s is Null!]) {
    return;
  }
  print(s);
}
```

The following code produces this diagnostic because the code is testing to
see whether the value of `s` is something other than `null` by using a type
check:

```dart
void f(String? s) {
  if ([!s is! Null!]) {
    print(s);
  }
}
```

#### Common fixes

Replace the type check with the equivalent comparison with `null`:

```dart
void f(String? s) {
  if (s == null) {
    return;
  }
  print(s);
}
```

### type_parameter_referenced_by_static

_Static members can't reference type parameters of the class._

#### Description

The analyzer produces this diagnostic when a static member references a
type parameter that is declared for the class. Type parameters only have
meaning for instances of the class.

#### Example

The following code produces this diagnostic because the static method
`hasType` has a reference to the type parameter `T`:

```dart
class C<T> {
  static bool hasType(Object o) => o is [!T!];
}
```

#### Common fixes

If the member can be an instance member, then remove the keyword `static`:

```dart
class C<T> {
  bool hasType(Object o) => o is T;
}
```

If the member must be a static member, then make the member be generic:

```dart
class C<T> {
  static bool hasType<S>(Object o) => o is S;
}
```

Note, however, that there isn't a relationship between `T` and `S`, so this
second option changes the semantics from what was likely to be intended.

### type_parameter_supertype_of_its_bound

_'{0}' can't be a supertype of its upper bound._

#### Description

The analyzer produces this diagnostic when the bound of a type parameter
(the type following the `extends` keyword) is either directly or indirectly
the type parameter itself. Stating that the type parameter must be the same
as itself or a subtype of itself or a subtype of itself isn't helpful
because it will always be the same as itself.

#### Examples

The following code produces this diagnostic because the bound of `T` is
`T`:

```dart
class C<[!T!] extends T> {}
```

The following code produces this diagnostic because the bound of `T1` is
`T2`, and the bound of `T2` is `T1`, effectively making the bound of `T1`
be `T1`:

```dart
class C<[!T1!] extends T2, T2 extends T1> {}
```

#### Common fixes

If the type parameter needs to be a subclass of some type, then replace the
bound with the required type:

```dart
class C<T extends num> {}
```

If the type parameter can be any type, then remove the `extends` clause:

```dart
class C<T> {}
```

### type_test_with_non_type

_The name '{0}' isn't a type and can't be used in an 'is' expression._

#### Description

The analyzer produces this diagnostic when the right-hand side of an `is`
or `is!` test isn't a type.

#### Example

The following code produces this diagnostic because the right-hand side is
a parameter, not a type:

```dart
typedef B = int Function(int);

void f(Object a, B b) {
  if (a is [!b!]) {
    return;
  }
}
```

#### Common fixes

If you intended to use a type test, then replace the right-hand side with a
type:

```dart
typedef B = int Function(int);

void f(Object a, B b) {
  if (a is B) {
    return;
  }
}
```

If you intended to use a different kind of test, then change the test:

```dart
typedef B = int Function(int);

void f(Object a, B b) {
  if (a == b) {
    return;
  }
}
```

### type_test_with_undefined_name

_The name '{0}' isn't defined, so it can't be used in an 'is' expression._

#### Description

The analyzer produces this diagnostic when the name following the `is` in a
type test expression isn't defined.

#### Example

The following code produces this diagnostic because the name `Srting` isn't
defined:

```dart
void f(Object o) {
  if (o is [!Srting!]) {
    // ...
  }
}
```

#### Common fixes

Replace the name with the name of a type:

```dart
void f(Object o) {
  if (o is String) {
    // ...
  }
}
```

### unchecked_use_of_nullable_value

_A nullable expression can't be used as a condition._

_A nullable expression can't be used as an iterator in a for-in loop._

_A nullable expression can't be used in a spread._

_A nullable expression can't be used in a yield-each statement._

_The function can't be unconditionally invoked because it can be 'null'._

_The method '{0}' can't be unconditionally invoked because the receiver can be
'null'._

_The operator '{0}' can't be unconditionally invoked because the receiver can be
'null'._

_The property '{0}' can't be unconditionally accessed because the receiver can
be 'null'._

#### Description

The analyzer produces this diagnostic when an expression whose type is
[potentially non-nullable][] is dereferenced without first verifying that
the value isn't `null`.

#### Example

The following code produces this diagnostic because `s` can be `null` at
the point where it's referenced:

```dart
void f(String? s) {
  if (s.[!length!] > 3) {
    // ...
  }
}
```

#### Common fixes

If the value really can be `null`, then add a test to ensure that members
are only accessed when the value isn't `null`:

```dart
void f(String? s) {
  if (s != null && s.length > 3) {
    // ...
  }
}
```

If the expression is a variable and the value should never be `null`, then
change the type of the variable to be non-nullable:

```dart
void f(String s) {
  if (s.length > 3) {
    // ...
  }
}
```

If you believe that the value of the expression should never be `null`, but
you can't change the type of the variable, and you're willing to risk
having an exception thrown at runtime if you're wrong, then you can assert
that the value isn't null:

```dart
void f(String? s) {
  if (s!.length > 3) {
    // ...
  }
}
```

### undefined_annotation

_Undefined name '{0}' used as an annotation._

#### Description

The analyzer produces this diagnostic when a name that isn't defined is
used as an annotation.

#### Example

The following code produces this diagnostic because the name `undefined`
isn't defined:

```dart
[!@undefined!]
void f() {}
```

#### Common fixes

If the name is correct, but it isn't declared yet, then declare the name as
a constant value:

```dart
const undefined = 'undefined';

@undefined
void f() {}
```

If the name is wrong, replace the name with the name of a valid constant:

```dart
@deprecated
void f() {}
```

Otherwise, remove the annotation.

### undefined_class

_Undefined class '{0}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a class but either isn't defined or isn't visible
in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because `Piont` isn't defined:

```dart
class Point {}

void f([!Piont!] p) {}
```

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a class that is defined. The example above can be corrected by
fixing the spelling of the class:

```dart
class Point {}

void f(Point p) {}
```

If the class is defined but isn't visible, then you probably need to add an
import.

### undefined_constructor_in_initializer

_The class '{0}' doesn't have a constructor named '{1}'._

_The class '{0}' doesn't have an unnamed constructor._

#### Description

The analyzer produces this diagnostic when a superclass constructor is
invoked in the initializer list of a constructor, but the superclass
doesn't define the constructor being invoked.

#### Examples

The following code produces this diagnostic because `A` doesn't have an
unnamed constructor:

```dart
class A {
  A.n();
}
class B extends A {
  B() : [!super()!];
}
```

The following code produces this diagnostic because `A` doesn't have a
constructor named `m`:

```dart
class A {
  A.n();
}
class B extends A {
  B() : [!super.m()!];
}
```

#### Common fixes

If the superclass defines a constructor that should be invoked, then change
the constructor being invoked:

```dart
class A {
  A.n();
}
class B extends A {
  B() : super.n();
}
```

If the superclass doesn't define an appropriate constructor, then define
the constructor being invoked:

```dart
class A {
  A.m();
  A.n();
}
class B extends A {
  B() : super.m();
}
```

### undefined_enum_constant

_There's no constant named '{0}' in '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier
that appears to be the name of an enum value, and the name either isn't
defined or isn't visible in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because `E` doesn't define a
constant named `c`:

```dart
enum E {a, b}

var e = E.[!c!];
```

#### Common fixes

If the constant should be defined, then add it to the declaration of the
enum:

```dart
enum E {a, b, c}

var e = E.c;
```

If the constant shouldn't be defined, then change the name to the name of
an existing constant:

```dart
enum E {a, b}

var e = E.b;
```

### undefined_enum_constructor

_The enum doesn't have a constructor named '{0}'._

_The enum doesn't have an unnamed constructor._

#### Description

The analyzer produces this diagnostic when the constructor invoked to
initialize an enum value doesn't exist.

#### Examples

The following code produces this diagnostic because the enum value `c`
is being initialized by the unnamed constructor, but there's no unnamed
constructor defined in `E`:

```dart
enum E {
  [!c!]();

  const E.x();
}
```

The following code produces this diagnostic because the enum value `c` is
being initialized by the constructor named `x`, but there's no constructor
named `x` defined in `E`:

```dart
enum E {
  c.[!x!]();

  const E.y();
}
```

#### Common fixes

If the enum value is being initialized by the unnamed constructor and one
of the named constructors should have been used, then add the name of the
constructor:

```dart
enum E {
  c.x();

  const E.x();
}
```

If the enum value is being initialized by the unnamed constructor and none
of the named constructors are appropriate, then define the unnamed
constructor:

```dart
enum E {
  c();

  const E();
}
```

If the enum value is being initialized by a named constructor and one of
the existing constructors should have been used, then change the name of
the constructor being invoked (or remove it if the unnamed constructor
should be used):

```dart
enum E {
  c.y();

  const E();
  const E.y();
}
```

If the enum value is being initialized by a named constructor and none of
the existing constructors should have been used, then define a constructor
with the name that was used:

```dart
enum E {
  c.x();

  const E.x();
}
```

### undefined_extension_getter

_The getter '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a getter, but the getter isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static getter is
referenced but isn't defined by the specified extension.

#### Examples

The following code produces this diagnostic because the extension `E`
doesn't declare an instance getter named `b`:

```dart
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').[!b!];
}
```

The following code produces this diagnostic because the extension `E`
doesn't declare a static getter named `a`:

```dart
extension E on String {}

var x = E.[!a!];
```

#### Common fixes

If the name of the getter is incorrect, then change it to the name of an
existing getter:

```dart
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').a;
}
```

If the name of the getter is correct but the name of the extension is
wrong, then change the name of the extension to the correct name:

```dart
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  F('c').b;
}
```

If the name of the getter and extension are both correct, but the getter
isn't defined, then define the getter:

```dart
extension E on String {
  String get a => 'a';
  String get b => 'z';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').b;
}
```

### undefined_extension_method

_The method '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a method, but the method isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static method is
referenced but isn't defined by the specified extension.

#### Examples

The following code produces this diagnostic because the extension `E`
doesn't declare an instance method named `b`:

```dart
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').[!b!]();
}
```

The following code produces this diagnostic because the extension `E`
doesn't declare a static method named `a`:

```dart
extension E on String {}

var x = E.[!a!]();
```

#### Common fixes

If the name of the method is incorrect, then change it to the name of an
existing method:

```dart
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').a();
}
```

If the name of the method is correct, but the name of the extension is
wrong, then change the name of the extension to the correct name:

```dart
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  F('c').b();
}
```

If the name of the method and extension are both correct, but the method
isn't defined, then define the method:

```dart
extension E on String {
  String a() => 'a';
  String b() => 'z';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').b();
}
```

### undefined_extension_operator

_The operator '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an operator is invoked on a
specific extension when that extension doesn't implement the operator.

#### Example

The following code produces this diagnostic because the extension `E`
doesn't define the operator `*`:

```dart
var x = E('') [!*!] 4;

extension E on String {}
```

#### Common fixes

If the extension is expected to implement the operator, then add an
implementation of the operator to the extension:

```dart
var x = E('') * 4;

extension E on String {
  int operator *(int multiplier) => length * multiplier;
}
```

If the operator is defined by a different extension, then change the name
of the extension to the name of the one that defines the operator.

If the operator is defined on the argument of the extension override, then
remove the extension override:

```dart
var x = '' * 4;

extension E on String {}
```

### undefined_extension_setter

_The setter '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a setter, but the setter isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static setter is
referenced but isn't defined by the specified extension.

#### Examples

The following code produces this diagnostic because the extension `E`
doesn't declare an instance setter named `b`:

```dart
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').[!b!] = 'd';
}
```

The following code produces this diagnostic because the extension `E`
doesn't declare a static setter named `a`:

```dart
extension E on String {}

void f() {
  E.[!a!] = 3;
}
```

#### Common fixes

If the name of the setter is incorrect, then change it to the name of an
existing setter:

```dart
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').a = 'd';
}
```

If the name of the setter is correct, but the name of the extension is
wrong, then change the name of the extension to the correct name:

```dart
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  F('c').b = 'd';
}
```

If the name of the setter and extension are both correct, but the setter
isn't defined, then define the setter:

```dart
extension E on String {
  set a(String v) {}
  set b(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').b = 'd';
}
```

### undefined_function

_The function '{0}' isn't defined._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a function but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because the name `emty` isn't
defined:

```dart
List<int> empty() => [];

void main() {
  print([!emty!]());
}
```

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a function that is defined. The example above can be corrected
by fixing the spelling of the function:

```dart
List<int> empty() => [];

void main() {
  print(empty());
}
```

If the function is defined but isn't visible, then you probably need to add
an import or re-arrange your code to make the function visible.

### undefined_getter

_The getter '{0}' isn't defined for the '{1}' function type._

_The getter '{0}' isn't defined for the type '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a getter but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because `String` has no member
named `len`:

```dart
int f(String s) => s.[!len!];
```

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a getter that is defined. The example above can be corrected by
fixing the spelling of the getter:

```dart
int f(String s) => s.length;
```

### undefined_hidden_name

_The library '{0}' doesn't export a member with the hidden name '{1}'._

#### Description

The analyzer produces this diagnostic when a hide combinator includes a
name that isn't defined by the library being imported.

#### Example

The following code produces this diagnostic because `dart:math` doesn't
define the name `String`:

```dart
import 'dart:math' hide [!String!], max;

var x = min(0, 1);
```

#### Common fixes

If a different name should be hidden, then correct the name. Otherwise,
remove the name from the list:

```dart
import 'dart:math' hide max;

var x = min(0, 1);
```

### undefined_identifier

_Undefined name '{0}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
either isn't defined or isn't visible in the scope in which it's being
referenced.

#### Example

The following code produces this diagnostic because the name `rihgt` isn't
defined:

```dart
int min(int left, int right) => left <= [!rihgt!] ? left : right;
```

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
an identifier that is defined. The example above can be corrected by
fixing the spelling of the variable:

```dart
int min(int left, int right) => left <= right ? left : right;
```

If the identifier is defined but isn't visible, then you probably need to
add an import or re-arrange your code to make the identifier visible.

### undefined_identifier_await

_Undefined name 'await' in function body not marked with 'async'._

#### Description

The analyzer produces this diagnostic when the name `await` is used in a
method or function body without being declared, and the body isn't marked
with the `async` keyword. The name `await` only introduces an await
expression in an asynchronous function.

#### Example

The following code produces this diagnostic because the name `await` is
used in the body of `f` even though the body of `f` isn't marked with the
`async` keyword:

```dart
void f(p) { [!await!] p; }
```

#### Common fixes

Add the keyword `async` to the function body:

```dart
void f(p) async { await p; }
```

### undefined_method

_The method '{0}' isn't defined for the '{1}' function type._

_The method '{0}' isn't defined for the type '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a method but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because the identifier
`removeMiddle` isn't defined:

```dart
int f(List<int> l) => l.[!removeMiddle!]();
```

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a method that is defined. The example above can be corrected by
fixing the spelling of the method:

```dart
int f(List<int> l) => l.removeLast();
```

### undefined_named_parameter

_The named parameter '{0}' isn't defined._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has a named argument, but the method or function being invoked doesn't
define a parameter with the same name.

#### Example

The following code produces this diagnostic because `m` doesn't declare a
named parameter named `a`:

```dart
class C {
  m({int? b}) {}
}

void f(C c) {
  c.m([!a!]: 1);
}
```

#### Common fixes

If the argument name is mistyped, then replace it with the correct name.
The example above can be fixed by changing `a` to `b`:

```dart
class C {
  m({int? b}) {}
}

void f(C c) {
  c.m(b: 1);
}
```

If a subclass adds a parameter with the name in question, then cast the
receiver to the subclass:

```dart
class C {
  m({int? b}) {}
}

class D extends C {
  m({int? a, int? b}) {}
}

void f(C c) {
  (c as D).m(a: 1);
}
```

If the parameter should be added to the function, then add it:

```dart
class C {
  m({int? a, int? b}) {}
}

void f(C c) {
  c.m(a: 1);
}
```

### undefined_operator

_The operator '{0}' isn't defined for the type '{1}'._

#### Description

The analyzer produces this diagnostic when a user-definable operator is
invoked on an object for which the operator isn't defined.

#### Example

The following code produces this diagnostic because the class `C` doesn't
define the operator `+`:

```dart
class C {}

C f(C c) => c [!+!] 2;
```

#### Common fixes

If the operator should be defined for the class, then define it:

```dart
class C {
  C operator +(int i) => this;
}

C f(C c) => c + 2;
```

### undefined_prefixed_name

_The name '{0}' is being referenced through the prefix '{1}', but it isn't
defined in any of the libraries imported using that prefix._

#### Description

The analyzer produces this diagnostic when a prefixed identifier is found
where the prefix is valid, but the identifier isn't declared in any of the
libraries imported using that prefix.

#### Example

The following code produces this diagnostic because `dart:core` doesn't
define anything named `a`:

```dart
import 'dart:core' as p;

void f() {
  p.[!a!];
}
```

#### Common fixes

If the library in which the name is declared isn't imported yet, add an
import for the library.

If the name is wrong, then change it to one of the names that's declared in
the imported libraries.

### undefined_referenced_parameter

_The parameter '{0}' isn't defined by '{1}'._

#### Description

The analyzer produces this diagnostic when an annotation of the form
[`UseResult.unless(parameterDefined: parameterName)`][meta-UseResult]
specifies a parameter name that isn't defined by the annotated function.

#### Example

The following code produces this diagnostic because the function `f`
doesn't have a parameter named `b`:

```dart
import 'package:meta/meta.dart';

@UseResult.unless(parameterDefined: [!'b'!])
int f([int? a]) => a ?? 0;
```

#### Common fixes

Change the argument named `parameterDefined` to match the name of one of
the parameters to the function:

```dart
import 'package:meta/meta.dart';

@UseResult.unless(parameterDefined: 'a')
int f([int? a]) => a ?? 0;
```

### undefined_setter

_The setter '{0}' isn't defined for the '{1}' function type._

_The setter '{0}' isn't defined for the type '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a setter but either isn't defined or isn't
visible in the scope in which the identifier is being referenced.

#### Example

The following code produces this diagnostic because there isn't a setter
named `z`:

```dart
class C {
  int x = 0;
  void m(int y) {
    this.[!z!] = y;
  }
}
```

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a setter that is defined. The example above can be corrected by
fixing the spelling of the setter:

```dart
class C {
  int x = 0;
  void m(int y) {
    this.x = y;
  }
}
```

### undefined_shown_name

_The library '{0}' doesn't export a member with the shown name '{1}'._

#### Description

The analyzer produces this diagnostic when a show combinator includes a
name that isn't defined by the library being imported.

#### Example

The following code produces this diagnostic because `dart:math` doesn't
define the name `String`:

```dart
import 'dart:math' show min, [!String!];

var x = min(0, 1);
```

#### Common fixes

If a different name should be shown, then correct the name. Otherwise,
remove the name from the list:

```dart
import 'dart:math' show min;

var x = min(0, 1);
```

### undefined_super_member

<a id="undefined_super_method" aria-hidden="true"></a>_(Previously known as `undefined_super_method`)_

_The getter '{0}' isn't defined in a superclass of '{1}'._

_The method '{0}' isn't defined in a superclass of '{1}'._

_The operator '{0}' isn't defined in a superclass of '{1}'._

_The setter '{0}' isn't defined in a superclass of '{1}'._

#### Description

The analyzer produces this diagnostic when an inherited member (method,
getter, setter, or operator) is referenced using `super`, but there's no
member with that name in the superclass chain.

#### Examples

The following code produces this diagnostic because `Object` doesn't define
a method named `n`:

```dart
class C {
  void m() {
    super.[!n!]();
  }
}
```

The following code produces this diagnostic because `Object` doesn't define
a getter named `g`:

```dart
class C {
  void m() {
    super.[!g!];
  }
}
```

#### Common fixes

If the inherited member you intend to invoke has a different name, then
make the name of the invoked member match the inherited member.

If the member you intend to invoke is defined in the same class, then
remove the `super.`.

If the member isn't defined, then either add the member to one of the
superclasses or remove the invocation.

### unknown_platform

_The platform '{0}' is not a recognized platform._

#### Description

The analyzer produces this diagnostic when an unknown platform name is
used as a key in the `platforms` map.
To learn more about specifying your package's supported platforms,
check out the [documentation on platform declarations](https://dart.dev/tools/pub/pubspec#platforms).

#### Example

The following `pubspec.yaml` produces this diagnostic because the platform
`browser` is unknown.

```yaml
name: example
platforms:
  [!browser:!]
```

#### Common fixes

If you can rely on automatic platform detection, then omit the
top-level `platforms` key.

```yaml
name: example
```

If you need to manually specify the list of supported platforms, then
write the `platforms` field as a map with known platform names as keys.

```yaml
name: example
platforms:
  # These are the known platforms
  android:
  ios:
  linux:
  macos:
  web:
  windows:
```

### unnecessary_cast

_Unnecessary cast._

#### Description

The analyzer produces this diagnostic when the value being cast is already
known to be of the type that it's being cast to.

#### Example

The following code produces this diagnostic because `n` is already known to
be an `int` as a result of the `is` test:

```dart
void f(num n) {
  if (n is int) {
    ([!n as int!]).isEven;
  }
}
```

#### Common fixes

Remove the unnecessary cast:

```dart
void f(num n) {
  if (n is int) {
    n.isEven;
  }
}
```

### unnecessary_dev_dependency

_The dev dependency on {0} is unnecessary because there is also a normal
dependency on that package._

#### Description

The analyzer produces this diagnostic when there's an entry under
`dev_dependencies` for a package that is also listed under `dependencies`.
The packages under `dependencies` are available to all of the code in the
package, so there's no need to also list them under `dev_dependencies`.

#### Example

The following code produces this diagnostic because the package `meta` is
listed under both `dependencies` and `dev_dependencies`:

```yaml
name: example
dependencies:
  meta: ^1.0.2
dev_dependencies:
  [!meta!]: ^1.0.2
```

#### Common fixes

Remove the entry under `dev_dependencies` (and the `dev_dependencies` key
if that's the only package listed there):

```yaml
name: example
dependencies:
  meta: ^1.0.2
```

### unnecessary_final

_The keyword 'final' isn't necessary because the parameter is implicitly
'final'._

#### Description

The analyzer produces this diagnostic when either a field initializing
parameter or a super parameter in a constructor has the keyword `final`.
In both cases the keyword is unnecessary because the parameter is
implicitly `final`.

#### Examples

The following code produces this diagnostic because the field initializing
parameter has the keyword `final`:

```dart
class A {
  int value;

  A([!final!] this.value);
}
```

The following code produces this diagnostic because the super parameter in
`B` has the keyword `final`:

```dart
class A {
  A(int value);
}

class B extends A {
  B([!final!] super.value);
}
```

#### Common fixes

Remove the unnecessary `final` keyword:

```dart
class A {
  A(int value);
}

class B extends A {
  B(super.value);
}
```

### unnecessary_import

_The import of '{0}' is unnecessary because all of the used elements are also
provided by the import of '{1}'._

#### Description

The analyzer produces this diagnostic when an import isn't needed because
all of the names that are imported and referenced within the importing
library are also visible through another import.

#### Example

Given a file `a.dart` that contains the following:

```dart
class A {}
```

And, given a file `b.dart` that contains the following:

```dart
export 'a.dart';

class B {}
```

The following code produces this diagnostic because the class `A`, which is
imported from `a.dart`, is also imported from `b.dart`. Removing the import
of `a.dart` leaves the semantics unchanged:

```dart
import [!'a.dart'!];
import 'b.dart';

void f(A a, B b) {}
```

#### Common fixes

If the import isn't needed, then remove it.

If some of the names imported by this import are intended to be used but
aren't yet, and if those names aren't imported by other imports, then add
the missing references to those names.

### unnecessary_nan_comparison

_A double can't equal 'double.nan', so the condition is always 'false'._

_A double can't equal 'double.nan', so the condition is always 'true'._

#### Description

The analyzer produces this diagnostic when a value is compared to
`double.nan` using either `==` or `!=`.

Dart follows the [IEEE 754] floating-point standard for the semantics of
floating point operations, which states that, for any floating point value
`x` (including NaN, positive infinity, and negative infinity),
- `NaN == x` is always false
- `NaN != x` is always true

As a result, comparing any value to NaN is pointless because the result is
already known (based on the comparison operator being used).

#### Example

The following code produces this diagnostic because `d` is being compared
to `double.nan`:

```dart
bool isNaN(double d) => d [!== double.nan!];
```

#### Common fixes

Use the getter `double.isNaN` instead:

```dart
bool isNaN(double d) => d.isNaN;
```

### unnecessary_non_null_assertion

_The '!' will have no effect because the receiver can't be null._

#### Description

The analyzer produces this diagnostic when the operand of the `!` operator
can't be `null`.

#### Example

The following code produces this diagnostic because `x` can't be `null`:

```dart
int f(int x) {
  return x[!!!];
}
```

#### Common fixes

Remove the null check operator (`!`):

```dart
int f(int x) {
  return x;
}
```

### unnecessary_no_such_method

_Unnecessary 'noSuchMethod' declaration._

#### Description

The analyzer produces this diagnostic when there's a declaration of
`noSuchMethod`, the only thing the declaration does is invoke the
overridden declaration, and the overridden declaration isn't the
declaration in `Object`.

Overriding the implementation of `Object`'s `noSuchMethod` (no matter what
the implementation does) signals to the analyzer that it shouldn't flag any
inherited abstract methods that aren't implemented in that class. This
works even if the overriding implementation is inherited from a superclass,
so there's no value to declare it again in a subclass.

#### Example

The following code produces this diagnostic because the declaration of
`noSuchMethod` in `A` makes the declaration of `noSuchMethod` in `B`
unnecessary:

```dart
class A {
  @override
  dynamic noSuchMethod(x) => super.noSuchMethod(x);
}
class B extends A {
  @override
  dynamic [!noSuchMethod!](y) {
    return super.noSuchMethod(y);
  }
}
```

#### Common fixes

Remove the unnecessary declaration:

```dart
class A {
  @override
  dynamic noSuchMethod(x) => super.noSuchMethod(x);
}
class B extends A {}
```

### unnecessary_null_assert_pattern

_The null-assert pattern will have no effect because the matched type isn't
nullable._

#### Description

The analyzer produces this diagnostic when a null-assert pattern is used
to match a value that isn't nullable.

#### Example

The following code produces this diagnostic because the variable `x` isn't
nullable:

```dart
void f(int x) {
  if (x case var a[!!!] when a > 0) {}
}
```

#### Common fixes

Remove the null-assert pattern:

```dart
void f(int x) {
  if (x case var a when a > 0) {}
}
```

### unnecessary_null_check_pattern

_The null-check pattern will have no effect because the matched type isn't
nullable._

#### Description

The analyzer produces this diagnostic when a null-check pattern is used to
match a value that isn't nullable.

#### Example

The following code produces this diagnostic because the value `x` isn't
nullable:

```dart
void f(int x) {
  if (x case var a[!?!] when a > 0) {}
}
```

#### Common fixes

Remove the null-check pattern:

```dart
void f(int x) {
  if (x case var a when a > 0) {}
}
```

### unnecessary_null_comparison

_The operand can't be null, so the condition is always 'false'._

_The operand can't be null, so the condition is always 'true'._

#### Description

The analyzer produces this diagnostic when it finds an equality comparison
(either `==` or `!=`) with one operand of `null` and the other operand
can't be `null`. Such comparisons are always either `true` or `false`, so
they serve no purpose.

#### Examples

The following code produces this diagnostic because `x` can never be
`null`, so the comparison always evaluates to `true`:

```dart
void f(int x) {
  if (x [!!= null!]) {
    print(x);
  }
}
```

The following code produces this diagnostic because `x` can never be
`null`, so the comparison always evaluates to `false`:

```dart
void f(int x) {
  if (x [!== null!]) {
    throw ArgumentError("x can't be null");
  }
}
```

#### Common fixes

If the other operand should be able to be `null`, then change the type of
the operand:

```dart
void f(int? x) {
  if (x != null) {
    print(x);
  }
}
```

If the other operand really can't be `null`, then remove the condition:

```dart
void f(int x) {
  print(x);
}
```

### unnecessary_question_mark

_The '?' is unnecessary because '{0}' is nullable without it._

#### Description

The analyzer produces this diagnostic when either the type `dynamic` or the
type `Null` is followed by a question mark. Both of these types are
inherently nullable so the question mark doesn't change the semantics.

#### Example

The following code produces this diagnostic because the question mark
following `dynamic` isn't necessary:

```dart
dynamic[!?!] x;
```

#### Common fixes

Remove the unneeded question mark:

```dart
dynamic x;
```

### unnecessary_set_literal

_Braces unnecessarily wrap this expression in a set literal._

#### Description

The analyzer produces this diagnostic when a function that has a return
type of `void`, `Future<void>`, or `FutureOr<void>` uses an expression
function body (`=>`) and the returned value is a literal set containing a
single element.

Although the language allows it, returning a value from a `void` function
isn't useful because it can't be used at the call site. In this particular
case the return is often due to a misunderstanding about the syntax. The
braces aren't necessary and can be removed.

#### Example

The following code produces this diagnostic because the closure being
passed to `g` has a return type of `void`, but is returning a set:

```dart
void f() {
  g(() => [!{1}!]);
}

void g(void Function() p) {}
```

#### Common fixes

Remove the braces from around the value:

```dart
void f() {
  g(() => 1);
}

void g(void Function() p) {}
```

### unnecessary_type_check

_Unnecessary type check; the result is always 'false'._

_Unnecessary type check; the result is always 'true'._

#### Description

The analyzer produces this diagnostic when the value of a type check (using
either `is` or `is!`) is known at compile time.

#### Example

The following code produces this diagnostic because the test `a is Object?`
is always `true`:

```dart
bool f<T>(T a) => [!a is Object?!];
```

#### Common fixes

If the type check doesn't check what you intended to check, then change the
test:

```dart
bool f<T>(T a) => a is Object;
```

If the type check does check what you intended to check, then replace the
type check with its known value or completely remove it:

```dart
bool f<T>(T a) => true;
```

### unqualified_reference_to_non_local_static_member

_Static members from supertypes must be qualified by the name of the defining
type._

#### Description

The analyzer produces this diagnostic when code in one class references a
static member in a superclass without prefixing the member's name with the
name of the superclass. Static members can only be referenced without a
prefix in the class in which they're declared.

#### Example

The following code produces this diagnostic because the static field `x` is
referenced in the getter `g` without prefixing it with the name of the
defining class:

```dart
class A {
  static int x = 3;
}

class B extends A {
  int get g => [!x!];
}
```

#### Common fixes

Prefix the name of the static member with the name of the declaring class:

```dart
class A {
  static int x = 3;
}

class B extends A {
  int get g => A.x;
}
```

### unqualified_reference_to_static_member_of_extended_type

_Static members from the extended type or one of its superclasses must be
qualified by the name of the defining type._

#### Description

The analyzer produces this diagnostic when an undefined name is found, and
the name is the same as a static member of the extended type or one of its
superclasses.

#### Example

The following code produces this diagnostic because `m` is a static member
of the extended type `C`:

```dart
class C {
  static void m() {}
}

extension E on C {
  void f() {
    [!m!]();
  }
}
```

#### Common fixes

If you're trying to reference a static member that's declared outside the
extension, then add the name of the class or extension before the reference
to the member:

```dart
class C {
  static void m() {}
}

extension E on C {
  void f() {
    C.m();
  }
}
```

If you're referencing a member that isn't declared yet, add a declaration:

```dart
class C {
  static void m() {}
}

extension E on C {
  void f() {
    m();
  }

  void m() {}
}
```

### unreachable_switch_case

_This case is covered by the previous cases._

#### Description

The analyzer produces this diagnostic when a `case` clause in a `switch`
statement doesn't match anything because all of the matchable values are
matched by an earlier `case` clause.

#### Example

The following code produces this diagnostic because the value `1` was
matched in the preceding case:

```dart
void f(int x) {
  switch (x) {
    case 1:
      print('one');
    [!case!] 1:
      print('two');
  }
}
```

#### Common fixes

Change one or both of the conflicting cases to match different values:

```dart
void f(int x) {
  switch (x) {
    case 1:
      print('one');
    case 2:
      print('two');
  }
}
```

### unused_catch_clause

_The exception variable '{0}' isn't used, so the 'catch' clause can be removed._

#### Description

The analyzer produces this diagnostic when a `catch` clause is found, and
neither the exception parameter nor the optional stack trace parameter are
used in the `catch` block.

#### Example

The following code produces this diagnostic because `e` isn't referenced:

```dart
void f() {
  try {
    int.parse(';');
  } on FormatException catch ([!e!]) {
    // ignored
  }
}
```

#### Common fixes

Remove the unused `catch` clause:

```dart
void f() {
  try {
    int.parse(';');
  } on FormatException {
    // ignored
  }
}
```

### unused_catch_stack

_The stack trace variable '{0}' isn't used and can be removed._

#### Description

The analyzer produces this diagnostic when the stack trace parameter in a
`catch` clause isn't referenced within the body of the `catch` block.

#### Example

The following code produces this diagnostic because `stackTrace` isn't
referenced:

```dart
void f() {
  try {
    // ...
  } catch (exception, [!stackTrace!]) {
    // ...
  }
}
```

#### Common fixes

If you need to reference the stack trace parameter, then add a reference to
it. Otherwise, remove it:

```dart
void f() {
  try {
    // ...
  } catch (exception) {
    // ...
  }
}
```

### unused_element

_A value for optional parameter '{0}' isn't ever given._

_The declaration '{0}' isn't referenced._

#### Description

The analyzer produces this diagnostic when a private declaration isn't
referenced in the library that contains the declaration. The following
kinds of declarations are analyzed:
- Private top-level declarations and all of their members
- Private members of public declarations
- Optional parameters of private functions for which a value is never
  passed

Not all references to an element will mark it as "used":
- Assigning a value to a top-level variable (with a standard `=`
  assignment, or a null-aware `??=` assignment) does not count as using
  it.
- Referring to an element in a doc comment reference does not count as
  using it.
- Referring to a class, mixin, or enum on the right side of an `is`
  expression does not count as using it.

#### Example

Assuming that no code in the library references `_C`, the following code
produces this diagnostic:

```dart
class [!_C!] {}
```

Assuming that no code in the library passes a value for `y` in any
invocation of `_m`, the following code produces this diagnostic:

```dart
class C {
  void _m(int x, [int? [!y!]]) {}

  void n() => _m(0);
}
```

#### Common fixes

If the declaration isn't needed, then remove it:

```dart
class C {
  void _m(int x) {}

  void n() => _m(0);
}
```

If the declaration is intended to be used, then add the code to use it.

### unused_field

_The value of the field '{0}' isn't used._

#### Description

The analyzer produces this diagnostic when a private field is declared but
never read, even if it's written in one or more places.

#### Example

The following code produces this diagnostic because the field
`_originalValue` isn't read anywhere in the library:

```dart
class C {
  final String [!_originalValue!];
  final String _currentValue;

  C(this._originalValue) : _currentValue = _originalValue;

  String get value => _currentValue;
}
```

It might appear that the field `_originalValue` is being read in the
initializer (`_currentValue = _originalValue`), but that is actually a
reference to the parameter of the same name, not a reference to the field.

#### Common fixes

If the field isn't needed, then remove it.

If the field was intended to be used, then add the missing code.

### unused_import

_Unused import: '{0}'._

#### Description

The analyzer produces this diagnostic when an import isn't needed because
none of the names that are imported are referenced within the importing
library.

#### Example

The following code produces this diagnostic because nothing defined in
`dart:async` is referenced in the library:

```dart
import [!'dart:async'!];

void main() {}
```

#### Common fixes

If the import isn't needed, then remove it.

If some of the imported names are intended to be used, then add the missing
code.

### unused_label

_The label '{0}' isn't used._

#### Description

The analyzer produces this diagnostic when a label that isn't used is
found.

#### Example

The following code produces this diagnostic because the label `loop` isn't
referenced anywhere in the method:

```dart
void f(int limit) {
  [!loop:!] for (int i = 0; i < limit; i++) {
    print(i);
  }
}
```

#### Common fixes

If the label isn't needed, then remove it:

```dart
void f(int limit) {
  for (int i = 0; i < limit; i++) {
    print(i);
  }
}
```

If the label is needed, then use it:

```dart
void f(int limit) {
  loop: for (int i = 0; i < limit; i++) {
    print(i);
    if (i != 0) {
      break loop;
    }
  }
}
```

### unused_local_variable

_The value of the local variable '{0}' isn't used._

#### Description

The analyzer produces this diagnostic when a local variable is declared but
never read, even if it's written in one or more places.

#### Example

The following code produces this diagnostic because the value of `count` is
never read:

```dart
void main() {
  int [!count!] = 0;
}
```

#### Common fixes

If the variable isn't needed, then remove it.

If the variable was intended to be used, then add the missing code.

### unused_result

_'{0}' should be used. {1}._

_The value of '{0}' should be used._

#### Description

The analyzer produces this diagnostic when a function annotated with
[`useResult`][meta-useResult] is invoked, and the value returned by that
function isn't used. The value is considered to be used if a member of the
value is invoked, if the value is passed to another function, or if the
value is assigned to a variable or field.

#### Example

The following code produces this diagnostic because the invocation of
`c.a()` isn't used, even though the method `a` is annotated with
[`useResult`][meta-useResult]:

```dart
import 'package:meta/meta.dart';

class C {
  @useResult
  int a() => 0;

  int b() => 0;
}

void f(C c) {
  c.[!a!]();
}
```

#### Common fixes

If you intended to invoke the annotated function, then use the value that
was returned:

```dart
import 'package:meta/meta.dart';

class C {
  @useResult
  int a() => 0;

  int b() => 0;
}

void f(C c) {
  print(c.a());
}
```

If you intended to invoke a different function, then correct the name of
the function being invoked:

```dart
import 'package:meta/meta.dart';

class C {
  @useResult
  int a() => 0;

  int b() => 0;
}

void f(C c) {
  c.b();
}
```

### unused_shown_name

_The name {0} is shown, but isn't used._

#### Description

The analyzer produces this diagnostic when a show combinator includes a
name that isn't used within the library. Because it isn't referenced, the
name can be removed.

#### Example

The following code produces this diagnostic because the function `max`
isn't used:

```dart
import 'dart:math' show min, [!max!];

var x = min(0, 1);
```

#### Common fixes

Either use the name or remove it:

```dart
import 'dart:math' show min;

var x = min(0, 1);
```

### uri_does_not_exist

_Target of URI doesn't exist: '{0}'._

#### Description

The analyzer produces this diagnostic when an import, export, or part
directive is found where the URI refers to a file that doesn't exist.

#### Example

If the file `lib.dart` doesn't exist, the following code produces this
diagnostic:

```dart
import [!'lib.dart'!];
```

#### Common fixes

If the URI was mistyped or invalid, then correct the URI.

If the URI is correct, then create the file.

### uri_has_not_been_generated

_Target of URI hasn't been generated: '{0}'._

#### Description

The analyzer produces this diagnostic when an import, export, or part
directive is found where the URI refers to a file that doesn't exist and
the name of the file ends with a pattern that's commonly produced by code
generators, such as one of the following:
- `.g.dart`
- `.pb.dart`
- `.pbenum.dart`
- `.pbserver.dart`
- `.pbjson.dart`
- `.template.dart`

#### Example

If the file `lib.g.dart` doesn't exist, the following code produces this
diagnostic:

```dart
import [!'lib.g.dart'!];
```

#### Common fixes

If the file is a generated file, then run the generator that generates the
file.

If the file isn't a generated file, then check the spelling of the URI or
create the file.

### uri_with_interpolation

_URIs can't use string interpolation._

#### Description

The analyzer produces this diagnostic when the string literal in an
`import`, `export`, or `part` directive contains an interpolation. The
resolution of the URIs in directives must happen before the declarations
are compiled, so expressions can't be  evaluated  while determining the
values of the URIs.

#### Example

The following code produces this diagnostic because the string in the
`import` directive contains an interpolation:

```dart
import [!'dart:$m'!];

const m = 'math';
```

#### Common fixes

Remove the interpolation from the URI:

```dart
import 'dart:math';

var zero = min(0, 0);
```

### use_of_native_extension

_Dart native extensions are deprecated and aren't available in Dart 2.15._

#### Description

The analyzer produces this diagnostic when a library is imported using the
`dart-ext` scheme.

#### Example

The following code produces this diagnostic because the native library `x`
is being imported using a scheme of `dart-ext`:

```dart
import [!'dart-ext:x'!];
```

#### Common fixes

Rewrite the code to use `dart:ffi` as a way of invoking the contents of the
native library.

### use_of_void_result

_This expression has a type of 'void' so its value can't be used._

#### Description

The analyzer produces this diagnostic when it finds an expression whose
type is `void`, and the expression is used in a place where a value is
expected, such as before a member access or on the right-hand side of an
assignment.

#### Example

The following code produces this diagnostic because `f` doesn't produce an
object on which `toString` can be invoked:

```dart
void f() {}

void g() {
  [!f()!].toString();
}
```

#### Common fixes

Either rewrite the code so that the expression has a value or rewrite the
code so that it doesn't depend on the value.

### values_declaration_in_enum

_A member named 'values' can't be declared in an enum._

#### Description

The analyzer produces this diagnostic when an enum declaration defines a
member named `values`, whether the member is an enum value, an instance
member, or a static member.

Any such member conflicts with the implicit declaration of the static
getter named `values` that returns a list containing all the enum
constants.

#### Example

The following code produces this diagnostic because the enum `E` defines
an instance member named `values`:

```dart
enum E {
  v;
  void [!values!]() {}
}
```

#### Common fixes

Change the name of the conflicting member:

```dart
enum E {
  v;
  void getValues() {}
}
```

### variable_pattern_keyword_in_declaration_context

_Variable patterns in declaration context can't specify 'var' or 'final'
keyword._

#### Description

The analyzer produces this diagnostic when a variable pattern is used
within a declaration context.

#### Example

The following code produces this diagnostic because the variable patterns
in the record pattern are in a declaration context:

```dart
void f((int, int) r) {
  var ([!var!] x, y) = r;
  print(x + y);
}
```

#### Common fixes

Remove the `var` or `final` keyword(s) within the variable pattern:

```dart
void f((int, int) r) {
  var (x, y) = r;
  print(x + y);
}
```

### variable_type_mismatch

_A value of type '{0}' can't be assigned to a const variable of type '{1}'._

#### Description

The analyzer produces this diagnostic when the evaluation of a constant
expression would result in a `CastException`.

#### Example

The following code produces this diagnostic because the value of `x` is an
`int`, which can't be assigned to `y` because an `int` isn't a `String`:

```dart
const dynamic x = 0;
const String y = [!x!];
```

#### Common fixes

If the declaration of the constant is correct, then change the value being
assigned to be of the correct type:

```dart
const dynamic x = 0;
const String y = '$x';
```

If the assigned value is correct, then change the declaration to have the
correct type:

```dart
const int x = 0;
const int y = x;
```

### workspace_field_not_list

_The value of the 'workspace' field is required to be a list of relative file
paths._

#### Description

The analyzer produces this diagnostic when the value of the `workspace` key
isn't a list.

#### Example

The following code produces this diagnostic because the value of the
`workspace` key is a string when a list is expected:

```yaml
name: example
workspace: [!notPaths!]
```

#### Common fixes

Change the value of the workspace field so that it's a list:

```yaml
name: example
workspace:
    - pkg/package_1
    - pkg/package_2
```

### workspace_value_not_string

_Workspace entries are required to be directory paths (strings)._

#### Description

The analyzer produces this diagnostic when a `workspace` list contains a
value that isn't a string.

#### Example

The following code produces this diagnostic because the `workspace` list
contains a map:

```yaml
name: example
workspace:
    - [!image.gif: true!]
```

#### Common fixes

Change the `workspace` list so that it only contains valid POSIX-style directory
paths:

```yaml
name: example
workspace:
    - pkg/package_1
    - pkg/package_2
```

### workspace_value_not_subdirectory

_Workspace values must be a relative path of a subdirectory of '{0}'._

#### Description

The analyzer produces this diagnostic when a `workspace` list contains a
value that is not a subdirectory of the directory containing the `pubspec.yaml`` file.

#### Example

The following code produces this diagnostic because the value in the `workspace` list is not a
relative path of a subdirectory of the directory containing the 'pubspec.yaml' file:

```yaml
name: example
workspace:
    - /home/my_package
```

#### Common fixes

Change the `workspace` list so that it only contains only subdirectory paths.

```yaml
name: example
workspace:
    - pkg/package_1
    - pkg/package_2
```

### wrong_number_of_parameters_for_operator

_Operator '-' should declare 0 or 1 parameter, but {0} found._

_Operator '{0}' should declare exactly {1} parameters, but {2} found._

#### Description

The analyzer produces this diagnostic when a declaration of an operator has
the wrong number of parameters.

#### Example

The following code produces this diagnostic because the operator `+` must
have a single parameter corresponding to the right operand:

```dart
class C {
  int operator [!+!](a, b) => 0;
}
```

#### Common fixes

Add or remove parameters to match the required number:

```dart
class C {
  int operator +(a) => 0;
}
```

### wrong_number_of_parameters_for_setter

_Setters must declare exactly one required positional parameter._

#### Description

The analyzer produces this diagnostic when a setter is found that doesn't
declare exactly one required positional parameter.

#### Examples

The following code produces this diagnostic because the setter `s` declares
two required parameters:

```dart
class C {
  set [!s!](int x, int y) {}
}
```

The following code produces this diagnostic because the setter `s` declares
one optional parameter:

```dart
class C {
  set [!s!]([int? x]) {}
}
```

#### Common fixes

Change the declaration so that there's exactly one required positional
parameter:

```dart
class C {
  set s(int x) {}
}
```

### wrong_number_of_type_arguments

_The type '{0}' is declared with {1} type parameters, but {2} type arguments
were given._

#### Description

The analyzer produces this diagnostic when a type that has type parameters
is used and type arguments are provided, but the number of type arguments
isn't the same as the number of type parameters.

The analyzer also produces this diagnostic when a constructor is invoked
and the number of type arguments doesn't match the number of type
parameters declared for the class.

#### Examples

The following code produces this diagnostic because `C` has one type
parameter but two type arguments are provided when it is used as a type
annotation:

```dart
class C<E> {}

void f([!C<int, int>!] x) {}
```

The following code produces this diagnostic because `C` declares one type
parameter, but two type arguments are provided when creating an instance:

```dart
class C<E> {}

var c = [!C<int, int>!]();
```

#### Common fixes

Add or remove type arguments, as necessary, to match the number of type
parameters defined for the type:

```dart
class C<E> {}

void f(C<int> x) {}
```

### wrong_number_of_type_arguments_constructor

_The constructor '{0}.{1}' doesn't have type parameters._

#### Description

The analyzer produces this diagnostic when type arguments are provided
after the name of a named constructor. Constructors can't declare type
parameters, so invocations can only provide the type arguments associated
with the class, and those type arguments are required to follow the name of
the class rather than the name of the constructor.

#### Example

The following code produces this diagnostic because the type parameters
(`<String>`) follow the name of the constructor rather than the name of the
class:

```dart
class C<T> {
  C.named();
}
C f() => C.named[!<String>!]();
```

#### Common fixes

If the type arguments are for the class' type parameters, then move the
type arguments to follow the class name:

```dart
class C<T> {
  C.named();
}
C f() => C<String>.named();
```

If the type arguments aren't for the class' type parameters, then remove
them:

```dart
class C<T> {
  C.named();
}
C f() => C.named();
```

### wrong_number_of_type_arguments_enum

_The enum is declared with {0} type parameters, but {1} type arguments were
given._

#### Description

The analyzer produces this diagnostic when an enum value in an enum that
has type parameters is instantiated and type arguments are provided, but
the number of type arguments isn't the same as the number of type
parameters.

#### Example

The following code produces this diagnostic because the enum value `c`
provides one type argument even though the enum `E` is declared to have
two type parameters:

```dart
enum E<T, U> {
  c[!<int>!]()
}
```

#### Common fixes

If the number of type parameters is correct, then change the number of
type arguments to match the number of type parameters:

```dart
enum E<T, U> {
  c<int, String>()
}
```

If the number of type arguments is correct, then change the number of type
parameters to match the number of type arguments:

```dart
enum E<T> {
  c<int>()
}
```

### wrong_number_of_type_arguments_extension

_The extension '{0}' is declared with {1} type parameters, but {2} type
arguments were given._

#### Description

The analyzer produces this diagnostic when an extension that has type
parameters is used and type arguments are provided, but the number of type
arguments isn't the same as the number of type parameters.

#### Example

The following code produces this diagnostic because the extension `E` is
declared to have a single type parameter (`T`), but the extension override
has two type arguments:

```dart
extension E<T> on List<T> {
  int get len => length;
}

void f(List<int> p) {
  E[!<int, String>!](p).len;
}
```

#### Common fixes

Change the type arguments so that there are the same number of type
arguments as there are type parameters:

```dart
extension E<T> on List<T> {
  int get len => length;
}

void f(List<int> p) {
  E<int>(p).len;
}
```

### wrong_number_of_type_arguments_method

_The method '{0}' is declared with {1} type parameters, but {2} type arguments
are given._

#### Description

The analyzer produces this diagnostic when a method or function is invoked
with a different number of type arguments than the number of type
parameters specified in its declaration. There must either be no type
arguments or the number of arguments must match the number of parameters.

#### Example

The following code produces this diagnostic because the invocation of the
method `m` has two type arguments, but the declaration of `m` only has one
type parameter:

```dart
class C {
  int m<A>(A a) => 0;
}

int f(C c) => c.m[!<int, int>!](2);
```

#### Common fixes

If the type arguments are necessary, then make them match the number of
type parameters by either adding or removing type arguments:

```dart
class C {
  int m<A>(A a) => 0;
}

int f(C c) => c.m<int>(2);
```

If the type arguments aren't necessary, then remove them:

```dart
class C {
  int m<A>(A a) => 0;
}

int f(C c) => c.m(2);
```

### yield_in_non_generator

_Yield statements must be in a generator function (one marked with either
'async*' or 'sync*')._

_Yield-each statements must be in a generator function (one marked with either
'async*' or 'sync*')._

#### Description

The analyzer produces this diagnostic when a `yield` or `yield*` statement
appears in a function whose body isn't marked with one of the `async*` or
`sync*` modifiers.

#### Examples

The following code produces this diagnostic because `yield` is being used
in a function whose body doesn't have a modifier:

```dart
Iterable<int> get digits {
  yield* [0, 1, 2, 3, 4, 5, 6, 7, 8, 9];
}
```

The following code produces this diagnostic because `yield*` is being used
in a function whose body has the `async` modifier rather than the `async*`
modifier:

```dart
Stream<int> get digits async {
  yield* [0, 1, 2, 3, 4, 5, 6, 7, 8, 9];
}
```

#### Common fixes

Add a modifier, or change the existing modifier to be either `async*` or
`sync*`:

```dart
Iterable<int> get digits sync* {
  yield* [0, 1, 2, 3, 4, 5, 6, 7, 8, 9];
}
```

### yield_of_invalid_type

_A yielded value of type '{0}' must be assignable to '{1}'._

_The type '{0}' implied by the 'yield*' expression must be assignable to '{1}'._

#### Description

The analyzer produces this diagnostic when the type of object produced by
a `yield` or `yield*` expression doesn't match the type of objects that
are to be returned from the `Iterable` or `Stream` types that are returned
from a generator (a function or method marked with either `sync*` or
`async*`).

#### Example

The following code produces this diagnostic because the getter `zero` is
declared to return an `Iterable` that returns integers, but the `yield` is
returning a string from the iterable:

```dart
Iterable<int> get zero sync* {
  yield [!'0'!];
}
```

#### Common fixes

If the return type of the function is correct, then fix the expression
following the keyword `yield` to return the correct type:

```dart
Iterable<int> get zero sync* {
  yield 0;
}
```

If the expression following the `yield` is correct, then change the return
type of the function to allow it:

```dart
Iterable<String> get zero sync* {
  yield '0';
}
```

### always_declare_return_types

_Missing return type on method._

#### Description

The analyzer produces this diagnostic when a method or function doesn't
have an explicit return type.

#### Example

The following code produces this diagnostic because the function `f`
doesn't have a return type:

```dart
[!f!]() {}
```

#### Common fixes

Add an explicit return type:

```dart
void f() {}
```

### always_put_control_body_on_new_line

_Statement should be on a separate line._

#### Description

The analyzer produces this diagnostic when the code being controlled by a
control flow statement (`if`, `for`, `while`, or `do`) is on the same line
as the control flow statement.

#### Example

The following code produces this diagnostic because the `return` statement
is on the same line as the `if` that controls whether the `return` will be
executed:

```dart
void f(bool b) {
  if (b) [!return!];
}
```

#### Common fixes

Put the controlled statement onto a separate, indented, line:

```dart
void f(bool b) {
  if (b)
    return;
}
```

### always_put_required_named_parameters_first

_Required named parameters should be before optional named parameters._

#### Description

The analyzer produces this diagnostic when required named parameters occur
after optional named parameters.

#### Example

The following code produces this diagnostic because the required parameter
`x` is after the optional parameter `y`:

```dart
void f({int? y, required int [!x!]}) {}
```

#### Common fixes

Reorder the parameters so that all required named parameters are before
any optional named parameters:

```dart
void f({required int x, int? y}) {}
```

### always_use_package_imports

_Use 'package:' imports for files in the 'lib' directory._

#### Description

The analyzer produces this diagnostic when an `import` in a library inside
the `lib` directory uses a relative path to import another library inside
the `lib` directory of the same package.

#### Example

Given that a file named `a.dart` and the code below are both inside the
`lib` directory of the same package, the following code produces this
diagnostic because a relative URI is used to import `a.dart`:

```dart
import [!'a.dart'!];
```

#### Common fixes

Use a package import:

```dart
import 'package:p/a.dart';
```

### annotate_overrides

_The member '{0}' overrides an inherited member but isn't annotated with
'@override'._

#### Description

The analyzer produces this diagnostic when a member overrides an inherited
member, but isn't annotated with `@override`.

#### Example

The following code produces this diagnostic because the method `m` in the
class `B` overrides the method with the same name in class `A`, but isn't
marked as an intentional override:

```dart
class A {
  void m() {}
}

class B extends A {
  void [!m!]() {}
}
```

#### Common fixes

If the member in the subclass is intended to override the member in the
superclass, then add an `@override` annotation:

```dart
class A {
  void m() {}
}

class B extends A {
  @override
  void m() {}
}
```

If the member in the subclass is not intended to override the member in
the superclass, then rename one of the members:

```dart
class A {
  void m() {}
}

class B extends A {
  void m2() {}
}
```

### avoid_empty_else

_Empty statements are not allowed in an 'else' clause._

#### Description

The analyzer produces this diagnostic when the statement after an `else`
is an empty statement (a semicolon).

For more information, see the documentation for
[`avoid_empty_else`](https://dart.dev/diagnostics/avoid_empty_else).

#### Example

The following code produces this diagnostic because the statement
following the `else` is an empty statement:

```dart
void f(int x, int y) {
  if (x > y)
    print("1");
  else [!;!]
    print("2");
}
```

#### Common fixes

If the statement after the empty statement is intended to be executed only
when the condition is `false`, then remove the empty statement:

```dart
void f(int x, int y) {
  if (x > y)
    print("1");
  else
    print("2");
}
```

If there is no code that is intended to be executed only when the
condition is `false`, then remove the whole `else` clause:

```dart
void f(int x, int y) {
  if (x > y)
    print("1");
  print("2");
}
```

### avoid_function_literals_in_foreach_calls

_Function literals shouldn't be passed to 'forEach'._

#### Description

The analyzer produces this diagnostic when the argument to
`Iterable.forEach` is a closure.

#### Example

The following code produces this diagnostic because the argument to the
invocation of `forEach` is a closure:

```dart
void f(Iterable<String> s) {
  s.[!forEach!]((e) => print(e));
}
```

#### Common fixes

If the closure can be replaced by a tear-off, then replace the closure:

```dart
void f(Iterable<String> s) {
  s.forEach(print);
}
```

If the closure can't be replaced by a tear-off, then use a `for` loop to
iterate over the elements:

```dart
void f(Iterable<String> s) {
  for (var e in s) {
    print(e);
  }
}
```

### avoid_init_to_null

_Redundant initialization to 'null'._

#### Description

The analyzer produces this diagnostic when a nullable variable is
explicitly initialized to `null`. The variable can be a local variable,
field, or top-level variable.

A variable or field that isn't explicitly initialized automatically gets
initialized to `null`. There's no concept of "uninitialized memory" in
Dart.

#### Example

The following code produces this diagnostic because the variable `f` is
explicitly initialized to `null`:

```dart
class C {
  int? [!f = null!];

  void m() {
    if (f != null) {
      print(f);
    }
  }
}
```

#### Common fixes

Remove the unnecessary initialization:

```dart
class C {
  int? f;

  void m() {
    if (f != null) {
      print(f);
    }
  }
}
```

### avoid_print

_Don't invoke 'print' in production code._

#### Description

The analyzer produces this diagnostic when the function `print` is invoked
in production code.

#### Example

The following code produces this diagnostic because the function `print`
can't be invoked in production:

```dart
void f(int x) {
  [!print!]('x = $x');
}
```

#### Common fixes

If you're writing code that uses Flutter, then use the function
[`debugPrint`][debugPrint], guarded by a test
using [`kDebugMode`][kDebugMode]:

```dart
import 'package:flutter/foundation.dart';

void f(int x) {
  if (kDebugMode) {
    debugPrint('x = $x');
  }
}
```

If you're writing code that doesn't use Flutter, then use a logging
service, such as [`package:logging`][package-logging], to write the
information.

### avoid_relative_lib_imports

_Can't use a relative path to import a library in 'lib'._

#### Description

The analyzer produces this diagnostic when the URI in an `import`
directive has `lib` in the path.

#### Example

Assuming that there is a file named `a.dart` in the `lib` directory:

```dart
class A {}
```

The following code produces this diagnostic because the import contains a
path that includes `lib`:

```dart
import [!'../lib/a.dart'!];
```

#### Common fixes

Rewrite the import to not include `lib` in the URI:

```dart
import 'a.dart';
```

### avoid_renaming_method_parameters

_The parameter name '{0}' doesn't match the name '{1}' in the overridden
method._

#### Description

The analyzer produces this diagnostic when a method that overrides a
method from a superclass changes the names of the parameters.

#### Example

The following code produces this diagnostic because the parameter of the
method `m` in `B` is named `b`, which is different from the name of the
overridden method's parameter in `A`:

```dart
class A {
  void m(int a) {}
}

class B extends A {
  @override
  void m(int [!b!]) {}
}
```

#### Common fixes

Rename one of the parameters so that they are the same:

```dart
class A {
  void m(int a) {}
}

class B extends A {
  @override
  void m(int a) {}
}
```

### avoid_return_types_on_setters

_Unnecessary return type on a setter._

#### Description

The analyzer produces this diagnostic when a setter has an explicit return
type.

Setters never return a value, so declaring the return type of one is
redundant.

#### Example

The following code produces this diagnostic because the setter `s` has an
explicit return type (`void`):

```dart
[!void!] set s(int p) {}
```

#### Common fixes

Remove the return type:

```dart
set s(int p) {}
```

### avoid_returning_null_for_void

_Don't return 'null' from a function with a return type of 'void'._

_Don't return 'null' from a method with a return type of 'void'._

#### Description

The analyzer produces this diagnostic when a function that has a return
type of `void` explicitly returns `null`.

#### Example

The following code produces this diagnostic because there is an explicit
return of `null` in a `void` function:

```dart
void f() {
  [!return null;!]
}
```

#### Common fixes

Remove the unnecessary explicit `null`:

```dart
void f() {
  return;
}
```

### avoid_shadowing_type_parameters

_The type parameter '{0}' shadows a type parameter from the enclosing {1}._

#### Description

The analyzer produces this diagnostic when a type parameter shadows a type
parameter from an enclosing declaration.

Shadowing a type parameter with a different type parameter can lead to
subtle bugs that are difficult to debug.

#### Example

The following code produces this diagnostic because the type parameter `T`
defined by the method `m` shadows the type parameter `T` defined by the
class `C`:

```dart
class C<T> {
  void m<[!T!]>() {}
}
```

#### Common fixes

Rename one of the type parameters:

```dart
class C<T> {
  void m<S>() {}
}
```

### avoid_single_cascade_in_expression_statements

_Unnecessary cascade expression._

#### Description

The analyzer produces this diagnostic when a single cascade operator is
used and the value of the expression isn't being used for anything (such
as being assigned to a variable or being passed as an argument).

#### Example

The following code produces this diagnostic because the value of the
cascade expression `s..length` isn't being used:

```dart
void f(String s) {
  [!s..length!];
}
```

#### Common fixes

Replace the cascade operator with a simple access operator:

```dart
void f(String s) {
  s.length;
}
```

### avoid_slow_async_io

_Use of an async 'dart:io' method._

#### Description

The analyzer produces this diagnostic when an asynchronous file I/O method
with a synchronous equivalent is used.

The following are the specific flagged asynchronous methods:

- `Directory.exists`
- `Directory.stat`
- `File.lastModified`
- `File.exists`
- `File.stat`
- `FileSystemEntity.isDirectory`
- `FileSystemEntity.isFile`
- `FileSystemEntity.isLink`
- `FileSystemEntity.type`

#### Example

The following code produces this diagnostic because the async method
`exists` is invoked:

```dart
import 'dart:io';

Future<void> g(File f) async {
  await [!f.exists()!];
}
```

#### Common fixes

Use the synchronous version of the method:

```dart
import 'dart:io';

void g(File f) {
  f.existsSync();
}
```

### avoid_type_to_string

_Using 'toString' on a 'Type' is not safe in production code._

#### Description

The analyzer produces this diagnostic when the method `toString` is
invoked on a value whose static type is `Type`.

#### Example

The following code produces this diagnostic because the method `toString`
is invoked on the `Type` returned by `runtimeType`:

```dart
bool isC(Object o) => o.runtimeType.[!toString!]() == 'C';

class C {}
```

#### Common fixes

If it's essential that the type is exactly the same, then use an explicit
comparison:

```dart
bool isC(Object o) => o.runtimeType == C;

class C {}
```

If it's alright for instances of subtypes of the type to return `true`,
then use a type check:

```dart
bool isC(Object o) => o is C;

class C {}
```

### avoid_types_as_parameter_names

_The parameter name '{0}' matches a visible type name._

#### Description

The analyzer produces this diagnostic when the name of a parameter in a
parameter list is the same as a visible type (a type whose name is in
scope).

This often indicates that the intended name of the parameter is missing,
causing the name of the type to be used as the name of the parameter
rather than the type of the parameter. Even when that's not the case (the
name of the parameter is intentional), the name of the parameter will
shadow the existing type, which can lead to bugs that are difficult to
diagnose.

#### Example

The following code produces this diagnostic because the function `f` has a
parameter named `int`, which shadows the type `int` from `dart:core`:

```dart
void f([!int!]) {}
```

#### Common fixes

If the parameter name is missing, then add a name for the parameter:

```dart
void f(int x) {}
```

If the parameter is intended to have an implicit type of `dynamic`, then
rename the parameter so that it doesn't shadow the name of any visible type:

```dart
void f(int_) {}
```

### avoid_unnecessary_containers

_Unnecessary instance of 'Container'._

#### Description

The analyzer produces this diagnostic when a widget tree contains an
instance of `Container` and the only argument to the constructor is
`child:`.

#### Example

The following code produces this diagnostic because the invocation of the
`Container` constructor only has a `child:` argument:

```dart
import 'package:flutter/material.dart';

Widget buildRow() {
  return [!Container!](
    child: Row(
      children: [
        Text('a'),
        Text('b'),
      ],
    )
  );
}
```

#### Common fixes

If you intended to provide other arguments to the constructor, then add
them:

```dart
import 'package:flutter/material.dart';

Widget buildRow() {
  return Container(
    color: Colors.red.shade100,
    child: Row(
      children: [
        Text('a'),
        Text('b'),
      ],
    )
  );
}
```

If no other arguments are needed, then unwrap the child widget:

```dart
import 'package:flutter/material.dart';

Widget buildRow() {
  return Row(
    children: [
      Text('a'),
      Text('b'),
    ],
  );
}
```

### avoid_web_libraries_in_flutter

_Don't use web-only libraries outside Flutter web plugin packages._

#### Description

The analyzer produces this diagnostic when a library in a package that
isn't a web plugin contains an import of a web-only library:
- `dart:html`
- `dart:js`
- `dart:js_util`
- `dart:js_interop`
- `dart:js_interop_unsafe`
- `package:js`
- `package:web`

#### Example

When found in a package that isn't a web plugin, the following code
produces this diagnostic because it imports `dart:html`:

```dart
import [!'dart:html'!];

import 'package:flutter/material.dart';

class C {}
```

#### Common fixes

If the package isn't intended to be a web plugin, then remove the import:

```dart
import 'package:flutter/material.dart';

class C {}
```

If the package is intended to be a web plugin, then add the following
lines to the `pubspec.yaml` file of the package:

```yaml
flutter:
  plugin:
    platforms:
      web:
        pluginClass: HelloPlugin
        fileName: hello_web.dart
```

See [Developing packages & plugins](https://docs.flutter.dev/packages-and-plugins/developing-packages)
for more information.

### await_only_futures

_Uses 'await' on an instance of '{0}', which is not a subtype of 'Future'._

#### Description

The analyzer produces this diagnostic when the expression after `await`
has any type other than `Future<T>`, `FutureOr<T>`, `Future<T>?`,
`FutureOr<T>?` or `dynamic`.

An exception is made for the expression `await null` because it is a
common way to introduce a microtask delay.

Unless the expression can produce a `Future`, the `await` is unnecessary
and can cause a reader to assume a level of asynchrony that doesn't exist.

#### Example

The following code produces this diagnostic because the expression after
`await` has the type `int`:

```dart
void f() async {
  [!await!] 23;
}
```

#### Common fixes

Remove the `await`:

```dart
void f() async {
  23;
}
```

### camel_case_extensions

_The extension name '{0}' isn't an UpperCamelCase identifier._

#### Description

The analyzer produces this diagnostic when the name of an extension
doesn't use the 'UpperCamelCase' naming convention.

#### Example

The following code produces this diagnostic because the name of the
extension doesn't start with an uppercase letter:

```dart
extension [!stringExtension!] on String {}
```

#### Common fixes

If the extension needs to have a name (needs to be visible outside this
library), then rename the extension so that it has a valid name:

```dart
extension StringExtension on String {}
```

If the extension doesn't need to have a name, then remove the name of the
extension:

```dart
extension on String {}
```

### camel_case_types

_The type name '{0}' isn't an UpperCamelCase identifier._

#### Description

The analyzer produces this diagnostic when the name of a type (a class,
mixin, enum, or typedef) doesn't use the 'UpperCamelCase' naming
convention.

#### Example

The following code produces this diagnostic because the name of the class
doesn't start with an uppercase letter:

```dart
class [!c!] {}
```

#### Common fixes

Rename the type so that it has a valid name:

```dart
class C {}
```

### cancel_subscriptions

_Uncancelled instance of 'StreamSubscription'._

#### Description

The analyzer produces this diagnostic when an instance of
`StreamSubscription` is created but the method `cancel` isn't invoked.

#### Example

The following code produces this diagnostic because the `subscription`
isn't canceled:

```dart
import 'dart:async';

void f(Stream stream) {
  // ignore: unused_local_variable
  var [!subscription = stream.listen((_) {})!];
}
```

#### Common fixes

Cancel the subscription:

```dart
import 'dart:async';

void f(Stream stream) {
  var subscription = stream.listen((_) {});
  subscription.cancel();
}
```

### close_sinks

_Unclosed instance of 'Sink'._

#### Description

The analyzer produces this diagnostic when an instance of `Sink` is
created but the method `close` isn't invoked.

#### Example

The following code produces this diagnostic because the `sink` isn't
closed:

```dart
import 'dart:io';

void g(File f) {
  var [!sink = f.openWrite()!];
  sink.write('x');
}
```

#### Common fixes

Close the sink:

```dart
import 'dart:io';

void g(File f) {
  var sink = f.openWrite();
  sink.write('x');
  sink.close();
}
```

### collection_methods_unrelated_type

_The argument type '{0}' isn't related to '{1}'._

#### Description

The analyzer produces this diagnostic when any one of several methods in
the core libraries are invoked with arguments of an inappropriate type.
These methods are ones that don't provide a specific enough type for the
parameter to allow the normal type checking to catch the error.

The arguments that are checked are:
- an argument to `Iterable<E>.contains` should be related to `E`
- an argument to `List<E>.remove` should be related to `E`
- an argument to `Map<K, V>.containsKey` should be related to `K`
- an argument to `Map<K, V>.containsValue` should be related to `V`
- an argument to `Map<K, V>.remove` should be related to `K`
- an argument to `Map<K, V>.[]` should be related to `K`
- an argument to `Queue<E>.remove` should be related to `E`
- an argument to `Set<E>.lookup` should be related to `E`
- an argument to `Set<E>.remove` should be related to `E`

#### Example

The following code produces this diagnostic because the argument to
`contains` is a `String`, which isn't assignable to `int`, the element
type of the list `l`:

```dart
bool f(List<int> l)  => l.contains([!'1'!]);
```

#### Common fixes

If the element type is correct, then change the argument to have the same
type:

```dart
bool f(List<int> l)  => l.contains(1);
```

If the argument type is correct, then change the element type:

```dart
bool f(List<String> l)  => l.contains('1');
```

### constant_identifier_names

_The constant name '{0}' isn't a lowerCamelCase identifier._

#### Description

The analyzer produces this diagnostic when the name of a constant doesn't
follow the lowerCamelCase naming convention.

#### Example

The following code produces this diagnostic because the name of the
top-level variable isn't a lowerCamelCase identifier:

```dart
const [!EMPTY_STRING!] = '';
```

#### Common fixes

Rewrite the name to follow the lowerCamelCase naming convention:

```dart
const emptyString = '';
```

### control_flow_in_finally

_Use of '{0}' in a 'finally' clause._

#### Description

The analyzer produces this diagnostic when a `finally` clause contains a
`return`, `break`, or `continue` statement.

#### Example

The following code produces this diagnostic because there is a `return`
statement inside a `finally` block:

```dart
int f() {
  try {
    return 1;
  } catch (e) {
    print(e);
  } finally {
    [!return 0;!]
  }
}
```

#### Common fixes

If the statement isn't needed, then remove the statement, and remove the
`finally` clause if the block is empty:

```dart
int f() {
  try {
    return 1;
  } catch (e) {
    print(e);
  }
}
```

If the statement is needed, then move the statement outside the `finally`
block:

```dart
int f() {
  try {
    return 1;
  } catch (e) {
    print(e);
  }
  return 0;
}
```

### curly_braces_in_flow_control_structures

_Statements in {0} should be enclosed in a block._

#### Description

The analyzer produces this diagnostic when a control structure (`if`,
`for`, `while`, or `do` statement) has a statement other than a block.

#### Example

The following code produces this diagnostic because the `then` statement
is not enclosed in a block:

```dart
int f(bool b) {
  if (b)
    [!return 1;!]
  return 0;
}
```

#### Common fixes

Add braces around the statement that should be a block:

```dart
int f(bool b) {
  if (b) {
    return 1;
  }
  return 0;
}
```

### dangling_library_doc_comments

_Dangling library doc comment._

#### Description

The analyzer produces this diagnostic when a documentation comment that
appears to be library documentation isn't followed by a `library`
directive. More specifically, it is produced when a documentation comment
appears before the first directive in the library, assuming that it isn't
a `library` directive, or before the first top-level declaration and is
separated from the declaration by one or more blank lines.

#### Example

The following code produces this diagnostic because there's a
documentation comment before the first `import` directive:

```dart
[!/// This is a great library.!]
import 'dart:core';
```

The following code produces this diagnostic because there's a
documentation comment before the first class declaration, but there's a
blank line between the comment and the declaration.

```dart
[!/// This is a great library.!]

class C {}
```

#### Common fixes

If the comment is library documentation, then add a `library` directive
without a name:

```dart
/// This is a great library.
library;

import 'dart:core';
```

If the comment is documentation for the following declaration, then remove
the blank line:

```dart
/// This is a great library.
class C {}
```

### depend_on_referenced_packages

_The imported package '{0}' isn't a dependency of the importing package._

#### Description

The analyzer produces this diagnostic when a package import refers to a
package that is not specified in the `pubspec.yaml` file.

Depending explicitly on packages that you reference ensures they will
always exist and allows you to put a dependency constraint on them to
guard against breaking changes.

#### Example

Given a `pubspec.yaml` file containing the following:

```yaml
dependencies:
  meta: ^3.0.0
```

The following code produces this diagnostic because there is no dependency
on the package `a`:

```dart
import 'package:a/a.dart';
```

#### Common fixes

Whether the dependency should be a regular dependency or dev dependency
depends on whether the package is referenced from a public library (one
under either `lib` or `bin`), or only private libraries, (such as one
under `test`).

If the package is referenced from at least one public library, then add a
regular dependency on the package to the `pubspec.yaml` file under the
`dependencies` field:

```yaml
dependencies:
  a: ^1.0.0
  meta: ^3.0.0
```

If the package is referenced only from private libraries, then add a
dev dependency on the package to the `pubspec.yaml` file under the
`dev_dependencies` field:

```yaml
dependencies:
  meta: ^3.0.0
dev_dependencies:
  a: ^1.0.0
```

### empty_catches

_Empty catch block._

#### Description

The analyzer produces this diagnostic when the block in a `catch` clause
is empty.

#### Example

The following code produces this diagnostic because the catch block is
empty:

```dart
void f() {
  try {
    print('Hello');
  } catch (exception) [!{}!]
}
```

#### Common fixes

If the exception shouldn't be ignored, then add code to handle the
exception:

```dart
void f() {
  try {
    print('We can print.');
  } catch (exception) {
    print("We can't print.");
  }
}
```

If the exception is intended to be ignored, then add a comment explaining
why:

```dart
void f() {
  try {
    print('We can print.');
  } catch (exception) {
    // Nothing to do.
  }
}
```

If the exception is intended to be ignored and there isn't any good
explanation for why, then rename the exception parameter:

```dart
void f() {
  try {
    print('We can print.');
  } catch (_) {}
}
```

### empty_constructor_bodies

_Empty constructor bodies should be written using a ';' rather than '{}'._

#### Description

The analyzer produces this diagnostic when a constructor has an empty
block body.

#### Example

The following code produces this diagnostic because the constructor for
`C` has a block body that is empty:

```dart
class C {
  C() [!{}!]
}
```

#### Common fixes

Replace the block with a semicolon:

```dart
class C {
  C();
}
```

### empty_statements

_Unnecessary empty statement._

#### Description

The analyzer produces this diagnostic when an empty statement is found.

#### Example

The following code produces this diagnostic because the statement
controlled by the `while` loop is an empty statement:

```dart
void f(bool condition) {
  while (condition)[!;!]
    g();
}

void g() {}
```

#### Common fixes

If there are no statements that need to be controlled, then remove both
the empty statement and the control structure it's part of (being careful
that any other code being removed doesn't have a side-effect that needs to
be preserved):

```dart
void f(bool condition) {
  g();
}

void g() {}
```

If there are no statements that need to be controlled but the control
structure is still required for other reasons, then replace the empty
statement with a block to make the structure of the code more obvious:

```dart
void f(bool condition) {
  while (condition) {}
  g();
}

void g() {}
```

If there are statements that need to be controlled, remove the empty
statement and adjust the code so that the appropriate statements are being
controlled, possibly adding a block:

```dart
void f(bool condition) {
  while (condition) {
    g();
  }
}

void g() {}
```

### file_names

_The file name '{0}' isn't a lower\_case\_with\_underscores identifier._

#### Description

The analyzer produces this diagnostic when the name of a `.dart` file
doesn't use lower_case_with_underscores.

#### Example

A file named `SliderMenu.dart` produces this diagnostic because the file
name uses the UpperCamelCase convention.

#### Common fixes

Rename the file to use the lower_case_with_underscores convention, such as
`slider_menu.dart`.

### hash_and_equals

_Missing a corresponding override of '{0}'._

#### Description

The analyzer produces this diagnostic when a class or mixin either
overrides the definition of `==` but doesn't override the definition of
`hashCode`, or conversely overrides the definition of `hashCode` but
doesn't override the definition of `==`.

Both the `==` operator and the `hashCode` property of objects must be
consistent for a common hash map implementation to function properly. As a
result, when overriding either method, both should be overridden.

#### Example

The following code produces this diagnostic because the class `C`
overrides the `==` operator but doesn't override the getter `hashCode`:

```dart
class C {
  final int value;

  C(this.value);

  @override
  bool operator [!==!](Object other) =>
      other is C &&
      other.runtimeType == runtimeType &&
      other.value == value;
}
```

#### Common fixes

If you need to override one of the members, then add an override of the
other:

```dart
class C {
  final int value;

  C(this.value);

  @override
  bool operator ==(Object other) =>
      other is C &&
      other.runtimeType == runtimeType &&
      other.value == value;

  @override
  int get hashCode => value.hashCode;
}
```

If you don't need to override either of the members, then remove the
unnecessary override:

```dart
class C {
  final int value;

  C(this.value);
}
```

### implementation_imports

_Import of a library in the 'lib/src' directory of another package._

#### Description

The analyzer produces this diagnostic when an import references a library
that's inside the `lib/src` directory of a different package, which
violates [the convention for pub
packages](https://dart.dev/tools/pub/package-layout#implementation-files).

#### Example

The following code, assuming that it isn't part of the `ffi` package,
produces this diagnostic because the library being imported is inside the
top-level `src` directory:

```dart
import [!'package:ffi/src/allocation.dart'!];
```

#### Common fixes

If the library being imported contains code that's part of the public API,
then import the public library that exports the public API:

```dart
import 'package:ffi/ffi.dart';
```

If the library being imported isn't part of the public API of the package,
then either find a different way to accomplish your goal, assuming that
it's possible, or open an issue asking the package authors to make it part
of the public API.

### implicit_call_tearoffs

_Implicit tear-off of the 'call' method._

#### Description

The analyzer produces this diagnostic when an object with a `call` method
is assigned to a function-typed variable, implicitly tearing off the
`call` method.

#### Example

The following code produces this diagnostic because an instance of
`Callable` is passed to a function expecting a `Function`:

```dart
class Callable {
  void call() {}
}

void callIt(void Function() f) {
  f();
}

void f() {
  callIt([!Callable()!]);
}
```

#### Common fixes

Explicitly tear off the `call` method:

```dart
class Callable {
  void call() {}
}

void callIt(void Function() f) {
  f();
}

void f() {
  callIt(Callable().call);
}
```

### invalid_use_of_do_not_submit_member

_Uses of '{0}' should not be submitted to source control._

#### Description

The analyzer produces this diagnostic when a member that is annotated with
[`@doNotSubmit`][meta-doNotSubmit] is referenced outside of a member
declaration that is also annotated with `@doNotSubmit`.

#### Example

Given a file `a.dart` containing the following declaration:

```dart
import 'package:meta/meta.dart';

@doNotSubmit
void emulateCrash() { /* ... */ }
```

The following code produces this diagnostic because the declaration is
being referenced outside of a member that is also annotated with
`@doNotSubmit`:

```dart
import 'a.dart';

void f() {
  [!emulateCrash!]();
}
```

#### Common fixes

Most commonly, when complete with local testing, the reference to the
member should be removed.

If building additional functionality on top of the member, annotate the
newly added member with `@doNotSubmit` as well:

```dart
import 'package:meta/meta.dart';

import 'a.dart';

@doNotSubmit
void emulateCrashWithOtherFunctionality() {
  emulateCrash();
  // do other things.
}
```

### library_names

_The library name '{0}' isn't a lower\_case\_with\_underscores identifier._

#### Description

The analyzer produces this diagnostic when the name of a library doesn't
use the lower_case_with_underscores naming convention.

#### Example

The following code produces this diagnostic because the library name
`libraryName` isn't a lower_case_with_underscores identifier:

```dart
library [!libraryName!];
```

#### Common fixes

If the library name is not required, then remove the library name:

```dart
library;
```

If the library name is required, then convert it to use the
lower_case_with_underscores naming convention:

```dart
library library_name;
```

### library_prefixes

_The prefix '{0}' isn't a lower\_case\_with\_underscores identifier._

#### Description

The analyzer produces this diagnostic when an import prefix doesn't use
the lower_case_with_underscores naming convention.

#### Example

The following code produces this diagnostic because the prefix
`ffiSupport` isn't a lower_case_with_underscores identifier:

```dart
import 'package:ffi/ffi.dart' as [!ffiSupport!];
```

#### Common fixes

Convert the prefix to use the lower_case_with_underscores naming
convention:

```dart
import 'package:ffi/ffi.dart' as ffi_support;
```

### library_private_types_in_public_api

_Invalid use of a private type in a public API._

#### Description

The analyzer produces this diagnostic when a type that is not part of the
public API of a library is referenced in the public API of that library.

Using a private type in a public API can make the API unusable outside the
defining library.

#### Example

The following code produces this diagnostic because the parameter `c` of
the public function `f` has a type that is library private (`_C`):

```dart
void f([!_C!] c) {}

class _C {}
```

#### Common fixes

If the API doesn't need to be used outside the defining library, then make
it private:

```dart
void _f(_C c) {}

class _C {}
```

If the API needs to be part of the public API of the library, then either
use a different type that's public, or make the referenced type public:

```dart
void f(C c) {}

class C {}
```

### literal_only_boolean_expressions

_The Boolean expression has a constant value._

#### Description

The analyzer produces this diagnostic when the value of the condition in
an `if` or loop statement is known to be either always `true` or always
`false`. An exception is made for a `while` loop whose condition is the
Boolean literal `true`.

#### Examples

The following code produces this diagnostic because the condition will
always evaluate to `true`:

```dart
void f() {
  [!if (true) {
    print('true');
  }!]
}
```

The lint will evaluate a subset of expressions that are composed of
constants, so the following code will also produce this diagnostic because
the condition will always evaluate to `false`:

```dart
void g(int i) {
  [!if (1 == 0 || 3 > 4) {
    print('false');
  }!]
}
```

#### Common fixes

If the condition is wrong, then correct the condition so that it's value
can't be known at compile time:

```dart
void g(int i) {
  if (i == 0 || i > 4) {
    print('false');
  }
}
```

If the condition is correct, then simplify the code to not evaluate the
condition:

```dart
void f() {
  print('true');
}
```

### no_adjacent_strings_in_list

_Don't use adjacent strings in a list literal._

#### Description

The analyzer produces this diagnostic when two string literals are
adjacent in a list literal. Adjacent strings in Dart are concatenated
together to form a single string, but the intent might be for each string
to be a separate element in the list.

#### Example

The following code produces this diagnostic because the strings `'a'` and
`'b'` are adjacent:

```dart
List<String> list = [[!'a' 'b'!], 'c'];
```

#### Common fixes

If the two strings are intended to be separate elements of the list, then
add a comma between them:

```dart
List<String> list = ['a', 'b', 'c'];
```

If the two strings are intended to be a single concatenated string, then
either manually merge the strings:

```dart
List<String> list = ['ab', 'c'];
```

Or use the `+` operator to concatenate the strings:

```dart
List<String> list = ['a' + 'b', 'c'];
```

### no_duplicate_case_values

_The value of the case clause ('{0}') is equal to the value of an earlier case
clause ('{1}')._

#### Description

The analyzer produces this diagnostic when two or more `case` clauses in
the same `switch` statement have the same value.

Any `case` clauses after the first can't be executed, so having duplicate
`case` clauses is misleading.

This diagnostic is often the result of either a typo or a change to the
value of a constant.

#### Example

The following code produces this diagnostic because two case clauses have
the same value (1):

```dart
// @dart = 2.14
void f(int v) {
  switch (v) {
    case 1:
      break;
    case [!1!]:
      break;
  }
}
```

#### Common fixes

If one of the clauses should have a different value, then change the value
of the clause:

```dart
void f(int v) {
  switch (v) {
    case 1:
      break;
    case 2:
      break;
  }
}
```

If the value is correct, then merge the statements into a single clause:

```dart
void f(int v) {
  switch (v) {
    case 1:
      break;
  }
}
```

### no_leading_underscores_for_library_prefixes

_The library prefix '{0}' starts with an underscore._

#### Description

The analyzer produces this diagnostic when the name of a prefix declared
on an import starts with an underscore.

Library prefixes are inherently not visible outside the declaring library,
so a leading underscore indicating private adds no value.

#### Example

The following code produces this diagnostic because the prefix `_core`
starts with an underscore:

```dart
import 'dart:core' as [!_core!];
```

#### Common fixes

Remove the underscore:

```dart
import 'dart:core' as core;
```

### no_leading_underscores_for_local_identifiers

_The local variable '{0}' starts with an underscore._

#### Description

The analyzer produces this diagnostic when the name of a local variable
starts with an underscore.

Local variables are inherently not visible outside the declaring library,
so a leading underscore indicating private adds no value.

#### Example

The following code produces this diagnostic because the parameter `_s`
starts with an underscore:

```dart
int f(String [!_s!]) => _s.length;
```

#### Common fixes

Remove the underscore:

```dart
int f(String s) => s.length;
```

### no_logic_in_create_state

_Don't put any logic in 'createState'._

#### Description

The analyzer produces this diagnostic when an implementation of
`createState` in a subclass of `StatefulWidget` contains any logic other
than the return of the result of invoking a zero argument constructor.

#### Examples

The following code produces this diagnostic because the constructor
invocation has arguments:

```dart
import 'package:flutter/material.dart';

class MyWidget extends StatefulWidget {
  @override
  MyState createState() => [!MyState(0)!];
}

class MyState extends State {
  int x;

  MyState(this.x);
}
```

#### Common fixes

Rewrite the code so that `createState` doesn't contain any logic:

```dart
import 'package:flutter/material.dart';

class MyWidget extends StatefulWidget {
  @override
  MyState createState() => MyState();
}

class MyState extends State {
  int x = 0;

  MyState();
}
```

### no_wildcard_variable_uses

_The referenced identifier is a wildcard._

#### Description

The analyzer produces this diagnostic when either a parameter or local
variable whose name consists of only underscores is referenced. Such
names will become non-binding in a future version of the Dart language,
making the reference illegal.

#### Example

The following code produces this diagnostic because the name of the
parameter consists of two underscores:

```dart
void f(int __) {
  print([!__!]);
}
```

The following code produces this diagnostic because the name of the
local variable consists of a single underscore:

```dart
void f() {
  int _ = 0;
  print([!_!]);
}
```

#### Common fixes

If the variable or parameter is intended to be referenced, then give it a
name that has at least one non-underscore character:

```dart
void f(int p) {
  print(p);
}
```

If the variable or parameter is not intended to be referenced, then
replace the reference with a different expression:

```dart
void f() {
  print(0);
}
```

### non_constant_identifier_names

_The variable name '{0}' isn't a lowerCamelCase identifier._

#### Description

The analyzer produces this diagnostic when the name of a class member,
top-level declaration, variable, parameter, named parameter, or named
constructor that isn't declared to be `const`, doesn't use the
lowerCamelCase convention.

#### Example

The following code produces this diagnostic because the top-level variable
`Count` doesn't start with a lowercase letter:

```dart
var [!Count!] = 0;
```

#### Common fixes

Change the name in the declaration to follow the lowerCamelCase
convention:

```dart
var count = 0;
```

### null_check_on_nullable_type_parameter

_The null check operator shouldn't be used on a variable whose type is a
potentially nullable type parameter._

#### Description

The analyzer produces this diagnostic when a null check operator is used
on a variable whose type is `T?`, where `T` is a type parameter that
allows the type argument to be nullable (either has no bound or has a
bound that is nullable).

Given a generic type parameter `T` which has a nullable bound, it is very
easy to introduce erroneous null checks when working with a variable of
type `T?`. Specifically, it is not uncommon to have `T? x;` and want to
assert that `x` has been set to a valid value of type `T`. A common
mistake is to do so using `x!`. This is almost always incorrect, because
if `T` is a nullable type, `x` may validly hold `null` as a value of type
`T`.

#### Example

The following code produces this diagnostic because `t` has the type `T?`
and `T` allows the type argument to be nullable (because it has no
`extends` clause):

```dart
T f<T>(T? t) => t[!!!];
```

#### Common fixes

Use the type parameter to cast the variable:

```dart
T f<T>(T? t) => t as T;
```

### overridden_fields

_Field overrides a field inherited from '{0}'._

#### Description

The analyzer produces this diagnostic when a class defines a field that
overrides a field from a superclass.

Overriding a field with another field causes the object to have two
distinct fields, but because the fields have the same name only one of the
fields can be referenced in a given scope. That can lead to confusion
where a reference to one of the fields can be mistaken for a reference to
the other.

#### Example

The following code produces this diagnostic because the field `f` in `B`
shadows the field `f` in `A`:

```dart
class A {
  int f = 1;
}

class B extends A {
  @override
  int [!f!] = 2;
}
```

#### Common fixes

If the two fields are representing the same property, then remove the
field from the subclass:

```dart
class A {
  int f = 1;
}

class B extends A {}
```

If the two fields should be distinct, then rename one of the fields:

```dart
class A {
  int f = 1;
}

class B extends A {
  int g = 2;
}
```

If the two fields are related in some way, but can't be the same, then
find a different way to implement the semantics you need.

### package_names

_The package name '{0}' isn't a lower\_case\_with\_underscores identifier._

#### Description

The analyzer produces this diagnostic when the name of a package doesn't
use the lower_case_with_underscores naming convention.

#### Example

The following code produces this diagnostic because the name of the
package uses the lowerCamelCase naming convention:

```yaml
name: [!somePackage!]
```

#### Common fixes

Rewrite the name of the package using the lower_case_with_underscores
naming convention:

```yaml
name: some_package
```

### package_prefixed_library_names

_The library name is not a dot-separated path prefixed by the package name._

#### Description

The analyzer produces this diagnostic when a library has a name that
doesn't follow these guidelines:

- Prefix all library names with the package name.
- Make the entry library have the same name as the package.
- For all other libraries in a package, after the package name add the
  dot-separated path to the library's Dart file.
- For libraries under `lib`, omit the top directory name.

For example, given a package named `my_package`, here are the library
names for various files in the package:


#### Example

Assuming that the file containing the following code is not in a file
named `special.dart` in the `lib` directory of a package named `something`
(which would be an exception to the rule), the analyzer produces this
diagnostic because the name of the library doesn't conform to the
guidelines above:

```dart
library [!something.special!];
```

#### Common fixes

Change the name of the library to conform to the guidelines.

### prefer_adjacent_string_concatenation

_String literals shouldn't be concatenated by the '+' operator._

#### Description

The analyzer produces this diagnostic when the `+` operator is used to
concatenate two string literals.

#### Example

The following code produces this diagnostic because two string literals
are being concatenated by using the `+` operator:

```dart
var s = 'a' [!+!] 'b';
```

#### Common fixes

Remove the operator:

```dart
var s = 'a' 'b';
```

### prefer_collection_literals

_Unnecessary constructor invocation._

#### Description

The analyzer produces this diagnostic when a constructor is used to create
a list, map, or set, but a literal would produce the same result.

#### Example

The following code produces this diagnostic because the constructor for
`Map` is being used to create a map that could also be created using a
literal:

```dart
var m = [!Map<String, String>()!];
```

#### Common fixes

Use the literal representation:

```dart
var m = <String, String>{};
```

### prefer_conditional_assignment

_The 'if' statement could be replaced by a null-aware assignment._

#### Description

The analyzer produces this diagnostic when an assignment to a variable is
conditional based on whether the variable has the value `null` and the
`??=` operator could be used instead.

#### Example

The following code produces this diagnostic because the parameter `s` is
being compared to `null` in order to determine whether to assign a
different value:

```dart
int f(String? s) {
  [!if (s == null) {
    s = '';
  }!]
  return s.length;
}
```

#### Common fixes

Use the `??=` operator instead of an explicit `if` statement:

```dart
int f(String? s) {
  s ??= '';
  return s.length;
}
```

### prefer_const_constructors

_Use 'const' with the constructor to improve performance._

#### Description

The analyzer produces this diagnostic when an invocation of a const
constructor isn't either preceded by `const` or in a [constant context][].

#### Example

The following code produces this diagnostic because the invocation of the
`const` constructor is neither prefixed by `const` nor in a
[constant context][]:

```dart
class C {
  const C();
}

C c = [!C()!];
```

#### Common fixes

If the context can be made a [constant context][], then do so:

```dart
class C {
  const C();
}

const C c = C();
```

If the context can't be made a [constant context][], then add `const`
before the constructor invocation:

```dart
class C {
  const C();
}

C c = const C();
```

### prefer_const_constructors_in_immutables

_Constructors in '@immutable' classes should be declared as 'const'._

#### Description

The analyzer produces this diagnostic when a non-`const` constructor is
found in a class that has the `@immutable` annotation.

#### Example

The following code produces this diagnostic because the constructor in `C`
isn't declared as `const` even though `C` has the `@immutable` annotation:

```dart
import 'package:meta/meta.dart';

@immutable
class C {
  final f;

  [!C!](this.f);
}
```

#### Common fixes

If the class really is intended to be immutable, then add the `const`
modifier to the constructor:

```dart
import 'package:meta/meta.dart';

@immutable
class C {
  final f;

  const C(this.f);
}
```

If the class is mutable, then remove the `@immutable` annotation:

```dart
class C {
  final f;

  C(this.f);
}
```

### prefer_const_declarations

_Use 'const' for final variables initialized to a constant value._

#### Description

The analyzer produces this diagnostic when a top-level variable, static
field, or local variable is marked as `final` and is initialized to a
constant value.

#### Examples

The following code produces this diagnostic because the top-level variable
`v` is both `final` and initialized to a constant value:

```dart
[!final v = const <int>[]!];
```

The following code produces this diagnostic because the static field `f`
is both `final` and initialized to a constant value:

```dart
class C {
  static [!final f = const <int>[]!];
}
```

The following code produces this diagnostic because the local variable `v`
is both `final` and initialized to a constant value:

```dart
void f() {
  [!final v = const <int>[]!];
  print(v);
}
```

#### Common fixes

Replace the keyword `final` with `const` and remove `const` from the
initializer:

```dart
class C {
  static const f = <int>[];
}
```

### prefer_const_literals_to_create_immutables

_Use 'const' literals as arguments to constructors of '@immutable' classes._

#### Description

The analyzer produces this diagnostic when a non-const list, map, or set
literal is passed as an argument to a constructor declared in a class
annotated with `@immutable`.

#### Example

The following code produces this diagnostic because the list literal
(`[1]`) is being passed to a constructor in an immutable class but isn't
a constant list:

```dart
import 'package:meta/meta.dart';

@immutable
class C {
  final f;

  const C(this.f);
}

C c = C([![1]!]);
```

#### Common fixes

If the context can be made a [constant context][], then do so:

```dart
import 'package:meta/meta.dart';

@immutable
class C {
  final f;

  const C(this.f);
}

const C c = C([1]);
```

If the context can't be made a [constant context][] but the constructor
can be invoked using `const`, then add `const` before the constructor
invocation:

```dart
import 'package:meta/meta.dart';

@immutable
class C {
  final f;

  const C(this.f);
}

C c = const C([1]);
```

If the context can't be made a [constant context][] and the constructor
can't be invoked using `const`, then add the keyword `const` before the
collection literal:

```dart
import 'package:meta/meta.dart';

@immutable
class C {
  final f;

  const C(this.f);
}

C c = C(const [1]);
```

### prefer_contains

_Always false because indexOf is always greater or equal -1._

_Always true because indexOf is always greater or equal -1._

_Unnecessary use of 'indexOf' to test for containment._

#### Description

The analyzer produces this diagnostic when the method `indexOf` is used and
the result is only compared with `-1` or `0` in a way where the semantics
are equivalent to using `contains`.

#### Example

The following code produces this diagnostic because the condition in the
`if` statement is checking to see whether the list contains the string:

```dart
void f(List<String> l, String s) {
  if ([!l.indexOf(s) < 0!]) {
    // ...
  }
}
```

#### Common fixes

Use `contains` instead, negating the condition when necessary:

```dart
void f(List<String> l, String s) {
  if (l.contains(s)) {
    // ...
  }
}
```

### prefer_final_fields

_The private field {0} could be 'final'._

#### Description

The analyzer produces this diagnostic when a private field is only
assigned one time. The field can be initialized in multiple constructors
and still be flagged because only one of those constructors can ever run.

#### Example

The following code produces this diagnostic because the field `_f` is only
assigned one time, in the field's initializer:

```dart
class C {
  int [!_f = 1!];

  int get f => _f;
}
```

#### Common fixes

Mark the field `final`:

```dart
class C {
  final int _f = 1;

  int get f => _f;
}
```

### prefer_for_elements_to_map_fromiterable

_Use 'for' elements when building maps from iterables._

#### Description

The analyzer produces this diagnostic when `Map.fromIterable` is used to
build a map that could be built using the `for` element.

#### Example

The following code produces this diagnostic because `fromIterable` is
being used to build a map that could be built using a `for` element:

```dart
void f(Iterable<String> data) {
  [!Map<String, int>.fromIterable(
    data,
    key: (element) => element,
    value: (element) => element.length,
  )!];
}
```

#### Common fixes

Use a `for` element to build the map:

```dart
void f(Iterable<String> data) {
  <String, int>{
    for (var element in data)
      element: element.length
  };
}
```

### prefer_function_declarations_over_variables

_Use a function declaration rather than a variable assignment to bind a function
to a name._

#### Description

The analyzer produces this diagnostic when a closure is assigned to a
local variable and the local variable is not re-assigned anywhere.

#### Example

The following code produces this diagnostic because the local variable `f`
is initialized to be a closure and isn't assigned any other value:

```dart
void g() {
  var [!f = (int i) => i * 2!];
  f(1);
}
```

#### Common fixes

Replace the local variable with a local function:

```dart
void g() {
  int f(int i) => i * 2;
  f(1);
}
```

### prefer_generic_function_type_aliases

_Use the generic function type syntax in 'typedef's._

#### Description

The analyzer produces this diagnostic when a typedef is written using the
older syntax for function type aliases in which the name being declared is
embedded in the function type.

#### Example

The following code produces this diagnostic because it uses the older
syntax:

```dart
typedef void [!F!]<T>();
```

#### Common fixes

Rewrite the typedef to use the newer syntax:

```dart
typedef F<T> = void Function();
```

### prefer_if_null_operators

_Use the '??' operator rather than '?:' when testing for 'null'._

#### Description

The analyzer produces this diagnostic when a conditional expression (using
the `?:` operator) is used to select a different value when a local
variable is `null`.

#### Example

The following code produces this diagnostic because the variable `s` is
being compared to `null` so that a different value can be returned when
`s` is `null`:

```dart
String f(String? s) => [!s == null ? '' : s!];
```

#### Common fixes

Use the if-null operator instead:

```dart
String f(String? s) => s ?? '';
```

### prefer_initializing_formals

_Use an initializing formal to assign a parameter to a field._

#### Description

The analyzer produces this diagnostic when a constructor parameter is used
to initialize a field without modification.

#### Example

The following code produces this diagnostic because the parameter `c` is
only used to set the field `c`:

```dart
class C {
  int c;

  C(int c) : [!this.c = c!];
}
```

#### Common fixes

Use an initializing formal parameter to initialize the field:

```dart
class C {
  int c;

  C(this.c);
}
```

### prefer_inlined_adds

_The addition of a list item could be inlined._

_The addition of multiple list items could be inlined._

#### Description

The analyzer produces this diagnostic when the methods `add` and `addAll`
are invoked on a list literal where the elements being added could be
included in the list literal.

#### Example

The following code produces this diagnostic because the `add` method is
being used to add `b`, when it could have been included directly in the
list literal:

```dart
List<String> f(String a, String b) {
  return [a]..[!add!](b);
}
```

The following code produces this diagnostic because the `addAll` method is
being used to add the elements of `b`, when it could have been included
directly in the list literal:

```dart
List<String> f(String a, List<String> b) {
  return [a]..[!addAll!](b);
}
```

#### Common fixes

If the `add` method is being used, then make the argument an element of
the list and remove the invocation:

```dart
List<String> f(String a, String b) {
  return [a, b];
}
```

If the `addAll` method is being used, then use the spread operator on the
argument to add its elements to the list and remove the invocation:

```dart
List<String> f(String a, List<String> b) {
  return [a, ...b];
}
```

### prefer_interpolation_to_compose_strings

_Use interpolation to compose strings and values._

#### Description

The analyzer produces this diagnostic when string literals and computed
strings are being concatenated using the `+` operator, but string
interpolation would achieve the same result.

#### Example

The following code produces this diagnostic because the elements of the
list `l` are being concatenated with other strings using the `+` operator:

```dart
String f(List<String> l) {
  return [!'(' + l[0] + ', ' + l[1] + ')'!];
}
```

#### Common fixes

Use string interpolation:

```dart
String f(List<String> l) {
  return '(${l[0]}, ${l[1]})';
}
```

### prefer_is_empty

_The comparison is always 'false' because the length is always greater than or
equal to 0._

_The comparison is always 'true' because the length is always greater than or
equal to 0._

_Use 'isEmpty' instead of 'length' to test whether the collection is empty._

_Use 'isNotEmpty' instead of 'length' to test whether the collection is empty._

#### Description

The analyzer produces this diagnostic when the result of invoking either
`Iterable.length` or `Map.length` is compared for equality with zero
(`0`).

#### Example

The following code produces this diagnostic because the result of invoking
`length` is checked for equality with zero:

```dart
int f(Iterable<int> p) => [!p.length == 0!] ? 0 : p.first;
```

#### Common fixes

Replace the use of `length` with a use of either `isEmpty` or
`isNotEmpty`:

```dart
void f(Iterable<int> p) => p.isEmpty ? 0 : p.first;
```

### prefer_is_not_empty

_Use 'isNotEmpty' rather than negating the result of 'isEmpty'._

#### Description

The analyzer produces this diagnostic when the result of invoking
`Iterable.isEmpty` or `Map.isEmpty` is negated.

#### Example

The following code produces this diagnostic because the result of invoking
`Iterable.isEmpty` is negated:

```dart
void f(Iterable<int> p) => [!!p.isEmpty!] ? p.first : 0;
```

#### Common fixes

Rewrite the code to use `isNotEmpty`:

```dart
void f(Iterable<int> p) => p.isNotEmpty ? p.first : 0;
```

### prefer_is_not_operator

_Use the 'is!' operator rather than negating the value of the 'is' operator._

#### Description

The analyzer produces this diagnostic when the prefix `!` operator is used
to negate the result of an `is` test.

#### Example

The following code produces this diagnostic because the result of testing
to see whether `o` is a `String` is negated using the prefix `!` operator:

```dart
String f(Object o) {
  if ([!!(o is String)!]) {
    return o.toString();
  }
  return o;
}
```

#### Common fixes

Use the `is!` operator instead:

```dart
String f(Object o) {
  if (o is! String) {
    return o.toString();
  }
  return o;
}
```

### prefer_iterable_wheretype

_Use 'whereType' to select elements of a given type._

#### Description

The analyzer produces this diagnostic when the method `Iterable.where` is
being used to filter elements based on their type.

#### Example

The following code produces this diagnostic because the method `where` is
being used to access only the strings within the iterable:

```dart
Iterable<Object> f(Iterable<Object> p) => p.[!where!]((e) => e is String);
```

#### Common fixes

Rewrite the code to use `whereType`:

```dart
Iterable<String> f(Iterable<Object> p) => p.whereType<String>();
```

This might also allow you to tighten the types in your code or remove
other type checks.

### prefer_null_aware_operators

_Use the null-aware operator '?.' rather than an explicit 'null' comparison._

#### Description

The analyzer produces this diagnostic when a comparison with `null` is
used to guard a member reference, and `null` is used as a result when the
guarded target is `null`.

#### Example

The following code produces this diagnostic because the invocation of
`length` is guarded by a `null` comparison even though the default value
is `null`:

```dart
int? f(List<int>? p) {
  return [!p == null ? null : p.length!];
}
```

#### Common fixes

Use a null-aware access operator instead:

```dart
int? f(List<int>? p) {
  return p?.length;
}
```

### prefer_relative_imports

_Use relative imports for files in the 'lib' directory._

#### Description

The analyzer produces this diagnostic when an `import` in a library inside
the `lib` directory uses a `package:` URI to refer to another library in
the same package.

#### Example

The following code produces this diagnostic because it uses a `package:`
URI when a relative URI could have been used:

```dart
import 'package:my_package/bar.dart';
```

#### Common fixes

Use a relative URI to import the library:

```dart
import 'bar.dart';
```

### prefer_typing_uninitialized_variables

_Prefer typing uninitialized variables and fields._

#### Description

The analyzer produces this diagnostic when a variable without an
initializer doesn't have an explicit type annotation.

Without either a type annotation or an initializer, a variable has the
type `dynamic`, which allows any value to be assigned to the variable,
often causing hard to identify bugs.

#### Example

The following code produces this diagnostic because the variable `r`
doesn't have either a type annotation or an initializer:

```dart
Object f() {
  var [!r!];
  r = '';
  return r;
}
```

#### Common fixes

If the variable can be initialized, then add an initializer:

```dart
Object f() {
  var r = '';
  return r;
}
```

If the variable can't be initialized, then add an explicit type
annotation:

```dart
Object f() {
  String r;
  r = '';
  return r;
}
```

### prefer_void_to_null

_Unnecessary use of the type 'Null'._

#### Description

The analyzer produces this diagnostic when `Null` is used in a location
where `void` would be a valid choice.

#### Example

The following code produces this diagnostic because the function `f` is
declared to return `null` (at some future time):

```dart
Future<[!Null!]> f() async {}
```

#### Common fixes

Replace the use of `Null` with a use of `void`:

```dart
Future<void> f() async {}
```

### provide_deprecation_message

_Missing a deprecation message._

#### Description

The analyzer produces this diagnostic when a `deprecated` annotation is
used instead of the `Deprecated` annotation.

#### Example

The following code produces this diagnostic because the function `f` is
annotated with `deprecated`:

```dart
[!@deprecated!]
void f() {}
```

#### Common fixes

Convert the code to use the longer form:

```dart
@Deprecated('Use g instead. Will be removed in 4.0.0.')
void f() {}
```

### recursive_getters

_The getter '{0}' recursively returns itself._

#### Description

The analyzer produces this diagnostic when a getter invokes itself,
resulting in an infinite loop.

#### Example

The following code produces this diagnostic because the getter `count`
invokes itself:

```dart
class C {
  int _count = 0;

  int get [!count!] => count;
}
```

#### Common fixes

Change the getter to not invoke itself:

```dart
class C {
  int _count = 0;

  int get count => _count;
}
```

### secure_pubspec_urls

_The '{0}' protocol shouldn't be used because it isn't secure._

#### Description

The analyzer produces this diagnostic when a URL in a `pubspec.yaml` file is
using a non-secure scheme, such as `http`.

#### Example

The following code produces this diagnostic because the `pubspec.yaml` file
contains an `http` URL:

```yaml
dependencies:
  example: any
    repository: [!http://github.com/dart-lang/example!]
```

#### Common fixes

Change the scheme of the URL to use a secure scheme, such as `https`:

```yaml
dependencies:
  example: any
    repository: https://github.com/dart-lang/example
```

### sized_box_for_whitespace

_Use a 'SizedBox' to add whitespace to a layout._

#### Description

The analyzer produces this diagnostic when a `Container` is created using
only the `height` and/or `width` arguments.

#### Example

The following code produces this diagnostic because the `Container` has
only the `width` argument:

```dart
import 'package:flutter/material.dart';

Widget buildRow() {
  return Row(
    children: <Widget>[
      const Text('...'),
      [!Container!](
        width: 4,
        child: Text('...'),
      ),
      const Expanded(
        child: Text('...'),
      ),
    ],
  );
}
```

#### Common fixes

Replace the `Container` with a `SizedBox` of the same dimensions:

```dart
import 'package:flutter/material.dart';

Widget buildRow() {
  return Row(
    children: <Widget>[
      Text('...'),
      SizedBox(
        width: 4,
        child: Text('...'),
      ),
      Expanded(
        child: Text('...'),
      ),
    ],
  );
}
```

### slash_for_doc_comments

_Use the end-of-line form ('///') for doc comments._

#### Description

The analyzer produces this diagnostic when a documentation comment uses
the block comment style (delimited by `/**` and `*/`).

#### Example

The following code produces this diagnostic because the documentation
comment for `f` uses a block comment style:

```dart
[!/**
 * Example.
 */!]
void f() {}
```

#### Common fixes

Use an end-of-line comment style:

```dart
/// Example.
void f() {}
```

### sort_child_properties_last

_The '{0}' argument should be last in widget constructor invocations._

#### Description

The analyzer produces this diagnostic when the `child` or `children`
argument isn't the last argument in an invocation of a widget class'
constructor. An exception is made if all of the arguments after the
`child` or `children` argument are function expressions.

#### Example

The following code produces this diagnostic because the `child` argument
isn't the last argument in the invocation of the `Center` constructor:

```dart
import 'package:flutter/material.dart';

Widget createWidget() {
  return Center(
    [!child: Text('...')!],
    widthFactor: 0.5,
  );
}
```

#### Common fixes

Move the `child` or `children` argument to be last:

```dart
import 'package:flutter/material.dart';

Widget createWidget() {
  return Center(
    widthFactor: 0.5,
    child: Text('...'),
  );
}
```

### sort_pub_dependencies

_Dependencies not sorted alphabetically._

#### Description

The analyzer produces this diagnostic when the keys in a dependency map in
the `pubspec.yaml` file aren't sorted alphabetically. The dependency maps
that are checked are the `dependencies`, `dev_dependencies`, and
`dependency_overrides` maps.

#### Example

The following code produces this diagnostic because the entries in the
`dependencies` map are not sorted:

```yaml
dependencies:
  path: any
  collection: any
```

#### Common fixes

Sort the entries:

```yaml
dependencies:
  collection: any
  path: any
```

### test_types_in_equals

_Missing type test for '{0}' in '=='._

#### Description

The analyzer produces this diagnostic when an override of the `==`
operator doesn't include a type test on the value of the parameter.

#### Example

The following code produces this diagnostic because `other` is not type
tested:

```dart
class C {
  final int f;

  C(this.f);

  @override
  bool operator ==(Object other) {
    return ([!other as C!]).f == f;
  }
}
```

#### Common fixes

Perform an `is` test as part of computing the return value:

```dart
class C {
  final int f;

  C(this.f);

  @override
  bool operator ==(Object other) {
    return other is C && other.f == f;
  }
}
```

### throw_in_finally

_Use of '{0}' in 'finally' block._

#### Description

The analyzer produces this diagnostic when a `throw` statement is found
inside a `finally` block.

#### Example

The following code produces this diagnostic because there is a `throw`
statement inside a `finally` block:

```dart
void f() {
  try {
    // ...
  } catch (e) {
    // ...
  } finally {
    [!throw 'error'!];
  }
}
```

#### Common fixes

Rewrite the code so that the `throw` statement isn't inside a `finally`
block:

```dart
void f() {
  try {
    // ...
  } catch (e) {
    // ...
  }
  throw 'error';
}
```

### type_init_formals

_Don't needlessly type annotate initializing formals._

#### Description

The analyzer produces this diagnostic when an initializing formal
parameter (`this.x`) or a super parameter (`super.x`) has an explicit type
annotation that is the same as the field or overridden parameter.

If a constructor parameter is using `this.x` to initialize a field, then
the type of the parameter is implicitly the same type as the field. If a
constructor parameter is using `super.x` to forward to a super
constructor, then the type of the parameter is implicitly the same as the
super constructor parameter.

#### Example

The following code produces this diagnostic because the parameter `this.c`
has an explicit type that is the same as the field `c`:

```dart
class C {
  int c;

  C([!int!] this.c);
}
```

The following code produces this diagnostic because the parameter
`super.a` has an explicit type that is the same as the parameter `a` from
the superclass:

```dart
class A {
  A(int a);
}

class B extends A {
  B([!int!] super.a);
}
```

#### Common fixes

Remove the type annotation from the parameter:

```dart
class C {
  int c;

  C(this.c);
}
```

### type_literal_in_constant_pattern

_Use 'TypeName \_' instead of a type literal._

#### Description

The analyzer produces this diagnostic when a type literal appears as a
pattern.

#### Example

The following code produces this diagnostic because a type literal is used
as a constant pattern:

```dart
void f(Object? x) {
  if (x case [!num!]) {
    // ...
  }
}
```

#### Common fixes

If the type literal is intended to match an object of the given type, then
use either a variable pattern:

```dart
void f(Object? x) {
  if (x case num _) {
    // ...
  }
}
```

Or an object pattern:

```dart
void f(Object? x) {
  if (x case num()) {
    // ...
  }
}
```

If the type literal is intended to match the type literal, then write it
as a constant pattern:

```dart
void f(Object? x) {
  if (x case const (num)) {
    // ...
  }
}
```

### unnecessary_brace_in_string_interps

_Unnecessary braces in a string interpolation._

#### Description

The analyzer produces this diagnostic when a string interpolation with
braces is used to interpolate a simple identifier and isn't followed by
alphanumeric text.

#### Example

The following code produces this diagnostic because the interpolation
element `${s}` uses braces when they are not necessary:

```dart
String f(String s) {
  return '"[!${s}!]"';
}
```

#### Common fixes

Remove the unnecessary braces:

```dart
String f(String s) {
  return '"$s"';
}
```

### unnecessary_const

_Unnecessary 'const' keyword._

#### Description

The analyzer produces this diagnostic when the keyword `const` is used in
a [constant context][]. The keyword isn't required because it's implied.

#### Example

The following code produces this diagnostic because the keyword `const` in
the list literal isn't needed:

```dart
const l = [!const!] <int>[];
```

The list is implicitly `const` because of the keyword `const` on the
variable declaration.

#### Common fixes

Remove the unnecessary keyword:

```dart
const l = <int>[];
```

### unnecessary_constructor_name

_Unnecessary '.new' constructor name._

#### Description

The analyzer produces this diagnostic when a reference to an unnamed
constructor uses `.new`. The only place where `.new` is required is in a
constructor tear-off.

#### Example

The following code produces this diagnostic because `.new` is being used
to refer to the unnamed constructor where it isn't required:

```dart
var o = Object.[!new!]();
```

#### Common fixes

Remove the unnecessary `.new`:

```dart
var o = Object();
```

### unnecessary_getters_setters

_Unnecessary use of getter and setter to wrap a field._

#### Description

The analyzer produces this diagnostic when a getter and setter pair
returns and sets the value of a field without any additional processing.

#### Example

The following code produces this diagnostic because the getter/setter pair
named `c` only expose the field named `_c`:

```dart
class C {
  int? _c;

  int? get [!c!] => _c;

  set c(int? v) => _c = v;
}
```

#### Common fixes

Make the field public and remove the getter and setter:

```dart
class C {
  int? c;
}
```

### unnecessary_late

_Unnecessary 'late' modifier._

#### Description

The analyzer produces this diagnostic when a top-level variable or static
field with an initializer is marked as `late`. Top-level variables and
static fields are implicitly late, so they don't need to be explicitly
marked.

#### Example

The following code produces this diagnostic because the static field `c`
has the modifier `late` even though it has an initializer:

```dart
class C {
  static [!late!] String c = '';
}
```

#### Common fixes

Remove the keyword `late`:

```dart
class C {
  static String c = '';
}
```

### unnecessary_new

_Unnecessary 'new' keyword._

#### Description

The analyzer produces this diagnostic when the keyword `new` is used to
invoke a constructor.

#### Example

The following code produces this diagnostic because the keyword `new` is
used to invoke the unnamed constructor from `Object`:

```dart
var o = [!new!] Object();
```

#### Common fixes

Remove the keyword `new`:

```dart
var o = Object();
```

### unnecessary_null_aware_assignments

_Unnecessary assignment of 'null'._

#### Description

The analyzer produces this diagnostic when the right-hand side of a
null-aware assignment is the `null` literal.

#### Example

The following code produces this diagnostic because the null aware
operator is being used to assign `null` to `s` when `s` is already `null`:

```dart
void f(String? s) {
  [!s ??= null!];
}
```

#### Common fixes

If a non-null value should be assigned to the left-hand operand, then
change the right-hand side:

```dart
void f(String? s) {
  s ??= '';
}
```

If there is no non-null value to assign to the left-hand operand, then
remove the assignment:

```dart
void f(String? s) {
}
```

### unnecessary_null_in_if_null_operators

_Unnecessary use of '??' with 'null'._

#### Description

The analyzer produces this diagnostic when the right operand of the `??`
operator is the literal `null`.

#### Example

The following code produces this diagnostic because the right-hand operand
of the `??` operator is `null`:

```dart
String? f(String? s) => s ?? [!null!];
```

#### Common fixes

If a non-null value should be used for the right-hand operand, then
change the right-hand side:

```dart
String f(String? s) => s ?? '';
```

If there is no non-null value to use for the right-hand operand, then
remove the operator and the right-hand operand:

```dart
String? f(String? s) => s;
```

### unnecessary_nullable_for_final_variable_declarations

_Type could be non-nullable._

#### Description

The analyzer produces this diagnostic when a final field or variable has a
nullable type but is initialized to a non-nullable value.

#### Example

The following code produces this diagnostic because the final variable `i`
has a nullable type (`int?`), but can never be `null`:

```dart
final int? [!i!] = 1;
```

#### Common fixes

Make the type non-nullable:

```dart
final int i = 1;
```

### unnecessary_overrides

_Unnecessary override._

#### Description

The analyzer produces this diagnostic when an instance member overrides an
inherited member but only invokes the overridden member with exactly the
same arguments.

#### Example

The following code produces this diagnostic because the method `D.m`
doesn't do anything other than invoke the overridden method:

```dart
class C {
  int m(int x) => x;
}

class D extends C {
  @override
  int [!m!](int x) => super.m(x);
}
```

#### Common fixes

If the method should do something more than what the overridden method
does, then implement the missing functionality:

```dart
class C {
  int m(int x) => x;
}

class D extends C {
  @override
  int m(int x) => super.m(x) + 1;
}
```

If the overridden method should be modified by changing the return type or
one or more of the parameter types, making one of the parameters
`covariant`, having a documentation comment, or by having additional
annotations, then update the code:

```dart
import 'package:meta/meta.dart';

class C {
  int m(int x) => x;
}

class D extends C {
  @mustCallSuper
  @override
  int m(int x) => super.m(x);
}
```

If the overriding method doesn't change or enhance the semantics of the
code, then remove it:

```dart
class C {
  int m(int x) => x;
}

class D extends C {}
```

### unnecessary_statements

_Unnecessary statement._

#### Description

The analyzer produces this diagnostic when an expression statement has no
clear effect.

#### Example

The following code produces this diagnostic because the addition of the
returned values from the two invocations has no clear effect:

```dart
void f(int Function() first, int Function() second) {
  [!first() + second()!];
}
```

#### Common fixes

If the expression doesn't need to be computed, then remove it:

```dart
void f(int Function() first, int Function() second) {
}
```

If the value of the expression is needed, then make use of it, possibly
assigning it to a local variable first:

```dart
void f(int Function() first, int Function() second) {
  print(first() + second());
}
```

If portions of the expression need to be executed, then remove the
unnecessary portions:

```dart
void f(int Function() first, int Function() second) {
  first();
  second();
}
```

### unnecessary_string_escapes

_Unnecessary escape in string literal._

#### Description

The analyzer produces this diagnostic when characters in a string are
escaped when escaping them is unnecessary.

#### Example

The following code produces this diagnostic because single quotes don't
need to be escaped inside strings delimited by double quotes:

```dart
var s = "Don[!\!]'t use a backslash here.";
```

#### Common fixes

Remove the unnecessary backslashes:

```dart
var s = "Don't use a backslash here.";
```

### unnecessary_string_interpolations

_Unnecessary use of string interpolation._

#### Description

The analyzer produces this diagnostic when a string literal contains a
single interpolation of a `String`-valued variable and no other
characters.

#### Example

The following code produces this diagnostic because the string literal
contains a single interpolation and doesn't contain any character outside
the interpolation:

```dart
String f(String s) => [!'$s'!];
```

#### Common fixes

Replace the string literal with the content of the interpolation:

```dart
String f(String s) => s;
```

### unnecessary_this

_Unnecessary 'this.' qualifier._

#### Description

The analyzer produces this diagnostic when the keyword `this` is used to
access a member that isn't shadowed.

#### Example

The following code produces this diagnostic because the use of `this` to
access the field `_f` isn't necessary:

```dart
class C {
  int _f = 2;

  int get f => [!this!]._f;
}
```

#### Common fixes

Remove the `this.`:

```dart
class C {
  int _f = 2;

  int get f => _f;
}
```

### unnecessary_to_list_in_spreads

_Unnecessary use of 'toList' in a spread._

#### Description

The analyzer produces this diagnostic when `toList` is used to convert an
`Iterable` to a `List` just before a spread operator is applied to the
list. The spread operator can be applied to any `Iterable`, so the
conversion isn't necessary.

#### Example

The following code produces this diagnostic because `toList` is invoked on
the result of `map`, which is an `Iterable` that the spread operator could
be applied to directly:

```dart
List<String> toLowercase(List<String> strings) {
  return [
    ...strings.map((String s) => s.toLowerCase()).[!toList!](),
  ];
}
```

#### Common fixes

Remove the invocation of `toList`:

```dart
List<String> toLowercase(List<String> strings) {
  return [
    ...strings.map((String s) => s.toLowerCase()),
  ];
}
```

### unrelated_type_equality_checks

_The type of the operand ('{0}') isn't a subtype or a supertype of the value
being matched ('{1}')._

_The type of the right operand ('{0}') isn't a subtype or a supertype of the
left operand ('{1}')._

#### Description

The analyzer produces this diagnostic when two objects are being compared
and neither of the static types of the two objects is a subtype of the
other.

Such a comparison will usually return `false` and might not reflect the
programmer's intent.

There can be false positives. For example, a class named `Point` might
have subclasses named `CartesianPoint` and `PolarPoint`, neither of which
is a subtype of the other, but it might still be appropriate to test the
equality of instances.

As a concrete case, the classes `Int64` and `Int32` from `package:fixnum`
allow comparing instances to an `int` provided the `int` is on the
right-hand side. This case is specifically allowed by the diagnostic, but
other such cases are not.

#### Example

The following code produces this diagnostic because the string `s` is
being compared to the integer `1`:

```dart
bool f(String s) {
  return s [!==!] 1;
}
```

#### Common fixes

Replace one of the operands with something compatible with the other
operand:

```dart
bool f(String s) {
  return s.length == 1;
}
```

### use_build_context_synchronously

_Don't use 'BuildContext's across async gaps, guarded by an unrelated 'mounted'
check._

_Don't use 'BuildContext's across async gaps._

#### Description

The analyzer produces this diagnostic when a `BuildContext` is referenced
by a `StatefulWidget` after an asynchronous gap without first checking the
`mounted` property.

Storing a `BuildContext` for later use can lead to difficult to diagnose
crashes. Asynchronous gaps implicitly store a `BuildContext`, making them
easy to overlook for diagnosis.

#### Example

The following code produces this diagnostic because the `context` is
passed to a constructor after the `await`:

```dart
import 'package:flutter/material.dart';

class MyWidget extends Widget {
  void onButtonTapped(BuildContext context) async {
    await Future.delayed(const Duration(seconds: 1));
    Navigator.of([!context!]).pop();
  }
}
```

#### Common fixes

If you can remove the asynchronous gap, do so:

```dart
import 'package:flutter/material.dart';

class MyWidget extends Widget {
  void onButtonTapped(BuildContext context) {
    Navigator.of(context).pop();
  }
}
```

If you can't remove the asynchronous gap, then use `mounted` to guard the
use of the `context`:

```dart
import 'package:flutter/material.dart';

class MyWidget extends Widget {
  void onButtonTapped(BuildContext context) async {
    await Future.delayed(const Duration(seconds: 1));
    if (context.mounted) {
      Navigator.of(context).pop();
    }
  }
}
```

### use_full_hex_values_for_flutter_colors

_Instances of 'Color' should be created using an 8-digit hexadecimal integer
(such as '0xFFFFFFFF')._

#### Description

The analyzer produces this diagnostic when the argument to the constructor
of the `Color` class is a literal integer that isn't represented as an
8-digit hexadecimal integer.

#### Example

The following code produces this diagnostic because the argument (`1`)
isn't represented as an 8-digit hexadecimal integer:

```dart
import 'package:flutter/material.dart';

Color c = Color([!1!]);
```

#### Common fixes

Convert the representation to be an 8-digit hexadecimal integer:

```dart
import 'package:flutter/material.dart';

Color c = Color(0x00000001);
```

### use_function_type_syntax_for_parameters

_Use the generic function type syntax to declare the parameter '{0}'._

#### Description

The analyzer produces this diagnostic when the older style function-valued
parameter syntax is used.

#### Example

The following code produces this diagnostic because the function-valued
parameter `f` is declared using an older style syntax:

```dart
void g([!bool f(String s)!]) {}
```

#### Common fixes

Use the generic function type syntax to declare the parameter:

```dart
void g(bool Function(String) f) {}
```

### use_key_in_widget_constructors

_Constructors for public widgets should have a named 'key' parameter._

#### Description

The analyzer produces this diagnostic when a constructor in a subclass of
`Widget` that isn't private to its library doesn't have a parameter named
`key`.

#### Example

The following code produces this diagnostic because the constructor for
the class `MyWidget` doesn't have a parameter named `key`:

```dart
import 'package:flutter/material.dart';

class MyWidget extends StatelessWidget {
  [!MyWidget!]({required int height});
}
```

The following code produces this diagnostic because the default
constructor for the class `MyWidget` doesn't have a parameter named `key`:

```dart
import 'package:flutter/material.dart';

class [!MyWidget!] extends StatelessWidget {}
```

#### Common fixes

Add a parameter named `key` to the constructor, explicitly declaring the
constructor if necessary:

```dart
import 'package:flutter/material.dart';

class MyWidget extends StatelessWidget {
  MyWidget({super.key, required int height});
}
```

### use_rethrow_when_possible

_Use 'rethrow' to rethrow a caught exception._

#### Description

The analyzer produces this diagnostic when a caught exception is thrown
using a `throw` expression rather than a `rethrow` statement.

#### Example

The following code produces this diagnostic because the caught exception
`e` is thrown using a `throw` expression:

```dart
void f() {
  try {
    // ...
  } catch (e) {
    [!throw e!];
  }
}
```

#### Common fixes

Use `rethrow` instead of `throw`:

```dart
void f() {
  try {
    // ...
  } catch (e) {
    rethrow;
  }
}
```

### use_string_in_part_of_directives

_The part-of directive uses a library name._

#### Description

The analyzer produces this diagnostic when a `part of` directive uses a
library name to refer to the library that the part is a part of.

#### Example

Given a file named `lib.dart` that contains the following:

```dart
library lib;

part 'test.dart';
```

The following code produces this diagnostic because the `part of`
directive uses the name of the library rather than the URI of the library
it's part of:

```dart
[!part of lib;!]
```

#### Common fixes

Use a URI to reference the library:

```dart
part of 'lib.dart';
```

### use_super_parameters

_Use super-initializer parameters where possible._

#### Description

The analyzer produces this diagnostic when a parameter to a constructor is
passed to a super constructor without being referenced or modified and a
`super` parameter isn't used.

#### Example

The following code produces this diagnostic because the parameters of the
constructor for `B` are only used as arguments to the super constructor:

```dart
class A {
  A({int? x, int? y});
}
class B extends A {
  [!B!]({int? x, int? y}) : super(x: x, y: y);
}
```

#### Common fixes

Use a `super` parameter to pass the arguments:

```dart
class A {
  A({int? x, int? y});
}
class B extends A {
  B({super.x, super.y});
}
```

### valid_regexps

_Invalid regular expression syntax._

#### Description

The analyzer produces this diagnostic when the string passed to the
default constructor of the class `RegExp` doesn't contain a valid regular
expression.

A regular expression created with invalid syntax will throw a
`FormatException` at runtime.

#### Example

The following code produces this diagnostic because the regular expression
isn't valid:

```dart
var r = RegExp([!r'('!]);
```

#### Common fixes

Fix the regular expression:

```dart
var r = RegExp(r'\(');
```

### void_checks

_Assignment to a variable of type 'void'._

#### Description

The analyzer produces this diagnostic when a value is assigned to a
variable of type `void`.

It isn't possible to access the value of such a variable, so the
assignment has no value.

#### Example

The following code produces this diagnostic because the field `value` has
the type `void`, but a value is being assigned to it:

```dart
class A<T> {
  T? value;
}

void f(A<void> a) {
  [!a.value = 1!];
}
```

The following code produces this diagnostic because the type of the
parameter `p` in the method `m` is `void`, but a value is being assigned
to it in the invocation:

```dart
class A<T> {
  void m(T p) { }
}

void f(A<void> a) {
  a.m([!1!]);
}
```

#### Common fixes

If the type of the variable is incorrect, then change the type of the
variable:

```dart
class A<T> {
  T? value;
}

void f(A<int> a) {
  a.value = 1;
}
```

If the type of the variable is correct, then remove the assignment:

```dart
class A<T> {
  T? value;
}

void f(A<void> a) {}
```
